#ifndef _PACE_COMMON_H_
#define _PACE_COMMON_H_

.syntax unified
.section .text.2	//main file will override this, other files will not, this makes sure main file starts first (required)

#define ASSUME_SR_ONLY_FLAGS			1				//set to 1 to assume that SR only ever contains XNZVC (allows some optimization)
#define EXIT_EMULATION_ADDR				0xFFFFFFF0		//JMP and RTS to this should instead exit the emulator

/*
	struct state {	//we are locked into this one by other parts of the OS, and our own code, as noted below
		uint32_t instr;	//0x00
		uint32_t d[8];	//0x04	//this actual ordering of D then A is used by "movem" code and EA code
		uint32_t a[8];	//0x24
		uint32_t pc;	//0x44
		uint32_t sr;	//0x48
	}
*/

//our reg use:
#define rD			r4
#define rA			r5
#define rSR			r6
#define rI			r7
#define rPC			r8

#define rNext		r11									//we store a pointer to "dispatch" here since "bx" is faster than "bl" and has no range limits
#define rT			r12

//SR flags
#define SR_C		0x01
#define SR_V		0x02
#define SR_Z		0x04
#define SR_N		0x08
#define SR_X		0x10

#define SR_ALL		(SR_C + SR_V + SR_Z + SR_N + SR_X)

//ea_t is 7 bits long, with the top 6 being the ea, and the bottom 1 being DNK)
.globl readEaB			//u8 (ea_t ea)
.globl readEaW			//u32 (ea_t ea)	//yes, return is 32 because An regs do not really ever read as 16 bits
.globl readEaL			//u32 (ea_t ea)
.globl writeEaB			//void (u8 val, ea_t ea)
.globl writeEaW			//void (u8 val, ea_t ea), will NOT return if write is to An
.globl writeEaL			//void (u8 val, ea_t ea), will NOT return if write is to An
.globl calcEaB			//void* (ea_t ea)
.globl calcEaW			//void* (ea_t ea)
.globl calcEaL			//void* (ea_t ea)
.globl calcEaGeneric	//void* (ea_t ea)

.globl instrsIllegal



////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////   MACROS   ////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////



.macro ubfx	Rd, Rs, lsb, numBits						//bitfield extract. negative values allowed
	lsls				\Rd, \Rs, #32 - (\lsb) - (\numBits)
	lsrs				\Rd, \Rd, #32 - (\numBits)
.endm

.macro ubfxs Rd, Rs, lsb, numBits						//bitfield extract. negative values allowed, set "Z" on result
	lsls				\Rd, \Rs, #32 - (\lsb) - (\numBits)
	lsrs				\Rd, \Rd, #32 - (\numBits)
.endm

.macro sbfx	Rd, Rs, lsb, numBits						//bitfield extract. negative values allowed
	lsls				\Rd, \Rs, #32 - (\lsb) - (\numBits)
	asrs				\Rd, \Rd, #32 - (\numBits)
.endm

.macro sbfxs Rd, Rs, lsb, numBits						//bitfield extract. negative values allowed, set "Z" on result
	lsls				\Rd, \Rs, #32 - (\lsb) - (\numBits)
	asrs				\Rd, \Rd, #32 - (\numBits)
.endm

.macro storeState Rstate, Rb							//needs two regs to clobber, Rstate will end up with a pointer to state
	subs				\Rstate, rD, #4					//get state pointer
	mov					\Rb, rPC						//get PC
	str					rI, [\Rstate, #0x00]			//store instr
	str					\Rb, [\Rstate, #0x44]			//store PC
	str					rSR, [\Rstate, #0x48]			//store sr
.endm

.macro loadState Rstate, Rb								//needs two regs to clobber, Rstate contains state on entry
	ldr					\Rb, [\Rstate, #0x44]			//get PC
	mov					rPC, \Rb
	ldr					rSR, [\Rstate, #0x48]			//get SR
	.if ASSUME_SR_ONLY_FLAGS
		ubfx			rSR, rSR, 0, 5					//make sure no rogue flags slip in
	.endif
	adds				rD, \Rstate, #4
	mov					rA, rD
	adds				rA, #0x20
.endm

.macro clearNZVC
	lsrs				rSR, #4
	lsls				rSR, #4
.endm

.macro clearXNZVC

	.if ASSUME_SR_ONLY_FLAGS
	
		movs			rSR, #0
	
	.else
	
		lsrs			rSR, #5
		lsls			rSR, #5
		
	.endif

.endm

.macro getX Rd											//Rd = SR.X
	.if ASSUME_SR_ONLY_FLAGS
		lsrs			\Rd, rSR, #4
	.else
		ubfx			\Rd, rSR, 4, 1
	.endif
.endm

.macro setCcNeg Ra, Rb, nBits, tabnm					//rB is clobber reg, rA is *INPUT*

.if	\nBits != 32
	lsls				\Ra, \Ra, #32 - \nBits
.endif

	.if ASSUME_SR_ONLY_FLAGS
	
		negs			\Ra, \Ra
		mrs				\Rb, APSR
		lsrs			\Rb, #28
		adr				\Ra, \tabnm
		ldrb			rSR, [\Ra, \Rb]

	.else
		clearXNZVC
		
		negs			\Ra, \Ra
		mrs				\Rb, APSR
		lsrs			\Rb, #28
		adr				\Ra, \tabnm
		ldrb			\Ra, [\Ra, \Rb]
		orrs			rSR, \Ra
	.endif
.endm

.macro setCcArith Ra, Rb, mathOp, nBits, setX, tabnm	//clobbers all regs

.if	\nBits != 32
	lsls				\Ra, \Ra, #32 - \nBits
	lsls				\Rb, \Rb, #32 - \nBits
.endif

	.if ASSUME_SR_ONLY_FLAGS && \setX
	
		\mathOp			\Ra, \Rb
		mrs				\Rb, APSR
		lsrs			\Rb, #28
		adr				\Ra, \tabnm
		ldrb			rSR, [\Ra, \Rb]

	.else
	
		.if \setX
			clearXNZVC
		.else
			clearNZVC
		.endif
		
		\mathOp			\Ra, \Rb
		mrs				\Rb, APSR
		lsrs			\Rb, #28
		adr				\Ra, \tabnm
		ldrb			\Ra, [\Ra, \Rb]
		orrs			rSR, \Ra
	.endif
.endm

.macro setCcLogicalNC Rval, nBits, andGoNextInstr		//destroys rVal, assumes sSR bits we need are clear
	lsls				\Rval, #32 - \nBits
	bmi					91f
	bne					92f
	adds				rSR, SR_Z
92:
	.if \andGoNextInstr
		nextInstr
	.else
		b				93f
	.endif
91:
	adds				rSR, SR_N
	.if \andGoNextInstr
		nextInstr
	.else
		93:
	.endif
.endm

.macro setCcLogical Rval, nBits, andGoNextInstr			//destroys rVal
	clearNZVC
	setCcLogicalNC		\Rval, \nBits, \andGoNextInstr
.endm

.macro write32 Rval, Radr, Rclobber						//val can be same as clobber
	rev					\Rclobber, \Rval
	strh				\Rclobber, [\Radr, #0]
	lsrs				\Rclobber, #16
	strh				\Rclobber, [\Radr, #2]
.endm

.macro read32 Rdst, Raddr, Rclobber						//dst can be same as addr, clober must differ from both
	ldrh				\Rclobber, [\Raddr, #0]
	ldrh				\Rdst, [\Raddr, #2]
	lsls				\Rdst, #16
	adds				\Rdst, \Rclobber
	rev					\Rdst, \Rdst
.endm

//generelly these next6 arent used and local code is, exceptions are macros, that should explaim why some of these next 4 take clobber regs and do not use them
.macro write16 Rval, Radr, Rclobber						//val can be same as clobber
	rev16				\Rclobber, \Rval
	strh				\Rclobber, [\Radr]
.endm

.macro read16 Rdst, Raddr, Rclobber						//clobber unused
	ldrh				\Rdst, [\Raddr]
	rev16				\Rdst, \Rdst
.endm

.macro read16s Rdst, Raddr, Rclobber					//clobber unused
	ldrh				\Rdst, [\Raddr]
	revsh				\Rdst, \Rdst
.endm

.macro write8 Rval, Radr, Rclobber						//clobber unused
	strb				\Rval, [\Radr]
.endm

.macro read8 Rdst, Raddr, Rclobber						//clobber unused
	ldrb				\Rdst, [\Raddr]
.endm

.macro read8sum Rdst, Raddr1, Raddr2
	ldrb				\Rdst, [\Raddr1, \Raddr2]
.endm

.macro read16sum Rdst, Raddr1, Raddr2
	ldrh				\Rdst, [\Raddr1, \Raddr2]
	rev16				\Rdst, \Rdst
.endm

.macro write8sum Rval, Raddr1, Raddr2
	strb				\Rval, [\Raddr1, \Raddr2]
.endm

.macro write16sum Rval, Raddr1, Raddr2, Rclobber		//clobber may match Rval
	rev16				\Rclobber, \Rval
	strh				\Rclobber, [\Raddr1, \Raddr2]
.endm

.macro getInstrByte Rdst								//slower than getInstrBytex, but does not need a temp reg
	movs				\Rdst, #2
	add					rPC, \Rdst
	negs				\Rdst, \Rdst
	add					\Rdst, rPC
	ldrb				\Rdst, [\Rdst, #1]
.endm

.macro getInstrBytex Rdst, Rclobber
	mov					\Rdst, rPC
	adds				\Rclobber, \Rdst, #2
	mov					rPC, \Rclobber
	ldrb				\Rdst, [\Rdst, #1]
.endm

.macro getInstrByteS Rdst								//slower than getInstrByteSx, but does not need a temp reg
	movs				\Rdst, #2
	add					rPC, \Rdst
	negs				\Rdst, \Rdst
	add					\Rdst, rPC
	ldrb				\Rdst, [\Rdst, #1]
	sxtb				\Rdst, \Rdst					//ldrsb lacks the addressing mode we'd like in v6M, so we're stuck using sxtb
.endm

.macro getInstrByteSx Rdst, Rclobber
	mov					\Rdst, rPC
	adds				\Rclobber, \Rdst, #2
	mov					rPC, \Rclobber
	movs				\Rclobber, #1
	ldrsb				\Rdst, [\Rdst, \Rclobber]
.endm

.macro getInstrWordLE Rdst								//slower than getInstrWordLEx, but does not need a temp reg
	movs				\Rdst, #2
	add					rPC, \Rdst
	negs				\Rdst, \Rdst
	add					\Rdst, rPC
	ldrh				\Rdst, [\Rdst]
.endm

.macro getInstrWordLEx Rdst, Rclobber
	mov					\Rdst, rPC
	adds				\Rclobber, \Rdst, #2
	mov					rPC, \Rclobber
	ldrh				\Rdst, [\Rdst]
.endm

.macro getInstrWordU Rdst								//slower than getInstrWordUx, but does not need a temp reg
	getInstrWordLE		\Rdst
	rev16				\Rdst, \Rdst
.endm

.macro getInstrWordUx Rdst, Rclobber
	getInstrWordLEx		\Rdst, \Rclobber
	rev16				\Rdst, \Rdst
.endm

.macro getInstrWordS Rdst								//slower than getInstrWordSx, but does not need a temp reg
	getInstrWordLE		\Rdst
	revsh				\Rdst, \Rdst
.endm

.macro getInstrWordSx Rdst, Rclobber
	getInstrWordLEx		\Rdst, \Rclobber
	revsh				\Rdst, \Rdst
.endm

.macro getInstrLong	Rdst, Rclobber
	mov					\Rdst, rPC
	adds				\Rclobber, \Rdst, #4
	mov					rPC, \Rclobber
	read32				\Rdst, \Rdst, \Rclobber
.endm


.macro nextInstr
	bx					rNext
.endm

.macro nextInstr1										//same as above, but MUST be precisely one thumb instruction long (thi smatters in case that
														//"nextInstr" ever gets bigger/cleverer)
	bx					rNext
.endm

.macro writeCCR	reg, Rclobber
	movs				\Rclobber, SR_ALL
	bics				rSR, \Rclobber
	ands				\reg, \Rclobber
	orrs				rSR, \reg
.endm

#endif
