/*
	(c) 2021 Dmitry Grinberg   https://dmitry.gr
	Non-commercial use only OR licensing@dmitry.gr
*/

#include <stdbool.h>
#include <string.h>
#include <stdint.h>
#include <stdio.h>
#include <boot.h>
#include "printf.h"
#include "sdHw.h"
#include "sd.h"


#pragma GCC optimize ("Os")

#define VERBOSE				0


struct SD {
	uint32_t numSec;
	uint32_t snum;
	uint32_t numSecLeft;		//for long operation
	uint16_t rca;
	uint16_t oid;
	uint8_t mid;
	union SdFlags flags;
	
	uint8_t lastErrorType;
	uint32_t lastErrorLoc, errData;
	uint32_t errCurBlock;
	
	struct SdHwData hwData;
};



#define ERR_NONE					0
#define ERR_WRITE_REPLY_TIMEOUT		1
#define ERR_READ_CMD17_REPLY		2
#define ERR_READ_CMD18_ERROR		3
#define ERR_READ_CMD23_ERROR		4
#define ERR_READ_DATA_TIMEOUT		5
#define ERR_READ_CRC_ERR			6
#define ERR_READ_FRAMING_ERR		7
#define ERR_READ_MISC_ERR			8
#define ERR_WRITE_ACMD23_ERROR		9
#define ERR_WRITE_CMD23_ERROR		10
#define ERR_WRITE_CMD24_ERROR		11
#define ERR_WRITE_CMD25_ERROR		12
#define ERR_WRITE_DATA_REPLY_ERROR	13
#define ERR_WRITE_BUSY_WAIT_TIMEOUT	14

void sdReportLastError(struct SD *sd)
{
	static const char mErrorNames[][32] = {
		[ERR_NONE] = "none",
		[ERR_WRITE_REPLY_TIMEOUT] = "write reply timeout",
		
		[ERR_READ_CMD17_REPLY] = "CMD17 error on read",
		[ERR_READ_CMD18_ERROR] = "CMD18 error on read",
		[ERR_READ_CMD23_ERROR] = "CMD23 error on read",
		[ERR_READ_DATA_TIMEOUT] = "read data timeout",
		[ERR_READ_CRC_ERR] = "read CRC error",
		[ERR_READ_FRAMING_ERR] = "read framing error",
		[ERR_READ_MISC_ERR] = "read misc error",
		
		[ERR_WRITE_ACMD23_ERROR] = "ACMD23 error on write",
		[ERR_WRITE_CMD23_ERROR] = "CMD23 error on write",
		[ERR_WRITE_CMD24_ERROR] = "CMD24 error on write",
		[ERR_WRITE_CMD25_ERROR] = "CMD25 error on write",
		[ERR_WRITE_DATA_REPLY_ERROR] = "reply token for write was bad",
		[ERR_WRITE_BUSY_WAIT_TIMEOUT] = "write busy wait took too long",
	};
	const char *errName = (sd->lastErrorType <= sizeof(mErrorNames) / sizeof(*mErrorNames)) ? mErrorNames[sd->lastErrorType] : NULL;
	if (!errName)
		errName = "UNKNOWN";
	
	loge("last error type %u(%s) at sector %u. Extra data: 0x%08x\n", sd->lastErrorType, errName, sd->lastErrorLoc, sd->errData);
}

static void sdPrvErrHappened(struct SD *sd, uint_fast8_t errType, uint32_t secNum, uint32_t extraData)
{
	sd->lastErrorType = errType;
	sd->lastErrorLoc = secNum;
	sd->errData = extraData;
}

enum SdHwCmdResult sdPrvSimpleCommand(struct SD *sd, uint_fast8_t cmd, uint32_t param, bool cmdCrc, enum SdHwRespType respTyp, void *respBufOut, enum SdHwDataDir dataDir, uint_fast16_t blockSz, uint32_t numBlocks)
{
	enum SdHwCmdResult ret;
	
	if (VERBOSE)
		logt("cmd %u (0x%08x)\n", cmd, param);
	
	ret = sdHwCmd(&sd->hwData, cmd, param, cmdCrc, respTyp, respBufOut, dataDir, blockSz, numBlocks);
	
	if (VERBOSE)
		logt(" -> %02x %02x\n", ret, respBufOut ? *(uint8_t*)respBufOut : 0xff);
	
	return ret;
}

static enum SdHwCmdResult sdPrvACMD(struct SD *sd, uint_fast8_t cmd, uint32_t param, bool cmdCrc, enum SdHwRespType respTyp, void *respBufOut, enum SdHwDataDir dataDir, uint_fast16_t blockSz, uint32_t numBlocks)
{
	enum SdHwCmdResult ret;
	uint8_t rsp;
	
	ret = sdPrvSimpleCommand(sd, 55, ((uint32_t)sd->rca) << 16, cmdCrc, SdRespTypeR1, &rsp, SdHwDataNone, 0, 0);
	if (ret != SdHwCmdResultOK)
		return ret;
	
	if (rsp &~ FLAG_IN_IDLE_MODE)
		return SdCmdInternalError;
	
	return sdPrvSimpleCommand(sd, cmd, param, cmdCrc, respTyp, respBufOut, dataDir, blockSz, numBlocks);
}

static bool sdPrvCardInit(struct SD *sd, bool isSdCard, bool signalHighCapacityHost)
{
	uint32_t param, time = 0;
	bool first = true;
	uint8_t rsp[4];
		
	param = (signalHighCapacityHost ? 0x40000000 : 0x00000000) | 0x00f80000;
	
	while (time++ < 1024) {
		
		enum SdHwCmdResult result;
		
		//busy bit is top bit of OCR, which is top bit of resp[1]
		//busy bit at 1 means init complete
		//see pages 85 and 26
		
		result = (isSdCard ? sdPrvACMD : sdPrvSimpleCommand)(sd, isSdCard ? 41 : 1, param, true, sd->flags.sdioIface ? SdRespTypeR3 : SdRespTypeR1, &rsp, SdHwDataNone, 0, 0);
		
		if (result != SdHwCmdResultOK)
			return false;
				
		if (sd->flags.sdioIface) {	//bytes sent MSB to LSB
			
			if (!first && (rsp[0] & 0x80)) {
				
				if (isSdCard)
					sd->flags.HC = !!(rsp[0] & 0x40);
				else
					sd->flags.HC = ((rsp[0] >> 5) & 3) == 2;
				
				return true;
			}
		}
		else {
			
			if (rsp[0] & FLAG_MISC_ERR)
				break;
			
			if (!first && !(rsp[0] & FLAG_IN_IDLE_MODE))
				return true;
		}
		first = false;
	}
	return false;
}

static uint32_t sdPrvGetBits(const uint8_t* data, uint_fast16_t numBytesInArray, int_fast16_t startBit, uint_fast16_t len)	//for CID and CSD data..
{
	uint32_t ret = 0;
	int_fast16_t i;
	
	for (i = startBit + len - 1; i >= startBit; i--)
		ret = (ret << 1) + ((data[numBytesInArray - 1 - i / 8] >> (i % 8)) & 1);
	
	return ret;
}

static bool sdPrvCmd6(struct SD *sd, uint32_t param, uint8_t *buf)
{
	enum SdHwCmdResult result;
	enum SdHwReadResult rr;
	uint32_t time;
	uint8_t rsp;
	
	//do not ask
	time = TimGetTicks();
	while (TimGetTicks() - time < TICKS_PER_SECOND / 100);
	
	result = sdPrvSimpleCommand(sd, 6, param, false, SdRespTypeR1, &rsp, SdHwDataRead, 64, 1);
	if (result != SdHwCmdResultOK)
		return false;

	rr = sdHwReadData(&sd->hwData, buf, 64);
	sdHwChipDeselect(&sd->hwData);
	if (rr != SdHwReadOK)
		return false;
	
	if (VERBOSE) {
				
		logt("CMD6(0x%08x) reply:", param);
		logBin(buf, 64);
	}
	
	return true;
}

bool __attribute__((noinline)) sdCardInit(struct SD *sd, uint8_t respBuf [static 64])
{
	uint32_t maxSpeed, hwFlags, readToTicks, writeToTicks, t;
	enum SdHwCmdResult result;
	enum SdHwReadResult rr;
	uint_fast16_t toBytes;
	bool hsMode = false;
	uint_fast8_t i;
	
	if (sizeof(struct SD) > sizeof(uint32_t) * STRUCT_SD_SIZE)
		return false;

	MemSet(sd, sizeof(struct SD), 0);
	
	hwFlags = sdHwInit(&sd->hwData);
	if (VERBOSE)
		logt("SD HW init flags: 0x%08x\n", hwFlags);
	if (!(hwFlags & SD_HW_FLAG_INITED))
		return false;	
	
	if (sdHwGetMaxBlocksAtOnce(&sd->hwData) < 1 || sdHwGetMaxBlockSize(&sd->hwData) < SD_BLOCK_SIZE)
		return false;
	
	sd->flags.sdioIface = !!(hwFlags & SD_HW_FLAG_SDIO_IFACE);
	
	t = TimGetTicks();
	while (TimGetTicks() - t < TICKS_PER_SECOND / 500);	//give it 2 ms to init
	
	sdHwGiveInitClocks(&sd->hwData);
	
	t = TimGetTicks();
	while (TimGetTicks() - t < TICKS_PER_SECOND / 1000);

	if (sd->flags.sdioIface) {	//reset has no reply, we send it once
		
		for (i = 0; i < 16; i++) {	//try CMD0 a few times
			
			if (SdHwCmdResultOK != sdPrvSimpleCommand(sd, 0, 0, true, SdRespTypeNone, NULL, SdHwDataNone, 0, 0))
				return false;
		}
	}
	else {						//reset has a reply - we try a few times
	
		for (i = 0; i < 64; i++) {	//try CMD0 a few times, then try it with an extra pulse
			
			if (SdHwCmdResultOK != sdPrvSimpleCommand(sd, (i < 32) ? 0 : 0x40, 0, true, SdRespTypeR1, respBuf, SdHwDataNone, 0, 0))
				continue;
			
			if (respBuf[0] == FLAG_IN_IDLE_MODE)
				break;
		}
		
		if (i == 16)
			return false;
	}
	
	result = sdPrvSimpleCommand(sd, 8, 0x000001AA, true, SdRespTypeR7, respBuf, SdHwDataNone, 0, 0);
	if (result == SdCmdInvalid)
		sd->flags.v2 = false;
	else if (result != SdHwCmdResultOK)
		return false;
	else if (respBuf[3] != 0xaa)
		return false;
	else
		sd->flags.v2 = true;
	
	logi("SD Card V%u\n", 1 + sd->flags.v2);
	
	result = sdPrvSimpleCommand(sd, 55, 0, true, SdRespTypeR1, respBuf, SdHwDataNone, 0, 0);
	if (result == SdCmdInvalid) {
		maxSpeed = 16000000;
		sd->flags.SD = false;
	}
	else if (result != SdHwCmdResultOK || (respBuf[0] &~ FLAG_IN_IDLE_MODE))
		return false;
	else
		sd->flags.SD = true;

	//only try high cap init if card replies to cmd8 as per spec
	if ((!sd->flags.v2 || !sdPrvCardInit(sd, sd->flags.SD, true)) && !sdPrvCardInit(sd, sd->flags.SD, false))
		return false;
	
	if (!sd->flags.sdioIface) {	//check for HC (by reading OCR), turn off CRC
	
		result = sdPrvSimpleCommand(sd, 58, 0, true, SdRespTypeR3, respBuf, SdHwDataNone, 0, 0);
		if (result != SdHwCmdResultOK)
			return false;
		
		sd->flags.HC = !!(respBuf[0] & 0x40);
		
		//only turn CRC off if the spi HAL doesn't want to do CRC, else turn it on (from unknown initial state)
		result = sdPrvSimpleCommand(sd, 59, (hwFlags & SD_HW_FLAG_SPI_DOES_CRC) ? 1 : 0, true, SdRespTypeR1, respBuf, SdHwDataNone, 0, 0);
		if (result != SdHwCmdResultOK || respBuf[0])
			return false;
	}
	
	//read CID
	if (sd->flags.sdioIface) {
		
		result = sdPrvSimpleCommand(sd, 2, 0, false, SdRespTypeSdR2, respBuf, SdHwDataNone, 0, 0);
		if (result != SdHwCmdResultOK)
			return false;
	}
	else {
		
		result = sdPrvSimpleCommand(sd, 10, 0, false, SdRespTypeR1, respBuf, SdHwDataRead, 16, 1);
		if (result != SdHwCmdResultOK || respBuf[0]) {
			loge("no read\n");
			return false;
		}
		rr = sdHwReadData(&sd->hwData, respBuf, 16);
		sdHwChipDeselect(&sd->hwData);
		if (rr != SdHwReadOK) {
			loge("read err\n");
			return false;
		}
	}

	//parse CID
	sd->mid = sdPrvGetBits(respBuf, 16, 120, 8);
	sd->oid = sdPrvGetBits(respBuf, 16, 104, sd->flags.SD ? 16 : 8);
	sd->snum = sdPrvGetBits(respBuf, 16, sd->flags.SD ? 24 : 16, 32);

	if (VERBOSE)
		logt("CARD ID: %02x %04x %u\n", sd->mid, sd->oid, sd->snum);

	//assign RCA
	if (sd->flags.sdioIface) {
		
		sd->rca = 2;
		result = sdPrvSimpleCommand(sd, 3, ((uint32_t)sd->rca) << 16, false, SdRespTypeSdR6, respBuf, SdHwDataNone, 0, 0);
		if (result != SdHwCmdResultOK)
			return false;
		
		if (sd->flags.SD)
			sd->rca = (((uint32_t)respBuf[0]) << 8) + respBuf[1];
		
		if (VERBOSE)
			logt("RCA %04x\n", sd->rca);
		
		sdHwNotifyRCA(&sd->hwData, sd->rca);
	}

	//read CSD
	if (sd->flags.sdioIface) {
		
		result = sdPrvSimpleCommand(sd, 9, ((uint32_t)sd->rca) << 16, false, SdRespTypeSdR2, respBuf, SdHwDataNone, 0, 0);
		if (result != SdHwCmdResultOK)
			return false;
	}
	else {
		
		result = sdPrvSimpleCommand(sd, 9, 0, false, SdRespTypeR1, respBuf, SdHwDataRead, 16, 1);
		if (result != SdHwCmdResultOK || respBuf[0])
			return false;
		rr = sdHwReadData(&sd->hwData, respBuf, 16);
		sdHwChipDeselect(&sd->hwData);
		if (rr != SdHwReadOK)
			return false;
	}
	
	if (VERBOSE) {
		logt("CSD:");
		logBin(respBuf, 16);
	}
	
	if (VERBOSE)
		logt("CSD ver %u\n", sdPrvGetBits(respBuf, 16, 126, 2) + 1);

	//calc capacity & figure out timeouts
	if (sd->flags.SD && sdPrvGetBits(respBuf, 16, 126, 2)) {	//v2 SD csd
		
		sd->numSec = 1024 * (1 + sdPrvGetBits(respBuf, 16, 48, 22));
		
		if (VERBOSE)
			logt("c_size: 0x%08x\n", sdPrvGetBits(respBuf, 16, 48, 22));
		
		readToTicks = (TICKS_PER_SECOND + 9) / 10;	//use 100ms
		writeToTicks = (TICKS_PER_SECOND + 1) / 2;	//use 500ms (sometimes 25, but this is simpler)
		toBytes = 0;
	}
	else {
		
		static const uint8_t taacMant[] = {0, 10, 12, 13, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60, 70, 80, };
		uint_fast8_t r2wFactor = sdPrvGetBits(respBuf, 16, 26, 3);
		uint32_t cSizeMult = sdPrvGetBits(respBuf, 16, 47, 3);
		uint32_t readBlLen = sdPrvGetBits(respBuf, 16, 80, 4);
		uint_fast8_t taac = sdPrvGetBits(respBuf, 16, 112, 8);
		uint_fast8_t nsac = sdPrvGetBits(respBuf, 16, 104, 8);
		uint32_t cSize = sdPrvGetBits(respBuf, 16, 62, 12);
		uint32_t shiftBy = readBlLen + cSizeMult + 2;
		
		if (!sd->flags.SD && sdPrvGetBits(respBuf, 16, 126, 2) >= 2 && sdPrvGetBits(respBuf, 16, 122, 4) >= 4) {
			
			if (VERBOSE)
				loge("MMC+ over 4GB not supported due to lack of testing\n");
			
			return false;
		}
		
		if (shiftBy < 9) {	//fatal
			if (VERBOSE)
				loge("card shift size invalid\n");
			return false;
		}
		shiftBy -= 9;
		
		sd->numSec = (cSize + 1) << shiftBy;
		
		(void)taac;
		(void)taacMant;
		//taac is now the "typical" access time in units of 0.1ns
		//it is expressed as a mantissa from this array: {0, 10, 12, 13, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60, 70, 80, }
		// and an exponent. this is "typical" time
		// and real timeout is 100x that, or 100ms (whichever is smaller)
		// to save on math we use 100ms!!!
		readToTicks = (TICKS_PER_SECOND + 9) / 10;	//use 100ms
		
		if (r2wFactor > 5)
			r2wFactor = 5;
		
		//250ms max
		writeToTicks = readToTicks << r2wFactor;
		if (writeToTicks > (TICKS_PER_SECOND + 3) / 4)
			writeToTicks = (TICKS_PER_SECOND + 3) / 4;
		
		//bytes
		toBytes = 256 / 8 * nsac;
	}
	
	if (VERBOSE)
		loge("read timeout: %u ticks, write %u + %u bytes\n", readToTicks, writeToTicks, toBytes);
	
	sdHwSetTimeouts(&sd->hwData, toBytes, readToTicks, writeToTicks);

	sd->flags.RO = false;
	
	if (sdPrvGetBits(respBuf, 16, 12, 1)) {
		
		if (VERBOSE)
			logi("Card temporarily write protected\n");
		sd->flags.RO = true;
	}
	
	if (sdPrvGetBits(respBuf, 16, 13, 1)) {
		
		if (VERBOSE)
			logi("Card permanently write protected\n");
		sd->flags.RO = true;
	}
	
	if (VERBOSE)
		logi("CARD capacity: %u 512-byte blocks\n", sd->numSec);
	
	if (sd->flags.sdioIface) {
		
		//select the card
		result = sdPrvSimpleCommand(sd, 7, ((uint32_t)sd->rca) << 16, false, SdRespTypeR1withBusy, respBuf, SdHwDataNone, 0, 0);
		if (result != SdHwCmdResultOK || respBuf[0])
			return false;
	}
	
	//some of the following code does rather large xfers. let's speed up to at least 8MHz (Safe) before doing that
	sdHwSetSpeed(&sd->hwData, 8000000, false);
	
	if (sd->flags.SD) {		//some SD-specific things
		
		result = sdPrvACMD(sd, 51, 0, false, SdRespTypeR1, respBuf, SdHwDataRead, 8, 1);
		if (result != SdHwCmdResultOK || respBuf[0])
			return false;
		rr = sdHwReadData(&sd->hwData, respBuf, 8);
		sdHwChipDeselect(&sd->hwData);
		if (rr != SdHwReadOK)
			return false;
		
		if (VERBOSE) {
			logt("SCR:");
			logBin(respBuf, 8);
		}
		
		if (!(respBuf[1] & 1)) {
			if (VERBOSE)
				loge("card claims to not support 1-bit mode...clearly SCR is wrong\n");
			return false;
		}

		if ((hwFlags & SD_HW_FLAG_SUPPORT_4BIT) && (respBuf[1] & 4)) {
			
			uint8_t resp;
			
			if (VERBOSE)
				logt("card support 4-wide bus. we'll do that\n");
			
			if (!sdHwSetBusWidth(&sd->hwData, true))
				return false;
			
			//disable pullup
			result = sdPrvACMD(sd, 42, 0, false, SdRespTypeR1, &resp, SdHwDataNone, 0, 0);
			if (result != SdHwCmdResultOK || resp)
				return false;
			
			//go to 4-bit-wide
			result = sdPrvACMD(sd, 6, 2, false, SdRespTypeR1, &resp, SdHwDataNone, 0, 0);
			if (result != SdHwCmdResultOK || resp)
				return false;
			
			if (VERBOSE)
				logt("now in 4-bit mode\n");
		}
		
		sd->flags.cmd23supported = !!(respBuf[3] & 2);
		if (VERBOSE)
			logt("CMD23 support: %d\n", sd->flags.cmd23supported);
		
		if (respBuf[0] & 0x0f) {	//at least spec 1.1 needed to use CMD6
			
			//SD v1.1 supports 25MHz
			maxSpeed = 25000000;

			if (!sdPrvCmd6(sd, 0x00000001, respBuf))
				return false;
			
			if ((hwFlags & SD_HW_FLAG_HS_SIGNALLING) && (respBuf[16] & 0x0f) == 1) {
				
				if (VERBOSE)
					logi("SD high speed mode supported\n sending cmd6 (to set HS mode)\n");
				
				if (!sdPrvCmd6(sd, 0x80000001, respBuf))
					return false;
				if ((respBuf[16] & 0x0f) == 1) {
					if (VERBOSE)
						logt("switch to high speed mode was a success\n");
					maxSpeed = 50000000;
					hsMode = true;
				}
			}
			else {				//high speed mode not supported - falling back to 12.5MHz
				if (VERBOSE)
					logt("high speed mode not supported\n");
			}
		}
		else {				//high speed mode not supported - falling back to 12.5MHz
			if (VERBOSE)
				logt("SD v1.0: high speed mode not known yet\n");
			maxSpeed = 12500000;
		}
		
		result = sdPrvACMD(sd, 13, 0, false, sd->flags.sdioIface ? SdRespTypeR1 : SdRespTypeSpiR2, respBuf, SdHwDataRead, 64, 1);
		if (result != SdHwCmdResultOK || respBuf[0])
			return false;

		rr = sdHwReadData(&sd->hwData, respBuf, 64);
		sdHwChipDeselect(&sd->hwData);
		if (rr != SdHwReadOK)
			return false;
		
		if (VERBOSE) {
			logt("SD_STATUS:");
			logBin(respBuf, 64);
		}
		
		sd->flags.hasDiscard = !!(respBuf[24] & 2);
	}
	else {	//mmc-specific init
		
		result = sdPrvSimpleCommand(sd, 6, 0x03b90100, false, SdRespTypeR1withBusy, respBuf, SdHwDataNone, 0, 0);
		if (result != SdHwCmdResultOK || respBuf[0])	//too old to go high speed
			maxSpeed = 20000000;
		else {											//at least mmc v4 - can go fast

			maxSpeed = 26000000;						//we could read EXT_CSD and see if 52MHz is an option, but that requires 512 bytes and isnt worth it.
			
			if (hwFlags & SD_HW_FLAG_SUPPORT_4BIT) {
				
				result = sdPrvSimpleCommand(sd, 6, 0x03b70100, false, SdRespTypeR1withBusy, respBuf, SdHwDataNone, 0, 0);
				if (result == SdHwCmdResultOK && !respBuf[0]) {		//success
					
					if (!sdHwSetBusWidth(&sd->hwData, true))
						return false;
				}
			}
		}
	}
	
	logi("Going to clock speed %u\n", maxSpeed);
	
	//speed the clock up
	sdHwSetSpeed(&sd->hwData, maxSpeed, hsMode);
	
	//set block size
	result = sdPrvSimpleCommand(sd, 16, SD_BLOCK_SIZE, false, SdRespTypeR1, respBuf, SdHwDataNone, 0, 0);
	if (result != SdHwCmdResultOK || respBuf[0])
		return false;
	
	sd->flags.inited = true;

	logi("SD init success\n");
	return true;
}

uint32_t sdGetNumSecs(struct SD *sd)
{
	return sd->numSec;
}

static bool sdPrvReadResultProcess(struct SD *sd, enum SdHwReadResult rr, uint32_t sec)
{
	uint_fast8_t errType;
	
	if (rr == SdHwReadOK)
		return true;
	
	switch (rr) {
		case SdHwReadOK:
			return true;
		
		case SdHwReadTimeout:
			errType = ERR_READ_DATA_TIMEOUT;
			break;
		
		case SdHwReadCrcErr:
			errType = ERR_READ_CRC_ERR;
			break;
		
		case SdHwReadFramingError:
		case SdHwReadInternalError:
		default:
			errType = ERR_READ_MISC_ERR;
			break;
	}
	
	sdPrvErrHappened(sd, errType, sec, 0);
	return false;
}

bool sdSecRead(struct SD *sd, uint32_t sec, uint8_t *dst)
{
	enum SdHwCmdResult result;
	enum SdHwReadResult rr;
	uint8_t rsp;
	
	result = sdPrvSimpleCommand(sd, 17, sd->flags.HC ? sec : sec << 9, false, SdRespTypeR1, &rsp, SdHwDataRead, SD_BLOCK_SIZE, 1);
	if (result != SdHwCmdResultOK || rsp) {
		
		sdPrvErrHappened(sd, ERR_READ_CMD17_REPLY, sec, rsp);
		return false;
	}

	rr = sdHwReadData(&sd->hwData, dst, SD_BLOCK_SIZE);
	sdHwChipDeselect(&sd->hwData);
	
	return sdPrvReadResultProcess(sd, rr, sec);
}

bool sdReadStart(struct SD *sd, uint32_t sec, uint32_t numSec)
{
	enum SdHwCmdResult result;
	uint8_t rsp = 0;

	if (!numSec)
		numSec = 0xffffffff;
	else if (sd->flags.cmd23supported || !sd->flags.SD) {	//all MMC cards support CMD23, only some SD cards do
	
		uint32_t maxCounter = sd->flags.SD ? ((1ul << 23) - 1) : ((1ul << 16) - 1);
	
		result = sdPrvSimpleCommand(sd, 23, numSec > maxCounter ? maxCounter : numSec, false, SdRespTypeR1, &rsp, SdHwDataNone, 0, 0);
		if (result != SdHwCmdResultOK || rsp) {
			
			sdPrvErrHappened(sd, ERR_READ_CMD23_ERROR, sec, rsp);
			return false;
		}
	}
	
	result = sdPrvSimpleCommand(sd, 18, sd->flags.HC ? sec : sec << 9, false, SdRespTypeR1, &rsp, SdHwDataRead, SD_BLOCK_SIZE, numSec);
	if (result != SdHwCmdResultOK || rsp) {
		
		sdPrvErrHappened(sd, ERR_READ_CMD18_ERROR, sec, rsp);
		return false;
	}
	
	sd->errCurBlock = sec;
	return true;
}

bool sdReadNext(struct SD *sd, uint8_t *dst)
{
	enum SdHwReadResult rr;
	
	rr = sdHwReadData(&sd->hwData, dst, SD_BLOCK_SIZE);
	sd->errCurBlock++;
	
	return sdPrvReadResultProcess(sd, rr, sd->errCurBlock - 1);
}

bool sdReadStop(struct SD *sd)
{
	bool readEndOk;
	uint8_t rsp;
	
	readEndOk = sdHwMultiBlockReadSignalEnd(&sd->hwData);
	sdHwChipDeselect(&sd->hwData);
	
	(void)sdPrvSimpleCommand(sd, 12, 0, false, SdRespTypeR1withBusy, &rsp, SdHwDataNone, 0, 0);
	
	if (!sd->flags.sdioIface) {
		//some card keep sending data a few more cycles on SPI...ignore it
		sdHwRxRawBytes(&sd->hwData, NULL, 8);
		sdHwChipDeselect(&sd->hwData);
	}
	
	return readEndOk;
}

bool sdWriteStart(struct SD *sd, uint32_t sec, uint32_t numSec)
{
	enum SdHwCmdResult result;
	uint8_t rsp;
	
	if (!numSec)
		numSec = 0xffffffff;
	else {
		//SD: ACMD23 is required, 23-bit counter
		//MMC: CMD23 is required, 16 bit counter
		
		if (sd->flags.SD)
			result = sdPrvACMD(sd, 23, (numSec >= (1ul << 23)) ? ((1ul << 23) - 1) : numSec, false, SdRespTypeR1, &rsp, SdHwDataNone, 0, 0);
		else
			result = sdPrvSimpleCommand(sd, 23, (numSec >= (1ul << 16)) ? ((1ul << 16) - 1) : numSec, false, SdRespTypeR1, &rsp, SdHwDataNone, 0, 0);
		if (result != SdHwCmdResultOK || rsp) {
			
			sdPrvErrHappened(sd, sd->flags.SD ? ERR_WRITE_ACMD23_ERROR : ERR_WRITE_CMD23_ERROR, sec, rsp);
			return false;
		}
	}
	
	result = sdPrvSimpleCommand(sd, 25, sd->flags.HC ? sec : sec << 9, false, SdRespTypeR1, &rsp, SdHwDataWrite, SD_BLOCK_SIZE, numSec);
	if (result != SdHwCmdResultOK || rsp) {
		
		sdPrvErrHappened(sd, ERR_WRITE_CMD25_ERROR, sec, rsp);
		return false;
	}
	
	sd->errCurBlock = sec;
	return true;
}

bool sdWriteNext(struct SD *sd, const uint8_t *src)
{
	enum SdHwWriteReply wr;
	bool busyDone;
	
	wr = sdHwWriteData(&sd->hwData, src, SD_BLOCK_SIZE, true);
	busyDone = sdHwPrgBusyWait(&sd->hwData);
	
	if (wr == SdHwTimeout) {
		
		sdPrvErrHappened(sd, ERR_WRITE_REPLY_TIMEOUT, sd->errCurBlock, wr);
		return false;
	}
	
	if (wr != SdHwWriteAccepted) {
		
		sdPrvErrHappened(sd, ERR_WRITE_DATA_REPLY_ERROR, sd->errCurBlock, wr);
		return false;
	}
	
	if (!busyDone) {
		
		sdPrvErrHappened(sd, ERR_WRITE_BUSY_WAIT_TIMEOUT, sd->errCurBlock, wr);
		return false;
	}
	
	sd->errCurBlock++;
	return true;
}

bool sdWriteStop(struct SD* sd)
{
	enum SdHwCmdResult result;
	bool writeEndOk, busyDone;
	uint8_t rsp;

	writeEndOk = sdHwMultiBlockWriteSignalEnd(&sd->hwData);
	busyDone = sdHwPrgBusyWait(&sd->hwData);
	
	sdHwChipDeselect(&sd->hwData);

	if (sd->flags.sdioIface) {
		result = sdPrvSimpleCommand(sd, 12, 0, false, SdRespTypeR1withBusy, &rsp, SdHwDataNone, 0, 0);
		if (result != SdHwCmdResultOK || rsp)
			return false;
	}
	
	return writeEndOk && busyDone;
}

bool sdSecWrite(struct SD *sd, uint32_t sec, const uint8_t *src)
{
	enum SdHwCmdResult result;
	enum SdHwWriteReply wr;
	uint8_t rsp = 0;
	bool busyDone;
	
	result = sdPrvSimpleCommand(sd, 24, sd->flags.HC ? sec : sec << 9, false,  SdRespTypeR1, &rsp, SdHwDataWrite, SD_BLOCK_SIZE, 1);
	if (result != SdHwCmdResultOK || rsp) {
		
		sdPrvErrHappened(sd, ERR_WRITE_CMD24_ERROR, sec, rsp);
		return false;
	}
	
	wr = sdHwWriteData(&sd->hwData, src, SD_BLOCK_SIZE, false);
	busyDone = sdHwPrgBusyWait(&sd->hwData);
	
	sdHwChipDeselect(&sd->hwData);
	
	if (wr == SdHwTimeout) {
		
		sdPrvErrHappened(sd, ERR_WRITE_REPLY_TIMEOUT, sd->errCurBlock, wr);
		return false;
	}
	
	if (wr != SdHwWriteAccepted) {
		
		sdPrvErrHappened(sd, ERR_WRITE_DATA_REPLY_ERROR, sd->errCurBlock, wr);
		return false;
	}
	
	if (!busyDone) {
		
		sdPrvErrHappened(sd, ERR_WRITE_BUSY_WAIT_TIMEOUT, sd->errCurBlock, wr);
		return false;
	}
	
	return true;
}

bool sdTrim(struct SD *sd, uint32_t firstBlock, uint32_t numBlocks)
{
	//discard support is signalled via b313 in SD_STATUS (ACMD13)
	//seq is: CMD32, CMD33, CMD38
	
	uint32_t addrStart, addrEnd;
	enum SdHwCmdResult ret;
	uint8_t rsp;
	
	if (sd->flags.RO)
		return false;
	
	if (!sd->flags.hasDiscard)
		return true;
	
	if (!numBlocks)
		return true;
		
	addrStart = sd->flags.HC ? firstBlock : (firstBlock * SD_BLOCK_SIZE);
	addrEnd = sd->flags.HC ? (firstBlock + numBlocks - 1) : ((firstBlock + numBlocks - 1) * SD_BLOCK_SIZE);
	
	ret = sdPrvSimpleCommand(sd, 32, addrStart, false, SdRespTypeR1, &rsp, SdHwDataNone, 0, 0);
	if (ret != SdHwCmdResultOK || rsp)
		return false;

	ret = sdPrvSimpleCommand(sd, 33, addrEnd, false, SdRespTypeR1, &rsp, SdHwDataNone, 0, 0);
	if (ret != SdHwCmdResultOK || rsp)
		return false;
	
	ret = sdPrvSimpleCommand(sd, 38, 1, false, SdRespTypeR1withBusy, &rsp, SdHwDataNone, 0, 0);
	if (ret != SdHwCmdResultOK || rsp)
		return false;
	
	return true;
}

void sdGetInfo(struct SD *sd, uint8_t *midP, uint16_t *oidP, uint32_t *snumP)
{
	if (midP)
		*midP = sd->mid;
	
	if (oidP)
		*oidP = sd->oid;
	
	if (snumP)
		*snumP = sd->snum;
}

uint8_t sdGetFlags(struct SD *sd)
{
	return sd->flags.value;
}

uint32_t sdPrvGetMaxSectorsAtOnce(struct SD *sd)
{
	return sdHwGetMaxBlocksAtOnce(&sd->hwData);
}

bool sdIsCardLockSwitchOn(struct SD *sd)
{
	return sdHwIsCardLockSwitchOn(&sd->hwData);
}

bool sdIsCardInserted(struct SD *sd)
{
	return sdHwIsCardInserted(&sd->hwData);
}

void sdCardPower(struct SD *sd, bool on)
{
	return sdHwCardPower(&sd->hwData, on);
}

void sdShutdown(struct SD *sd)
{
	sdHwShutdown(&sd->hwData);
}

void sdSleep(struct SD *sd)
{
	sdHwSleep(&sd->hwData);
}

void sdWake(struct SD *sd)
{
	sdHwWake(&sd->hwData);
}

