#include <kernel/drivers/platforms/xscale/xscale.h>
#include <kernel/drivers/platforms/platform.h>
#include <kernel/drivers/aximX3/aximPwrCtl.h>
#include "common.h"
#include "printf.h"
#include "sdUtil.h"
#include <boot.h>
#include "sdHw.h"
#include <dal.h>
#include <kal.h>



//XXX: support for write protect switch
//XXX: tapewave fs lib shouldn't suck


#define SD_BASE			(0x89000000)

enum W86IndirectReg {
	W86IndRegExtStaAndSettings = 0b00000,
	W86IndRegSdioCtl = 0b00010,
	W86IndRegMasterDataFmt = 0b00100,
	W86IndRegMasterBlockCnt = 0b00110,
	W86IndRegSlaveDataFmt = 0b01000,
	W86IndRegSlaveBlockCnt = 0b01010,
	W86IndRegNakTO = 0b01100,
	W86IndRegErrSta = 0b01110,
	W86IndRegHostIfaceAndBufferSvcLen = 0b10000,
	W86IndRegTest = 0b10010,
	W86IndRegID = 0b10100,
};

struct W86 {
	volatile uint16_t cmdRspFifo;		//0000
	volatile uint16_t ctrlSta;			//0001
	volatile uint16_t dataFifo;			//0010
	volatile uint16_t intCtl;			//0011
	volatile uint16_t gpioCtl;			//0100
	volatile uint16_t gpioIntCtl;		//0101
	volatile uint16_t indirectAddr;		//0110
	volatile uint16_t indirectData;		//0111

	//W86L488AY only
	volatile uint16_t globalSta;		//1000
	volatile uint16_t globalCtl;		//1001
	volatile uint16_t readyAndDataSz;	//1010
};

#define W86L488Y_ID_CODE		0x488d	//rev in t3
#define W86L488Y_ID_CODE_OLDER	0x488c	//rev found in axim



#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC push_options
	#pragma GCC target ("arm")
#endif

static uint32_t __attribute__((noinline)) irqsOff(void)
{
	uint32_t dummy, prevSta;
	
	asm volatile(
		"	mrs %0, CPSR		\n\t"
		"	orr %1, %0, #0x80	\n\t"
		"	msr CPSR, %1		\n\t"
		:"=r"(prevSta), "=r"(dummy)
		:
		:"memory"
	);
	
	return prevSta;
}

static void __attribute__((noinline)) irqsRestore(uint32_t prev)
{
	asm volatile(
		"	msr CPSR_c, %0		\n\t"
		:
		:"r"(prev)
		:"memory"
	);
}


#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC pop_options
#endif



static void sdInsertionIrqHandler(void *userData)
{
	struct PxaGpio *gpio = (struct PxaGpio*)repalmPlatPeriphP2V(PXA_BASE_GPIO);
	struct W86 *sd = (struct W86*)SD_BASE;
	
	//i do not want to talk about this
	(void)sd->gpioIntCtl;		//must be read
	(void)sd->gpioCtl;			//must be read
	sd->gpioIntCtl &= 0xfe;
	sd->gpioIntCtl |= 1;
	sd->gpioIntCtl |= 0xff00;

	sd->ctrlSta = 0x0100;
	while (sd->ctrlSta & 0x60);
	sd->gpioIntCtl = 0x0001;		//irq on insert/remove
	sd->intCtl = 0x11;
	sd->gpioCtl = 0x000e;	//some are outputs and must be zeroes
	
	logt("6 gpio state %u\n", (gpio->GPLR[0] >> 12) & 1);
	logt("6 sd ctrlSta 0x%04x, intCtl 0x%04x,  gpioIntCtl 0x%04x\n", sd->ctrlSta, sd->intCtl, sd->gpioIntCtl);
	
	gpio->GEDR[0] = 1 << 12;	//clear edge detect bit
	
	SysNotifyBroadcastFromInterrupt(MY_CRID, MY_CRID, NULL);
}

static bool __attribute__((naked)) sdPrvReadDataBlock(uint8_t *dst, struct W86 *sd)
{
	asm volatile(
		"	push {r4}				\n\t"
		"	mov  r12, #0x900000		\n\t"
		"	mov  r4, #4				\n\t"
		"1:							\n\t"
		"	subs r12, #1			\n\t"
		"	beq  timeout			\n\t"
		"	ldrh r2, [r1, %0]		\n\t"
		"	cmp  r2, #0xc000		\n\t"
		"	bcc  1b					\n\t"
		
		"	pld  [r0, #0x00]		\n\t"
		"	pld  [r0, #0x20]		\n\t"
		"	pld  [r0, #0x40]		\n\t"
		"	pld  [r0, #0x60]		\n\t"
		
		".rept 32					\n\t"
		"	ldrh r2, [r1, %1]		\n\t"
		"	ldrh r3, [r1, %1]		\n\t"
		"	strb r2, [r0, #1]		\n\t"
		"	lsr  r2, #8				\n\t"
		"	strb r3, [r0, #3]		\n\t"
		"	lsr  r3, #8				\n\t"
		"	strb r3, [r0, #2]		\n\t"
		"	strb r2, [r0], #4		\n\t"
		".endr						\n\t"
		
		"	subs r4, #1				\n\t"
		"	bne  1b					\n\t"
		
		"	mov  r0, %2				\n\t"
		
		"out:						\n\t"
		"	pop  {r4}				\n\t"
		"	bx   lr					\n\t"
		
		"timeout:					\n\t"
		"	mov  r0, %3				\n\t"
		"	b    out				\n\t"
		:
		:"I"(offsetof(struct W86, indirectData)), "I"(offsetof(struct W86, dataFifo)), "I"(true), "I"(false)
		:"memory", "cc", "r0", "r1", "r2", "r3", "lr", "r12"
	);
	
	//not reached
	return 0;
}

//expects sd->indirectAddr == W86IndRegHostIfaceAndBufferSvcLen;
static bool sdPrvReadData(uint8_t *dst, struct W86 *sd, uint32_t dataBlockSz)	//TODO: assemblify-this
{
	uint32_t sta, i;
	
	//wait for top bit in W86IndRegHostIfaceAndBufferSvcLen
	while (1) {
		
		i = 10000000;
		while (!((sta = sd->indirectData) & 0x8000) || !(sta & 0x7f00)) {
			
			if (!--i)
				return false;
		}
		
		logt("sta finaly 0x%08x\n", sta);
		
		sta >>= 8;
		sta &= 0x7f;
		
		while (sta && dataBlockSz >= 2) {
			
			i = sd->dataFifo;
			*dst++ = i >> 8;
			*dst++ = i;
			if (!(dataBlockSz -= 2))
				return true;
			sta--;
		}
		if (sta && dataBlockSz == 1) {
			
			i = sd->dataFifo;
			*dst++ = i;
			if (!--dataBlockSz)
				return true;
		}
	}
}

//expects sd->indirectAddr == W86IndRegHostIfaceAndBufferSvcLen;
static bool sdPrvWriteData(const uint8_t *src, struct W86 *sd, uint32_t dataBlockSz)	//TODO: assemblify-this
{
	uint32_t sta, i;
	
	//wait for top bit in W86IndRegHostIfaceAndBufferSvcLen
	while (1) {
		
		i = 10000000;
		while (!((sta = sd->indirectData) & 0x8000) || !(sta & 0x7f00)) {
			
			if (!--i)
				return false;
		}
		
		logt("sta finaly 0x%08x\n", sta);
		
		sta >>= 8;
		sta &= 0x7f;
		
		while (sta && dataBlockSz >= 2) {
			
			i = *src++;
			i <<= 8;
			i += *src++;
			sd->dataFifo = i;
			if (!(dataBlockSz -= 2))
				return true;
			sta--;
		}
		if (sta && dataBlockSz == 1) {
			
			sd->dataFifo = ((uint32_t)*src++) << 8;
			if (!--dataBlockSz)
				return true;
		}
	}
}

static bool __attribute__((naked)) sdPrvWriteDataBlock(const uint8_t *src, struct W86 *sd)
{
	asm volatile(
		"	push {r4-r6}			\n\t"
		"	mov  r12, #0x900000		\n\t"
		"	mov  r4, #4				\n\t"
		"1:							\n\t"
		"	subs r12, #1			\n\t"
		"	beq  timeout2			\n\t"
		"	ldrh r2, [r1, %0]		\n\t"
		"	cmp  r2, #0xc000		\n\t"
		"	bcc  1b					\n\t"
		
		"	pld  [r0, #0x00]		\n\t"
		"	pld  [r0, #0x20]		\n\t"
		"	pld  [r0, #0x40]		\n\t"
		"	pld  [r0, #0x60]		\n\t"
		
		".rept 32					\n\t"
		"	ldrb r6, [r0, #3]		\n\t"
		"	ldrb r5, [r0, #2]		\n\t"
		"	ldrb r3, [r0, #1]		\n\t"
		"	ldrb r2, [r0], #4		\n\t"
		"	add  r6, r5, lsl #8		\n\t"
		"	add  r3, r2, lsl #8		\n\t"
		"	strh r3, [r1, %1]		\n\t"
		"	strh r6, [r1, %1]		\n\t"
		".endr						\n\t"
		
		"	subs r4, #1				\n\t"
		"	bne  1b					\n\t"
		
		"	mov  r0, %2				\n\t"
		
		"out2:						\n\t"
		"	pop  {r4-r6}			\n\t"
		"	bx   lr					\n\t"
		
		"timeout2:					\n\t"
		"	mov  r0, %3				\n\t"
		"	b    out2				\n\t"
		:
		:"I"(offsetof(struct W86, indirectData)), "I"(offsetof(struct W86, dataFifo)), "I"(true), "I"(false)
		:"memory", "cc", "r0", "r1", "r2", "r3", "lr", "r12"
	);
	
	//not reached
	return 0;
}

uint32_t sdHwGetMaxBlocksAtOnce(struct SdHwData *hwData)
{
	return 0x1ff;
}

uint32_t sdHwGetMaxBlockSize(struct SdHwData *hwData)
{
	return 0x7ff;
}

static bool sdHwPrvBusyWait(struct W86 *sd)
{
	uint32_t i;
	
	sd->indirectAddr = W86IndRegExtStaAndSettings;
	i = 10000000;
	while (sd->indirectData & 0x4000) {	//XXX: might be inverted...
		
		logt("busy wait 0x%04x\n", sd->indirectData);
		if (!--i) {
			loge("busy wait timeout\n");
			return false;
		}
	}
	return true;
}

static void sdHwPrvPostCmdCleanup(struct W86 *sd)
{
	//post-command cleanup
	sd->ctrlSta |= 0x60;
	while (sd->ctrlSta & 0x60);
	
	sd->indirectAddr = W86IndRegErrSta;
	sd->indirectData = 0xffff;
	sd->intCtl = (sd->intCtl & 0x00ff) | 0xc800;
}

enum SdHwReadResult sdHwReadData(struct SdHwData *hwData, uint8_t *data, uint_fast16_t sz)
{
	struct W86 *sd = (struct W86*)SD_BASE;
	enum SdHwReadResult rr;
	bool llReadResult;
	uint32_t i, err;
	
	sd->indirectAddr = W86IndRegHostIfaceAndBufferSvcLen;
	llReadResult = (sz == 512) ? sdPrvReadDataBlock(data, sd) : sdPrvReadData(data, sd, sz);
	if (!llReadResult) {
		rr = SdHwReadTimeout;
		goto out_err;
	}
		
	logt("2 sta = 0x%04x  intSta=0x%04x\n", sd->ctrlSta, sd->intCtl);
	logt("sta = 0x%04x\n", sd->indirectData);
	
	i = 1000000;
	while (!(sd->intCtl & 0x4000)) {	//wait for CRC rx and status
		if (!--i) {
			rr = SdHwReadTimeout;
			goto out_err;
		}
	}
	sd->indirectAddr = W86IndRegErrSta;
	err = sd->indirectData;
	
	if (err & 0x0f00)					//crc error
		rr = SdHwReadCrcErr;
	else if (err & 0xffc0) {
		rr = SdHwReadInternalError;
		logt("err = 0x%08x\n", err);
	}
	else
		rr = SdHwReadOK;

out_err:
	sdHwPrvPostCmdCleanup(sd);
	return rr;
}

enum SdHwWriteReply sdHwWriteData(struct SdHwData *hwData, const uint8_t *data, uint_fast16_t sz, bool isMultiblock)
{
	struct W86 *sd = (struct W86*)SD_BASE;
	enum SdHwWriteReply ret;
	bool writeResult;
	uint32_t i, err;
	
	sd->indirectAddr = W86IndRegHostIfaceAndBufferSvcLen;
	writeResult = (sz == 512) ? sdPrvWriteDataBlock(data, sd) : sdPrvWriteData(data, sd, sz);
	if (!writeResult) {
		ret = SdHwCommErr;
		goto out_err;
	}
	
	logt("2 sta = 0x%04x   intSta=0x%04x   err=0x%04x\n", sd->ctrlSta, sd->intCtl, err);
	logt("sta = %u\n", sd->indirectData);
	
//	while (((i = sd->indirectData) & 0x8000) && (i & 0x7f00))
//		logt("overread 0x%04x\n", sd->dataFifo);
	
	//wait till card replies to data
	i = 1000000;
	while (!(sd->intCtl & 0x0800)) {
		
		if (!--i) {
			ret = SdHwTimeout; 
			goto out_err;
		}
	}
	
	//let's see what the reply was
	sd->indirectAddr = W86IndRegErrSta;
	err = sd->indirectData;
	
	if (err & 0x4000) {		//card sent back a "programing error" token
		ret = SdHwWriteError;
		goto out_err;
	}
	if (err & 0x2000) {		//card sent back a "crc error" token
		ret = SdHwWriteCrcErr;
		goto out_err;
	}
	ret = SdHwWriteAccepted;
	
out_err:
	sdHwPrvPostCmdCleanup(sd);
	return ret;
}

bool sdHwPrgBusyWait(struct SdHwData *hwData)
{
	struct W86 *sd = (struct W86*)SD_BASE;
	
	return sdHwPrvBusyWait(sd);
}

void sdHwChipDeselect(struct SdHwData *hwData)
{
	struct W86 *sd = (struct W86*)SD_BASE;
	
	sdHwPrvPostCmdCleanup(sd);
}

enum SdHwCmdResult sdHwCmd(struct SdHwData *hwData, uint_fast8_t cmd, uint32_t param, bool cmdCrcRequired, enum SdHwRespType respTyp, void *respBufOut, enum SdHwDataDir dataDir, uint_fast16_t blockSz, uint32_t numBlocks)
{
	bool waitForBusy = respTyp == SdRespTypeR1withBusy, useFakeCmd = false;
	struct W86 *sd = (struct W86*)SD_BASE;
	enum SdHwCmdResult ret;
	uint32_t i, err;
	
	logt("cmd %u(0x%08x) with resp %u\n", cmd, param, respTyp);
	
	sd->indirectAddr = W86IndRegHostIfaceAndBufferSvcLen;
	sd->indirectData = 0;	//mainly we want H_HIS low
	
	sd->indirectAddr = W86IndRegErrSta;
	sd->indirectData = 0xffff;
	sd->intCtl = (sd->intCtl & 0x00ff) | 0xc800;
	
	sd->ctrlSta |= 0x60;
	while (sd->ctrlSta & 0x60);
	
	//check for sanity
	if (blockSz >= 0x0800 || numBlocks >= 0x200 || (respTyp == SdRespTypeR1withBusy && dataDir != SdHwDataNone)) {
		
		ret = SdCmdInternalError;
		goto out;
	}
	else if (dataDir != SdHwDataNone) {
		
		sd->indirectAddr = W86IndRegMasterDataFmt;
		sd->indirectData = (sd->indirectData & 0x8000) | blockSz;
		
		sd->indirectAddr = W86IndRegMasterBlockCnt;
		sd->indirectData = 0x8000 + numBlocks;
		
		if (cmd == 6) {
			
			//our chip is too clever for its own good. IT sees CMD6 and assumed no data to be sent, no matter what we program!
			//to get around this we send CMD6 without data RX, but immediatly after send a CMD17 with a data RX. The trick is that
			//we malform CMD17 just so that SD card ignored it but our chip buys it and does our DATA RX
			
			if (numBlocks != 1)
				fatal("fake cmd only good for one block\n");
			
			useFakeCmd = true;
			logw("these commands will not work on the chip as is. we cn try to force it, but it is unlikely to be worth it or work well\n");
		}
	}
	
	sd->cmdRspFifo = 0x4000 | (((uint32_t)cmd) << 8) | (param >> 24);
	sd->cmdRspFifo = param >> 8;
	if (cmd == 12) {
		/*
		 * When the CMD12 write to the command pipe register, the time period from
		 * the first word write action to the end of third word write action must
		 * longer than the clock period of the system clock.
		 */
		uint32_t t;
		
		asm volatile(
			"1:				\n\t"
			"	subs %0, #1	\n\t"
			"	bne  1b		\n\t"
			:"=r"(t)
			:"0"(100)
			:"cc", "memory"
		);
	}
	sd->cmdRspFifo = (param << 8) | 1;
	
	i = 1000000;
	while (!(sd->intCtl & 0x8000)){	//wait for command complete
		if (!--i) {
			logt("complete timeout\n");
			ret = SdHwCmdResultRespTimeout;
			goto out;
		}
	}

	if (respTyp != SdRespTypeNone) {
				
		i = 1000000;
		while (!(sd->ctrlSta & 0x1000)){	//wait for resp avail
			if (!--i) {
				logt("reply timeout\n");
				ret = SdHwCmdResultRespTimeout;
				goto out;
			}
		}
		
		if (useFakeCmd) {
			sd->cmdRspFifo = 0x8000 | (17 << 8);	//to force read
			sd->cmdRspFifo = 0;
			sd->cmdRspFifo = 0;	//force it to be ignored by missing the 1
			
			sd->indirectAddr = W86IndRegMasterDataFmt;
			sd->indirectData = (sd->indirectData & 0x8000) | blockSz;
			
			sd->indirectAddr = W86IndRegMasterBlockCnt;
			sd->indirectData = 0x8000 + numBlocks;
			
			logi("fake cmd\n");
		}
		
		logt("sta = 0x%04x   intSta=0x%04x\n", sd->ctrlSta, sd->intCtl);
			
		if ((sd->ctrlSta & 0x2000) && respTyp != SdRespTypeSdR2) {	//got long reply but did not expect it
			ret = SdCmdInternalError;
			goto out;
		}
		if (!(sd->ctrlSta & 0x2000) && respTyp == SdRespTypeSdR2) {	//expected long reply but did not get it
			ret = SdCmdInternalError;
			goto out;
		}
		
		if (respTyp == SdRespTypeSdR2) {	//long reply
			
			uint8_t *respBufOut8 = (uint8_t*)respBufOut;
			uint32_t hi = sd->cmdRspFifo;
		
			for (i = 0; i < 4; i++) {
				
				uint32_t mid = sd->cmdRspFifo;
				uint32_t lo = sd->cmdRspFifo;
				
				respBufOut8[(3 - i) * 4 + 0] = lo >> 8;
				respBufOut8[(3 - i) * 4 + 1] = mid;
				respBufOut8[(3 - i) * 4 + 2] = mid >> 8;
				respBufOut8[(3 - i) * 4 + 3] = hi;
				
				hi = lo;
			}
		}
		else {
			
			uint8_t *respBufOut8 = (uint8_t*)respBufOut;
			uint32_t hi = sd->cmdRspFifo;
			uint32_t mid = sd->cmdRspFifo;
			uint32_t lo = sd->cmdRspFifo;
			
			switch (respTyp) {
				case SdRespTypeR1:
				case SdRespTypeR1withBusy:
					*respBufOut8 = sdPrvR1toSpiR1((hi << 24) | (mid << 8) | (lo >> 8));
					break;
				
				case SdRespTypeR3:
				case SdRespTypeR7:
				case SdRespTypeSdR6:
					*respBufOut8++ = hi;
					*respBufOut8++ = mid >> 8;
					*respBufOut8++ = mid;
					*respBufOut8++ = lo >> 8;
					break;
				
				default:
					break;
			}
		}
	}
	
	sd->indirectAddr = W86IndRegErrSta;
	err = sd->indirectData;
	logt("sta = 0x%04x   intSta=0x%04x   err=0x%04x\n", sd->ctrlSta, sd->intCtl, err);
	
	if (err & 0x1000) {
		ret = SdCmdInternalError;
		goto out;
	}
	
	//done...cleanup
	
	if (waitForBusy && !sdHwPrvBusyWait(sd))
		ret = SdHwCmdResultRespTimeout;
	else
		ret = SdHwCmdResultOK;

out:
	sdHwPrvPostCmdCleanup(sd);
	return ret;
}

static void setupInsertIrq(void)
{
	struct PxaGpio *gpio = (struct PxaGpio*)repalmPlatPeriphP2V(PXA_BASE_GPIO);
	uint32_t sta;
	
	HALInterruptSetHandler(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_GPIO(12)), sdInsertionIrqHandler, NULL);
	sta = irqsOff();
	gpio->GFER[0] |= 1 << 12;
	irqsRestore(sta);
}

static void disableInsertIrq(void)
{
	struct PxaGpio *gpio = (struct PxaGpio*)repalmPlatPeriphP2V(PXA_BASE_GPIO);
	uint32_t sta;
	
	sta = irqsOff();
	gpio->GFER[0] &=~ 1 << 12;
	irqsRestore(sta);
	HALInterruptSetHandler(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_GPIO(12)), NULL, NULL);
}

uint32_t sdHwInit(struct SdHwData *hwData)		//should set speed to 400khz
{
	struct W86 *sd = (struct W86*)SD_BASE;
	uint32_t i, t;
	
	i = 1;
	repalmDalPwrCtl(AXIM_PWR_BIT_SD_CHIP, &i, NULL);		//XXX: we used to set AXIM_CPLD_BIT_SD_CHIP_POWER | 0x6000. might mater

	//reset chip
	i = 0;
	repalmDalPwrCtl(AXIM_PWR_BIT_SD_CHIP_RST, &i, NULL);
	SysTaskDelay(10);
	i = 1;
	repalmDalPwrCtl(AXIM_PWR_BIT_SD_CHIP_RST, &i, NULL);
	SysTaskDelay(5);
	
	setupInsertIrq();
	
	//do not ask. rom does this
	for (t = 0; t < 20; t++) {
		sd->intCtl = 0xa5;
		sd->indirectAddr = 0x0c;
		sd->intCtl = 0x5a;
		sd->indirectAddr = 0x0c;
	}
	for (t = 0; t < 10; t++)
		sd->indirectAddr = W86IndRegID;
	//end rom weirdness
	
	
	sd->ctrlSta = 0;
	sd->gpioIntCtl = 0x0001;		//irq on insert/remove
	sd->intCtl = 0x11;
	sd->gpioCtl = 0x000e;	//some are outputs and must be zeroes
	
	
	t = sd->indirectData;
	logt("SD chip version: 0x%04x\n", t);
	
	if (t == W86L488Y_ID_CODE || t == W86L488Y_ID_CODE_OLDER)
		logt(" found W86L488Y\n");
	else
		return 0;
	
	sd->indirectAddr = W86IndRegExtStaAndSettings;
	sd->indirectData = 0x0079;		//spi off, crc generated, largest timeout
	
	sdHwSetSpeed(hwData, 400000);
	
	return SD_HW_FLAG_INITED | SD_HW_FLAG_SUPPORT_4BIT | SD_HW_FLAG_SDIO_IFACE;
}

void sdHwShutdown(struct SdHwData *hwData)
{
	struct W86 *sd = (struct W86*)SD_BASE;
	uint32_t i = 0;
	
	sd->ctrlSta = 0x0000;
	
	disableInsertIrq();
	
	//reset pin low (to not waste power)
	repalmDalPwrCtl(AXIM_PWR_BIT_SD_CHIP_RST, &i, NULL);		//XXX: we used to set AXIM_CPLD_BIT_SD_CHIP_POWER | 0x6000. might mater

	//power off
	repalmDalPwrCtl(AXIM_PWR_BIT_SD_CHIP, &i, NULL);		//XXX: we used to set AXIM_CPLD_BIT_SD_CHIP_POWER | 0x6000. might mater
}

void sdHwGiveInitClocks(struct SdHwData *hwData)
{
	struct W86 *sd = (struct W86*)SD_BASE;
	
	sd->indirectAddr = W86IndRegSdioCtl;
	sd->indirectData = 0x10;
	
	sd->ctrlSta = 0x0019;			//clock on, 400 KHz
	
	logt("driving clock\n");
	while (sd->indirectData & 0x10);
	logt("drove clock\n");
	
	sd->indirectData = 0x04;		//SDCS
}

bool sdHwIsCardInserted(struct SdHwData *hwData)
{
	struct W86 *sd = (struct W86*)SD_BASE;
	
	return !!(sd->gpioCtl & 0x0100);
}

bool sdHwIsCardLockSwitchOn(struct SdHwData *hwData)
{
	struct W86 *sd = (struct W86*)SD_BASE;
	
	return !(sd->gpioCtl & 0x1000);
}

void sdHwSetSpeed(struct SdHwData *hwData, uint32_t maxHz)
{
	struct W86 *sd = (struct W86*)SD_BASE;
	uint32_t i, busSpeed = 25000000;
	uint8_t clks = 1;

	for (i = 0; i < 15; i++) {
		if (maxHz >= (busSpeed >> i)) {
			clks = 15 - i;
			break;
		}
	}
	
	logi("speed setting for %u hz is %u\n", maxHz, clks);
	sd->ctrlSta = (sd->ctrlSta & 0xe0) | 0x10 | clks;
}

bool sdHwSetBusWidth(struct SdHwData *hwData, bool useFourWide)
{
	struct W86 *sd = (struct W86*)SD_BASE;
	
	sd->indirectAddr = W86IndRegMasterDataFmt;
	if (useFourWide)
		sd->indirectData |= 0x8000;
	else
		sd->indirectData &=~ 0x8000;
	
	return true;
}

void sdHwCardPower(struct SdHwData *hwData, bool on)
{
	//nothing
}

void sdHwSleep(struct SdHwData *hwData)
{
	//really should do this
}

void sdHwWake(struct SdHwData *hwData)
{
	//really should do this
}

void sdHwRxRawBytes(struct SdHwData *hwData, void *dst /* can be NULL*/, uint_fast16_t numBytes)
{
	//not used for SDIO HWs
}

void sdHwNotifyRCA(struct SdHwData *hwData, uint_fast16_t rca)
{
	//we do not need it
}

bool sdHwMultiBlockWriteSignalEnd(struct SdHwData *hwData)
{
	//not needed
	return true;
}

bool sdHwMultiBlockReadSignalEnd(struct SdHwData *hwData)
{
	//not needed
	return true;
}

void sdHwSetTimeouts(struct SdHwData *hwData, uint_fast16_t timeoutBytes, uint32_t rdTimeoutTicks, uint32_t wrTimeoutTicks)
{
	//i really should honor these...
}