#include <kernel/drivers/platforms/xscale/xscale.h>
#include <kernel/drivers/platforms/platform.h>
#include <kernel/drivers/aximX3/aximPwrCtl.h>
#include "printf.h"
#include "common.h"
#include "port.h"
#include <boot.h>
#include <dal.h>
#include <kal.h>

#define PORT_NAME_IR				'sir0'



struct OpenPort {
	
	struct PxaUart *uart;
	struct PxaGpio *gpio;
	const struct VdrvRecvQueue *recvQ;
	uint32_t txTimeout;
	uint32_t settings;
	uint16_t baudrate;
};

struct Globals {					//limited to 4 bytes by 'amdi' resource. change that to get more
	struct OpenPort* sir;	//NULL if not open
};

static struct Globals* __attribute__((const)) serGlobalsGet(void)
{
	void**** ret;
	
	asm ("mov %0, r9":"=r"(ret));
	return (struct Globals*)&ret[0][MY_LIB_ID / 4][0x10/4];
}



#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC push_options
	#pragma GCC target ("arm")
#endif

static uint32_t __attribute__((noinline)) irqsOff(void)
{
	uint32_t dummy, prevSta;
	
	asm volatile(
		"	mrs %0, CPSR		\n\t"
		"	orr %1, %0, #0x80	\n\t"
		"	msr CPSR, %1		\n\t"
		:"=r"(prevSta), "=r"(dummy)
		:
		:"memory"
	);
	
	return prevSta;
}

static void __attribute__((noinline)) irqsRestore(uint32_t prev)
{
	asm volatile(
		"	msr CPSR_c, %0		\n\t"
		:
		:"r"(prev)
		:"memory"
	);
}


#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC pop_options
#endif



static bool serHwPower(struct OpenPort *prt, bool on)
{
	uint32_t onVal = on ? 1 : 0, prevSta;
	
	
	prevSta = irqsOff();
	
	if (on)  {
		
		repalmDalPwrCtl(AXIM_PWR_BIT_IR_UART_CLK, &onVal, NULL);
		
		prt->gpio->GPCR[47 / 32] = 1 << (47 % 32);				//TX pin low when gpio
		prt->gpio->GPDR[47 / 32] |= 1 << (47 % 32);				//TX is out
		prt->gpio->GPDR[46 / 32] &=~ (1 << (46 % 32));			//RX is in
		prt->gpio->GAFR[47 / 16] &=~ (3 << ((47 % 16) * 2));
		prt->gpio->GAFR[46 / 16] &=~ (3 << ((46 % 16) * 2));
		prt->gpio->GAFR[47 / 16] |= (1 << ((47 % 16) * 2));		//make TX a TX pin
		prt->gpio->GAFR[46 / 16] |= (2 << ((46 % 16) * 2));		//make RX a RX pin
	}
	else {
		
		prt->gpio->GAFR[47 / 16] &=~ (3 << ((47 % 16) * 2));		//make TX a gpio pin
		prt->gpio->GAFR[46 / 16] &=~ (3 << ((46 % 16) * 2));		//make RX a gpio pin
		
		repalmDalPwrCtl(AXIM_PWR_BIT_IR_UART_CLK, &onVal, NULL);
	}	
	
	irqsRestore(prevSta);
	
	return true;
}

static void portRxEnable(struct OpenPort *prt, bool on)
{
	bool sta = HALInterruptSetState(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_STUART), false);
	
	if (on) {
		prt->uart->FCR = 7;
		prt->uart->FCR = 1;
		prt->uart->IER |= 1;
	}
	else
		prt->uart->IER &=~ 1;
	
	HALInterruptSetState(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_STUART), sta);
}

static void portUartIrqHandler(void *drvrData)
{
	struct OpenPort *prt = (struct OpenPort*)drvrData;
	uint32_t ptr, lsr;
	uint8_t val[128];
	
	while ((lsr = prt->uart->LSR) & 0x01) {
		
		uint32_t space = prt->recvQ->RecvQueueGetQueueSpaceF(prt->recvQ->userData);
		if (!space) {
			//sw rx overflow
			
			prt->recvQ->RecvQueueBlockRxedF(prt->recvQ->userData, NULL, 0, VDRV_LINE_ERR_SW_RX_OVERFLOW);
			portRxEnable(prt, false);
			break;
		}
		
		if (lsr & 2) {	//hw overflow
			
			//hw rx overflow
			prt->recvQ->RecvQueueBlockRxedF(prt->recvQ->userData, NULL, 0, VDRV_LINE_ERR_HW_RX_OVERFLOW);
		}
		
		if (space > sizeof(val))
			space = sizeof(val);
		
		for (ptr = 0; ptr < space && (prt->uart->LSR & 0x01); ptr++)
			val[ptr] = prt->uart->RBR;
		
		prt->recvQ->RecvQueueBlockRxedF(prt->recvQ->userData, val, ptr, 0);
	}
}

static Err portCfgBaudrate(struct OpenPort *prt, uint32_t baud)	//sets and returns it
{
	int32_t baseRate = repalmDalGetClockRate(UartUnitClockRate);
	uint32_t brr, ier;
	bool sta;
	
	if (prt->baudrate == baud)
		return errNone;
	
	if (baseRate < 0)
		fatal("cannot go on without knowing the clock rate\n");
	
	brr = (baseRate + baud * 16 / 2) / (baud * 16);
	
	sta = HALInterruptSetState(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_STUART), false);
	ier = prt->uart->IER;
	prt->uart->IER = ier &~ 1;	//ints off while chaging DLAB
	prt->uart->LCR |= 0x80;
	prt->uart->DLL = brr & 0xff;
	prt->uart->DLH = brr >> 8;
	prt->uart->LCR &=~ 0x80;
	prt->uart->IER = ier;
	HALInterruptSetState(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_STUART), sta);
	
	prt->baudrate = baud;
	
	return errNone;
}

static Err portCfgSettings(struct OpenPort *prt, uint32_t cfg)
{
	uint32_t lcr;
	bool sta;
	
	
	sta = HALInterruptSetState(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_STUART), false);
	lcr = prt->uart->LCR & 0xe0;
	
	if (cfg & VDRV_UART_SETTING_2_STOP_BITS)
		lcr |= 0x04;
	
	if (cfg & VDRV_UART_SETTING_PARITY_ON)
		lcr |= 0x08;
	
	if (!(cfg & VDRV_UART_SETTING_PARITY_EVEN))
		lcr |= 0x10;
	
	switch (cfg & VDRV_UART_SETTING_CHAR_SZ_MASK) {
		case VDRV_UART_SETTING_5_BIT_CHARS:
			break;
		
		case VDRV_UART_SETTING_6_BIT_CHARS:
			lcr |= 1;
			break;
		
		case VDRV_UART_SETTING_7_BIT_CHARS:
			lcr |= 2;
			break;
		
		case VDRV_UART_SETTING_8_BIT_CHARS:
			lcr |= 3;
			break;
		
		default:
			return serErrNotSupported;
	}
	
	prt->uart->LCR = lcr;
	HALInterruptSetState(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_STUART), sta);
	
	return errNone;
}

static void portIrCtl(struct OpenPort *prt, bool portEna, bool portInTxMode)
{
	//see chip manual as to why this is so complicated
	uint32_t prevSta, desired = portInTxMode ? 0x05 : 0x16;
	
	if (desired != prt->uart->ISR) {
		
		prevSta = irqsOff();
		
		if (portInTxMode) {
			
			prt->uart->ISR = 0x05;
			prt->gpio->GAFR[47 / 16] |= (1 << ((47 % 16) * 2));		//make it a TX pin
		}
		else {
			
			prt->gpio->GAFR[47 / 16] &=~ (3 << ((47 % 16) * 2));	//make it a gpio
			prt->uart->ISR = 0x16;
		}
		
		irqsRestore(prevSta);
	}
}

Err portOpen(void **drvrDataP /* OUT */, const struct VdrvConfig *cfg, const struct VdrvRecvQueue* recvQ)
{
	struct Globals *g = serGlobalsGet();
	struct OpenPort *prt, **prtP;
	Err e;
	
	if (!drvrDataP || !cfg || !recvQ || !cfg->baudrate)
		return serErrBadParam;
	
	if (cfg->portName != PORT_NAME_IR)
		return serErrBadParam;
	
	prtP = &g->sir;
	if (*prtP)
		return serErrAlreadyOpen;
	
	prt = MemChunkNew(0, sizeof(struct OpenPort), 0x200);
	if (!prt)
		return memErrNotEnoughSpace;
	MemSet(prt, sizeof(struct OpenPort), 0);
	prt->uart = repalmPlatPeriphP2V(PXA_BASE_STUART);
	prt->gpio = repalmPlatPeriphP2V(PXA_BASE_GPIO);
	
	//logt("Opening IR port with baudrate %u\n", cfg->baudrate);
	serHwPower(prt, true);

	e = portCfgSettings(prt, VDRV_UART_SETTING_8_BIT_CHARS);
	if (e) {
		MemChunkFree(prt);
		return e;
	}
	portCfgBaudrate(prt, cfg->baudrate);

	*prtP = prt;
	
	prt->txTimeout = 5000;
	prt->recvQ = recvQ;
	prt->uart->IER = 0;
	prt->uart->FCR = 1;
	prt->uart->MCR = 0x08;	//irqs on
	prt->uart->IER = 0x40;
	HALInterruptSetHandler(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_STUART), portUartIrqHandler, prt);
	HALInterruptSetState(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_STUART), true);
	portRxEnable(prt, true);

	*drvrDataP = prt;
	return errNone;
}

struct OpenPort* portVerify(void *drvrData)
{
	if (!drvrData || serGlobalsGet()->sir != drvrData)
		return NULL;
	
	return (struct OpenPort*)drvrData;
}

Err portClose(void *drvrData)
{
	struct OpenPort *prt = portVerify(drvrData);
	struct Globals *g = serGlobalsGet();
	
	//logt("Closing port\n");
	
	if (!prt)
		return serErrBadPort;
	
	HALInterruptSetState(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_STUART), false);
	HALInterruptSetHandler(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_STUART), NULL, NULL);
	portRxEnable(prt, false);
	serHwPower(prt, false);
	if (g->sir == prt)
		g->sir = NULL;
	MemChunkFree(drvrData);
	
	return errNone;
}

Err portControl(void *drvrData, uint32_t controlOp /* VDRV_CTL_OP_* */, void* data, uint16_t *controlDataLenP /* never used */)
{
	struct OpenPort *prt = portVerify(drvrData);
	uint32_t t, *data32 = (uint32_t*)data;
	uint16_t *data16 = (uint16_t*)data;
	uint8_t *data8 = (uint8_t*)data;
	Err e = errNone;
	
	if (!prt)
		return serErrBadPort;
	
	if (!data || !controlDataLenP)
		return serErrBadParam;
	
	switch (controlOp) {
		
		case VDRV_CTL_OP_SET_BAUDRATE:
			if (!*data32)
				e = serErrBadParam;
			else {
				//logt("setting baudrate to %u\n",  *data32);
				e = portCfgBaudrate(prt, *data32);
			}
			break;
			
		case VDRV_CTL_OP_SET_SETTINGS:
			e = portCfgSettings(prt, prt->settings = *data32);
			break;
		
		case VDRV_CTL_OP_SET_CTS_TIMEOUT:
			prt->txTimeout = *data32;
			break;
		
		case VDRV_CTL_OP_CLEAR_ERR:
			(void)prt->uart->LSR;
			break;
		
		case VDRV_CTL_OP_SLEEP:
			portRxEnable(prt, false);
			HALInterruptSetState(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_STUART), false);
			break;
		
		case VDRV_CTL_OP_WAKE:
			e = portCfgSettings(prt, prt->settings);
			HALInterruptSetState(REPALM_IRQ_NO_MANGLE(XSCALE_IRQ_NO_STUART), true);
			portRxEnable(prt, true);
			break;
		
		case VDRV_CTL_OP_GET_TX_FIFO_USED:
			if (prt->uart->LSR & 0x40)		//empty? no byte sin fifo
				*data32 = 0;
			else if (prt->uart->LSR & 0x20)	//drq? at most half a fifo
				*data32 = 2;
			else							//non empty and more than half
				*data32 = 32;
			break;
		
		case VDRV_CTL_OP_START_BREAK:
			prt->uart->LCR |= 0x40;
			break;
		
		case VDRV_CTL_OP_STOP_BREAK:
			prt->uart->LCR &=~ 0x40;
			break;
		
		case VDRV_CTL_OP_FLUSH_TX_FIFO:
			while (!(prt->uart->LSR & 0x40));
			break;
		
		case VDRV_CTL_OP_FLUSH_RX_FIFO:
			//nothing
			break;
		
		case VDRV_CTL_OP_SEND_QUEUED_DATA:
			t = HALTimeGetSystemTime();
			while (!(prt->uart->LSR & 0x40) && HALTimeGetSystemTime() - t <= prt->txTimeout);
			e = (prt->uart->LSR & 0x40) ? errNone : serErrTimeOut;
			break;
		
		case VDRV_CTL_OP_GET_BEST_TX_BLOCK_SZ:
			*data32 = 64;
			break;
		
		case VDRV_CTL_OP_GET_MAX_RX_BLOCK_SZ:
			*data32 = 62;
			break;
		
		case VDRV_CTL_OP_SET_DTR:
			e = *data8 ? errNone : serErrNotSupported;
			break;
		
		case VDRV_CTL_OP_GET_DTR:
			*data8 = 1;
			break;
		
		case VDRV_CTL_OP_WAIT_FOR_CONFIG:
			break;
		
		case VDRV_CTL_OP_USB_GET_DEVICE_DESCR:
		case VDRV_CTL_OP_USB_GET_CONFIG_DESCR:
			e = serErrNotSupported;
			break;
		
		case VDRV_CTL_OP_IRDA_ENABLE:
			portIrCtl(prt, true, false);
			break;
		
		case VDRV_CTL_OP_IRDA_DISABLE:
			portIrCtl(prt, false, 0 /* do not case */);
			break;
		
		case VDRV_CTL_OP_IR_RX_ENA:
			portIrCtl(prt, true, false);
			break;
			
		case VDRV_CTL_OP_IR_RX_DIS:
			portIrCtl(prt, true, true);
			break;

		case VDRV_CTL_OP_IRQ_DISABLE:
			portRxEnable(prt, false);
			break;
		
		case VDRV_CTL_OP_IRQ_ENABLE:
			portRxEnable(prt, true);
			break;
		
		case VDRV_CTL_OP_SET_NEW_RECV_QUEUE:
			prt->recvQ = data;
			//fallthrough
			
		case VDRV_CTL_OP_NOTIF_RX_BYTES_CONSUMED:
			portRxEnable(prt, !!prt->recvQ->RecvQueueGetQueueSpaceF(prt->recvQ->userData));
			break;
		
		case VDRV_CTL_OP_GET_IR_SPEED_BITMAPS:
			//we could use higher speeds, but palms never did, so deal with it later
			*data16 = VDRV_IR_SPEED_2K2 | VDRV_IR_SPEED_9K6 | VDRV_IR_SPEED_19K2 | VDRV_IR_SPEED_38K4 | VDRV_IR_SPEED_57K6 | VDRV_IR_SPEED_115K;
			break;
		
		default:
			e = serErrBadParam;
			break;
	}
	
	return e;
}

uint32_t portWrite(void *drvrData, const void *buf, uint32_t size, Err *errP)
{
	struct OpenPort *prt = portVerify(drvrData);
	const uint8_t *src = (const uint8_t*)buf;
	uint32_t time;
	
	if (!prt)
		return serErrBadPort;
	
	if (!errP || (size && !buf))
		return serErrBadParam;
	
	*errP = errNone;
	
	if (!size)
		return 0;
	
	portIrCtl(prt, true, true);
	
	time = HALTimeGetSystemTime();
	while (size) {
		
		while ((prt->uart->LSR & 0x20) && size) {
			prt->uart->THR = *src++;
			size--;
		}
		
		if (HALTimeGetSystemTime() - time > prt->txTimeout) {
			*errP = serErrTimeOut;
			break;
		}
	}
	
	while(!(prt->uart->LSR & 0x40));	//thi shas fixed duration, so no problem doing this
	
	portIrCtl(prt, true, false);
	
	return src - ((const uint8_t*)buf);
}

Err portGetStatus(void *drvrData, uint16_t* statusP /* VDRV_STA_* bitfield */ )
{
	struct OpenPort *prt = portVerify(drvrData);
	
	if (!prt)
		return serErrBadPort;
	
	if (!statusP)
		return serErrBadParam;
	
	//IrDA has no status
	return serErrNotSupported;
}

Err portCustomControl(void *drvrData, ...)
{
	return serErrNotSupported;
}

Err portGetInfo(struct VdrvPortInfo* info)
{
	const char* portNameSrc;
	char* portNameDst;
	
	if (!info)
		return serErrBadParam;
	
	switch (info->portIdx) {
		case 0:
			info->portName = PORT_NAME_IR;
			info->portFlags = VDRV_PORT_FLAG_VISIBLE_IN_CNC_MGR | VDRV_PORT_FLAG_IRDA_CAPABLE | VDRV_PORT_FLAG_CAN_RUN_IN_BACKGROUND;
			info->maxBaudrate = 115200;
			info->handshakeBaudrate = 19200;
			portNameSrc = "AximX3 IR port";
			break;
		
		default:
			return serErrBadPort;
	}
	
	info->driverVer = VDRV_VERSION_5;
	info->dbCrid = MY_CRID;
	portNameDst = info->portNameStr;
	if (portNameDst)
		while ((*portNameDst++ = *portNameSrc++));
	
	return errNone;
}

