#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "parsePrcPdb.h"
#include "util.h"


//this code will not correctly handle zero-sized resources/records
//i think this is ok since the spec for PRC/PDB file never really stated how to
//represent them (in-orderness) of records inheader and memory is not guranteed
//and thus if two pointers point to the same place, which one is zero sized
//the only way to handle that is to have all zero-sized records/resrouces
//point to the end of the file. This representation we DO support.

struct PRCFILE_RsrcEntry {						//as it is on host filesystem. this is BIG ENDIAN and packed

	uint32_t					type; 					// 0x00
	uint16_t					id; 					// 0x04
	uint32_t					ofstInFile; 			// 0x06

} __attribute__((packed));

struct PRCFILE_RecEntry {						//as it is on host filesystem. this is BIG ENDIAN and packed

	uint32_t					ofstInFile; 			// 0x00
	uint8_t						attrs; 					// 0x04
	uint8_t						uniqId[3]; 				// 0x05		ACTUALLY stored LE

} __attribute__((packed));

struct PRCFILE_RecordList {						//as it is on host filesystem. this is BIG ENDIAN and packed

	uint32_t					nextRecordListID_ofst;	// 0x00
	uint16_t					numRecords;				// 0x04
	
	//struct PRCFILE_RsrcEntry	res[];					// 0x06
	// OR
	//struct PRCFILE_RecEntry	rec[];					// 0x06
} __attribute__((packed));

struct PRCFILE_DatabaseHdrType {				//as it is on host filesystem. this is BIG ENDIAN and packed

	char						name[32];				// 0x00 
	uint16_t					attributes;				// 0x20
	uint16_t					version;				// 0x22
	uint32_t					creationDate;			// 0x24
	uint32_t					modificationDate;		// 0x28
	uint32_t					lastBackupDate;			// 0x2C
	uint32_t					modificationNumber;		// 0x30
	uint32_t					appInfoID_offset;		// 0x34
	uint32_t					sortInfoID_offset;		// 0x38
	uint32_t					type;					// 0x3C
	uint32_t					creator;				// 0x40
	uint32_t					uniqueIDSeed;			// 0x44
	struct PRCFILE_RecordList	recordList;				// 0x48

} __attribute__((packed));

static void* readfile(const char *path, uint32_t *lenP)	//yes i know this will fall all over itself for large files. don't care! Palms never supported PRC files over 4GB
{
	FILE *f = fopen(path, "rb");
	void *buf;
	long len;
	
	
	if (!f)
		goto out;
	
	if (fseek(f, 0, SEEK_END)) {
		fprintf(stderr, "cannot seek to end");
		goto out_close;
	}
	
	len = ftell(f);
	if (len < 0) {
		fprintf(stderr, "cannot get position");
		goto out_close;
	}
	
	if (fseek(f, 0, SEEK_SET)) {
		fprintf(stderr, "cannot seek to start");
		goto out_close;
	}
	
	buf = calloc(len, 1);
	if (!buf) {
		fprintf(stderr, "cannot allocate %lu bytes", len);
		goto out_close;
	}
	
	if(len != (long)fread(buf, 1, len, f)){
		fprintf(stderr, "cannot read file");
		goto out_free;
	}

	*lenP = len;
	fclose(f);
	return buf;

out_free:
	free(buf);

out_close:
	fclose(f);

out:
	return NULL;
}

static uint32_t getChunkSizeCheckCand(uint32_t curCand, uint32_t potentialCand, uint32_t ourchunk)
{
	if (potentialCand && potentialCand > ourchunk && potentialCand < curCand)
		curCand = potentialCand;
	
	return curCand;
}

//yup, O(n^2). too bad. databases are small and my time is valuable. you'll live...
static int32_t getChunkSize(const struct PRCFILE_DatabaseHdrType *hdr, uint32_t fileSz, bool prc, uint32_t chunkStartOfst, const char *friendlyNameForErrorMsg)
{
	//find smallest other offset into file larger than ours. start with assuming file size
	const struct PRCFILE_RsrcEntry* fileRes = (const struct PRCFILE_RsrcEntry*)(hdr + 1);
	const struct PRCFILE_RecEntry* fileRec = (const struct PRCFILE_RecEntry*)(hdr + 1);
	uint32_t cand = fileSz;
	uint32_t i;
	
	//sanity check
	if (chunkStartOfst > fileSz) {
		fprintf(stderr, "chunks cannot start past the end of the file for %s\n", friendlyNameForErrorMsg);
		return -1;
	}
	
	//try the two offsets in the header
	cand = getChunkSizeCheckCand(cand, __builtin_bswap32(hdr->appInfoID_offset), chunkStartOfst);
	cand = getChunkSizeCheckCand(cand, __builtin_bswap32(hdr->sortInfoID_offset), chunkStartOfst);
	
	//try the records/resources
	for (i = 0; i < __builtin_bswap16(hdr->recordList.numRecords); i++) {
			
		if (prc)
			cand = getChunkSizeCheckCand(cand, __builtin_bswap32(fileRes[i].ofstInFile), chunkStartOfst);
		else
			cand = getChunkSizeCheckCand(cand, __builtin_bswap32(fileRec[i].ofstInFile), chunkStartOfst);
	}
	
	//done
	return cand - chunkStartOfst;
}

static bool loadChunkToBuf(struct Buffer *dst, const struct PRCFILE_DatabaseHdrType *hdr, uint32_t fileSz, bool prc, uint32_t chunkStartOfst, const char *friendlyNameForErrorMsg)
{
	int32_t chunkSz;
	
	if (!chunkStartOfst) {	//no such chunk exists
		dst->data = NULL;
		dst->sz = 0;
		return true;
	}
	
	chunkSz = getChunkSize(hdr, fileSz, prc, chunkStartOfst, friendlyNameForErrorMsg);
	
	if (chunkSz < 0)
		return false;
	
	dst->data = malloc(chunkSz ? chunkSz : 1 /* always alloc a byte so that zero-sized chunks are distinguishable form chunks that did not exist at all */);
	if (!dst->data) {
		fprintf(stderr, "allocation error for %s\n", friendlyNameForErrorMsg);
		return false;
	}
	
	dst->sz = chunkSz;
	memcpy(dst->data, ((uint8_t*)hdr) + chunkStartOfst, chunkSz);
	return true;
}

struct PalmDb *parsePrcPdb(const char *path)
{
	const struct PRCFILE_DatabaseHdrType *fileHdr;
	const struct PRCFILE_RsrcEntry *fileRes;
	const struct PRCFILE_RecEntry *fileRec;
	uint32_t fileSz, i, numChildren;
	struct PalmDb *ret = NULL;
	void* fileContents;
	bool prc;
	
	if (strlen(path) < 3) {
		fprintf(stderr, "path '%s' is too short to contain a valid PRC/PDB file name\n", path);
		return NULL;
	}
	
	if (!strcasecmp(path + strlen(path) - 3, "prc"))
		prc = true;
	else if (!strcasecmp(path + strlen(path) - 3, "pdb"))
		prc = false;
	else {
		fprintf(stderr, "path '%s' does not end in 'prc' or 'pdb' and thus cannot be parsed\n", path);
		return NULL;
	}
	
	fileContents = readfile(path, &fileSz);
	if (!fileContents)
		return NULL;
	
	fileHdr = (const struct PRCFILE_DatabaseHdrType*)fileContents;
	
	//verify record list is not being weird
	if (readBE32(&fileHdr->recordList.nextRecordListID_ofst)) {
		fprintf(stderr, "split record lists are not supported by this application. sorry.\n");
		free(fileContents);
		return NULL;
	}
	
	//sort out how much space to alloc and do so
	numChildren = readBE16(&fileHdr->recordList.numRecords);
	ret = calloc(1, sizeof(struct PalmDb) + (prc ? sizeof(struct PalmRes) : sizeof(struct PalmRec)) * numChildren);
	if (!ret) {
		fprintf(stderr, "cannot alloc db hdr\n");
		free(fileContents);
		return NULL;
	}
	
	//copy in the header
	strncpy(ret->name, fileHdr->name, sizeof(ret->name));
	ret->name[sizeof(ret->name) - 1] = 0;
	ret->attributes			= readBE16(&fileHdr->attributes);
	ret->version			= readBE16(&fileHdr->version);
	ret->creationDate		= readBE32(&fileHdr->creationDate);
	ret->modificationDate	= readBE32(&fileHdr->modificationDate);
	ret->lastBackupDate		= readBE32(&fileHdr->lastBackupDate);
	ret->modificationNumber	= readBE32(&fileHdr->modificationNumber);
	ret->type				= readBE32(&fileHdr->type);
	ret->creator			= readBE32(&fileHdr->creator);
	ret->uniqueIDSeed		= readBE32(&fileHdr->uniqueIDSeed);
	
	//set our values
	ret->isPrc				= prc;
	ret->numChildren		= readBE16(&fileHdr->recordList.numRecords);
	
	//verify name
	if (((ret->attributes & PALM_DB_ATTR_RES_DB) && !prc) || (!(ret->attributes & PALM_DB_ATTR_RES_DB) && prc)) {
		fprintf(stderr, "DB flags say it is a P%s while name says it is a P%s. Giving up\n", (ret->attributes & PALM_DB_ATTR_RES_DB) ? "RC" : "DB", prc ? "RC" : "DB");
		goto fail;
	}
	
	if (prc)
		ret->res			= (struct PalmRes*)(ret + 1);
	else
		ret->rec			= (struct PalmRec*)(ret + 1);
	
	//copy in the app info and sort info chunks
	if (!loadChunkToBuf(&ret->appInfo, fileHdr, fileSz, prc, readBE32(&fileHdr->appInfoID_offset), "appinfo")) {
		fprintf(stderr, "Failed to read AppInfo chunk\n");
		goto fail;
	}
	
	if (!loadChunkToBuf(&ret->sortInfo, fileHdr, fileSz, prc, readBE32(&fileHdr->sortInfoID_offset), "sortinfo")) {
		fprintf(stderr, "Failed to read SortInfo chunk\n");
		goto fail;
	}

	//get pointers
	fileRes = (const struct PRCFILE_RsrcEntry*)(fileHdr + 1);
	fileRec = (const struct PRCFILE_RecEntry*)(fileHdr + 1);

	//copy in rec/res entries
	for (i = 0; i < ret->numChildren; i++) {
		
		char name[64];
		
		if (prc) {
			
			ret->res[i].type = readBE32(&fileRes[i].type);
			ret->res[i].id = readBE16(&fileRes[i].id);
			
			sprintf(name, "res ('%c%c%c%c', %d)", (ret->res[i].type >> 24) & 0xff, (ret->res[i].type >> 16) & 0xff, (ret->res[i].type >> 8) & 0xff, (ret->res[i].type >> 0) & 0xff, ret->res[i].id);
			if (!loadChunkToBuf(&ret->res[i].buf, fileHdr, fileSz, prc, readBE32(&fileRes[i].ofstInFile), name)) {
				fprintf(stderr, "Failed to read resource (0x%08X, %u)\n", ret->res[i].type, ret->res[i].id);
				goto fail;
			}
		}
		else {
			
			ret->rec[i].uniq = readLE24(fileRec[i].uniqId);	//this is correct. LE not BE
			ret->rec[i].attrs = fileRec[i].attrs;
			sprintf(name, "rec %u", ret->rec[i].uniq);
			if (!loadChunkToBuf(&ret->rec[i].buf, fileHdr, fileSz, prc, readBE32(&fileRec[i].ofstInFile), name)) {
				fprintf(stderr, "Failed to read record 0x%06X\n", ret->rec[i].uniq);
				goto fail;
			}
		}
	}
	
	//success!
	free(fileContents);
	return ret;

fail:
	free(fileContents);
	freePrcPdb(ret);
	return NULL;
}

struct PalmDb *createPrcPdb(const char *name, uint16_t attr, uint16_t ver, uint32_t datCr, uint32_t datMod, uint32_t datBck, uint32_t modNum, uint32_t type, uint32_t crid, uint32_t uniqDbSeed)
{
	struct PalmDb *ret = calloc(1, sizeof(struct PalmDb));
	
	if (ret) {
		strncpy(ret->name, name, sizeof(ret->name) - 1);	//terminator is built-in thanks to calloc
		ret->attributes = attr;
		ret->version = ver;
		ret->creationDate = datCr;
		ret->modificationDate = datMod;
		ret->lastBackupDate = datBck;
		ret->modificationNumber = modNum;
		ret->type = type;
		ret->creator = crid;
		ret->uniqueIDSeed = uniqDbSeed;
		ret->isPrc = !!(attr & PALM_DB_ATTR_RES_DB);
	}
	return ret;
}

void freePrcPdb(struct PalmDb *fil)
{
	uint32_t i;
	
	//free rec/res entries
	for (i = 0; i < fil->numChildren; i++) {
		if (fil->isPrc)
			free(fil->res[i].buf.data);
		else
			free(fil->rec[i].buf.data);
	}
	
	//free sortInfo and appInfo
	free(fil->appInfo.data);
	free(fil->sortInfo.data);
	
	//free struct itself
	free(fil);
}

struct PalmDb *appendRes(struct PalmDb *db, uint32_t type, uint16_t id, struct Buffer* buf)
{
	struct PalmDb *ret;
	
	ret = realloc(db, sizeof(struct PalmDb) + (db->numChildren + 1) * sizeof(struct PalmRes));
	if (ret) {
		
		ret->res = (struct PalmRes*)(ret + 1);
		ret->res[db->numChildren].type = type;
		ret->res[db->numChildren].id = id;
		ret->res[db->numChildren].buf = *buf;
		ret->numChildren++;
	}
	
	return ret;
}

struct PalmDb *appendRec(struct PalmDb *db, uint8_t attr, uint32_t uniqId, struct Buffer* buf)
{
	struct PalmDb *ret;
	
	ret = realloc(db, sizeof(struct PalmDb) + (db->numChildren + 1) * sizeof(struct PalmRec));
	if (ret) {
		
		ret->rec = (struct PalmRec*)(ret + 1);
		ret->rec[db->numChildren].uniq = uniqId;
		ret->rec[db->numChildren].attrs = attr;
		ret->rec[db->numChildren].buf = *buf;
		ret->numChildren++;
	}
	
	return ret;
}

static bool allocateDataChunk(FILE* f, const struct Buffer *buf, uint32_t *curDataPtrP, uint32_t totalData, bool missingAllowed, uint32_t *addrP /* non-null on first pass to get addr to write, null on second pass */)
{
	if (buf->sz) {				//a chunk with size
		
		*addrP = *curDataPtrP;
		(*curDataPtrP) += buf->sz;
	}
	else if (buf->data) {		//zero sized but existing chunk
			
		*addrP = totalData;
	}
	else if (missingAllowed) {	//no chunk but this is ok
		
		*addrP = 0;
	}
	else {						//no chunk and not allowed
		fprintf(stderr, "No chunk where a chunk was expected\n");
		return false;
	}
	return true;
}

static bool writeDataChunk(FILE* f, const struct Buffer *buf)
{
	if (!buf->sz || buf->sz == fwrite(buf->data, 1, buf->sz, f))
		return true;
	
	fprintf(stderr, "Failure to write data chunk\n");
	return false;
}

bool writePrcPdb(const struct PalmDb *db, const char *filename)
{
	uint32_t dataPtr = sizeof(struct PRCFILE_DatabaseHdrType) + (db->isPrc ? sizeof(struct PRCFILE_RsrcEntry) : sizeof(struct PRCFILE_RecEntry)) * db->numChildren;
	uint32_t recListPtr = sizeof(struct PRCFILE_DatabaseHdrType);
	struct PRCFILE_DatabaseHdrType hdr = {0,};
	uint32_t i, addr, totalData;
	bool ret = false;
	FILE *f;
	
	//open the file
	f = fopen(filename, "wb");
	if (!f)
		return false;
	
	//sum up all the "data" bytes
	totalData = db->appInfo.sz + db->sortInfo.sz;
	for (i = 0; i < db->numChildren; i++) {
		if (db->isPrc)
			totalData += db->res[i].buf.sz;
		else
			totalData += db->rec[i].buf.sz;
	}
	
	//add in the header itself
	totalData += sizeof(struct PRCFILE_DatabaseHdrType) + db->numChildren * (db->isPrc ? sizeof(struct PRCFILE_RsrcEntry) : sizeof(struct PRCFILE_RecEntry));
	
	//add in the 2 bytes of zeroes that OS3.5 and below NEED so we produce PRC files they can understand (in case we want ot beam to them)
	totalData += 2;
	dataPtr += 2;
	
	//create header
	strncpy(hdr.name, db->name, sizeof(hdr.name));
	writeBE16(&hdr.attributes, db->attributes);
	writeBE16(&hdr.version, db->version);
	writeBE32(&hdr.creationDate, db->creationDate);
	writeBE32(&hdr.modificationDate, db->modificationDate);
	writeBE32(&hdr.lastBackupDate, db->lastBackupDate);
	writeBE32(&hdr.modificationNumber, db->modificationNumber);
	if (!allocateDataChunk(f, &db->appInfo, &dataPtr, totalData, true, &addr)) {
		fprintf(stderr, "Failure to allocate appInfo\n");
		goto out;
	}
	writeBE32(&hdr.appInfoID_offset, addr);
	if (!allocateDataChunk(f, &db->sortInfo, &dataPtr, totalData, true, &addr)) {
		fprintf(stderr, "Failure to allocate sortInfo\n");
		goto out;
	}
	writeBE32(&hdr.sortInfoID_offset, addr);
	writeBE32(&hdr.type, db->type);
	writeBE32(&hdr.creator, db->creator);
	writeBE32(&hdr.uniqueIDSeed, db->uniqueIDSeed);
	writeBE32(&hdr.recordList.nextRecordListID_ofst, 0);
	writeBE16(&hdr.recordList.numRecords, db->numChildren);
	
	//write header
	if (sizeof(struct PRCFILE_DatabaseHdrType) != fwrite(&hdr, 1, sizeof(struct PRCFILE_DatabaseHdrType), f)) {
		fprintf(stderr, "Failure to write header\n");
		goto out;
	}
	
	//write record/resrouce lists
	if (db->isPrc) {
		struct PRCFILE_RsrcEntry rsrc;
		
		for (i = 0; i < db->numChildren; i++) {
			
			if (!allocateDataChunk(f, &db->res[i].buf, &dataPtr, totalData, false, &addr)) {
				fprintf(stderr, "Failure to allocate res %u\n", i);
				goto out;
			}
			writeBE32(&rsrc.type, db->res[i].type);
			writeBE16(&rsrc.id, db->res[i].id);
			writeBE32(&rsrc.ofstInFile, addr);
			
			if (sizeof(struct PRCFILE_RsrcEntry) != fwrite(&rsrc, 1, sizeof(struct PRCFILE_RsrcEntry), f)) {
				fprintf(stderr, "Failure to write rsrc header %u\n", i);
				goto out;
			}
		}
	}
	else {
		struct PRCFILE_RecEntry rec;
		
		for (i = 0; i < db->numChildren; i++) {
			
			if (!allocateDataChunk(f, &db->rec[i].buf, &dataPtr, totalData, false, &addr)) {
				fprintf(stderr, "Failure to allocate rec %u\n", i);
				goto out;
			}
			rec.attrs = db->rec[i].attrs;
			writeLE24(rec.uniqId, db->rec[i].uniq);	//yes LE not BE
			writeBE32(&rec.ofstInFile, addr);
			
			if (sizeof(struct PRCFILE_RecEntry) != fwrite(&rec, 1, sizeof(struct PRCFILE_RecEntry), f)) {
				fprintf(stderr, "Failure to write rec header %u\n", i);
				goto out;
			}
		}
	}
	
	//write the 2 bytes of zeroes that OS3.5 and below NEED so we produce PRC files they can understand (in case we want ot beam to them)
	fputc(0, f);
	fputc(0, f);
	
	//write data
	if (!writeDataChunk(f, &db->appInfo)) {
		fprintf(stderr, "Failure to write AppInof data %u\n", i);
		goto out;
	}
	
	if (!writeDataChunk(f, &db->sortInfo)) {
		fprintf(stderr, "Failure to write SortInfo data %u\n", i);
		goto out;
	}
	
	if (db->isPrc) {
		struct PRCFILE_RsrcEntry rsrc;
		
		for (i = 0; i < db->numChildren; i++) {
			
			if (!writeDataChunk(f, &db->res[i].buf)) {
				fprintf(stderr, "Failure to write rsrc data %u\n", i);
				goto out;
			}
		}
	}
	else {
		struct PRCFILE_RecEntry rec;
		
		for (i = 0; i < db->numChildren; i++) {
			
			if (!writeDataChunk(f, &db->rec[i].buf)) {
				fprintf(stderr, "Failure to write rec data %u\n", i);
				goto out;
			}
		}
	}
	
	//success!
	ret = true;
	
out:
	fclose(f);
	return ret;
}

