#include <sys/stat.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdint.h>
#include <stddef.h>
#include <errno.h>
#include <ctype.h>
#include <stdio.h>
#include <time.h>
#include "parsePrcPdb.h"
#include "osStructs.h"
#include "util.h"


static const uint32_t saneRomStartSignificantBitsOS5 = 0xFFFE0000UL;
static const uint32_t saneRomStartSignificantBitsOS4 = 0xFFFF8000UL;
static const uint32_t saneRomStartSignificantBitsOS2 = 0xFFFFF000UL;
static bool mIsOS5 = false;
static bool mIsCobalt = false;

static bool mTolerateFlashProImages = false;

enum RomType {
	RomTypeBig,
	RomTypeSmall,
	RomTypeSpringboard,
};

static void show4cc(uint32_t code)
{
	fprintf(stderr, "0x%08lx '%c%c%c%c'", (unsigned long)code,
		(char)((code >> 24) & 0xff), (char)((code >> 16) & 0xff), (char)((code >> 8) & 0xff), (char)((code >> 0) & 0xff));
}

static int getChunkInfoV1(const uint8_t *rom, uint32_t romSz, uint32_t romVa, uint32_t chunkVa, uint32_t *szP, uint8_t *lockCtP, uint8_t *ownerP, bool *freeP)
{
	const struct HeapChunkHeaderV1 *chunkHdr = (const struct HeapChunkHeaderV1*)(rom + chunkVa - romVa - sizeof(struct HeapChunkHeaderV1));
	uint32_t chunkW0, chunkW1, chunkSz, chunkSzExtra;
	
	if (!chunkVa) {
		fprintf(stderr, "Unexpected NULL chunk\n");
		return -99;
	}
	
	if (chunkVa < romVa || chunkVa - romVa < sizeof(struct HeapChunkHeaderV1) || chunkVa > romVa + romSz || (mIsOS5 && (chunkVa & 3)) || (!mIsOS5 && (chunkVa & 1))) {
		fprintf(stderr, "Invalid chunk pointer 0x%08lx\n", (unsigned long)chunkVa);
		return -98;
	}
	
	chunkSz = readSE16(&chunkHdr->size);
	chunkSzExtra = chunkHdr->flagsAndSizeExtra & 0x0F;
	
	if (chunkSz < sizeof(struct HeapChunkHeaderV1) + chunkSzExtra) {
		fprintf(stderr, "chunk sizes do not match up sz=%lu xtra=%u\n", (unsigned long)chunkSz, (unsigned)chunkSzExtra);
		return -97;
	}
	
	if (chunkVa + chunkSz - sizeof(struct HeapChunkHeaderV1) - romVa > romSz + 4 /* SLOP to allow small dals to properly decompose - they do this */) {
		fprintf(stderr, "chunk end is past the end of the heap (0x%08lx + 0x%08lx - 0x%08lx > 0x%08lx)\n",
			(unsigned long)chunkVa, (unsigned long)chunkSz, (unsigned long)romVa, (unsigned long)romSz);
		return -96;
	}
	
	if (szP)
		*szP = chunkSz - chunkSzExtra - sizeof(struct HeapChunkHeaderV1);
	if (lockCtP)
		*lockCtP = chunkHdr->lockCtAndOwner >> 4;
	if (ownerP)
		*ownerP = chunkHdr->lockCtAndOwner & 0x0F;
	if (freeP)
		*freeP = !!(chunkHdr->flagsAndSizeExtra & 0x80);
	
	return 0;
}

static int getChunkInfo(const uint8_t *rom, uint32_t romSz, uint32_t romVa, uint32_t chunkVa, uint32_t *szP, uint8_t *lockCtP, uint8_t *ownerP, bool *freeP)
{
	const struct HeapChunkHeader *chunkHdr = (const struct HeapChunkHeader*)(rom + chunkVa - romVa - sizeof(struct HeapChunkHeader));
	uint32_t chunkW0, chunkW1, chunkSz, chunkSzExtra;
	
	if (!chunkVa) {
		fprintf(stderr, "Unexpected NULL chunk\n");
		return -99;
	}
	
	if (chunkVa < romVa || chunkVa - romVa < sizeof(struct HeapChunkHeader) || chunkVa > romVa + romSz || (mIsOS5 && (chunkVa & 3)) || (!mIsOS5 && (chunkVa & 1))) {
		fprintf(stderr, "Invalid chunk pointer 0x%08lx\n", (unsigned long)chunkVa);
		return -98;
	}
	
	chunkW0 = readSE32(&chunkHdr->w0);
	chunkW1 = readSE32(&chunkHdr->w1);
	chunkSz = mIsCobalt ? ((chunkW0 & OS6_CHUNK_HDR_W0_SIZE_MASK) >> OS6_CHUNK_HDR_W0_SIZE_SHIFT) : ((chunkW0 & CHUNK_HDR_W0_SIZE_MASK) >> CHUNK_HDR_W0_SIZE_SHIFT);
	chunkSzExtra = mIsCobalt ? ((chunkW0 & OS6_CHUNK_HDR_W0_SIZE_XTRA_MASK) >> OS6_CHUNK_HDR_W0_SIZE_XTRA_SHIFT) : ((chunkW0 & CHUNK_HDR_W0_SIZE_XTRA_MASK) >> CHUNK_HDR_W0_SIZE_XTRA_SHIFT);
	
//	fprintf(stderr, "Chunk 0x%08lx has word 0x%08lx 0x%08lx\n", chunkVa, chunkW0, chunkW1);

	if (chunkSz < sizeof(struct HeapChunkHeader) + chunkSzExtra) {
		fprintf(stderr, "chunk sizes do not match up sz=%lu xtra=%u\n", (unsigned long)chunkSz, (unsigned)chunkSzExtra);
		return -97;
	}
	
	if (chunkVa + chunkSz - sizeof(struct HeapChunkHeader) - romVa > romSz + 4 /* SLOP to allow small dals to properly decompose - they do this */) {
		fprintf(stderr, "chunk end is past the end of the heap (0x%08lx + 0x%08lx - 0x%08lx > 0x%08lx)\n",
			(unsigned long)chunkVa, (unsigned long)chunkSz, (unsigned long)romVa, (unsigned long)romSz);
		return -96;
	}
	
	if (szP)
		*szP = chunkSz - chunkSzExtra - sizeof(struct HeapChunkHeader);
	if (lockCtP)
		*lockCtP = mIsCobalt ? ((chunkW1 & OS6_CHUNK_HDR_W1_LOCK_CT_MASK) >> OS6_CHUNK_HDR_W1_LOCK_CT_SHIFT) : ((chunkW1 & CHUNK_HDR_W1_LOCK_CT_MASK) >> CHUNK_HDR_W1_LOCK_CT_SHIFT);
	if (ownerP)
		*ownerP = mIsCobalt ? ((chunkW1 & OS6_CHUNK_HDR_W1_OWNER_MASK) >> OS6_CHUNK_HDR_W1_OWNER_SHIFT) : ((chunkW1 & CHUNK_HDR_W1_OWNER_MASK) >> CHUNK_HDR_W1_OWNER_SHIFT);
	if (freeP)
		*freeP = !!(chunkW0 & CHUNK_HDR_W0_FREE_MASK);
	
	return 0;
}

static int verifyUsedChunkInRom(const uint8_t *rom, uint32_t romSz, uint32_t romVa, bool isV1Heap, uint32_t chunkVa, uint32_t *szP, uint8_t expectedOwner)
{
	uint8_t lockCt, owner;
	bool isFree;
	int ret;
	
	ret = isV1Heap ? getChunkInfoV1(rom, romSz, romVa, chunkVa, szP, &lockCt, &owner, &isFree) : getChunkInfo(rom, romSz, romVa, chunkVa, szP, &lockCt, &owner, &isFree);
	if (ret)
		return ret;

	if (lockCt != CHUNK_LOCK_CT_NONMOVABLE) {
		fprintf(stderr, "Chunk (0x%08lx) lock count is %u\n", (unsigned long)chunkVa, (unsigned)lockCt);
		return -95;
	}
	if (isFree)  {
		fprintf(stderr, "chunk unexpectedly marked as free\n");
		return -94;
	}
	if (!mTolerateFlashProImages) {
		if (owner != expectedOwner) {
			fprintf(stderr, "chunk owner %u is not what we expected (%u)\n", (unsigned)owner, (unsigned)expectedOwner);
			return -93;
		}
	}
	
	return 0;
}

static int readDataChunk(const uint8_t *rom, uint32_t romSz, uint32_t romVa, bool isV1Heap, uint32_t chunkVa, struct Buffer* dst)
{
	uint32_t size;
	int ret;
	
	ret = verifyUsedChunkInRom(rom, romSz, romVa, isV1Heap, chunkVa, &size, mIsCobalt ? OS6_CHUNK_OWNER_DM_REC_AND_RES : CHUNK_OWNER_DM_REC_AND_RES);
	if (ret)
		return ret;

	dst->data = malloc(size);
	if (!dst->data) {
		fprintf(stderr, "Allocation failue reading chunk\n");
		return -90;
	}
	dst->sz = size;
	memcpy(dst->data, rom + chunkVa - romVa, size);
	
	return 0;
}

static int dumpDb(const uint8_t *rom, uint32_t romSz, uint32_t romVa, int romStructVer, bool isV1Heap, uint32_t dbHdrVa)
{
	const struct ResourceListEntryOS5 *resOS5;
	const struct ResourceListEntryOS4 *resOS4;
	const struct DatabaseHeaderOS5 *dbRomOS5;
	const struct DatabaseHeaderOS4 *dbRomOS4;
	const struct DatabaseHeaderCommon *dbRom;
	const struct RecordListEntry *rec;
	uint32_t sz, num, i, t;
	struct PalmDb *dbRam;
	char filename[128];
	struct Buffer buf;
	int ret;
	
	if (!mIsOS5 && !mIsCobalt && romStructVer < 4 && dbHdrVa)
		dbHdrVa = (dbHdrVa & 0x00FFFFFFul) | (romVa & 0xFF000000ul);
	
	dbRomOS5 = (const struct DatabaseHeaderOS5*)(rom + dbHdrVa - romVa);
	dbRomOS4 = (const struct DatabaseHeaderOS4*)(rom + dbHdrVa - romVa);
	dbRom = (const struct DatabaseHeaderCommon*)(rom + dbHdrVa - romVa);
	rec = (const struct RecordListEntry*)(mIsOS5 || mIsCobalt ? (void*)(dbRomOS5 + 1) : (void*)(dbRomOS4 + 1));
	resOS5 = (const struct ResourceListEntryOS5*)(dbRomOS5 + 1) ;
	resOS4 = (const struct ResourceListEntryOS4*)(dbRomOS4 + 1);
	
	ret = verifyUsedChunkInRom(rom, romSz, romVa, isV1Heap, dbHdrVa, &sz, mIsCobalt ? OS6_CHUNK_OWNER_DM_MGMNT_DATA : CHUNK_OWNER_DM_MGMNT_DATA);
	if (ret)
		return ret;
	
	if (sz < (mIsOS5 || mIsCobalt ? sizeof(struct DatabaseHeaderOS5) : sizeof(struct DatabaseHeaderOS4))) {
		fprintf(stderr, "DB hdr chunk is %lu bytes unexpectedly\n", (unsigned long)sz);
		return -50;
	}
	
	t = readSE32(&dbRom->nextRecordListPtr);
	if (t) {
		fprintf(stderr, "DB unexpectedly has a record list linked list: 0x%08lx\n", (unsigned long)t);
		return -51;
	}
	if ((mIsOS5 || mIsCobalt) && readSE16(&dbRomOS5->val0xD2CE) != 0xD2CE) {
		fprintf(stderr, "DB unexpectedly has a non 0xD2CE value in last halfword :0x%04x\n", (unsigned)readSE16(&dbRomOS5->val0xD2CE));
		return -52;
	}
	
	num = readSE16(&dbRom->numChildren);
	t = ((readSE16(&dbRom->attributes) & PALM_DB_ATTR_RES_DB) ? (mIsOS5 || mIsCobalt ? sizeof(struct ResourceListEntryOS5) : sizeof(struct ResourceListEntryOS4)) : sizeof(struct RecordListEntry)) * num + (mIsOS5 || mIsCobalt ? sizeof(struct DatabaseHeaderOS5) : sizeof(struct DatabaseHeaderOS4));
	if (sz < t) {
		fprintf(stderr, "DB hdr chunk is %lu bytes unexpectedly (expected %lu)\n", (unsigned long)sz, (unsigned long)t);
		return -53;
	}
	
	dbRam = createPrcPdb(
			dbRom->name,
			readSE16(&dbRom->attributes),
			readSE16(&dbRom->version),
			readSE32(&dbRom->creationDate),
			readSE32(&dbRom->modificationDate),
			readSE32(&dbRom->lastBackupDate),
			readSE32(&dbRom->modificationNumber),
			readSE32(&dbRom->type),
			readSE32(&dbRom->creator),
			readSE32(&dbRom->uniqueIDSeed));
	if (!dbRam) {
		fprintf(stderr, "Failed to allocate db header\n");
		return -54;
	}
	
	fprintf(stderr, "   DB is '%s' type={", dbRam->name);
	show4cc(dbRam->type);
	fprintf(stderr, "} crid={");
	show4cc(dbRam->creator);
	fprintf(stderr, "}\n");
	
	t = readSE32(&dbRom->appInfoPtr);
	if (t) {
		if (!mIsOS5 && !mIsCobalt && romStructVer < 4)
			t = (t & 0x00FFFFFFul) | (romVa & 0xFF000000ul);
		
		ret = readDataChunk(rom, romSz, romVa, isV1Heap, t, &dbRam->appInfo);
		if (ret) {
			fprintf(stderr, "Cannot read appinfo\n");
			goto out;
		}
	}
	
	t = readSE32(&dbRom->sortInfoPtr);
	if (t) {
		if (!mIsOS5 && !mIsCobalt && romStructVer < 4)
			t = (t & 0x00FFFFFFul) | (romVa & 0xFF000000ul);
		
		ret = readDataChunk(rom, romSz, romVa, isV1Heap, t, &dbRam->sortInfo);
		if (ret) {
			fprintf(stderr, "Cannot read sortinfo\n");
			goto out;
		}
	}
	
	for (i = 0; i < num; i++) {
		struct PalmDb *tmp;
		
		if (dbRam->isPrc) {
			uint32_t dataVa = readSE32(mIsOS5 || mIsCobalt ? &resOS5[i].dataPtr : &resOS4[i].dataPtr);
			uint32_t type = readSE32(mIsOS5 || mIsCobalt ? &resOS5[i].type : &resOS4[i].type);
			uint32_t id = readSE16(mIsOS5 || mIsCobalt ? &resOS5[i].id : &resOS4[i].id);
			
			if (!mIsOS5 && !mIsCobalt && romStructVer < 4 && dataVa)
				dataVa = (dataVa & 0x00FFFFFFul) | (romVa & 0xFF000000ul);
		
			fprintf(stderr, "    Resource (");
			show4cc(type);
			fprintf(stderr, ", %u) has data pointer 0x%08lx\n", (unsigned)id, (unsigned long)dataVa);
				
			if (!dataVa) {
				fprintf(stderr, "Resource has NULL data pointer\n");
				ret = -55;
				goto out;
			}
			
			ret = readDataChunk(rom, romSz, romVa, isV1Heap, dataVa, &buf);
			if (ret) {
				fprintf(stderr, "Cannot read resource\n");
				ret = -56;
				goto out;
			}
			
			tmp = appendRes(dbRam, type, id, &buf);
			if (!tmp) {
				fprintf(stderr, "Cannot read resource\n");
				ret = -57;
				goto out;
			}
			
			dbRam = tmp;
		}
		else {
			uint32_t dataVa = readSE32(&rec[i].dataPtr);
			uint32_t uniqId = readLE24(rec[i].uniqId);		//always LE
			uint8_t attr = rec[i].attr;
			
			if (!mIsOS5 && !mIsCobalt && romStructVer < 4 && dataVa)
				dataVa = (dataVa & 0x00FFFFFFul) | (romVa & 0xFF000000ul);
		
			ret = readDataChunk(rom, romSz, romVa, isV1Heap, dataVa, &buf);
			if (ret) {
				fprintf(stderr, "Cannot read record\n");
				ret = -58;
				goto out;
			}
			
			tmp = appendRec(dbRam, attr, uniqId, &buf);
			if (!tmp) {
				fprintf(stderr, "Cannot read record\n");
				ret = -59;
				goto out;
			}
			
			dbRam = tmp;
		}
	}
	
	//now create the filename
	for (i = 0, filename[0] = 0; i < strlen(dbRam->name); i++) {
		char ch = dbRam->name[i];
		if (!ch || isalnum(ch) || isblank(ch) || ch == '_' || ch == '-' || ch == '.')
			sprintf(filename + strlen(filename), "%c", ch);
		else
			sprintf(filename + strlen(filename), "%%%02x", ch);
	}
	strcat(filename, dbRam->isPrc ? ".prc" : ".pdb");
	fprintf(stderr, "   Filename will be '%s'\n", filename);
	
	//now dump the file
	if (!writePrcPdb(dbRam, filename)) {
		fprintf(stderr, "Failed to write out the file\n");
		ret = -60;
		goto out;
	}
	
	ret = 0;
	
out:
	freePrcPdb(dbRam);
	return ret;
}


static int dumpRom(const uint8_t *rom, uint32_t bytes, uint32_t va, enum RomType romType)
{
	uint32_t t, tokenStore = 0, tokensMaxSz = 0, romCardSz, code4cc, storeOfst, numDbs;
	uint32_t allegedBootCodeRes, allegedBootDataRes, allegedBootInfoRes;
	uint32_t allegedDalCodeRes, allegedDalDataRes, allegedDalInfoRes;
	const struct CardHeader *ch = (const struct CardHeader*)rom;
	bool haveRomTokensInOurImage, isV1Heap = false;
	uint32_t allegedBootScreenPtrs[3] = {0,};
	const struct DatabaseListOS5 *dbListOS5;
	const struct DatabaseListOS4 *dbListOS4;
	const struct HeapListOS5 *heapListOS5;
	const struct HeapListOS4 *heapListOS4;
	const struct HeapHeaderV1 *heapV1;
	const struct StoreHeader *sh;
	char romStr[33] = {0,};
	int ret, romStructVer;
	time_t time;
	
	if (mIsCobalt) {
		
		ch = NULL;		//to catch st
		romStructVer = 7;	//why not
		tokenStore = 0;
		tokensMaxSz = 0;
		haveRomTokensInOurImage = false;
	}
	else {
		if (bytes < sizeof(struct CardHeader)) {
			fprintf(stderr, "ROM is too small (%u)\n", bytes);
			return -5;
		}
		
		if (readSE32(&ch->signature) != CARD_HEADER_SIG) {
			fprintf(stderr, "Card signature is invalid: 0x%08lx != 0x%08lx\n", (unsigned long)readSE32(&ch->signature), (unsigned long)CARD_HEADER_SIG);
			return -6;
		}
		
		fprintf(stderr, " Card header (@va 0x%08x):\n", va);
		
		if (mIsOS5) {
			
			if((readSE32(&ch->entry.os5.jumpToCode) & 0xFF000000) != 0xEA000000) {
				fprintf(stderr, "OS5 Card jump instr is invalid: 0x%08lx\n", (unsigned long)readSE32(&ch->entry.os5.jumpToCode));
				return -7;
			}
			allegedDalCodeRes = 8 * (0x00FFFFFF & readSE32(&ch->entry.os5.jumpToCode)) + offsetof(struct CardHeader, entry.os5.jumpToCode) + 8 + va;
			if (readSE32(&ch->entry.os5.zeroNum0)) {
				fprintf(stderr, "Unexpected non-zero value in field 'zeroNum0': 0x%08lx\n", (unsigned long)readSE32(&ch->entry.os5.zeroNum0));
				return -9;
			}
		}
		else {
			allegedDalCodeRes = readSE32(&ch->entry.os4.initialPc);
		}
		
		
		fprintf(stderr, "  DAL code begins at 0x%08lx\n", (unsigned long)allegedDalCodeRes);
		
		romStructVer = readSE16(&ch->hdrVersion);
		fprintf(stderr, "  Card header is version %u\n", (unsigned)romStructVer);
		if (romStructVer < 1 || romStructVer > 6)
			return -8;
		
		if (mIsOS5 && romStructVer < 5) {
			fprintf(stderr, "OS5 does not support such old card headers\n");
			return -8;
		}
		else if (!mIsOS5&& romStructVer > 5) {
			fprintf(stderr, "OS4- does not support such new card headers\n");
			return -8;
		}
		
		strncpy(romStr, ch->name, 32);
		fprintf(stderr, "  ROM name: '%s'\n", romStr);
		strncpy(romStr, ch->manuf, 32);
		fprintf(stderr, "  ROM manufacturer: '%s'\n", romStr);
		if (mIsOS5)
			strncpy(romStr, ch->diff2.os5.romVersionString, 32);
		else
			strncpy(romStr, ch->diff2.os4.romVersionString, 32);
		fprintf(stderr, "  ROM header checksum: 0x%08lx\n", (unsigned long)(mIsOS5 ? readSE32(&ch->diff2.os5.romChecksum) : readSE16(&ch->diff2.os4.romChecksum)));
		fprintf(stderr, "  ROM flags: 0x%04x\n", (unsigned)readSE16(&ch->flags));
		fprintf(stderr, "  ROM card hdr ver: 0x%04x\n", (unsigned)readSE16(mIsOS5 ? &ch->diff1.os5.version : &ch->diff1.os4.version));
		fprintf(stderr, "  ROM num ram blocks: 0x%04x\n", (unsigned)readSE16(mIsOS5 ? &ch->diff1.os5.numRAMBlocks : &ch->diff1.os4.numRAMBlocks));
		if (romStructVer >= 2) {
			
			t = mIsOS5 ? readSE32(&ch->diff1.os5.creationDate) : readSE32(&ch->diff1.os4.creationDate);
			time = t - PALMOS_TIME_OFFSET;
			fprintf(stderr, "  ROM creation date: 0x%08lx => %s", (unsigned long)t, ctime(&time));
		
			if (readSE32(&ch->readWriteParmsOffset) && mIsOS5) {	//not expected in OS5
				fprintf(stderr, "Unexpected non-zero value in field 'readWriteParmsOffset': 0x%08lx\n", (unsigned long)readSE32(&ch->readWriteParmsOffset));
				return -9;
			}
			if (readSE32(&ch->readWriteParmsSize) && mIsOS5) {	//not expected in OS5
				fprintf(stderr, "Unexpected non-zero value in field 'readWriteParmsSize': 0x%08lx\n", (unsigned long)readSE32(&ch->readWriteParmsSize));
				return -9;
			}
			t = readSE32(mIsOS5 ? &ch->diff2.os5.readWriteWorkingOffset : &ch->diff2.os4.readWriteWorkingOffset);
			if (t && mIsOS5) {					//not expected in OS5
				fprintf(stderr, "Unexpected non-zero value in field 'readWriteWorkingOffset': 0x%08lx\n", (unsigned long)t);
				return -9;
			}
			t = readSE32(mIsOS5 ? &ch->diff2.os5.readWriteWorkingSize : &ch->diff2.os4.readWriteWorkingSize);
			if (t && mIsOS5) {					//not expected in OS5
				fprintf(stderr, "Unexpected non-zero value in field 'readWriteWorkingSize': 0x%08lx\n", (unsigned long)t);
				return -9;
			}
			
			tokenStore = (unsigned long)readSE32(&ch->romTokenStorePtr);
			tokensMaxSz = bytes - (tokenStore - va);
			haveRomTokensInOurImage = tokenStore >= va && tokenStore < va + bytes;
			fprintf(stderr, "  ROM token store is at 0x%08lx (it is %sa part of this ROM section)\n",
				(unsigned long)tokenStore, haveRomTokensInOurImage ? "" : "not ");
			
			if ((mIsOS5 && (tokenStore & 3)) || (!mIsOS5 && (tokenStore & 1))) {
				fprintf(stderr, "Token store pointer misaligned: 0x%08lx\n", (unsigned long)tokenStore);
				return -10;
			}
			
			romCardSz = readSE32(&ch->totalRomSz);
			fprintf(stderr, "  ROM claimed size: 0x%08lx bytes\n", (unsigned long)romCardSz);
			if (romCardSz > bytes) {
				fprintf(stderr, "ROM total size goes beyond this section 0x%08lx > 0x%08lx\n", (unsigned long)romCardSz, (unsigned long)bytes);
				return -11;
			}
		
			if (!mTolerateFlashProImages && romType != RomTypeSpringboard) {
				//shrink the ROM bytes we're considering down to the size we were just told about
				//flashpro messes this up and we need to handle that. Also pringboard roms all
				//somehow claim to be 4K in size
				if (romStructVer >= 4)		//in v3 we've seen this be too small
					bytes = romCardSz;
			}
		}
		
		if (romType != RomTypeSpringboard && (allegedDalCodeRes < va || allegedDalCodeRes >= va + bytes || (mIsOS5 && (allegedDalCodeRes & 3)) || (!mIsOS5 && (allegedDalCodeRes & 1)))) {
			fprintf(stderr, "DAL code pointer 0x%08lx is invalid\n", (unsigned long)allegedDalCodeRes);
			return -12;
		}
		
		if (romStructVer >= 4) {
			t = readSE32(mIsOS5 ? &ch->diff2.os5.dalAmddRsrcOfst : &ch->diff2.os4.halCodeOffset);
			allegedDalDataRes = t + va;
			fprintf(stderr, "  DAL data begins at offset 0x%08lx (va 0x%08lx)\n", (unsigned long)t, (unsigned long)allegedDalDataRes);
			if (t > bytes) {
				fprintf(stderr, "DAL data offset 0x%08lx is invalid\n", (unsigned long)t);
				return -13;
			}
		}
		
		if (romStructVer >= 5) {
			
			if (mIsOS5) {
				t = readSE32(&ch->diff2.os5.dalAmdiRsrcOfst);
				allegedDalInfoRes = t + va;
				fprintf(stderr, "  DAL info begins at offset 0x%08lx (va 0x%08lx)\n", (unsigned long)t, (unsigned long)allegedDalInfoRes);
				if (t > bytes) {
					fprintf(stderr, "DAL info offset 0x%08lx is invalid\n", (unsigned long)t);
					return -14;
				}
			}
			
			fprintf(stderr, "  ROM Company ID: ");
			show4cc(readSE32(mIsOS5 ? &ch->diff2.os5.companyID : &ch->diff2.os4.companyID));
			fprintf(stderr, "\n");
			fprintf(stderr, "  ROM HAL ID: ");
			show4cc(readSE32(mIsOS5 ? &ch->diff2.os5.halID : &ch->diff2.os4.halID));
			fprintf(stderr, "\n");
			
			fprintf(stderr, "  ROM version str: '%s'\n", romStr);
			fprintf(stderr, "  ROM version num: 0x%08lx\n", (unsigned long)readSE32(mIsOS5 ? &ch->diff2.os5.romVersion : &ch->diff2.os4.romVersion));
		}
		
		if (romStructVer >= 2) {
			t = readSE32(&ch->bigRomPtr);
			if (!mIsOS5 && romStructVer < 4)
				t = (t & 0x00FFFFFFul) | (va & 0xFF000000ul);
			fprintf(stderr, "  BigRom VA: 0x%08lx\n", (unsigned long)t);
			
			if (romType == RomTypeSmall) {
				if (t >= va && t < va + bytes) {
					fprintf(stderr, "BigROM cannot be part of SmallROM\n");
					return -15;
				}
			}
			else if (romType == RomTypeBig) {
				if (t != va) {
					fprintf(stderr, "BigROM's VA does not reference itself\n");
					return -15;
				}
			}
		}
	}
	
	for (storeOfst = sizeof(struct CardHeader); storeOfst < 8192 && storeOfst < bytes; storeOfst++) {
		
		sh = (const struct StoreHeader*)(rom + storeOfst);
		if (readSE32(&sh->signature) == STORE_HEADER_SIG)
			goto storeFound;
	}
	
	fprintf(stderr, "Store not found in rom\n");
	return -16;
	
storeFound:
	fprintf(stderr, " Store header (found at va 0x%08lx):\n", (unsigned long)(storeOfst + va));
	fprintf(stderr, "  Store flags: 0x%04x\n", (unsigned)readSE16(&sh->flags));
	fprintf(stderr, "  Store version: 0x%04x\n", (unsigned)readSE16(&sh->version));
	strncpy(romStr, sh->name, 32);
	fprintf(stderr, "  Store name: '%s'\n", romStr);

	if (readSE32(&sh->zeroNum0)) {
		//XXX: Handera 330c has a nonzero number here
		fprintf(stderr, "Unexpected non-zero value in field 'zeroNum0': 0x%08lx\n", (unsigned long)readSE32(&sh->zeroNum0));
		return -17;
	}
	if (readSE32(&sh->zeroNum1)) {
		fprintf(stderr, "Unexpected non-zero value in field 'zeroNum1': 0x%08lx\n", (unsigned long)readSE32(&sh->zeroNum1));
		return -17;
	}
	if (readSE32(&sh->zeroNum2) && !mIsCobalt) {
		fprintf(stderr, "Unexpected non-zero value in field 'zeroNum2': 0x%08lx\n", (unsigned long)readSE32(&sh->zeroNum2));
		return -17;
	}
	if (readSE32(&sh->zeroNum3)) {
		fprintf(stderr, "Unexpected non-zero value in field 'zeroNum3': 0x%08lx\n", (unsigned long)readSE32(&sh->zeroNum3));
		return -17;
	}
	if (readSE32(&sh->crcUnused)) {
		fprintf(stderr, "Unexpected non-zero value in field 'crcUnused': 0x%08lx\n", (unsigned long)readSE32(&sh->crcUnused));
		return -17;
	}
	for (t = 0; t < sizeof(sh->zeros1) / sizeof(*sh->zeros1); t++) {
		
		if (mIsCobalt && !t)	//first word used in cobalt
			continue;
		
		if (readSE32(&sh->zeros1[t])) {
			fprintf(stderr, "Unexpected non-zero value in field 'zeros1[%u]': 0x%08lx\n", t, (unsigned long)readSE32(&sh->zeros1[t]));
			return -17;
		}
	}
	for (t = 0; t < sizeof(sh->zeros2) / sizeof(*sh->zeros2); t++) {
		
		if (mIsCobalt && t < 2)	//first 2 words used in cobalt and point somewhere...
			continue;
		
		if (readSE32(&sh->zeros2[t])) {
			fprintf(stderr, "Unexpected non-zero value in field 'zeros2[%u]': 0x%08lx\n", t, (unsigned long)readSE32(&sh->zeros2[t]));
			return -17;
		}
	}
	
	if (romStructVer >= 3) {
		
		uint16_t lang, country;
		
		lang = readSE16(mIsOS5 ? &sh->os5.romLanguage : &sh->os4.romLanguage);
		country = readSE16(mIsOS5 ? &sh->os5.romCountry : &sh->os4.romCountry);
		
		if (mIsCobalt) {
			fprintf(stderr, "  Language ID: %c%c\n", lang >> 8, lang & 0xff);
			fprintf(stderr, "  Country ID: %c%c\n", country >> 8, country & 0xff);
		}
		else {
			fprintf(stderr, "  Language ID: %u\n", lang);
			fprintf(stderr, "  Country ID: %u\n", country);
		}
	}
	
	t = readSE32(&sh->heapListPtr);
	if (!mIsOS5 && romStructVer < 4 && t)
		t = (t & 0x00FFFFFFul) | (va & 0xFF000000ul);
		
	fprintf(stderr, "  Heap List: 0x%08x\n", t);
	if (t < va || t >= va + bytes) {
		fprintf(stderr, "Heap list pointer is not in range: 0x%08lx\n", (unsigned long)t);
		return -19;
	}
	heapListOS5 = (const struct HeapListOS5*)(rom + t - va);
	heapListOS4 = (const struct HeapListOS4*)(rom + t - va);
	
	fprintf(stderr, "  Num Heaps: %u\n", mIsOS5 || mIsCobalt ? readSE16(&heapListOS5->numItems) : readSE16(&heapListOS4->numItems));
	t = mIsOS5 || mIsCobalt ? readSE32(&heapListOS5->heapHeaderPtrs[0]) : readSE32(&heapListOS4->heapHeaderPtrs[0]);
	if (!mIsOS5 && romStructVer < 4 && t)
		t = (t & 0x00FFFFFFul) | (va & 0xFF000000ul);
	if (t < va || t >= va + bytes) {
		fprintf(stderr, "Heap pointer is not in range: 0x%08lx\n", (unsigned long)t);
		return -20;
	}
	heapV1 = (const struct HeapHeaderV1*)(rom + t - va);
	//check if really v1 heap (diff chunk header then)
	t = readSE16(&heapV1->flags);
	fprintf(stderr, "  ROM heap flags 0x%04lx ", (unsigned long)t);
	if (t & HEAP_FLAGS_IS_V6)
		fprintf(stderr, "(V6)\n");
	else if (t & HEAP_FLAGS_IS_V4)
		fprintf(stderr, "(V4)\n");
	else if (t & HEAP_FLAGS_IS_V3)
		fprintf(stderr, "(V3)\n");
	else if (t & HEAP_FLAGS_IS_V2)
		fprintf(stderr, "(V2)\n");
	else {
		fprintf(stderr, "(V1)\n");
		isV1Heap = true;
	}
	
	t = readSE32(&sh->databaseListPtr);
	if (!mIsOS5 && !mIsCobalt && romStructVer < 4 && t)
		t = (t & 0x00FFFFFFul) | (va & 0xFF000000ul);
	fprintf(stderr, "  ROM database list is at 0x%08lx\n", (unsigned long)t);
	if (t < va || t - va >= bytes) {
		fprintf(stderr, "Database list pointer is not in range: 0x%08lx (expected 0x%08x...0x%08x)\n", (unsigned long)t, va, va + bytes - 1);
		return -21;
	}
	dbListOS4 = (const struct DatabaseListOS4*)(rom + t - va);
	dbListOS5 = (const struct DatabaseListOS5*)(rom + t - va);
	
	t = readSE32(mIsOS5 || mIsCobalt ? &dbListOS5->nextDatabaseDir : &dbListOS4->nextDatabaseDir);
	if (t) {
		fprintf(stderr, "Database list has a continuation: 0x%08lx. This is technically valid, but not supported by this util\n", (unsigned long)t);
		return -22;
	}
	
	numDbs = mIsOS5 || mIsCobalt ? readSE32(&dbListOS5->numDatabases) : readSE16(&dbListOS4->numDatabases);
	if (!numDbs) {
		fprintf(stderr, "Database list has no items. This is technically valid, but unexpected in a ROM\n");
		return -23;
	}
	fprintf(stderr, "  ROM contains %lu database%s\n", (unsigned long)numDbs, (numDbs == 1 ? "" : "s"));
	
	if (romStructVer >= 3) {
		allegedBootDataRes = readSE32(&sh->bootRsrsBoot10001);
		if (!mIsOS5 && romStructVer < 4 && allegedBootDataRes)
			allegedBootDataRes = (allegedBootDataRes & 0x00FFFFFFul) | (va & 0xFF000000ul);
		if (romType == RomTypeSmall && allegedBootDataRes) {
			fprintf(stderr, "SmallROM has nonzero Boot data reference: 0x%08lx\n", (long)allegedBootDataRes);
		}
		else if (romType == RomTypeBig && !allegedBootDataRes && !mIsCobalt) {
			fprintf(stderr, "BigROM has zero Boot data reference\n");
		}
		if (allegedBootDataRes && (allegedBootDataRes < va || allegedBootDataRes >= va + bytes)) {
			fprintf(stderr, " Boot data [Res ('boot' 10001)] pointer is not in range: 0x%08lx\n", (unsigned long)allegedBootDataRes);
		}
		fprintf(stderr, "  Boot data begins at va 0x%08lx\n", (unsigned long)allegedBootDataRes);
		
		allegedBootInfoRes = readSE32(&sh->bootRsrsBoot10002);
		if (!mIsOS5 && romStructVer < 4 && allegedBootInfoRes )
			allegedBootInfoRes = (allegedBootInfoRes & 0x00FFFFFFul) | (va & 0xFF000000ul);
		if (romType == RomTypeSmall && allegedBootInfoRes) {
			fprintf(stderr, "SmallROM has nonzero Boot info reference: 0x%08lx\n", (long)allegedBootInfoRes);
		}
		else if (romType == RomTypeBig && !allegedBootInfoRes && !mIsCobalt) {
			fprintf(stderr, "BigROM has zero Boot info reference\n");
		}
		if (allegedBootInfoRes && (allegedBootInfoRes < va || allegedBootInfoRes >= va + bytes)) {
			fprintf(stderr, " Boot info [Res ('boot' 10002)] pointer is not in range: 0x%08lx\n", (unsigned long)allegedBootInfoRes);
		}
		fprintf(stderr, "  Boot info begins at va 0x%08lx\n", (unsigned long)allegedBootInfoRes);
	
		allegedBootCodeRes = readSE32(&sh->bootBootRsrcPtr);
		if (!mIsOS5 && romStructVer < 4 && allegedBootCodeRes)
			allegedBootCodeRes = (allegedBootCodeRes & 0x00FFFFFFul) | (va & 0xFF000000ul);
		if (romType == RomTypeSmall && allegedBootCodeRes) {
			fprintf(stderr, "SmallROM has nonzero Boot code reference: 0x%08lx\n", (long)allegedBootCodeRes);
		}
		else if (romType == RomTypeBig && !allegedBootCodeRes && !mIsCobalt) {
			fprintf(stderr, "BigROM has zero Boot code reference\n");
		}
		if (allegedBootCodeRes && (allegedBootCodeRes < va || allegedBootCodeRes >= va + bytes)) {
			fprintf(stderr, " Boot code [Res ('boot' 10003)] pointer is not in range: 0x%08lx\n", (unsigned long)allegedBootCodeRes);
		}
		fprintf(stderr, "  Boot code begins at va 0x%08lx\n", (unsigned long)allegedBootCodeRes);
	}
	
	if (romStructVer >= 4) {
		allegedBootScreenPtrs[0] = readSE32(mIsOS5 ? &sh->os5.bootScreen0ptr : &sh->os4.bootScreen0ptr);
		if (romType == RomTypeSmall && allegedBootScreenPtrs[0]) {
			fprintf(stderr, "SmallROM has nonzero Boot screen\n");
		}
		else if (romType == RomTypeBig && !allegedBootScreenPtrs[0] && !mIsCobalt) {
			fprintf(stderr, "BigROM has zero Boot screen\n");
		}
		if (allegedBootScreenPtrs[0] && (allegedBootScreenPtrs[0] < va || allegedBootScreenPtrs[0] >= va + bytes)) {
			fprintf(stderr, " Boot screen [Res ('absb' 19000)] pointer is not in range: 0x%08lx\n", (unsigned long)allegedBootScreenPtrs[0]);
		}
		fprintf(stderr, "  Boot screen 0 begins at va 0x%08lx\n", (unsigned long)allegedBootScreenPtrs[0]);
		
		allegedBootScreenPtrs[1] = readSE32(mIsOS5 ? &sh->os5.bootScreen1ptr : &sh->os4.bootScreen1ptr);
		if (romType == RomTypeSmall && allegedBootScreenPtrs[1]) {
			fprintf(stderr, "SmallROM has nonzero Boot screen\n");
		}
		else if (romType == RomTypeBig && !allegedBootScreenPtrs[1] && !mIsCobalt) {
			fprintf(stderr, "BigROM has zero Boot screen\n");
		}
		if (allegedBootScreenPtrs[1] && (allegedBootScreenPtrs[1] < va || allegedBootScreenPtrs[1] >= va + bytes)) {
			fprintf(stderr, " Boot screen [Res ('absb' 19001)] pointer is not in range: 0x%08lx\n", (unsigned long)allegedBootScreenPtrs[1]);
		}
		fprintf(stderr, "  Boot screen 1 begins at va 0x%08lx\n", (unsigned long)allegedBootScreenPtrs[1]);
	
		if (mIsOS5 || mIsCobalt) {
			allegedBootScreenPtrs[2] = readSE32(&sh->bootScreen2ptr);
			if (romType == RomTypeSmall && allegedBootScreenPtrs[2]) {
				fprintf(stderr, "SmallROM has nonzero Boot screen\n");
			}
			else if (romType == RomTypeBig && !allegedBootScreenPtrs[2] && !mIsCobalt) {
				fprintf(stderr, "BigROM has zero Boot screen\n");
			}
			if (allegedBootScreenPtrs[2] && (allegedBootScreenPtrs[2] < va || allegedBootScreenPtrs[2] >= va + bytes)) {
				fprintf(stderr, " Boot screen [Res ('absb' 19002)] pointer is not in range: 0x%08lx\n", (unsigned long)allegedBootScreenPtrs[2]);
			}
			fprintf(stderr, "  Boot screen 2 begins at va 0x%08lx\n", (unsigned long)allegedBootScreenPtrs[2]);
		}
	}
	
	//we have dbList left to process
	for (t = 0; t < numDbs; t++) {
		fprintf(stderr, "  Dumping file %u/%u\n", t + 1, numDbs);
		ret = dumpDb(rom, bytes, va, romStructVer, isV1Heap, readSE32(mIsOS5 || mIsCobalt ? &dbListOS5->databaseHdr[t] : &dbListOS4->databaseHdr[t]));
		if (ret)
			return ret;
	}
	fprintf(stderr, "  ROM dumped\n");
	
	return 0;
}

static int dumpRomToFolder(const uint8_t *rom, uint32_t bytes, uint32_t va, enum RomType romType, const char *dir)
{
	int ret, chdirRet;
	
	if (mkdir(dir, 0755)) {
		fprintf(stderr, "Failed to create directry '%s': %s\n", dir, strerror(errno));
		return -1;
	}
	
	if (chdir(dir)) {
		fprintf(stderr, "Failed to change to the directry '%s': %s\n", dir, strerror(errno));
		return -2;
	}
	
	ret = dumpRom(rom, bytes, va, romType);
	
	chdirRet = chdir("..");
	if (chdirRet) {
		fprintf(stderr, "Failed to change out of the directry '%s': %s\n", dir, strerror(errno));
		return -3;
	}
	
	if (!ret && chdirRet)
		ret = -4;
	
	return ret;
}

int main(int argc, char** argv)
{
	const unsigned long maxCardSearchOffset = 8192;
	long romSz = 0, romUsedSz = 0, i, maxCardSearch = maxCardSearchOffset;
	bool found = false, springboard = false;
	const struct CardHeaderCobalt *chCob;
	uint32_t suggestedBaseVa, bigRomVa;
	const struct CardHeader *ch;
	int c, ret, prospectiveVer;
	uint8_t *rom = NULL;
	
	
	if (argc != 2) {
		fprintf(stderr, "USAGE: %s outdir < rom.bin\n", argv[0]);
		return -1;
	}
	
	if (mkdir(argv[1], 0755)) {
		fprintf(stderr, "Failed to create output directry '%s': %s\n", argv[1], strerror(errno));
		return -2;
	}
	
	if (chdir(argv[1])) {
		fprintf(stderr, "Failed to change to the main output directry '%s': %s\n", argv[1], strerror(errno));
		return -3;
	}
	
	while ((c = getchar()) != EOF) {
		
		if (romUsedSz == romSz) {
			long newSz = romUsedSz * 9 / 8 + 1;
			void *tmp = realloc(rom, newSz);
			if (!tmp) {
				fprintf(stderr, "Failed to reallocate rom buffer to %lu bytes\n", newSz);
				ret = -4;
				goto out;
			}
			rom = tmp;
			romSz = newSz;
		}
		rom[romUsedSz++] = c;
	}
	
	fprintf(stderr, "Read %lu bytes of ROM\n", romUsedSz);
	
	if (romUsedSz < 0x200) {
		fprintf(stderr, "Read ROM is too small\n");
		ret = -5;
		goto out;
	}
	
	//there may be a header before the first rom (like on some NVFS devices) - skip it if it exists. search a limited number of options before giving up
	//as far as addresses are converned, this header acts as if it does not exist. That is ROM will point to 0x20000100 and in the PA0 dump this data
	//will be at 0x00000160 is the header was 0x60 bytes big. That being said, the offset to the big rom is real
	for (i = 0; i < maxCardSearch && i < romUsedSz; i++) {
		chCob = (const struct CardHeaderCobalt*)(rom + i);
		ch = (const struct CardHeader*)(rom + i);
		
		if (readLE32(&ch->signature) == CARD_HEADER_SIG || readBE32(&ch->signature) == CARD_HEADER_SIG) {
			found = true;
			break;
		}
		
		if (readLE32(&chCob->signature) == CARD_HEADER_COBALT_SMALLROM) {
			
			i = readLE32(&chCob->bigRomSearchStartOffset) - 1 /* make up for our "i++" up top */;
			maxCardSearch = i + maxCardSearchOffset;
			
			fprintf(stderr, "Found a cobalt smallrom. skipping it as it has no files - it is a flat binary\n");
			fprintf(stderr, "ROM suggest we resume search at 0x%08x\n", (unsigned)i);
			continue;
		}
		
		if (readLE32(&chCob->signature) == CARD_HEADER_COBALT_BIGROM) {
			fprintf(stderr, "Potentially found a cobalt image at offset 0x%08lx\n", i);
			mIsCobalt = true;
			seSetLittleEndian(true);
			found = true;
			break;
		}
	}
	
	if (!found) {
		fprintf(stderr, "Failed to find a card header in the first %lu bytes\n", maxCardSearch);
		ret = -6;
		goto out;
	}
	fprintf(stderr, "Found card header at offset 0x%lx\n", i);

	if (mIsCobalt) {
		uint32_t ver = readSE32(&chCob->osVersionBcd);
		char romVer[33] = {0,};
		char romOSVer[33] = {0,};
		
		memcpy(romOSVer, chCob->osVersionString, 32);
		memcpy(romVer, chCob->romVersionString, 32);
		
		fprintf(stderr, "Cobalt OS version: %u.%u.%u (%s)\n", (ver >> 24) & 0xff, (ver >> 20) & 0x0f, (ver >> 16) & 0x0f, romOSVer);
		fprintf(stderr, "ROM ver: '%s'\n", romVer);
		
		fprintf(stderr, "ROM type: ");
		show4cc(readSE32(&chCob->romType));
		fprintf(stderr, "\n");
		
		fprintf(stderr, "ROM crid: ");
		show4cc(readSE32(&chCob->romCrid));
		fprintf(stderr, "\n");
		
		fprintf(stderr, "ROM OS crid: ");
		show4cc(readSE32(&chCob->osCrid));
		fprintf(stderr, "\n");
		
		fprintf(stderr, "ROM base addr: 0x%08lx\n", (long)readSE32(&chCob->romBase));
		
		ret = dumpRom(rom + i, romUsedSz - i, readSE32(&chCob->romBase), false);
	}
	else {
		//sort out our endianness
		if (readLE32(&ch->signature) == CARD_HEADER_SIG) {
			fprintf(stderr, "OS5+ LE image found\n");
			seSetLittleEndian(true);
			mIsOS5 = true;
		}
		else {
			fprintf(stderr, "OS4- BE image found\n");
			seSetLittleEndian(false);
		}
		
		prospectiveVer = readSE16(&ch->hdrVersion);
		
		if (mIsOS5)
			suggestedBaseVa = readSE32(&ch->blockListPtr) & saneRomStartSignificantBitsOS5;
		else {
			suggestedBaseVa = readSE32(&ch->entry.os4.initialPc);
			if (prospectiveVer < 3)
				suggestedBaseVa &= saneRomStartSignificantBitsOS2;
			else
				suggestedBaseVa &= saneRomStartSignificantBitsOS4;
		}
		fprintf(stderr, "Likely ROM base VA: 0x%08lx\n", (unsigned long)suggestedBaseVa);
		
		//now guess at the VA of the ROM (we can only do this in V2+ as before this BigRomPtr doesnt exist)
		if (prospectiveVer == 1) {
			//offset if 0x3000 ot bigrom. see if our size is that big and see if we find a header there
			
			uint32_t offset;
			
			bigRomVa = 0;
			for (offset = 0x400; offset <= 0x4000; offset += 0x400) {
				
				const struct CardHeader *maybeBigRom;
				maybeBigRom = (const struct CardHeader*)(((const uint8_t*)ch) + offset);
				if (readSE32(&maybeBigRom->signature) == CARD_HEADER_SIG) {
					bigRomVa = suggestedBaseVa + offset;
					fprintf(stderr, "V1romLogic: found bigrom at offset 0x%08lx\n", (long)offset);
					break;
				}
			}
			
			if (!bigRomVa) {
				fprintf(stderr, "Best guess at big rom finding failed for V1 image\n");
				bigRomVa = suggestedBaseVa;	//not found
			}
			
		}
		else if (prospectiveVer >= 2) {
			bigRomVa = readSE32(&ch->bigRomPtr);
			if (!mIsOS5 && prospectiveVer < 4)
				bigRomVa = (bigRomVa & 0x00FFFFFFul) | (suggestedBaseVa & 0xFF000000ul);
			
			fprintf(stderr, "Claimed BigROM VA: 0x%08lx\n", (unsigned long)bigRomVa);
			
			//springboard
			if (suggestedBaseVa == 0x00000000 && bigRomVa == 0x00c08000) {
				
				suggestedBaseVa = bigRomVa = 0x08000000;
				springboard = true;
				fprintf(stderr, "This is likely a springboard rom. Acting accordingly\n");
			}
			else if (bigRomVa < suggestedBaseVa || bigRomVa - suggestedBaseVa + sizeof(struct CardHeader) > romUsedSz - i) {
				fprintf(stderr, "Big rom VA 0x%08lx is implausible for a ROM starting at VA 0x%08lx and running for 0x%08lx bytes\n", (unsigned long)bigRomVa, (unsigned long)suggestedBaseVa, (unsigned long)romUsedSz);
				ret = -7;
				goto out;
			}
		}
		else {
			
			fprintf(stderr, "No way to tell if big/small/combo rom for such early rom header version. Split it yourself!\n");
			bigRomVa = suggestedBaseVa;
		}
		
		if (springboard) {
			
			ret = dumpRom(rom + i, romUsedSz - (bigRomVa - suggestedBaseVa), suggestedBaseVa, RomTypeSpringboard);
		}
		else if (bigRomVa == suggestedBaseVa) {
			fprintf(stderr, "This ROM appears to only contain a BigROM. Dumping that\n");
			ret = dumpRom(rom + i, romUsedSz - i, suggestedBaseVa, RomTypeBig);
		}
		else {
			fprintf(stderr, "This ROM appears to contain a BigROM and a SmallROM. Dumping each...\n");
			fprintf(stderr, "Dumping SmallROM (0x%08lx)\n", (unsigned long)suggestedBaseVa);
			ret = dumpRomToFolder(rom + i, bigRomVa - suggestedBaseVa - i, suggestedBaseVa, RomTypeSmall, "SmallROM");
			if (ret) {
				ret -= 50;
				goto out;
			}
	
			fprintf(stderr, "Dumping BigROM (0x%08lx)\n", (unsigned long)bigRomVa);
			ret = dumpRomToFolder(rom + (bigRomVa - suggestedBaseVa), romUsedSz - (bigRomVa - suggestedBaseVa), bigRomVa, RomTypeBig, "BigROM");
			if (ret) {
				ret -= 150;
				goto out;
			}
		}
	}

out:
	free(rom);
	return ret;
}
