// Copyright (C) 1998-1999 Scott Cutler
// Please see the "readme.txt" file for license details

#include "include.h"


Tex::Tex(const uint32 address, const XGLTexture &tex) {
	DDSURFACEDESC2 DDSD;
	HRESULT href;

	vlog("Called Tex::Tex");

	vlog("  Texture:");
	vlog("    smalllod=" << tex.smalllod);
	vlog("    largelod=" << tex.largelod);
	vlog("    aspect=" << tex.aspect);
	vlog("    format=" << tex.format);


	_palhash = -1;
	_base = address;
	_lod = tex.largelod;
	_aspect = tex.aspect;
	_format = tex.format;
	_data = 0;


	getwidthheight(_lod, _aspect, _width, _height);

	_pf = xglpixelformats[tex.format];

	// Auto mipmap needs to be turned on, we need a valid mipmap creation function,
	// and we can't be in paletted mode, unless we are doing palette emulation.
	// The reason is that creating mipmaps from paletted textures is very hard,
	// but if we're converting to a non-palletted mode, then we can easily use them.
	_mipmapping = Settings.automipmap && mipmapfunc_lut[tex.format] && (!_pf.paletted || _pf.palemu);

//	log("Using texture format: " << formatstostrings[tex.format]);

	if (xglpixelformats[tex.format].set && !xglpixelformats[tex.format].supported) {
		log("Using emulated format: " << formatstostrings[tex.format]);
	}
	if (!xglpixelformats[tex.format].set) {
		log("Using unsupported format: " << formatstostrings[tex.format]);
	}


	memset(&DDSD, 0, sizeof(DDSURFACEDESC2));
	DDSD.dwSize = sizeof(DDSURFACEDESC2);
	DDSD.ddpfPixelFormat = _pf.pf;

	DDSD.ddsCaps.dwCaps2 = DDSCAPS2_TEXTUREMANAGE;
	DDSD.dwWidth = _width;
	DDSD.dwHeight = _height;
	DDSD.dwTextureStage = 0;
	DDSD.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT | DDSD_PIXELFORMAT;
	DDSD.ddsCaps.dwCaps = DDSCAPS_TEXTURE;

	if (_mipmapping) DDSD.ddsCaps.dwCaps |= DDSCAPS_MIPMAP | DDSCAPS_COMPLEX;


	if ((href = DX.DD->CreateSurface(&DDSD, &_texDDS, 0)) != DD_OK) {
		_texDDS = 0;
		log("Error code " << (href&0xFF) << " in Tex::Tex on CreateSurface, line " << __LINE__ << ", file " << __FILE__);
		return;
	}


	if (_mipmapping)	
		copytexturemipmap(tex);
	else
		copytexture(tex);


	if (_pf.paletted && !_pf.palemu) createpalette();

	setcolorkey();

	if (_texDDS->QueryInterface(IID_IDirect3DTexture2, (void**)&_tex) != D3D_OK) {
		_tex = 0;
		log("Error querying interface");
	}


}



void Tex::copytexture(const XGLTexture &tex) {
	// Ok, now we copy the texture to the surface
	DDSURFACEDESC2 DDSD;
	RECT r = {0, 0, _width, _height};

	DDSD.dwSize = sizeof(DDSURFACEDESC2);

	if (_pf.palemu) {
		_data = new uchar[_width*_height];
		memcpy(_data, (uchar *)tex.data, _width*_height);

	} else if (_texDDS->Lock(&r, &DDSD, DDLOCK_WAIT, 0) == DD_OK) {

		vlog("   width=" << DDSD.dwWidth);
		vlog("   height=" << DDSD.dwHeight);
		vlog("   pitch=" << DDSD.lPitch);

		if (_pf.texcc) {
			// If there's a texture conversion function defined, use that to convert the texture
			for (uint32 i=0; i<DDSD.dwHeight; i++) {
				for (uint32 j=0; j<DDSD.dwWidth; j++) {
					_pf.texcc((uchar *)DDSD.lpSurface+(i*DDSD.lPitch)+j*_pf.bpp, (uchar *)tex.data+((i*DDSD.dwWidth+j)*_pf.sbpp));
				}
			}

		} else {
			// Otherwise, a straight copy
			for (uint32 i=0; i<DDSD.dwHeight; i++) {
				memcpy((char *)DDSD.lpSurface+(i*DDSD.lPitch), (char *)tex.data+(i*DDSD.dwWidth*_pf.bpp), DDSD.dwWidth*_pf.bpp);
			}
		}
		
		if (_texDDS->Unlock(&r) != DD_OK) log("Could not unlock surface on line " << __LINE__ << ", file " << __FILE__);

	} else {
		_tex = 0;
		log("Could not lock texture surface on line " << __LINE__);
		return;
	}
}


uchar temptexbuf[256*384*4];

void Tex::copytexturemipmap(const XGLTexture &tex) {
	// Ok, now we copy the texture to the surface
	uint32 dbpp, sbpp;
	uint32 i, j;
	uchar *sp, *dp;

	dbpp = _pf.bpp;
	sbpp = _pf.sbpp;

	sp = tex.data;
	dp = temptexbuf;

	if (_pf.texcc) {
		// If there's a texture conversion function defined, use that to convert the texture
		for (i=0; i<_height; i++) {
			for (j=0; j<_width; j++) {
				_pf.texcc(dp, sp);
				sp += dbpp;
				dp += sbpp;
			}
		}
	} else {
		// Otherwise, a straight copy
		memcpy(temptexbuf, tex.data, _width*_height*dbpp);
	}


	uint size = (_width<_height) ? _width : _height;
	uint dwidth = _width;
	uint dheight = _height;
	uint spitch;
	uint dpitch = _width*dbpp; 

	void (*mipmapfunc)(void *, void *, void *, void *, void *) = mipmapfunc_lut[tex.format];

	sp = temptexbuf;
	dp = temptexbuf + _width*_height*dbpp;

	// Generate the mipmap levels
	while (size > 0) {		// Quit after we get a small side with length 1
		spitch = dpitch;
		dpitch /= 2;
		dheight /= 2;
		dwidth /= 2;

		for (i=0; i<dheight; i++) {
			for (j=0; j<dwidth; j++) {
				mipmapfunc(dp, sp, sp+dbpp, sp+spitch, sp+spitch+dbpp);
				sp += 2*dbpp;
				dp += dbpp;
			}
			sp += spitch;	// Skip next source texture line
		}
		
		size /= 2;			// Divide small side length by 2
	}


	
	LPDIRECTDRAWSURFACE4 DDLevel, DDNextLevel; 
	DDSCAPS2 ddsCaps; 
	HRESULT ddres; 

	ddsCaps.dwCaps = DDSCAPS_TEXTURE | DDSCAPS_MIPMAP;
	
	DDSURFACEDESC2 DDSD;
	DDSD.dwSize = sizeof(DDSURFACEDESC2);

	ddres = DD_OK;
	dwidth = _width;
	dheight = _height;

	DDLevel = _texDDS;
	DDLevel->AddRef();

	sp = temptexbuf;
	while (ddres == DD_OK) {     // Process this level 
		DDLevel->Lock(0, &DDSD, DDLOCK_WAIT, 0);

//		log(" Locked surface: width=" << DDSD.dwWidth << "  height=" << DDSD.dwHeight);

		dp = (uchar *)DDSD.lpSurface;
		spitch = dwidth*dbpp;

		for (i=0; i<dheight; i++) {
			memcpy(dp, sp, spitch);
			dp += DDSD.lPitch;
			sp += spitch;
		}
		DDLevel->Unlock(0);

		dwidth /= 2;
		dheight /= 2;

		ddres = DDLevel->GetAttachedSurface(&ddsCaps, &DDNextLevel); 
		DDLevel->Release();
		DDLevel = DDNextLevel; 
	} 

	if ((ddres != DD_OK) && (ddres != DDERR_NOTFOUND)) {
		// Code to handle the error goes here
	}
}



void Tex::createpalette() {
	LPDIRECTDRAWPALETTE DDP;

	if (DX.DD->CreatePalette(DDPCAPS_8BIT | DDPCAPS_ALLOW256 | DDPCAPS_INITIALIZE , Voodoo.palette, &DDP, 0) != DD_OK) {
		_tex = 0;
		log("Error creating palette");
		return;
	}
	if (_texDDS->SetPalette(DDP) != DD_OK) {
		_tex = 0;
		log("Error setting palette");
		return;
	}
	DDP->Release();
}

void Tex::setcolorkey() {
	DDCOLORKEY colorKey;
	DWORD c = 0;

	// Perform the color key conversion if necessary
	if (_pf.ckcc)
		_pf.ckcc((uchar *)&c, (uchar *)&Voodoo.colorKey);
	else
		c = Voodoo.colorKey;
	
	colorKey.dwColorSpaceLowValue = c;
	colorKey.dwColorSpaceHighValue = c;

	_texDDS->SetColorKey(DDCKEY_SRCBLT, &colorKey);
}


Tex::~Tex() {
	vlog("Called Tex::~Tex");
	if (_tex) _tex->Release();
	if (_texDDS) _texDDS->Release();
	if (_data) delete [] _data;
}


#define d3dassert(func) if (FAILED(func)) log("  Could not perform: " #func)

void Tex::setpalette() {

	if (Voodoo.palhash != _palhash) {
		_palhash = Voodoo.palhash;

		if (!_pf.palemu) {
			LPDIRECTDRAWPALETTE DDP;

			if (_texDDS->GetPalette(&DDP) != DD_OK) log("Error getting palette");
			if (DDP->SetEntries(0, 0, 256, Voodoo.palette) != DD_OK) log("Error setting palette");
			DDP->Release();
		} else {
			vlog("Doing paletted texture conversion");

			// This should never happen, but who knows...
			if (!_data) {
				log("  _data not initialized in Tex::setpalette");
				return;
			}

			uint32 i, j;

			DDSURFACEDESC2 DDSD;
			DDSD.dwSize = sizeof(DDSURFACEDESC2);
			RECT r = {0, 0, _width, _height};

			d3dassert(_texDDS->Lock(&r, &DDSD, DDLOCK_WAIT, 0));

			uchar *ds = (uchar *)DDSD.lpSurface;
			uint32 dpitch = DDSD.lPitch;
			uint32 spitch = _width;


			for (i=0; i<_height; i++) {
				for (j=0; j<_width; j++) {
					_pf.texcc(ds+(i*dpitch + j*_pf.bpp), _data+(i*spitch + j));
				}
			}

			d3dassert(_texDDS->Unlock(&r));
		}
	}
}


void getwidthheight(uint32 &lod, uint32 &aspect, uint32 &width, uint32 &height) {
	switch (lod) {
	case XGLLOD_256:	width=256;	height=256;	break;
	case XGLLOD_128:	width=128;	height=128;	break;
	case XGLLOD_64:		width=64;	height=64;	break;
	case XGLLOD_32:		width=32;	height=32;	break;
	case XGLLOD_16:		width=16;	height=16;	break;
	case XGLLOD_8:		width=8;	height=8;	break;
	case XGLLOD_4:		width=4;	height=4;	break;
	case XGLLOD_2:		width=2;	height=2;	break;
	case XGLLOD_1:		width=1;	height=1;	break;
	default:
		log("  Error in large LOD getwidthheight: " << lod);
		width=1; height=1;
		break;
	}

	switch(aspect)	{
	case XGLASPECT_8x1:		height >>= 3; 	break;
	case XGLASPECT_4x1:		height >>= 2; 	break;
	case XGLASPECT_2x1:		height >>= 1; 	break;
	case XGLASPECT_1x1:		break;
	case XGLASPECT_1x2:		width >>= 1; 	break;
	case XGLASPECT_1x4:		width >>= 2; 	break;
	case XGLASPECT_1x8:		width >>= 3; 	break;
	default:
		log("  Error in aspect ratio in getwidthheight: " << aspect);
		aspect = XGLASPECT_1x1;
		break;
	}
}



