// DMACNT.C
// This program reads one buffer of 8 bit PCM sound data
// from a file. This buffer is played back under DMA.
// The results of the function dmaBytesRemaining() are printed
// to the screen immediately after dmaPlay() is called and
// again after DMA has completed.
//
// The buffers that are used in this program are allocated 
// using cvxBufferAlloc. Each buffer is set to 32K. 

#include <stdio.h>
#include "cvxdigi.h"
#include "cvxutil.h"

#define _BUFFER_SIZE  0x8000     // Size of DMA buffer.
#define _PCM_FILE     "TEST1.V8"  // File containing sound data.

extern _dmaInProgressFlag;          
extern _dmaDevice;

VOID main( VOID )
{
   HANDLE  pcmFileHandle;
   LPSTR   pcmBuffer;
   LONG    bytesAvailable;
   LONG    bytesRead;
   WORD    portAddress;
   BYTE    dmaRate;
   WORD    dmaTempCount;
   WORD    repeatCount = 1;
   WORD    initError;


   // Allocate memory for buffer used to read 8 bit PCM
   // data from file.
   pcmBuffer =  cvxBufferAlloc( ( LONG )_BUFFER_SIZE, &bytesAvailable );

   if( pcmBuffer == _NULL )
   {
      printf( "ERROR : Cannot allocate memory!\n" );
      exit( 0 );
   }

   // Initialize DMA. Setting each parameter to _AUTODETECT 
   // causes dmaInit to perform a search for the Port,
   // DMA channel, and IRQ setting respectively.
   initError = dmaInit( _AUTODETECT, _AUTODETECT, _AUTODETECT, 
                        &portAddress );

   // If the variable _dmaDevice equals 0 then the DMA
   // sub-system was not initialized correctly.
   if( _dmaDevice == 0 )
      printf( "ERROR = %d : dmaInit failed\n", initError );
   else
   {
      // Open a file containing 8 bit PCM data.
      if( cvxFileOpen( _PCM_FILE, _OPEN_R_ONLY, &pcmFileHandle ) )
         printf( "ERROR : %s not found.\n", _PCM_FILE );
      else
      {
         // Get header information, if it exists, from file.
         cvxFileRead( pcmFileHandle, pcmBuffer,
                      ( LONG )_HEADER_LENGTH, &bytesRead );

         // Get rate from header.
         dmaRate = pcmBuffer[ _HEADER_RATE_OFFSET ];

         // Fill buffer with sound data.
         cvxFileRead( pcmFileHandle, pcmBuffer,
                      ( LONG )_BUFFER_SIZE, &bytesRead );

         // Insert buffer into DMA queue.
         dmaPlay( ( LPSTR )pcmBuffer, ( LONG )bytesRead, 
                  dmaRate, repeatCount );

         // Loop until DMA has completed.
         while( _dmaInProgressFlag );

         printf( "dmaBytesRemaining() = %d\n", dmaBytesRemaining() );
      }
   }

   // Clear all requests from the Play and Record Queue.
   if( dmaFlushQueue() )
      printf( "DMA uninit failure : could not flush queue.\n" );

   // Uninitialize the DMA system.   
   if( dmaUninit() )
      printf( "DMA uninit failure.\n" );
   
   // Close the sound file.
   cvxFileClose( pcmFileHandle );

   // Free memory used by 8 bit PCM buffer.
   if( cvxBufferFree( pcmBuffer ) )
      printf( "ERROR : Cannot free memory!\n" );
}


