// HARDPOLL.C
// This program uses the 8253 timer counter 2 to perform (non-DMA)
// polling. The contents of the file TEST1.V8 are output to 
// 0x22F.

#include <dos.h>
#include <stdio.h>
#include <conio.h>
#include <io.h>
#include "cvxutil.h"

#define _SOUND_FILE      "TEST1.V8"  // File containing sound data.
#define _RATE_IN_HERTZ    9622       // Rate at which to play sound data.


VOID main( VOID )
{
   HANDLE    fileHandle;      // File handle of TEST1.V8
   LONG      bytesRead;       // Bytes read from TEST1.V8
   LONG      bytesAvailable;  // Total bytes available in system.
   LPSTR     bufferStart;     // Far pointer to sound data read from TEST1.V8
   LPSTR     bufferEnd;       // End marker of sound data.
   BYTE      terminalCount;   // Count used with timer 2.
   WORD      vmPort;          // Covox DAC.
   BYTE      outByte;         // Storage variable for a byte of sound data.
   WORD      port61;          // Used to store Programmable Peripheral Interface contents.


   // Open the file containing sound data.
   if( cvxFileOpen( _SOUND_FILE, _OPEN_R_ONLY, &fileHandle ) )
      printf( "ERROR : %s not found.\n", _SOUND_FILE );
   else
   {
      // Find maximum amount of memory available in system.
      cvxBufferAlloc( -1L, &bytesAvailable );
   
      // If fileSize is less than bytes available in system then only
      // allocate enough memory to read entire file.
      if( filelength( fileHandle ) < bytesAvailable )
         bytesAvailable = filelength( fileHandle );

      printf("Allocating %lu bytes for playback buffer.\n", bytesAvailable );

      // Allocate maximum amount of memory for playback buffer.
      bufferStart = cvxBufferAlloc( bytesAvailable, &bytesAvailable );

      if( bufferStart == _NULL )
         printf( "ERROR : Cannot Allocate Memory!!\n" );
      else
      {
         // Read sound data from file.
         cvxFileRead( fileHandle, bufferStart, 
                      bytesAvailable, &bytesRead );

         // Point bufferEnd to the last byte in the sound data buffer.
         bufferEnd = ( LPSTR )(( HPSTR )bufferStart + bytesRead);

         // Calculate the value to count down to.
         terminalCount = ( BYTE )(256 - ( BYTE )( 1193180 / _RATE_IN_HERTZ ) );

         // Set DAC port value.
         vmPort = 0x22F;

         // Prompt user to hit a key to begin playback.
         printf( "Hit any key to begin playback.\n" );
         
         // Wait for keystroke.
         while( !kbhit() );
         
         // Clear keystroke(s) from keyboard buffer.
         while( kbhit() )
            getch();
         
         // Notify user that playback has begun.
         printf( "Playing sound file ..." );

         // Save off current contents of programmable peripheral
         // interface (PPI). (to be restored at end of program)
         port61 = inp( 0x61 );

         // Enable timer 2 clock input. Bit 0 of the PPI (port 0x61)
         // must be set in order to use timer 2.
         outp( 0x61, (port61 | 1 ) );
   
         // Initialize the 8254 timer on channel 2 (port 0x43).
         // 0x98 = Select counter 2.
         //        Prepare for read/write LSB of count down value.
         //        Mode 4 : software triggered strobe.
         //        Binary counter.
         outp( 0x43, 0x98 );

         // Set timer 2 to maximum count value. (LSB only)
         // The timer pre-decrements the counter value before it 
         // is examined, therefore the timer decrements 0x00 to
         // 0xFF on its first clock pulse.
         outp( 0x42, 0xFF );

         // Disable timer 0 (int 8). Setting bit 0 of the Interupt Mask
         // Register (port 0x21) disable interrupt 8.
         outp( 0x21, (inp( 0x21 ) | 1) );

         // Loop here and output all PCM bytes in the buffer 
         // pointed to by bufferstart. Typecast to HPSTR to handle
         // segment rollover.
         while( bufferStart != bufferEnd )
         {
            // NOTE : Some in-line assembly is used during this loop
            // to cut back on required CPU overhead.  The equivilent
            // C code is shown in the comments.

            // Get next sound data byte.
            outByte = *bufferStart;

            // Increment buffer to next byte. Typecast to HPSTR to
            // handle segment rollover.
            bufferStart = ( LPSTR )( ( HPSTR )bufferStart + 1 );


            // Loop here until timer 2 count value is less than
            // terminalCount.
            //
            // while( inp( 0x42 ) > terminalCount );
            //
            WAIT_LOOP:
            _asm   in   al, 42h
            _asm   cmp  al, terminalCount
            _asm   ja   WAIT_LOOP

            // Output sound data byte and increment buffer to
            // next byte.
            //
            // outp( vmPort, *bufferStart++ );
            //
            _asm   mov  al, outByte
            _asm   mov  dx, word ptr vmPort
            _asm   out  dx, al
            

            // Reset timer 2 to maximum countdown value.
            //
            // outp( 0x42, 0x00 );
            //
            _asm   xor  al, al
            _asm   out  42h, al
         }

         // Initialize the 8254 timer on channel 2 (0x43).
         // 0xB6 = Select counter 2.
         //        Prepare for read/write LSB/MSB of count down value.
         //        Mode 3 : square wave triggered.
         //        Binary counter.
         outp( 0x43, 0xB6 );

         // Write the LSB of the original countdown value.
         outp( 0x42, 0xD0 );

         // Write the MSB of the original countdown value.
         outp( 0x42, 0x04 );
 
         // Restore original contents of programmable peripheral
         // interface.
         outp( 0x61, port61 );

         // Renable timer 0 (int 8).
         outp( 0x21, (inp( 0x21 ) & 0xFE) );

         // Notify user that playback has completed.
         printf( " complete.\n" );

         // Point bufferStart back to beginning so that the memory
         // can be freed.
         bufferStart = ( LPSTR )(( HPSTR )bufferStart - bytesRead);
                          
         // Free memory used by buffer.
         if( cvxBufferFree( bufferStart ) )
            printf( "ERROR : Cannot Free Memory!!\n" );

      }
   
      // Close file containing sound data.
      cvxFileClose( fileHandle );
   }
}



