// UNPACK3S.C
// This program reads 3 bit sound data from a file.
// The 3 bit sound data is converted to 8 bit PCM encoding
// using the unpackFirst and unpackNext routines.
// The 8 bit PCM data is then written to a new file.
//
// The 2 buffers that are used in this program are allocated 
// using cvxBufferAlloc. The size of both buffers is set to 32K.

#include <stdio.h>
#include <bios.h>
#include <dos.h>
#include <errno.h>
#include <fcntl.h>
#include "cvxdigi.h"
#include "cvxutil.h"

#define  _BUFFER_SIZE    0x8000
#define  _PCM_FILE       "TEST1.V8"
#define  _3_BIT_FILE     "TEST1.V3S"

VOID main( VOID )
{
   LONG            recordedLength;
   HANDLE          v3sFileHandle;
   HANDLE          pcmFileHandle;
   LONG            bytesAvailable;
   LONG            bytesWritten;
   LONG            bytesRead;
   _CONVERT_DATA   cnvrtData;
   LPSTR           v3sBuffer;
   LPSTR           conversionBuffer;
   WORD            i;
   LONG            sourceSize         = 0;
   BOOL            exitFlag           = _FALSE;


   // Allocate memory for buffers used to read 3 bit
   // data from file.
   v3sBuffer = cvxBufferAlloc( ( LONG )_BUFFER_SIZE, &bytesAvailable );

   if( v3sBuffer == _NULL )
   {
      printf( "ERROR : Cannot allocate memory for v3s buffer!\n" );
      exitFlag = _TRUE;
   }

   // Allocate memory for buffer used for unpacking 3 bit with
   // silence encoding to 8 bit PCM .
   //
   conversionBuffer = cvxBufferAlloc( ( LONG )_BUFFER_SIZE, &bytesAvailable );
 
   if( conversionBuffer == _NULL )
   {
      printf( "ERROR : Cannot allocate memory for unpack buffer!\n" );
      exitFlag = _TRUE;
   }

   // Open file containg 3 bit data.
   if( cvxFileOpen( _3_BIT_FILE, _OPEN_R_ONLY, &v3sFileHandle ) )
   {
      printf( "ERROR : %s not found.\n", _3_BIT_FILE );
      exitFlag = _TRUE;
   }
   else
   {
      // Read header information from file. 
      cvxFileRead( v3sFileHandle, v3sBuffer, 
                   ( LONG )_HEADER_LENGTH, &bytesRead );

      // Point _CONVERT_DATA structure element to same location as
      // conversionBuffer. This buffer will contain the unpacked data
      cnvrtData.destinationPointer = ( LPVOID )conversionBuffer;
   
      // Point _CONVERT_DATA structure element to same location as
      // buffer containg 3 bit data.
      cnvrtData.sourcePointer = ( LPVOID )v3sBuffer;

      // Open a new file to write 8 bit sound data.
      if( cvxFileCreate( _PCM_FILE, _CREATE_NORMAL, &pcmFileHandle ) )
      {
         printf( "ERROR : Cannot create file %s.\n", _PCM_FILE );
         exitFlag = _TRUE;
      }
   }
   
   if( !exitFlag )
   {
      // Put header information into structure for call
      // to unpackFirst.
      cnvrtData.sampleRate   = ( BYTE )*( v3sBuffer + _HEADER_RATE_OFFSET );
      cnvrtData.bufferFormat = _FORMAT_V3S;
      cnvrtData.trebleFlag   = _FALSE;
      cnvrtData.noiseFlag    = _TRUE;

      cnvrtData.destinationLength = (LONG)_HEADER_LENGTH;
      cnvrtData.sourceLength      = (LONG)_HEADER_LENGTH;

      // Features not needed during unpack.
      cnvrtData.silenceRange      = 0;

      // Make call to unpack first 16 bytes of 3 bit buffer.
      if( unpackFirst( ( _CONVERT_DATA far * )&cnvrtData ) )
      {
         printf( "ERROR : Function unpackFirst failed.\n" );
         exitFlag = _TRUE;
      }
      else
      {
         // Write the 16 bit header returned by unpackFirst()
         // to file.
         cvxFileWrite( pcmFileHandle, cnvrtData.destinationPointer, 
                       ( LONG )_HEADER_LENGTH, &bytesWritten );

         // Increment v3sBuffer past the 16 byte header
         v3sBuffer = v3sBuffer + (WORD)_HEADER_LENGTH;

         // Setup loop to process contents of v3sBuffer.
         do
         {
            // Read 3 bit data from file.
            cvxfileRead( v3sFileHandle,
                         (LPSTR)(((LONG)cnvrtData.sourcePointer) 
                         + sourceSize),(LONG)( (LONG)_BUFFER_SIZE - sourceSize ), &bytesRead );

            // Calculate the amount of 3 bit data we have left 
            // to unpack in the current buffer.
            sourceSize = sourceSize + bytesRead;

            // Put information into _CONVERT_DATA structure for 
            // unpackNext.
            cnvrtData.sourceLength      = sourceSize;
            cnvrtData.destinationLength = ( LONG )_BUFFER_SIZE;

            // Call function to unpack a buffer of 3 bit data.
            if( unpackNext( ( _CONVERT_DATA far * )&cnvrtData ) )
            {
               printf( "ERROR : Data unpacking failure.\n" );
               exitFlag = _TRUE;
            }
            else
               cvxFileWrite( pcmFileHandle,
                             cnvrtData.destinationPointer, 
                             cnvrtData.destinationFilled,
                             &bytesWritten );

            // Move left over source data to the begining of the 
            // conversion buffer.
            sourceSize = sourceSize - cnvrtData.sourceUsed;

            for( i = 0; i < sourceSize; i++ )
                 *( cnvrtData.sourcePointer + i ) = *( cnvrtData.sourcePointer +  cnvrtData.sourceUsed + i );
                
            // Check to see if we are done unpacking.
            if( ( sourceSize < 16L ) && eof( v3sFileHandle ) )
               exitFlag = _TRUE;

         } while( !exitFlag );
      }

      // Close files.
      cvxFileClose( v3sFileHandle );
      cvxFileClose( pcmFileHandle );
   }

   // Free memory used for 3 bit sound data.
   if( cvxBufferFree( conversionBuffer ) )
      printf( "ERROR : Cannot free memory!\n" );

   // Free memory used by 8 bit PCM buffer.
   if( cvxBufferFree( v3sBuffer ) )
      printf( "ERROR : Cannot free memory used for PCM buffer!\n" );

}
