//
// File        : playfile.c
//
// Description : Function to playback a file of Covox ADPCM and PCM data.
//
// Written by  : Ryan Hanlon
// Date        : Feb 19, 1992
//
//----------------------------------------------------------------------------
//             Copyright (c) 1992, Covox, Inc. All Rights Reserved                   
//----------------------------------------------------------------------------
// 415 571 1800 ext 3541 Al Nies
// for playFile and dmaPlayFile.

#include <io.h>
#include "cvxdigi.h"
#include "cvxutil.h"

extern _ioStopFlag;

// This module will playback the contents of a sound file.
// If bufferSize is sent as zero then the entire contents of the file
// will be played back.
//
WORD playFile( HANDLE fileHandle, BYTE fileFormat, BYTE playbackRate,
               WORD   portOut,    BOOL noiseFlag,  BOOL trebleFlag,
               BOOL   hookKBFlag, LONG bufferSize )
{              
   static BYTE   tempRate;
   static BYTE   tempFormat;

   WORD   i;
   LONG   fileSize;
   LONG   tempSize;

   LPSTR  buffer;
   LONG   bytesAvailable;
   LONG   bytesRead;

   WORD   returnValue = _ERROR_NONE;


   // Get length of file.
   //
   fileSize = filelength( fileHandle );

   // See if bufferSize has been requested.
   if( bufferSize && (bufferSize < fileSize ) )
      fileSize = bufferSize;

   // Allocate memory for reading file.
   //
   buffer = cvxBufferAlloc( fileSize, &bytesAvailable );
   
   if( buffer == _NULL )
   {
      buffer = cvxBufferAlloc( bytesAvailable, &tempSize );
   
      bufferSize = bytesAvailable;
   
      if( buffer == _NULL )
         returnValue =  _ERROR_MEMORY_ALLOC;
   }
   else
   {
      bufferSize = fileSize;
   }

   
   if( !returnValue )
   {
      // Read data from sound file for cvxHeaderParse.
      //
      returnValue = cvxFileRead( fileHandle, buffer, bufferSize, &bytesRead );

      // Get format and playback of file.
      //
      cvxHeaderParse( buffer, &tempFormat, &tempRate );

      // If no file format was specified use return from cvxHeaderParse.
      //
      if( !fileFormat )
         ( BYTE )fileFormat = ( BYTE )tempFormat;
      else
         // If specified file format is different from format found in
         // header then return with error.
         //
         if ( ( BYTE )fileFormat != ( BYTE )tempFormat )
           returnValue = _ERROR_INVALID_FORMAT;

      // If no playback rate was specified then set rate to rate found
      // in header.
      //
      if( !playbackRate )
         ( BYTE )playbackRate = ( BYTE )tempRate;

      // If a VMF format file is found then bypass the 128 byte header.
      //
      if( fileFormat == _FORMAT_VMF )
      {
         fileSize -= _HEADER_LENGTH_VMF;
      }
         
      // Hook int 9h (keyboard interrupt).  This hook will set the flag
      // _ioStopFlag when a key has been struck.
      //
      if( hookKBFlag )
         hookKeyboard();

      // Loop here until entire file is played back.
      while( fileSize && !returnValue && !_ioStopFlag )
      {
         // Calculate number of bytes left in file.
         //
         if( fileSize - bytesRead <= 0)
            fileSize = 0;
         else
            fileSize = fileSize - bytesRead;

         // Call function to play file of specified format.
         //
         switch( fileFormat )
         {
            case _FORMAT_V2S:
               returnValue = play2s ( buffer,  bytesRead,  playbackRate,
                                      portOut, trebleFlag, noiseFlag );
               break;
   
            case _FORMAT_V3S:
               returnValue = play3s ( buffer,  bytesRead,  playbackRate,
                                      portOut, trebleFlag, noiseFlag );
               break;
   
            case _FORMAT_V4S:
               returnValue = play4s ( buffer,  bytesRead,  playbackRate,
                                      portOut, trebleFlag, noiseFlag );
               break;
   
            case _FORMAT_V8S:
               returnValue = play8s ( buffer,  bytesRead,  playbackRate,
                                      portOut, trebleFlag, noiseFlag );
               break;
   
            case _FORMAT_VMF:
            case _FORMAT_V8:
               returnValue = play8  ( buffer,  bytesRead,  playbackRate,
                                      portOut, trebleFlag );
               break;

            default:
               returnValue = _ERROR_INVALID_FORMAT;
   
               break;
         }

         // Read sound data from file.
         //
         if( fileSize )
            returnValue = cvxFileRead( fileHandle, buffer, 
                                       bufferSize, &bytesRead );
      }
         
      // Reset keyboard ISR vector.
      //
      if( hookKBFlag )
         unhookKeyboard();
   }

   if( cvxBufferFree( buffer ) )
   {
      returnValue = _ERROR_MEMORY_DEALLOC;
   }


   return( returnValue );
}

