//
//    RECORD.EXE - (c) 1992 Covox Inc., Eugene, Oregon USA
//
//    File to record Covox ADPCM and PCM recordings on any Covox hardware
//    from the DOS command line.
//
//      This program or parts of it may be used in other programs written
//      to work with the Covox external digitizer freely and with out prior
//      consent of Covox Inc. The speech output through the parallel printer
//      is a Covox US Patent Number 4812847. Other US and foreign patents
//      are pending. Specifications subject to change without notice.
//
//      RECORD.EXE can be compiled using either MicroSoft C 6.0 or Borland
//      C 2.0.  See RECORD.MAK for more information.
//
//----------------------------------------------------------------------------
//             Copyright (c) 1992, Covox, Inc. All Rights Reserved                   
//----------------------------------------------------------------------------

#include <conio.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "parse.h"
#include "cvxdigi.h"
#include "cvxutil.h"


//  Version string for RECORD.C
//
PSTR    recordVersionString  = "4.01"; 

// Error codes for RECORD.C
//
enum
{
   _ERROR_RATE_EXCEEDED = 1,       // Rate specified on command line exceeded 209
   _ERROR_INVALID_FILE_NAME,       // File name did not begin with a character.
   _ERROR_NO_COMMAND_LINE_OPTIONS, // No options were found on command line.
   _ERROR_INVALID_JUMPER,          // Invalid dma jumper specified on command line.
   _ERROR_INVALID_IRQ_NUMBER,      // Invalid irq specified on command line.
   _ERROR_INVALID_DMA_CHANNEL,     // Invalid dma channel specified on command line.
   _ERROR_INVALID_PORT             // Invalid port specified on command line.
};


// Covox ADC hardware types.
//
#define  _HARDWARE_INTERNAL_NONDMA  0    // Sound Master II, Voice Master Key System
#define  _HARDWARE_INTERNAL_DMA     1    // Sound Master II, Voice Master Key System
#define  _HARDWARE_EXTERNAL_LPT     4    // Voice Master Key System II


// If EXTERNAL is defined at compile time then the default
// output device will be LPT1 otherwise non-DMAl Voice Master Key 
// System is the default.
//
#ifdef EXTERNAL
#define   _HARDWARE_DEFAULT      _HARDWARE_EXTERNAL_LPT
#else
#define   _HARDWARE_DEFAULT      _HARDWARE_INTERNAL_NONDMA
#endif

// Global variables.
//
WORD hardwareType = _HARDWARE_DEFAULT;  // Covox recording device.

WORD portIn       = _HARDWARE_DEFAULT;  // Port for non-DMA record.

WORD dmaJumper    = _AUTODETECT;        // Base port for dma recording.
WORD dmaChannel   = _AUTODETECT;        // DMA channel of Sound Master II or Voice Master.
WORD irqNumber    = _AUTODETECT;        // IRQ of Sound Master II or Voice Master.

WORD silenceRange = _SILENCE_3;         // Silence threshold for ADPCM recording.

LONG bufferSize   = 0L;                 // Size of record buffer(s).
WORD bufferCount  = 5;                  // Number of buffers to use for dmaPlay().

BYTE recordRate   = 0;                  // Covox record rate (0 to _CVX_RATE_IN_MAXIMUM)

BYTE fileName[80] = "";                 // File path and name.
BYTE fileFormat;                        // Sound data format of recording.

BOOL quietFlag    = _FALSE;             // If _TRUE, suppress all screen output
                                        //    except for user prompt (see autoStart).

WORD autoStart    = _FALSE;             // If _TRUE, recording will start without 
                                        //    prompting user to 'Hit any key to begin'

BOOL hookKBFlag   = _TRUE;              // Used by recordFile() and dmaRecordFile()
                                        //   to determine use of HOOK_KB.C functions.

// Flag used to determine when i/o is complete.
//
extern BYTE _ioStopFlag;

//------------------------------------------------------------------------
// RECORD.C function prototypes

// Function to display valid options to the screen.
//
VOID  displayCommandLineSyntax( VOID );

// This function displays the appropriate recording information
// to the screen.
//
WORD displayRecordFileInfo( VOID );

// This procedure insures that the correct file format and extension
// are on the fileName
//
WORD checkFileName( VOID );

// Command line parse function prototypes.
//
WORD  parseCommandLine (WORD argc, PSTR argv[]);
WORD  parseFilename    ( PSTR );
WORD  parseRate        ( PSTR );
WORD  parseMaxMemory   ( PSTR );
WORD  parsePort        ( PSTR );
WORD  parseJumper      ( PSTR );
WORD  parseChannel     ( PSTR );
WORD  parseSilence     ( PSTR );
WORD  parseFormat      ( PSTR );
WORD  parseQuiet       ( PSTR );
WORD  parseAutoStart   ( PSTR );
WORD  parseIrqNumber   ( PSTR );

// Used to take over <CTRL-C> and <CTRL-BREAK>
//
VOID ( interrupt far * oldCtrlBreak ) ();  
VOID interrupt far controlBreakHandler( VOID );  

     
// ------------------------------------------------------------------------
// --------------------------   BEGIN MAIN   ------------------------------
VOID main(WORD argc, PSTR argv[])
{
   HANDLE fileHandle;
   PSTR   string;
   BYTE   fileOverwrite;
   WORD   returnValue = _ERROR_NONE;

   // Get all options from the command line that have been entered
   // by user. If illegal options have been used then display  
   // commandLineSyntaxMessage and return control to DOS.
   //
   if( parseCommandLine(argc, argv) )
   {
      if( !quietFlag )
         displayCommandLineSyntax();

      exit(0);
   }


   if( !strlen( fileName ) )
   {
      if( !quietFlag )
         displayCommandLineSyntax();

      exit(0);
   }

   // Trap <CTRL-C> and <CTRL-BREAK>
   // Save off orignal ISR vector address and set temporary
   // handler address.
   //
   oldCtrlBreak = _dos_getvect( 0x1B );
   _dos_setvect( 0x1B, controlBreakHandler );

   // This procedure insures that the correct file format and extension
   // are on the fileName
   //    
   checkFileName();


   if( !quietFlag )
   {
      // Print out say version and ID
      printf( "\nCovox Inc (c) - Sound file recording utility - Version %s\n\n", recordVersionString );

      // Test if file exist.
      //
      if( !cvxFileOpen( fileName, _OPEN_R_ONLY, &fileHandle ) )
      {
         cvxFileClose( fileHandle );

         printf("File \"%s\" exists, overwrite (Y/N)? ", fileName);
         fileOverwrite = getch();

         if( toupper( fileOverwrite ) != 'Y' )
            exit(1);
             
         printf("\n");
      }
   }

   returnValue =  cvxFileCreate( fileName, _CREATE_NORMAL, &fileHandle );

   if( !returnValue )
   {
      if( !quietFlag )
      {
         displayRecordFileInfo();
      }

      if( !autoStart )
      {
         printf("\nPress any key to start recording\n");
         getch();
         printf("Press any key to stop recording\n");
      }


      if( hardwareType == _HARDWARE_INTERNAL_DMA ) 
      {
         returnValue = dmaRecordFile( fileHandle,   fileFormat, recordRate,
                                      dmaJumper,    dmaChannel, irqNumber,
                                      silenceRange, hookKBFlag,
                                      bufferSize,   bufferCount );
      }
      else
      {
         returnValue = recordFile( fileHandle,   fileFormat,   recordRate,  
                                   silenceRange, hookKBFlag,  
                                   portIn,       bufferSize );
      }

      // Close file containg recorded sound data.
      //
      cvxFileClose( fileHandle );
   }

   // If an error occured print out error message.
   //
   if( returnValue )
   {         
      string = cvxGetErrorString( returnValue );
      printf("\nERROR = %d \n", returnValue );
      printf("%s\n", string);
   }

   // Restore orignal <CTRL-C> and <CTRL-BREAK> vector.
   //
   _dos_setvect( 0x1B, oldCtrlBreak );


   //  Clear all keystrokes from the keyboard buffer.
   //
   while( kbhit() )
   {
     getch();
   }
}
// ---------------------------    END MAIN    --------------------------------
//----------------------------------------------------------------------------

// This function displays the appropriate recording information
// to the screen.
//
WORD displayRecordFileInfo( VOID )
{
   BYTE   line[10];

   printf("Recording File:   %s\n",fileName);
   printf("Input device:     ");
   if( hardwareType == _HARDWARE_INTERNAL_NONDMA )
   {
      printf("Voice Master, port %s\n", itoa(portIn,line,10) );
   }
   else if( hardwareType == _HARDWARE_EXTERNAL_LPT )
   {
      printf("VM System II, port LPT%s\n", itoa(portIn-3,line,10) );
   }
   else if( hardwareType == _HARDWARE_INTERNAL_DMA )
   {
      printf("DMA\n");
   }

   printf("Recording format: ");
   switch( fileFormat )
   {
      case _FORMAT_V2S:
         printf("Two bit ADPCM w/ silence encoding\n");
         break;

      case _FORMAT_V3S:
         printf("Three bit ADPCM w/ silence encoding\n");
         break;

      case _FORMAT_V4S:
         printf("Four bit ADPCM w/ silence encoding\n");
         break;

      case _FORMAT_V8S:
         printf("Eight bit PCM w/ silence encoding\n");
         break;

      case _FORMAT_V8:
         printf("Eight bit PCM\n");
         break;
   }

   // If record rate is set to zero display default rate (132).
   if( recordRate )
   {
      // Print out Covox and Hertz rate to screen.
      //
      printf("Rate:             %u  (", recordRate );
      printf("%uHz)\n", cvxRateToHz( recordRate) );
   }
   else
   {
      // Print out Covox and Hertz rate to screen.
      //
      printf("Rate:             %u  (", _CVX_RATE_DEFAULT );
      printf("%uHz)\n", cvxRateToHz( _CVX_RATE_DEFAULT ) );
   }


    return( _ERROR_NONE );
}


// This procedure insures that the correct file format and extension
// are on the fileName
//
WORD checkFileName( VOID )
{
   // Convert fileName to all lower case characters.
   //
   strlwr( fileName );

   // Insure fileName does not begin with a period.
   //
   if( strcspn(fileName, ".") == 0 )
      return( _ERROR_INVALID_FILE_NAME );

   // If no extension and no fileFormat then default to .v8
   //
   if( ( *( fileName + strlen(fileName) - 1 ) == '.' ) && !fileFormat )
   {
      strcat(fileName,"v8");
      fileFormat = _FORMAT_V8;
   }
   else
   {
      // If the last character in fileName is a period then
      // remove it.
      //
      if( *( fileName + strlen(fileName) - 1 ) == '.' )
         *( fileName + strlen(fileName) - 1 ) == _NULL;

      // Test for absence of file extention.
      //
      if( strrchr(fileName,'.') == _NULL )
      {
         // If no fileFormat then append .v8
         // Otherwise append specfied format extension on file.
         //
         if( !fileFormat )
         {
            strcat(fileName,".v8");
         }
         else
         {
            if( fileFormat == _FORMAT_V8 )
               strcat( fileName,".v8" );
            else if( fileFormat == _FORMAT_V8S )
               strcat( fileName,".v8s" );
            else if( fileFormat == _FORMAT_V4S )
               strcat( fileName,".v4s" );
            else if( fileFormat == _FORMAT_V3S )
               strcat( fileName,".v3s" );
            else if( fileFormat == _FORMAT_V2S )
               strcat( fileName,".v2s" );
         }
      }
   
      // Determine fileFormat from file extension.
      //
      if( strstr(fileName,".v2s" ) )
      {
         fileFormat = _FORMAT_V2S;
      }
      else if( strstr(fileName,".v3s" ) )
      {
         fileFormat = _FORMAT_V3S;
      }
      else if( strstr(fileName,".v4s" ) )
      {
         fileFormat = _FORMAT_V4S;
      }
      else if( strstr(fileName,".v8s" ) )
      {
         fileFormat = _FORMAT_V8S;
      }
      else // Default to .v8
      {
         fileFormat = _FORMAT_V8;
      }
   }
}


// Parse the options entered by user of program.
//
WORD parseCommandLine( WORD argc, PSTR argv[] )
{
   BYTE  commandLineToken;
   unsigned short commandLineCount;
   WORD  returnValue;

   static PARSE_INFO recordParse[] =
   {
      { 0,  0 , parseFilename},
      { 1, 'R', parseRate},
      { 0, 'Q', parseQuiet},
      { 1, 'S', parseSilence},
      { 1, 'M', parseMaxMemory},
      { 1, 'P', parsePort},
      { 1, 'F', parseFormat},
      { 0, 'A', parseAutoStart},
      { 1, 'J', parseJumper},
      { 1, 'C', parseChannel},
      { 1, 'I', parseIrqNumber},
   };

   commandLineToken = '/';

   commandLineCount = sizeof( recordParse ) / sizeof(PARSE_INFO);

   SetParse(commandLineToken, commandLineCount, recordParse );

   if( argc == 1 )
   {
      returnValue = _ERROR_NO_COMMAND_LINE_OPTIONS;
   }
   else
   {
      returnValue = Parse(argc, argv);
   }

   return( returnValue );

}


WORD parseFilename( PSTR x )
{
   // Copy parse string into global variable.
   //
   strcpy( fileName, x );
   return( _ERROR_NONE );
}                      

WORD parseRate( PSTR x )
{
   recordRate = (BYTE)atoi(x);
   if(recordRate > _CVX_RATE_IN_MAXIMUM)
   {
      return( _ERROR_RATE_EXCEEDED );
   }
   return( _ERROR_NONE );
}

WORD parseSilence( PSTR x )
{
   silenceRange = atoi(x);
   return( _ERROR_NONE );
}

WORD parseQuiet( PSTR x )
{
   quietFlag = 1;
   return( _ERROR_NONE );
}

WORD parseMaxMemory( PSTR x )
{
   // Convert pages to bytes.
   //
   bufferSize = ( LONG )(atoi(x) << 4);

   return( _ERROR_NONE );
}

WORD parseFormat( PSTR x )
{
   PSTR string1;

   strlwr(x);
   if(strstr(x,"8s") != 0)
   {
      fileFormat = _FORMAT_V8S;
   }
   else if(strstr(x,"4s") != 0)
   {
      fileFormat = _FORMAT_V4S;
   }
   else if(strstr(x,"3s") != 0)
   {
      fileFormat = _FORMAT_V3S;
   }
   else if(strstr(x,"2s") != 0)
   {
      fileFormat = _FORMAT_V2S;
   }
   else if(strstr(x,"8") != 0)
   {
      fileFormat = _FORMAT_V8;
   }
   else
      return( _ERROR_INVALID_FORMAT );

   return( _ERROR_NONE );
}

WORD parsePort( PSTR x )
{

   strlwr(x);
   if( strstr(x,"vm") )
   {
      hardwareType = _HARDWARE_INTERNAL_NONDMA;
      portIn = atoi(x+2);
   }
   else if( strstr(x,"lpt") )
   {
      hardwareType = _HARDWARE_EXTERNAL_LPT;

      // Add three to lpt port since routines expect 4 - 6 as
      // lpt1 - lpt3
      portIn = atoi(x+3) + 3;
   }
   else if( strstr(x,"dma") )
   {
      hardwareType = _HARDWARE_INTERNAL_DMA;
   }
   else
      return( _ERROR_INVALID_PORT );

   return( _ERROR_NONE );
}

WORD parseAutoStart( PSTR x )
{

   autoStart = _TRUE;
   return( _ERROR_NONE );
}

WORD parseJumper( PSTR x )
{
   portIn = atoi(x);

   if( (dmaJumper == _AUTODETECT ) ||
       (dmaJumper == _CVX_VM0    ) ||
       (dmaJumper == _CVX_VM1    ) ||
       (dmaJumper == _CVX_VM2    ) ||
       (dmaJumper == _CVX_VM3    ) )
   {
      return( _ERROR_NONE );
   }
   else
   {
      return( _ERROR_INVALID_JUMPER );
   }
}

WORD parseChannel( PSTR x )
{
   dmaChannel = atoi(x);

   if( ( dmaChannel == _AUTODETECT    ) ||
       ( dmaChannel == _DMA_CHANNEL_1 ) ||
       ( dmaChannel == _DMA_CHANNEL_3 ) )     
   {                         
      return( _ERROR_NONE );
   }
   else
   {
      return( _ERROR_INVALID_DMA_CHANNEL );
   }
}

WORD parseIrqNumber( PSTR x )
{
   irqNumber = atoi(x);
   if( (irqNumber == _AUTODETECT) ||
       (irqNumber == _IRQ_2)  || 
       (irqNumber == _IRQ_3)  || 
       (irqNumber == _IRQ_4)  || 
       (irqNumber == _IRQ_5)  || 
       (irqNumber == _IRQ_7) )   
   {
      return( _ERROR_NONE );
   }
   else
   {
      return( _ERROR_INVALID_IRQ_NUMBER );
   }
}

// ISR for control break.
//
VOID interrupt far controlBreakHandler( VOID )
{
   _asm nop
}


// Display to the screen the legal command line options avaiable.
//
VOID displayCommandLineSyntax( VOID )
{
   WORD  i, j;
   WORD  messageLength;

   static PSTR commandLineSyntaxMessage[] =
      {
      "        RECORD Version ",
      "",
#ifdef EXTERNAL     // insert an x in the external record
      "x"
#endif
      " - Covox Inc (c) 1991\n",
      "Record any a file of any Covox sound format.\n",
      "RECORD <filename> [<switches>]\n",
      "/Rxxx - Where xxx is a record rate from 0-209.  0 = default rate of 132.\n",
      "/Sxx  - Silence threshold (0-5), default 3.\n",
      "/A    - Start recording without prompting the user.\n",
      "/Q    - Quiet mode. (For no screen output use /A also).\n",
      "/Mxxx - Maximum KB to allocate for recording, 0 = default = all avalable RAM.\n",
      "/Fxxx - Recording format, file extension overrides if present (default is V8).\n",
      "          (S for silence encoding)       8  - Eight bit PCM\n",
      "          8S - Eight PCM bit encoding    4S - Four bit ADPCM encoding\n",
      "          3S - Three bit ADPCM encoding  2S - Two bit ADPCM encoding\n",
      "/Pxxx - The port to use for input, xxx can be one of the following:\n",
      "          VM0 "
#ifndef EXTERNAL
      "(default)"
#endif
      ", VM1, VM2, VM3 for Voice Master or Sound Master II, or\n",
      "          LPT1 "
#ifdef EXTERNAL
      "(default)"
#endif
      ", LPT2, LPT3 port for the VM System II, or\n",
      "          DMA unlimited recording using Voice Master or Sound Master II.\n",
      "The following apply only to Voice Master or Sound Master II with DMA.\n",
      "/Jx   - Select DMA port setting (0-3).  -1 (default) for sequential search.\n"
      "/Cx   - Select DMA channel (1 or 3).  -1 (default) checks 1 then 3.\n"
      "/Ix   - Select IRQ (2,3,4,5 or 7).  -1 (default) for sequential search.\n\n"
      "RECORD birds.v4s /PLpt2\n",
      "        Record to file bird.v4s 4 bit speech coding from the\n",
      "        VM System II attached to LPT2."
      };

   // Put version into array.
   commandLineSyntaxMessage[ 1 ] = recordVersionString;
   
   messageLength = sizeof(commandLineSyntaxMessage)/sizeof(PSTR);

   for(i = 0 ; ( (i < messageLength) && (i != 22) ) ; i++)
   {
      printf(commandLineSyntaxMessage[i]);

   }
}



