;;===========================================================================
;;          Sample MASM Macros to get byte from external digitizer
;;              16 Feb 1990 Covox Inc., Eugene Oregon USA
;;
;;      These macros can be used by assembler routines written in 
;;      MicroSoft MASM 5.1 or greater, or with Borland TASM. 
;;
;;      This program or parts of it may be used in other programs written
;;      to work with the Covox external digitizer freely and with out prior
;;      consent of Covox Inc. The speech output through the parallel printer
;;      is a Covox US Patent Number 4812847. Other US and foreign patents
;;      are pending. Specifications subject to change without notice.
;;
;;      The routines:
;;          * Select the Covox external digitizer
;;          * Get one byte of A to D data from the Covox external digitizer
;;          * Deselect the Covox external digitizer
;;          * Get LPT port address given LPT port number 1 to 3
;;      
;;===========================================================================


;;**
;;**  Macro to read one byte of data from the Covox external digitizer
;;**
;;**  Passed:   lpt_port     Registor or Address with lpt port number
;;**            out_reg      8 bit registor for output, can not be al,
;;**                         default is ah.
;;**
;;**  Uses:     al,dx,out_reg
;;**
;;**  Returns:  out_reg       8 bit A/D data in a CPU registor, ah by 
;;**            default
;;**
get_x_byte   MACRO   lpt_port, out_reg
      LOCAL  WaitForDataToStabilize
      .ERRB   <lpt_port>

      IFDIFI <port>,<dx>
         mov   dx,lpt_port     ;; only load port address if all 
      ENDIF                    ;;  ready not in dx
         inc   dx              ;; port + 1
         in    al, dx          ;; get ls nibble
         shr   al, 1
         shr   al, 1
         shr   al, 1
         shr   al, 1

      IFB   <out_reg>          ;; uses default output
         mov   ah, al
      ELSE
         mov     out_reg,al    ;; uses alternate output register
      ENDIF
         inc   dx              ;; port + 2 for switch
         mov   al, 5           ;; switch so now reading 
         out   dx, al          ;; upper nibble

         ; This jump is need to give the data in the multiplexer time to 
         ; stabilize before an 'in' to read the upper nibble is performed.
         ; This fix will only be noticable on AT's.
         ;
         jmp SHORT WaitForDataToStabilize

      WaitForDataToStabilize:
         dec   dx              ;; port + 1
         in    al, dx
         and   al, 0F0h        ;; make sure only nybble

      IFB   <out_reg>          ;; uses default output
         or    ah, al
         xor   ah, 088h        ;; flip bits with negative logic
      ELSE
         or    out_reg, al     ;; uses alternate output registor
         xor   out_reg, 088h   ;; flip bits with negative logic
      ENDIF

         inc   dx
         mov   al, 4           ;; restart D to A convertion and 
         out   dx, al          ;; switch to reading LS Nibble
         dec   dx              ;; restore dx to value when entered
         dec   dx
      ENDM


;;**
;;**  Macro to select Covox external digitizer
;;**
;;**  Passed:   lpt_port        Registor or Address with lpt port number
;;**
;;**  Uses:     al,dx
;;**
;;**  Returns:  Nothing
;;**
external_on   MACRO   lpt_port
      .ERRB   <lpt_port>
      IFDIFI <lpt_port>,<dx>
         mov   dx,lpt_port
      ENDIF
         mov   al, 080h        ;; center external D/A
         out   dx, al
         add   dx, 2
         mov   al, 0100b       ;; Reset printer
         out   dx, al
         mov   al, 0100b       ;; set so upper nybble to be read
         out   dx, al
      ENDM

;;**
;;**  Macro to deselect Covox external digitizer
;;**
;;**  Passed:   lpt_port Registor or Address with lpt port number
;;**
;;**  Uses:     al,dx
;;**
;;**  Returns:  Nothing
;;**
external_off   MACRO   lpt_port
      .ERRB   <lpt_port>

      IFDIFI <lpt_port>,<dx>
         mov   dx,lpt_port
      ENDIF
         add     dx, 2
         mov     al, 01100b    ;; Reset printer and turn off external digitizer
         out     dx, al
ENDM

;;**
;;**  Get lpt port number from system memory
;;**
;;**  Passed:   lpt_port_number   LPT port number 1 to 3
;;**
;;**  Uses:     ax,bx,2 bytes of stack
;;**
;;**  Returns:  ax   LPT port address, should check if not zero
;;**                 if value zero then port not installed
;;**
get_lpt_address MACRO   lpt_port_number
      .ERRB   <lpt_port_number>

      IFDIFI <lpt_port_number>,<bx>
         mov    bx,lpt_port_number
      ENDIF

         push   es
         mov    ax, 040h
         mov    es, ax
         sub    bx, 01           ;;convert from 1-3 to 0-2
         shl    bx, 01           ;;Now convert to a pointer
         mov    ax, es:[bx + 8]
         pop    es
ENDM




