;****************************************************************************
;
;  File              : internal.asm
;
;  Date Created      : 5/18/90
;
;  Description       : Driver for midi hardware
;
;  Programmer(s)     : Nick Skrepetos, Don Fowler
;
;  Last Modification : 7-7-91
;
;  Additional Notes  : This is the driver to control the midi hardware.
;
;****************************************************************************
;            Copyright (c) 1992,  Covox, Inc.  All Rights Reserved
;****************************************************************************
                                
ifdef  SMODEL
        .model SMALL,c
endif

ifdef  MMODEL
        .model MEDIUM,c
endif

ifdef  CMODEL
        .model COMPACT,c
endif

ifdef  LMODEL
        .model LARGE,c
endif

                        ;
                        ; Compile in masm 5.1 mode.
                        ;
                        MASM51

                        ;
                        ; Handle the turbo quirks.
                        ;
                        QUIRKS

                        ;
                        ; Handle local labels.
                        ;
                        LOCALS

                        ;
                        ; Declare these functions public to all linked modules.
                        ;
                        public  midiByteReady   
                        public  midiBytesInQueue 


; Size of the midi byte queue
MIDI_BUFFERSIZE         equ     0400h


                        .DATA

;
; Port used to access internal midi interface. 
;
midiPort                dw      0

;
; Flag used to indicate that a midi byte is ready, this flag will be set to
; zero if no byte is ready, ie. midiFetchIndex == midiInternalIndex
;
midiByteReady           db      0

;
; Circular buffer to collect midi data. 
;
midiBuffer              db      MIDI_BUFFERSIZE dup( 0 )

;
; Index maintained by the interrupt service routine, the external application
; must not affect with this value.
;
midiInternalIndex       dw      0

;
; Index maintained by the midi fetch byte routine that passes back bytes
; to the calling program.
;
midiFetchIndex          dw      0

;
; Number of bytes currently in the midi queue buffer, this can be used to 
; grab 'packets' of data.
;
midiBytesInQueue        dw      0


;
; Interrupt request line passed to initialization function
;
irqNumber          db     0

;
; Storage for replaced interrupt vector
;
irqOffset              dw      ?
irqSegment             dw      ?

;
; Flag for the interrupt handler to skip the first interrupt 
;
interruptSkipFlag      db       0

 
                        .CODE

                        ;
                        ; Assume access to the code segment to be relative to cs
                        ; and access to the data segment to be relative to ds
                        ;
                        ASSUME  cs:@CODE,ds:@DATA

;****************************************************************************
;
; Function         : midiPollUART
;
; Description      : Waits until the UART has processed data.
;
; Entry Parameters : None
;
; Returns          : Nothing
;
;****************************************************************************

midiPollUART            PROC USES AX DX
               
                        ; 
                        ; Setup dx with the midi port
                        ;
                        mov     dx, midiPort
        
                        ;
                        ; Increment dx to set it to the control register on the
                        ; UART.
                        ;
                        inc     dx

                        ;
                        ; Read in the value at the control register.
                        ;
@@Poll:                 in     al,dx

                        ;
                        ; And it with a 2h to look at bit 1
                        ;
                        and     al, 2h

                        ;
                        ; If it is not set then loop until it is set.
                        ; If it is set then the UART is ready to receive more
                        ; data.
                        ;
                        cmp     al,0
                        je     @@Poll

                        ;
                        ; Return
                        ;
                        ret

midiPollUART            ENDP     

;****************************************************************************
;
; Function         : midiOutByte
;
; Description      : Ouputs a byte to the midi device
;
; Entry Parameters : Byte
;
; Returns          : Nothing
;
;****************************************************************************

midiOutByte             PROC   USES DX AX, midiData:BYTE

                        ;
                        ; Wait until the uart is ready to send data
                        ;
                        call    midiPollUART
                
                        ;
                        ; Ouput the byte
                        ;
                        mov     dx, midiPort
                        mov     al, midiData
                        out     dx, al

                        ;
                        ; return
                        ;
                        ret

midiOutByte             ENDP
                    
;****************************************************************************
;
; Function         : midiInit
;
; Description      : sets up irq vector and printer ports
;
; Entry Parameters : void midiInitialize(  unsigned  midiPort,  unsigned char irqVector )
;
; Returns          : Nothing
;
;****************************************************************************

midiInit                PROC USES ES DI SI,\
                                midi_Port  : WORD, \
                                irq_Number : BYTE

                        ;
                        ; Store the passed midi port
                        ;
                        mov     ax, word ptr midi_Port
                        mov     midiPort, ax

                        ;
                        ; Store the passed interrupt request number
                        ;
                        mov     al, byte ptr irq_Number
                        mov     byte ptr irqNumber, al

                        ;
                        ; Initialize the circular buffer for data transfer. Also
                        ; initialize the buffer pointer, and fetch pointer. Zero 
                        ; out bytes in queue.
                        ;
                        mov     BYTE PTR midiByteReady,     0h
                        mov     WORD PTR midiInternalIndex, 0h
                        mov     WORD PTR midiFetchIndex,    0h
                        mov     WORD PTR midiBytesInQueue,  0h

                        ;
                        ; Read interrupt vector table to retrieve the previous
                        ; interrupt vector.      
                        ;
                        ; Interrupt 21 function 35 returns the vector to the currently set 
                        ; interrupt handler.
                        ;
                        mov     ah, 35h      

                        ;
                        ; Pass the interrupt request number. Add eight to the passed number because
                        ; interrupt request vectors start at eight. 
                        ;
                        mov     al, byte ptr irq_Number   
                        add     al, 08h                
                                         
                        ;
                        ; Call the interrupt function to get the interrupt request
                        ; vector.
                        ;
                        int     21h

                        ;
                        ; Store the segment and offset of the returned vector.
                        ;
                        mov     irqOffset, bx
                        mov     irqSegment, es

                        ;
                        ; Disable hardware interrupt to prevent irq firing while
                        ; setting new vector.
                        ;
                        cli

                        ;
                        ; Interrupt 21 function 25 sets an interrupt vector
                        ;
                        mov     ah,25h

                        ;
                        ; Pass the interrupt request number. Add eight to the passed number because
                        ; interrupt request vectors start at eight. 
                        ;
                        mov     al,byte ptr irq_Number
                        add     al,08h

                        ;
                        ; ds is used in passing parameters to this function, so it needs
                        ; to be stored on the stack.        
                        ; 
                        push    ds

                        ;
                        ; Setup ds with the segment of the new interrupt handler.
                        ;
                        mov     dx,SEG midiIrqHandler
                        mov     ds,dx

                        ;
                        ; Setup dx with the offset of the new interrupt handler.
                        ;
                        mov     dx,OFFSET midiIrqHandler

                        ;
                        ; Call the set interrupt vector function.
                        ;
                        int     21h

                        ;
                        ; Restore ds
                        ;
                        pop     ds

                        ;
                        ; Reenable the harware interrupts.
                        ;
                        sti

                        ;
                        ; Enable the 8259( interrupt controller ) interrupt 
                        ; request line.
                        ;
                        ; Was irq 2 passed?
                        ;
                        mov     al, byte ptr irq_Number
                        cmp     al,02h

                        ;
                        ; No. Check to see if irq 3 was passed.
                        ;
                        jne     @@Irq3

                        ;
                        ; Enable interrupt request line 2
                        ;
                        ; Read in the value at the interrupt request port.
                        ;
                        in      al, 21h

                        ;
                        ; And off the bit to enable irq 2.
                        ;
                        and     al, 0fbh 

                        ;
                        ; Output the value to the interrupt request port.
                        ;
                        out     21h, al

                        ;
                        ; Jump to reset device.
                        ;
                        jmp     @@ResetDevice

                        ;
                        ; Was irq 3 passed?
                        ;
@@Irq3:                  mov     al, byte ptr irq_Number
                        cmp     al, 03h

                        ;
                        ; No. Check to see if irq 5 was passed.
                        ;
                        jne     @@Irq5

                        ;
                        ;
                        ; Enable interrupt request line 3
                        ;
                        ; Read in the value at the interrupt request port.
                        ;
                        in     al, 21h

                        ;
                        ; And off the bit to enable irq 3.
                        ;
                        and       al, 0f7h

                        ;
                        ; Output the value to the interrupt controller
                        ;
                        out     21h, al

                        ;
                        ; Jump to reset device.
                        ;
                        jmp     @@ResetDevice

                        ;
                        ; Was irq 5 passed?
                        ;
@@Irq5:                 mov     al, byte ptr irq_Number
                        cmp     al, 05h

                        ;
                        ; No. Check to see if irq 7 was passed.
                        ;
               jne     @@Irq7

                        ;
                        ;
                        ; Enable interrupt request line 5
                        ;
                        ; Read in the value at the interrupt request port.
                        ;
                        in     al, 21h

                        ;
                        ; And off the bit to enable irq 5.
                        ;
                        and       al, 0dfh

                        ;
                        ; Output the value to the interrupt controller
                        ;
                        out     21h, al

                        ;
                        ; Jump to reset device.
                        ;
                        jmp     @@ResetDevice

                        ;
                        ; Was irq 7 passed?
                        ;
@@Irq7:                 mov     al, byte ptr irq_Number
                        cmp     al, 07h

                        ;
                        ; No. Then reset device, however if the check falls
                        ; through here for the interrupt line then no interrupts
                        ; will happen.
                        ;
                        jne     @@ResetDevice

                        ;
                        ; Enable interrupt request line 7
                        ;
                        ; Read in the value at the interrupt request port.
                        ;
                        in     al, 21h

                        ;
                        ; And off the bit to enable irq 7.
                        ;
                        and       al, 07fh

                        ;
                        ; Output the value to the interrupt controller
                        ;
                        out     21h, al

                        ;
                        ; Reset internal midi device.
                        ;
                        ; Setup dx with the port to access.
                        ;
@@ResetDevice:          mov     dx, midiPort

                        ;
                        ; Add one to dx to allow us to access to port + 1
                        ; which is the UART'S control register.
                        inc     dx

                        ;
                        ; Outputting a 3 to this register resets the UART.
                        ;
                        mov     al, 03h          
                        out     dx, al
               
                        ;
                        ; Setup the UART to interrupt on new data from the external
                        ; midi device. e.g. Keyboard, Guitar...
                        ;
                        ; Writing a 95h to the control port setup up the UART to interrupt
                        ; on new data.
                        ;
                        mov     al, 95h   
                        out     dx, al
               
                        ;
                        ; Set the midi device to accept data on channel 1 by 
                        ; sending a note on control change on channel 1.
                        ;
                        ; Call the poll UART function to make sure the UART is
                        ; ready to accept data.
                        call    midiPollUART

                        ;
                        ; Set the port to access to the original midi port.
                        ;
                        dec      dx

                        ;
                        ; Send the 90h value ( note on ) to the UART.
                        ;
                        mov     al, 90h
                        out     dx, al

                        ;
                        ; Signal to the interrupt controller that there has been an 
                        ; interrupt recieved to make sure that the next interrupt will
                        ; happen.
                        ;
                        mov     al, 20h
                        out     20h, al

                        ;
                        ; Return from the initialization function.
                        ;
                        ret

midiInit                ENDP


;****************************************************************************
;
; Function         : midiUninit
;
; Description      : Resets orignal interrupt servicing information.
;
; Entry Parameters : Nothing
;
; Returns          : Nothing
;
;****************************************************************************

midiUninit              PROC USES DI SI

                        ;
                        ; Disable all hardware interrupts to ensure that no 
                        ; interrupt will happen while we are changing the 
                        ; vectors.
                        ;
                        cli

                        ;
                        ; Write out a 3 to the control port of the UART to 
                        ; disable it from firing any further interupts.
                        ;
                        ; Setup dx with the port.
                        ;
                        mov     dx, midiPort

                        ;
                        ; Increment dx to access the control register.
                        ;
                        inc     dx

                        ;
                        ; Output a 3 to the control register to reset the UART.
                        ; 
                        mov     al, 3h
                        out     dx, al


                        ;
                        ; Disable interrupt request line on the interrupt controller.
                        ;
                        ; Is the used interrupt line irq 2?
                        ;
                        mov     al, byte ptr irqNumber
                        cmp     al, 02h

                        ;
                        ; No. Check for irq 3.
                        ;
                        jne     @@Irq3

                        ;
                        ; Disable interrupt request line 2
                        ;
                        ; Read in the interrupt control register from the interrupt controller.
                        ;
                        in      al, 21h

                        ;
                        ; Or it with a 4 to disable irq 2.
                        ;
                        or      al, 04h

                        ; 
                        ; Write out the value to the interrupt controller.
                        ;
                        out     21h,al

                        ;
                        ; Jump to reset vector.
                        ;
                        jmp     @@ResetVector

                        ;
                        ; Is the used interrupt line irq 3?
                        ;
@@Irq3:                 mov     al, byte ptr irqNumber
                        cmp     al, 03h

                        ;
                        ; No. Check for irq 5.
                        ;
                        jne     @@Irq5

                        ;
                        ; Disable interrupt request line 3
                        ;
                        ; Read in the interrupt control register from the interrupt controller.
                        ;
                        in      al, 21h

                        ;
                        ; Or it with a 8 to disable irq 3.
                        ;
                        or      al, 08h

                        ; 
                        ; Write out the value to the interrupt controller.
                        ;
                        out     21h,al

                        ;
                        ; Jump to reset vector.
                        ;
                        jmp     @@ResetVector

                        ;
                        ; Is the used interrupt line irq 5?
                        ;
@@Irq5:                 mov     al, byte ptr irqNumber
                        cmp     al, 05h

                        ;
                        ; No. Check for irq 7.
                        ;
                        jne     @@Irq7

                        ;
                        ; Disable interrupt request line 5
                        ;
                        ; Read in the interrupt control register from the interrupt controller.
                        ;
                        in      al, 21h

                        ;
                        ; Or it with a 20 to disable irq 5.
                        ;
                        or      al, 20h

                        ; 
                        ; Write out the value to the interrupt controller.
                        ;
                        out     21h,al

                        ;
                        ; Jump to reset vector.
                        ;
                        jmp     @@ResetVector

                        ;
                        ; Is the used interrupt line irq 7?
                        ;
@@Irq7:                  mov     al, byte ptr irqNumber
                        cmp     al, 07h

                        ;
                        ; No. Then no valid interrupt numbers were passed
                        ;
                        jne     @@ResetVector
                        
                        ;
                        ; Disable interrupt request line 7
                        ;
                        ; Read in the interrupt control register from the interrupt controller.
                        ;
                        in      al, 21h

                        ;
                        ; Or it with a 80 to disable irq 7.
                        ;
                        or      al, 80h

                        ; 
                        ; Write out the value to the interrupt controller.
                        ;
                        out     21h,al

                        ;
                        ; Reset the interrupt vector.
                        ;
                        ; Interrupt 21 function 25 will set a new vector.
                        ;
@@ResetVector:          mov     ah, 25h

                        ;
                        ; Pass the interrupt request number. Add eight to the passed number because
                        ; interrupt request vectors start at eight. 
                        ;
                        mov     al,byte ptr irqNumber
                        add     al,08h

                        ;
                        ; ds is used in passing parameters to this function
                        ; so it needs to be stored on the stack.
                        ;
                        push    ds

                        ;
                        ; Setup ds with the segment of the old interrupt handler.
                        ;
                        mov     dx, irqSegment
                        mov     ds, dx

                        ;
                        ; Setup dx with the offset of old interrupt handler.
                        ;
                        mov     dx, irqOffset

                        ;
                        ; Call the set interupt function.
                        ;
                        int     21h

                        ;
                        ; Restore ds.
                        ;
                        pop     ds

                        ;
                        ; Enable previously disabled hardware interrupts
                        ;
                        sti

                        ; 
                        ; Return from function.
                        ;
                        ret

midiUninit              ENDP

;****************************************************************************
;
; Function         : midiFetchByte
;
; Description      : Fetches a byte from the circular buffer.
;
; Entry Parameters : None
;
; Returns          : al - Byte from circular buffer at present buffer position
;
;****************************************************************************

midiFetchByte           PROC USES SI BX

                        ; 
                        ; Setup si with the offset off the midi buffer.
                        ;
                        mov     si, offset midiBuffer

                        ;
                        ; Setup bx with the current fetch index.
                        ;
                        mov     bx, midiFetchIndex

                        ;
                        ; Get the byte at the offset of the buffer + current
                        ; fetch index.
                        ;
                        mov     al, byte ptr [ si + bx ]

                        ;
                        ; Check to see if the current fetch index is at the end of the buffer.
                        ;
                        cmp     bx, MIDI_BUFFERSIZE

                        ;
                        ; If not at end of buffer then jump to index ok
                        ;
                        jne     @@midiIndexOK

                        ;
                        ; Set the fetch index to -1.
                        ;
                        mov     WORD PTR midiFetchIndex, -1h

@@midiIndexOK:
                        ;
                        ; Increment the fetch index.
                        ;
                        add     WORD PTR midiFetchIndex, 1h

                        ;
                        ; Subtract from the number of bytes in the buffer.
                        ;
                        sub     WORD PTR midiBytesInQueue, 1h

                        ;
                        ; If the fetch index catches up to the internal index
                        ; then set the byte ready flag to zero.
                        ;
                        mov     bx, WORD PTR midiFetchIndex
                        cmp     bx, WORD PTR midiInternalIndex
                        jne      @@Exit

                        ;
                        ; Set the byte ready flag to 0
                        ;
                        mov     BYTE PTR midiByteReady, 0h
@@Exit:
                        ;
                        ; Return from function.
                        ;
                        ret

midiFetchByte           ENDP

;****************************************************************************
;
; Function         : midiGetByte
;
; Description      : Reads in a byte from the midi port(  UART  ).
;
; Entry Parameters : None
;
; Returns          : al - Byte read from midi port.
;
;****************************************************************************

midiIntGetByte          PROC

                        ;
                        ; Setup dx with the port to read from. 
                        ;
                        mov     dx, midiPort

                        ;
                        ; Read in the byte.
                        ;
                        in     al, dx

                        ;
                        ; Return from the function.
                        ;
                        ret

midiIntGetByte          ENDP

;****************************************************************************
;
; Function         : midiIrqHandler
;
; Description      : Interrupt request handler to get a byte from the midi device,
;                    and handle the circular buffer, when a new byte is ready.   
;
; Entry Parameters : None
;
; Returns          : Nothing
;
;****************************************************************************

midiIrqHandler          PROC

                        ;
                        ; Push all of the affected registers.
                        ;
                        push     ds
                        push     es
                        push     si
                        push     di
                        push     dx
                        push     cx
                        push     bx
                        push     ax


                        ;
                        ; Setup ds relative to the data segment.
                        ;
                        mov     ax, @DATA
                        mov     ds, ax

                        ;
                        ; Get the byte from the midi device.( returned in al )
                        ;
@@getByte:              call    midiIntGetByte

                        ;
                        ; Setup si with the offset of the circular buffer.
                        ;
                        mov     si, offset midiBuffer

                        ;
                        ; Setup bx with the internal index.
                        ;
                        mov     bx, midiInternalIndex

                        ;
                        ; Store the byte at the offset of the buffer + internal
                        ; index.
                        ;
                        mov     BYTE PTR  [ si + bx ], al

                        ;
                        ; Check to see if the internal index is greater than the size of
                        ; the buffer.
                        ;
                        cmp     bx, MIDI_BUFFERSIZE

                        ;
                        ; If not then jump to index ok.
                        ;
                        jne     @@midiIndexOK

                        ;
                        ; Set the internal index to -1 to signal that 
                        ; the queue is full.
                        ;
                        mov     WORD PTR midiInternalIndex, -1

                        ;
                        ; Add one to the internal index.
                        ;
@@midiIndexOK:          add     WORD PTR midiInternalIndex, 1h

                        ;
                        ; Increment the bytes in buffer counter.
                        ;
                        add     WORD PTR midiBytesInQueue, 1h

                        ;
                        ; Set the midi byte ready flag.
                        ;
                        mov     BYTE PTR midiByteReady, 1h

                        ;
                        ; Check to see if there is another byte ready.
                        ; Bit 0 of the status register( port + 1 ) will
                        ; be high when a byte is ready.
                        ; In most cases there will not be another byte
                        ; ready within the interrupt handler; however,
                        ; in cases where there's a large amount of time
                        ; taken up by the handler another byte may have
                        ; time to come in.  In the interest of good
                        ; coding, the only processing that should be 
                        ; done in the interrupt handler is to get a byte
                        ; and store it.  All other processing, such as
                        ; filtering of the bytes, should be done in the 
                        ; main programming loop.
                        ; 
                        mov     dx, midiPort
                        inc     dx              ; status register 
                        in      al, dx
                        and     al, 1h          ; bit 0

                        ;
                        ; No byte was ready, exit.
                        ;
                        jz      @@Exit

                        ;
                        ; Jump back to the beginning of the handler and
                        ; get the byte that is ready.
                        ;
                        jmp     @@getByte

                        ;
                        ; Signal the interrupt controller that the interrupt has
                        ; been received.
                        ;
@@Exit:                 mov     al, 20h
                        out     20h, al

                        ;
                        ; Restore all affected registers.
                        ;
                        pop       ax
                        pop       bx
                        pop       cx
                        pop       dx
                        pop       di
                        pop       si
                        pop       es
                        pop       ds

                        ;
                        ; Restore the C stack frame
                        ;
                        mov     sp, bp
                        pop     bp

                        ;
                        ; Return from interrupt.
                        ;
                        iret

midiIrqHandler          ENDP

                        END
