;****************************************************************************
;
;  File              : playset.asm
;  Date Created      : 3/20/89
;  Description       : Module to initializing the hardware timer on the PC
;                      for playback (non-DMA).
;
;  Last Modification : February 8, 1992
;
;  Additional Notes  :
;
;****************************************************************************
;            Copyright (c) 1992,  Covox, Inc.  All Rights Reserved
;****************************************************************************

IFNDEF NO_DOS_SEG
  DOSSEG
ENDIF


      INCLUDE model.inc

      INCLUDE timer.inc
      INCLUDE system.inc
      INCLUDE getxbyte.inc


      EXTRN   _sr_reset_8253_timer:PROC
      EXTRN   _sr_set_8253_timer:PROC

      .DATA

VM_USED_MASK      EQU   1      ; bit mask if VM card used
LPT_USED_MASK     EQU   2      ; bit mask if SpeechThing or External used
ISP_USED_MASK     EQU   8      ; bit mask if Internal Speaker used
VALUE_USED_MASK   EQU   16      ; bit mask if using port address passed

vmII_table        DW   22fh, 24fh, 28fh, 2cfh
dawrite           DW   0
hardware_type     DB   0         ; bit flags of what hardware type is being used
saved_portb       DB   0         ; Save hardware port b if timer not used

no_timer          DB   0
treble_flag       DB   0
sample_rate       DB   0
diffbyte1         DB   0
diffbyte2         DB   0


      .CODE

      PUBLIC playByteInit
      PUBLIC playByteUninit
      PUBLIC playByte

;*********************************************************************
;**
;**  playByteInit:  Initialize for playing
;**
;**  Call to initialize the hardware before any plays, sets up port numbers
;**  and varables, as well as hardware.
;**
;**  Called by C as: playByteInit(port,rate,no_timer_flag,treble_flag);
;**  C declaration:  int playByteInit(int, int, int, int);
;**
;**  Return:
;**   The phyical port number used
;**  Passed: 
;**   port:   The port where sound is to come in.  If the number is 0-3
;**      then the Voice Master II ports 22f, 24f, 28f, or 2cf are used
;**      respectively.  Otherwise the value is considered to be the
;**      actual port address.
;**      In the Tandy version of this library, the port variable is
;**      ignored.  For calling compatibility, the parameter is still
;**      passed.
;**             Added Jan 90: If port number between 4-6 then external Covox
;**             digitizer is used.
;**   rate:   If 0, default (132) is used.  If greater then 229 uses 229
;**      Otherwise the value is used as is.
;**   no_timer_flag: If zero uses timer2 of the IBM 8253, if not zero don't
;**      wait for timer when reading bytes.
;**   treble:   A flag to indicate that differentiation is to be used on
;**      output.  0 = No, any other number means yes.
;**************************************************************************

playByteInit   PROC   USES   DS, \
      port_out:WORD, \
      rate:BYTE, \
      no_timer_flag:WORD, \
      treble:WORD

      mov   ax,@data
      mov   ds,ax

      mov   bx,port_out
      mov   hardware_type,0      ; Clear assumed hardware type
      cmp   bx,04
      jb    SHORT use_vmII_ports
      cmp   bx,07
      jb    SHORT use_lpt123_ports
      cmp   bx,8
      jne   SHORT @F
      jmp   use_isp
@@:
      mov   hardware_type,VALUE_USED_MASK   ; Set hardware to using given port number
      mov   dawrite,bx
      jmp   end_translation

use_vmII_ports:
      mov   hardware_type,VM_USED_MASK   ; Set hardware to using Voice Master Card
      shl   bx,01
      mov   ax,[vmII_table + bx]
      mov   dawrite,ax
      mov   dx,ax
      sub   dx,2
      out   dx,al              ; on old VM set to voice channel
      jmp   end_translation

use_lpt123_ports:
      ASSUME   es:system_data
      push  es
      mov   ax,system_data
      mov   es,ax
      sub   bx,04         ;convert from 4-6 to 0-2
      shl   bx,01         ;Now convert to a pointer
      mov   ax,es:[bx + lpt1]
      pop     es

      ASSUME   es:NOTHING
      or    ax,ax
      jnz   SHORT @F
      jmp   init_ret                       ; Return error if LPT not present.
@@:
      mov   hardware_type, LPT_USED_MASK   ; Set hardware to using external or SpeechThing
      mov   dawrite, ax

      external_on ax                       ; Set printer port address for macro
      jmp   end_translation

use_isp:
      mov   hardware_type, ISP_USED_MASK   ; Set hardware to using internal speaker
      mov   dawrite, 8                     ; uses logical port number

end_translation:
      mov   ax, no_timer_flag
      or    ax, ax             ; make sure use timer flag 0 or 1
      jnz   SHORT @F           ; jump if not using timer

      call  _sr_set_8253_timer
      xor   al,al
      jmp   SHORT set_use_flag
@@:
      in    al,061h           ; Save portb to be restored because internal speaker
                              ; could be used with no timer
      mov   saved_portb, al
      mov   al, 1

set_use_flag:
      mov   no_timer, al

      mov   ax, treble
      or    ax, ax               ; make sure use treble flag 0 or 1
      jz    SHORT @F             ; jump if not treble
      mov   diffbyte2, 080h      ; reset diff byte to center line
      mov   treble_flag, 1
      jmp   SHORT done_treble
@@:
      mov   treble_flag,0

done_treble:
      xor   ah, ah
      mov   al,rate
      cmp   al,0
      jne   SHORT @F
      xor   ah, ah
      mov   al, 132         ; set default rate 
      jmp   SHORT rate_done
@@:
      xor   ah, ah
      cmp   al, 229
      jbe   SHORT rate_done
      xor   ah, ah
      mov   al, 229         ; if rate over 229 set to 229

rate_done:
      mov   sample_rate, al
      
      mov   ax, dawrite     ; return the phyical port address

init_ret:
      ret

playByteInit   ENDP


;*********************************************************************
;**
;**  playByteUninit:  Uninitialize from playing, call after playByteInit
;**
;**  Called by C as: playByteUninit();
;**  C declaration:  int playByteUninit(void);
;**
;**************************************************************************

playByteUninit   PROC   USES   DS

      mov   ax,@data
      mov   ds,ax

      cmp   no_timer,0
      jnz   SHORT @F
      call   _sr_reset_8253_timer
      jmp   SHORT done_timer
@@:
      mov   al,saved_portb      ; Resore port b since not restored by
                  ; timer reset
      out   061h,al
done_timer:
           test   hardware_type,LPT_USED_MASK
      jz   SHORT exit
      external_off dawrite
      jmp   SHORT exit
exit:      
      ret
playByteUninit   ENDP

;*********************************************************************
;**
;**  playByte:  Put data from device of data selected with 
;**       init_recording
;**
;**  Called by C as: playByte(int);
;**  C declaration:  int playByte(int);
;**
;**  Assembly language programs that call this function assume that
;**  only AX, and DX are changed, and only lower 8 bits of out_value are used.
;**
;**************************************************************************

playByte   PROC   \
      out_value:BYTE

   IF      @DataSize                      
      push   DS         ; if more then one data segment
      mov   dx,@data
      mov   ds,dx
   ENDIF

      cmp   no_timer,0
      jne   SHORT after_wait   ; skip wait loop if no hardware timer to be used

wait_loop:
      in     al,timer2       ; read lsb from counter register.
      cmp    al,sample_rate  ; compare timer value to sampling/playback rate.
      ja     SHORT wait_loop   ; keep waiting if not.
      mov    al,0            ; set timer again
      out    timer2,al                 

after_wait:
      xor    ah, ah
      mov    al,out_value

      cmp    treble_flag,0   ; former difference macro
      je     SHORT done_diff

      mov   diffbyte1,al
      sub   al,diffbyte2
      mov   dl,diffbyte1
      mov   diffbyte2,dl
      neg   al
      add   al,80h
      cmp   al,40h
      ja   SHORT differ1
      mov   al,0
      jmp   SHORT done_diff
differ1:
      cmp   al,0C0h
      jb   SHORT differ2
      mov   al,0FFh
      jmp   SHORT done_diff
differ2:
      shl   al,1
      xor   al,80h
done_diff:
      test   hardware_type,ISP_USED_MASK
      jz   SHORT outbyte3   ; jump if not internal speaker

      cmp   al,080h+4   ; Only flip bits when above theshold
      jae   SHORT outbyte_above   ; jump if above threshold
      cmp   al,080h-4
      ja   SHORT outbyte4   ; no output, within center bounds
               ; Below threshold
      in   al,061h      ; Get IBM status info to flip bit if needed, port b
      or   al,02h      ; set speaker bit because above theshold
      and   al,0FEh      ; mask so direct speaker control
      jmp   SHORT outbyte2
outbyte_above:
      in   al,061h      ; Get IBM status info to flip bit if needed, port b
      and   al,0FCh      ; clear bit speaker bit because below threshold
outbyte2:
      out   061h,al      ; port b
      or   al,1      ; Bit #0 must be reset to continue countdown.
      out   061h,al
      jmp   SHORT outbyte4
outbyte3:
      mov   dx,dawrite
      out   dx,al
outbyte4:

      IF @DataSize                      
        pop  DS  ; if more then one data segment
      ENDIF

      ret

playByte   ENDP


      END
