//
//
// File           : tsrinit.c
// Date Created   : 4/23/91
// Last Updated   : 2/11/92
//
// Description    : This program parses the command line for legal options
//                  for TSRPLAY.  If the command line has no errors then
//                  calls to load or change an already loaded copy of
//                  TSRPLAY are executed.
//
//                  This program was designed to be complied with Microsoft
//                  C version 6.0.  See TSRPLAY.MAK make file for complete
//                  compilation requirements.
//
// Programmer(s)  : Ryan Hanlon
// 
//----------------------------------------------------------------------------
//             Copyright (c) 1992, Covox, Inc. All Rights Reserved                   
//----------------------------------------------------------------------------

#include <bios.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include <io.h>
#include <fcntl.h>
#include <conio.h>
#include "cvxdigi.h"
#include "cvxutil.h"
#include "parse.h"


// Error codes found during parsing of command line.
//
enum
{
   _ERROR_ILLEGAL_PLAYBACK_RATE = 1000,
   _ERROR_FILE_COUNT_EXCEEDED,
   _ERROR_ILLEGAL_PLAYBACK_COUNT,      // An illegal number of file playback count was requested.
   _ERROR_REPEAT_WITHOUT_FILE,         // A repeat count was specified before any files were sepcified.
   _ERROR_RATE_WITHOUT_FILE            // A rate specified before any files were sepcified.
};

BOOL  tsrUninstallFlag      = _FALSE;  // Flag variable used to indicate that 
                                       // user wants to uninstall tsr.
                             
BOOL  tsrPauseFlag          = _FALSE;  // Flag variable used to indicate that
                                       // user wants to pause tsr playback.


BOOL  tsrUnpauseFlag        = _FALSE;  // Flag variable used to indicate that
                                       // user wants to unpause tsr playback.

BOOL  tsrFlushFilesFlag     = _FALSE;  // Flag variable used to indicate that
                                       // user wants to flush all files that   
                                       // are currently being played back.

BOOL  tsrDisplayOptionsFlag = _FALSE;  // Flag used to indicate whether to put
                                       // command line syntax message to the
                                       // screen.

BOOL  silentFlag            = _FALSE;  // Variable used in this program that 
                                       // indicates no screen output.

// Functions used to parse command line options.
//
WORD  parseCommandLine(WORD argc, PSTR argv[]);
WORD  parseFilename( PSTR );
WORD  parsePlaybackRate( PSTR );
WORD  parseBuffers( PSTR );
WORD  parseBufferSize( PSTR );
WORD  parseFileRepeatCount( PSTR option );
WORD  parsePause( PSTR );
WORD  parseUnpause( PSTR );
WORD  parseFlushFiles( PSTR );
WORD  parseIRQNumber( PSTR );
WORD  parseSilent( PSTR );
WORD  parseUninstall( PSTR );
WORD  parseJumper( PSTR );
WORD  parseChannel( PSTR );

// Function to display valid options to the screen.
//
VOID  displayCommandLineSyntax( VOID );

// 
PSTR  tsrPlayVersionString;
PSTR  tsrPlayProgramID;

// ------------------------------------------------------------------------
// --------------------------   BEGIN MAIN   ------------------------------
VOID main( WORD argc, PSTR argv[] )
{
   BOOL tsrResidentFlag;
   WORD returnValue;
   PSTR string;

   // Call function to get version of TSRPLAY.
   tsrPlayProgramID = tsrGetProgramID();

   // Call function to get version of TSRPLAY.
   tsrPlayVersionString = tsrGetVersionString();

   // Get all options from the command line that have been entered
   // by user. If illegal options have been used then display  
   // commandLineSyntaxMessage and return control to DOS.
   //
   if( parseCommandLine(argc, argv) )
   {
      if( !silentFlag )
         displayCommandLineSyntax();
      exit(0);
   }

   // Check to see if a request to display option screen has been made.
   //
   if( tsrDisplayOptionsFlag )
   {
      if( !silentFlag )
         displayCommandLineSyntax();
      exit(0);
   }

 
   // Set flag that indicates whether TSRLAY.EXE is resident.
   //
   tsrResidentFlag = tsrResident( );

   // Check to see if a request has been made to uninstall resident tsr.
   // If it is resident and a request to uninstall has been then display a 
   // message, uninstall INT1ATSR and return to dos.
   //
   if ( tsrResidentFlag && tsrUninstallFlag )
   {
      if( !silentFlag )
      {
         printf("Uninstalling %s\n", tsrPlayProgramID );
      }

      tsrRemoveSystem();

      exit(0);
   }

   // Check to see if a request to uninstall has been made and
   // tsr has not been previously installed.
   //
   if ( !tsrResidentFlag && tsrUninstallFlag)
   {
      if( !silentFlag )
      {
         printf("%s has not been previously installed.\n", tsrPlayProgramID);
      }
      exit(0);
   }

   // Check to see if tsr is already resident and user is requesting to pause 
   // playback.
   //
   if ( tsrResidentFlag && tsrPauseFlag )
   {
      if( !silentFlag )
      {
         printf("Pausing playback of %s.\n", tsrPlayProgramID );
      }
      tsrPause();
      exit(0);
   }


   // Check to see if tsr is not resident and user is requesting to pause 
   // playback.
   //
   if ( !tsrResidentFlag && tsrPauseFlag )
   {
      if( !silentFlag )
      {
         printf("%s has not been previously installed.\n", tsrPlayProgramID);
      }
      exit(0);
   }
 

   // Check to see if tsr is already resident and user is requesting to unpause
   // playback.
   //
   if ( tsrResidentFlag && tsrUnpauseFlag)
   {
      if( !silentFlag )
      {
         printf("Continuing playback of %s.\n", tsrPlayProgramID );
      }
      tsrUnpause();
      exit(0);
   }
 
   // Check to see if user is requesting to unpause when TSRPLAY.EXE is not
   // paused.
   //
   if ( !tsrResidentFlag && !tsrPauseFlag && tsrUnpauseFlag)
   {
      if( !silentFlag )
      { 
         printf("%s has not been previously paused.\n", tsrPlayProgramID );
      }
      exit(0);
   }

   // At this point if tsr is resident then the only option we have left is to
   // replave the files that are being played back in the tsr.
   //
   if ( tsrResidentFlag  && tsrFlushFilesFlag )
   {
      if( !silentFlag )
      {
         printf("Flushing all files from %s.\n", tsrPlayProgramID );
      }
      tsrFlushFiles();
      exit(0);
   }

   // At this point if tsr is resident then the only option we have left is to
   // replave the files that are being played back in the tsr.
   //
   if ( tsrResidentFlag )
   {
      if( !silentFlag )
      {
         printf("Replacing playback files in resident %s.\n", tsrPlayProgramID );
      }

      tsrSetupNewFiles();
      exit(0);
   }



   if( !silentFlag )
   {
      printf("Installing %s Version %s\n", tsrPlayProgramID, tsrPlayVersionString);
   }
  
   // Hook all interrupts and start tsr.
   //
   returnValue = tsrStart();


   // If tsrStart returns then an error occured.
   //
   if( !silentFlag )
   {
      printf("\nERROR : %d \n", returnValue );

      string = cvxGetErrorString( returnValue );

      if( string != _NULL )
      {
         printf("%s\n", string);
      }
      else
      {
         printf("Unknown TSRPLAY failure.\n" );
      }
   
      printf(" ... Installation aborted.\n");
   }
}

// ---------------------------    END MAIN    --------------------------------
//----------------------------------------------------------------------------


// Parse the options entered by user of program.
//
WORD parseCommandLine(WORD argc, PSTR argv[])
{

   static PARSE_INFO commandLineParse[] =
   {
      { 0 , 0   , parseFilename},
      { 1 , 'R' , parsePlaybackRate},
      { 1 , 'K' , parseBufferSize},
      { 1 , 'N' , parseFileRepeatCount},
      { 0 , 'Z' , parsePause},
      { 0 , 'W' , parseUnpause},
      { 1 , 'B' , parseBuffers},
      { 0 , 'F' , parseFlushFiles},
      { 1 , 'J' , parseJumper},
      { 1 , 'C' , parseChannel},
      { 1 , 'I' , parseIRQNumber},
      { 0 , 'S' , parseSilent},
      { 0 , 'Q' , parseUninstall}
   };

   BYTE  commandLineToken;
   unsigned short commandLineCount;

   commandLineToken = '/';

   commandLineCount = sizeof(commandLineParse) / sizeof(PARSE_INFO);

   SetParse(commandLineToken, commandLineCount, commandLineParse);

   if(argc == 1)
   {
      return( 1 );
   }
   else
   {
      if( Parse(argc, argv) )
      {
         return( 1 );
      }
      else
      {
         return( 0 );
      }
   }

}


// Function to capture request to print out legal options.
//
WORD parseFilename(PSTR option)
{
   if( _tsrPlayInfo.fileCount == _FILE_COUNT_MAX )
   {
      return(_ERROR_FILE_COUNT_EXCEEDED);
   }
   else
   {
      strcpy(_tsrPlayInfo.fileName[_tsrPlayInfo.fileCount++], option);

      return(_ERROR_NONE);
   }
}


// Function to capture request to print out legal options.
//
WORD parsePlaybackRate(PSTR option)
{
   // Check to make sure that file count does not equal 0. If it does then
   // a rate was specified before a file was specified. Exit with error
   // if this condition is true.
   //
   if( _tsrPlayInfo.fileCount == 0 )
   {
      return( _ERROR_RATE_WITHOUT_FILE );
   }


   _tsrPlayInfo.fileRate[ (_tsrPlayInfo.fileCount - 1) ] = atoi(option);

   if( (_tsrPlayInfo.fileRate[ (_tsrPlayInfo.fileCount - 1) ] < _RATE_MINIMUM) || (_tsrPlayInfo.fileRate[ (_tsrPlayInfo.fileCount - 1) ] > _RATE_MAXIMUM) )
   {
      return(_ERROR_ILLEGAL_PLAYBACK_RATE);
   }
   else
   {
      return(_ERROR_NONE);
   }

}


// Function to capture the number of buffers to use for DMA playback.
//
WORD parseBuffers(PSTR option)
{
   _tsrPlayInfo.bufferCount = atoi(option);
   if( (_tsrPlayInfo.bufferCount > 0) && (_tsrPlayInfo.bufferCount <= _BUFFER_COUNT_MAX) )
   {
      return(_ERROR_NONE);
   }
   else
   {
      return(_ERROR_ILLEGAL_BUFFER_SIZE);
   }

}


// Function to capture size of each playback buffer.
//
WORD parseBufferSize(PSTR option)
{
   _tsrPlayInfo.bufferSize = atoi(option);
   if( (_tsrPlayInfo.bufferSize > 0) && (_tsrPlayInfo.bufferSize <= _BUFFER_SIZE_MAX) )
   {
      // Change _tsrPlayInfo.bufferSize from K bytes to bytes.
      //
      _tsrPlayInfo.bufferSize = _tsrPlayInfo.bufferSize;

      return(_ERROR_NONE);
   }
   else
   {
      return(_ERROR_ILLEGAL_BUFFER_SIZE);
   }
}


// Function to capture number of times to playback each file.
//
WORD parseFileRepeatCount(PSTR option)
{
   // Check to make sure that file count does not equal 0. If it does then
   // a repeatCount was specified before a file was specified. Exit with
   // error if this condition is true.
   //
   if( _tsrPlayInfo.fileCount == 0 )
   {
      return( _ERROR_REPEAT_WITHOUT_FILE );
   }

   _tsrPlayInfo.fileRepeat[ (_tsrPlayInfo.fileCount - 1) ] = atoi(option);

   if( (_tsrPlayInfo.fileRepeat[(_tsrPlayInfo.fileCount - 1)] == -1) ||
       ((_tsrPlayInfo.fileRepeat[(_tsrPlayInfo.fileCount - 1)] > 0) && (_tsrPlayInfo.fileRepeat[(_tsrPlayInfo.fileCount - 1)] <= _FILE_REPEAT_MAX)) )
   {
      return(_ERROR_NONE);
   }
   else
   {
      return(_ERROR_ILLEGAL_PLAYBACK_COUNT);
   }
}


// Function to capture request to pause playback.
//
WORD parsePause(PSTR option)
{
   tsrPauseFlag = _TRUE;
   return(_ERROR_NONE);
}


// Function to capture request to pause playback.
//
WORD parseUnpause(PSTR option)
{
   tsrUnpauseFlag = _TRUE;
   return(_ERROR_NONE);
}


// Function to capture request to flush all files being played back.
//
WORD parseFlushFiles( PSTR option )
{
   tsrFlushFilesFlag = _TRUE;
   return(_ERROR_NONE);
}


// Function to capture jumper assignment for DMA i/o of sound data.
//
WORD parseJumper(PSTR option)
{
   _tsrPlayInfo.dmaPort = atoi(option);

   return(_ERROR_NONE);
}

// Function to capture channel assignment for DMA i/o of sound data.
//
WORD parseChannel(PSTR option)
{
   _tsrPlayInfo.dmaChannel = atoi(option);

   return(_ERROR_NONE);
}


// Function to capture irq assignment for DMA i/o of sound data.
//
WORD parseIRQNumber(PSTR option)
{
   _tsrPlayInfo.dmaIRQNumber = atoi(option);

   return(_ERROR_NONE);
}


// Function to capture command line option to uninstall tsr.
//
WORD parseUninstall(PSTR option)
{
   tsrUninstallFlag = _TRUE;

   return(_ERROR_NONE);
}


// Function to capture request to suppress all information messages.
// (i.e, intializing and unitializing messages)
//
WORD parseSilent(PSTR option)
{
   silentFlag = _TRUE;

   return(_ERROR_NONE);
}


// Display to the screen the legal command line options avaiable.
//
VOID displayCommandLineSyntax( VOID )
{
   WORD  i, j;
   WORD  messageLength;

   static PSTR commandLineSyntaxMessage[] =
   {
      "COVOX TSR sound file playback program - Version ", 
      "",
      " - Covox Inc (c) 1992\n\n",
      "DESCRIPTION : This program will play back any format Covox sound file(s)\n",
      "              as a background (tsr), task.  As many as 35 files may be\n",
      "              specified when installing TSRPLAY.EXE.  If no file extension\n",
      "              is specified then 8 bit (.V8), is the assumed default.\n\n",
      "SYNTAX : TSRPLAY file1 [file2...file35] [OPTIONS]\n\n",
      "OPTIONS:\n",
      "/Rx - Rate in hertz to play back file (4679-44191). 9622 is the default.\n",
      "/Bx - Number of playback buffers (2-4). 3 is the default.\n",
      "/Kx - Size in kilobytes of each of the buffers (1-65). 32 is the default.\n",
      "/Nx - Number of times to play file. 1 is default. -1 for infinite. 255 is max.\n",
      "/Z  - Pause output.\n",
      "/W  - Restart paused output.\n",
      "/F  - Flush all files being played back.\n",
      "/Q  - Uninstall TSRPLAY.EXE program.\n",
      "/S  - Install/Uninstall quietly (no screen output). \n",
      "The following Voice Master and Sound Master II DMA settings can be selected.\n",
      "/Jx - Select jumper setting (0-3).  -1 (default) for sequential search.\n",
      "/Cx - Select DMA channel (1 or 3).  -1 (default) checks 1 then 3.\n",
      "/Ix - Select IRQ (2,3,4,5 or 7).  -1 (default) for sequential search.\n",
      "EXAMPLES\n",
      "        - Install TSRPLAY.EXE to play back 2 sound files.\n",
      "          C:\\TSRPLAY>TSRPLAY songs.v8 voice.v3s\n",
      "\n",
      "        - Install TSRPLAY.EXE to play back 1 sound file. Set the buffer\n",
      "          size to 16 kilobytes.\n",
      "          C:\\TSRPLAY>TSRPLAY C:\\songs.v8 /K16\n",
      "\n",
      "        - Install TSRPLAY.EXE to play back 1 sound file. Set the number\n",
      "          of buffers to 4.\n",
      "          C:\\TSRPLAY>TSRPLAY songs.v8 /B4\n",
      "\n",
      "        - Install TSRPLAY.EXE to play back 2 sound files. The first file is\n",
      "          played 3 times and the second is played once.\n",
      "          C:\\TSRPLAY>TSRPLAY C:\\f1.v8 /N3 C:\\f2.v3s \n",
      "\n",
      "        - Install TSRPLAY.EXE to play back 3 sound files. Each file is\n",
      "          specified with a playback rate.\n",
      "          C:\\TSRPLAY>TSRPLAY C:\\f1.v8 /R8000 C:\\f2.v3s /R10000 C:\\f3.v4s /R12000\n"
   };

   
   // Put version into above array.
   commandLineSyntaxMessage[ 1 ] = tsrPlayVersionString;

   messageLength = sizeof(commandLineSyntaxMessage)/sizeof(PSTR);

   for(i = 0 ; ( (i < messageLength) && (i != 22) ) ; i++)
   {
      printf(commandLineSyntaxMessage[i]);

   }

   if( i < messageLength )
   {
      printf("                                             - Press any key for MORE -\n");
      getch();
      printf("                                                                       \n");

      printf(commandLineSyntaxMessage[0]);
      printf(commandLineSyntaxMessage[1]);
      printf(commandLineSyntaxMessage[2]);
      for(j = i ; j < messageLength ; j++)
      {
         printf(commandLineSyntaxMessage[j]);

      }
   }




}


