
/*=========================================================================*/
/* Konzept        : DATA BECKERs Sound Blaster Superbuch                   */
/* Modul DMA      : Stellt Funktionen zur Programmierung des DMA-Control-  */
/*                  lers 8237A zur Verfgung. In einem PC-XT ist nur ein   */
/*                  DMA-Controller vorhanden, in einem PC-AT zwei. In die- */
/*                  sem Modul werden beide Controller untersttzt. Die Ka- */
/*                  nle 4-7 des zweiten DMA-Controllers entsprechen im    */
/*                  folgenden den Kanlen 0-3 (wie beim ersten Control-    */
/*                  ler).                                                  */
/*                  Zu Beginn eines Programms, das dieses Modul benutzt,   */
/*                  sollte immer die Funktion CheckComputer zur Feststel-  */
/*                  lung des Rechnertyps aufgerufen werden.                */
/*                                                                         */
/*                  HINWEIS: Im Zusammenhang mit der Programmierung der    */
/*                           DMA-Kanle wird darauf hingewiesen, da der   */
/*                           Kanal 2 (DMA 1) vom Disketten/Festplatten-    */
/*                           Controller standardmig belegt ist. Der Ka-  */
/*                           nal 1 wird hufig von der Sound-Blaster-Karte */
/*                           belegt.                                       */
/*=========================================================================*/
/* Autor          : Arthur Burda                                           */
/* Dateiname      : DMA.C                                                  */
/* entwickelt am  : 04.07.1993                                             */
/* letztes Update : 01.09.1993                                             */
/* Version        : 1.05                                                   */
/* Compiler       : Turbo C, Turbo C++, Borland C++                        */
/*=========================================================================*/

#include <stdio.h>
#include <dos.h>
#include "dma.h"

int Computer;                                  /* speichert den Rechnertyp */

/*=========================================================================*/
/* CheckChannel: Setzt die Kanalnummer auf 0, falls eine andere Kanalnum-  */
/*               mer als 0, 1, 2 oder 3 gewhlt wurde.                     */
/*=========================================================================*/
/* Eingabe: Nmb = Nummer des Kanals                                        */
/* Ausgabe: Kanalnummer, falls 0, 1, 2 oder 3 gewhlt wurde, sonst 0       */
/*-------------------------------------------------------------------------*/

Byte CheckChannel(Byte Nmb)

{
  if (Nmb > 3)                                         /* Kanalnummer > 3? */
    return 0;                                     /* ja, Kanalnummer ist 0 */
  else                                                             /* nein */
    return Nmb;      /* Kanalnummer ist gleich der angegebenen Kanalnummer */
}

/*=========================================================================*/
/* PtrToLongInt: Wandelt einen Zeiger-Ausdruck in einen LongInt-Wert um.   */
/*               Diese Funktion wird benutzt, wenn Speicheradressen in     */
/*               Form von Zeigern in LongInt-Zahlen umgewandelt werden     */
/*               sollen.                                                   */
/*=========================================================================*/
/* Eingabe: P = Zeiger-Ausdruck                                            */
/* Ausgabe: LongInt-Zahl (32 Bit)                                          */
/*-------------------------------------------------------------------------*/

LongInt PtrToLongInt(void far *P)

{
  return ((LongInt)FP_SEG(P) << 4)+FP_OFF(P);
}

/*=========================================================================*/
/* LowWord: Liefert das niederwertige Wort einer LongInt-Zahl zurck.      */
/*=========================================================================*/
/* Eingabe: LI = LongInt-Wert                                              */
/* Ausgabe: das niederwertige Wort von LI                                  */
/*-------------------------------------------------------------------------*/

Word LowWord(LongInt LI)

{
  return (LI & 0xFFFF);
}

/*=========================================================================*/
/* HighWord: Liefert das hherwertige Wort einer LongInt-Zahl zurck.      */
/*=========================================================================*/
/* Eingabe: LI = LongInt-Wert                                              */
/* Ausgabe: das hherwertige Wort von LI                                   */
/*-------------------------------------------------------------------------*/

Word HighWord(LongInt LI)

{
  return (LI >> 16);
}

/*=========================================================================*/
/* CheckComputer: Stellt den Rechnertyp fest und speichert das Ergebnis in */
/*                der Variablen Computer.                                  */
/*=========================================================================*/
/* Eingabe: keine                                                          */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void CheckComputer()

{
  Byte far *P;                                                   /* Zeiger */

  P = MK_FP(0xF000, 0xFFFE);
  if (*P == 0xFC)                                                /* PC-AT? */
    Computer = AT;                                                   /* ja */
  else                                                             /* nein */
    Computer = XT;                                                /* PC-XT */
}

/*=========================================================================*/
/* SetCommandReg: Setzt das Befehlsregister. Dadurch werden verschiedene   */
/*                Arbeitsweisen des DMA-Controllers gewhlt.               */
/*=========================================================================*/
/* Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                */
/*          BitMask = Bitmaske                                             */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void SetCommandReg(Byte DMA, Byte BitMask)

{
  if (DMA == 1)                                  /* erster DMA-Controller? */
    outportb(0x08, BitMask);                        /* ja, Bitmaske setzen */
  else                                     /* nein, zweiter DMA-Controller */
    if (Computer == AT)                 /* Handelt es sich um einen PC-AT? */
      outportb(0xD0, BitMask);                      /* ja, Bitmaske setzen */
}

/*=========================================================================*/
/* GetStatusReg: Liefert den Inhalt des Statusregisters zurck.            */
/*=========================================================================*/
/* Eingabe: DMA = Nummer des DMA-Controllers (1 oder 2)                    */
/* Ausgabe: Bitmaske                                                       */
/*-------------------------------------------------------------------------*/

Byte GetStatusReg(Byte DMA)

{
  if (DMA == 1)
    return inportb(0x08);
  else
    if (Computer == AT)
      return inportb(0xD0);
    else
      return 0;
}

/*=========================================================================*/
/* SetRequestReg: Setzt das Start-, auch Anforderungsregister des DMA-     */
/*                Controllers. Dadurch wird die Anforderung fr einen      */
/*                bestimmten Kanal gesetzt (DMA-Start) oder gelscht.      */
/*=========================================================================*/
/* Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                */
/*          BitMask = Bitmaske                                             */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void SetRequestReg(Byte DMA, Byte BitMask)

{
  if (DMA == 1)
    outportb(0x09, BitMask);
  else
    if (Computer == AT)
      outportb(0xD2, BitMask);
}

/*=========================================================================*/
/* SetSingleMaskReg: Setzt das Einzelmaskierungsregister des DMA-Control-  */
/*                   lers.                                                 */
/*=========================================================================*/
/* Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                */
/*          BitMask = Bitmaske                                             */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void SetSingleMaskReg(Byte DMA, Byte BitMask)

{
  if (DMA == 1)
    outportb(0x0A, BitMask);
  else
    if (Computer == AT)
      outportb(0xD4, BitMask);
}

/*=========================================================================*/
/* SetMultiMaskReg: Setzt das Gesamtmaskierungsregister.                   */
/*=========================================================================*/
/* Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                */
/*          BitMask = Bitmaske                                             */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void SetMultiMaskReg(Byte DMA, Byte BitMask)

{
  if (DMA == 1)
    outportb(0x0F, BitMask);
  else
    if (Computer == AT)
      outportb(0xDE, BitMask);
}

/*=========================================================================*/
/* SetModeReg: Setzt das Modusregister des DMA-Controllers.                */
/*=========================================================================*/
/* Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                */
/*          BitMask = Bitmaske                                             */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void SetModeReg(Byte DMA, Byte BitMask)

{
  if (DMA == 1)
    outportb(0x0B, BitMask);
  else
    if (Computer == AT)
      outportb(0xD6, BitMask);
}

/*=========================================================================*/
/* StartDMA: Fordert den DMA-Controller auf, seine Arbeit zu beginnen.     */
/*           Dazu wird die Funktion SetRequestReg aufgerufen.              */
/*=========================================================================*/
/* Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                */
/*          Channel = Nummer des Kanals (0-3)                              */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void StartDMA(Byte DMA, Byte Channel)

{
  Byte Mask;

  Channel = CheckChannel(Channel); /* Kanalnummer prfen, ggf. korrigieren */
  Mask = Channel+dma_ReqReg_SetRequestBit;            /* Bitmaske erzeugen */
  SetRequestReg(DMA, Mask);
}

/*=========================================================================*/
/* SetMaskBit: Setzt das Maskierungsbit fr den angegebenen Kanal. Die     */
/*             DMA-Anforderung fr diesen Kanal wird damit beendet. Intern */
/*             wird die Funktion SetSingleMaskReg aufgerufen.              */
/*=========================================================================*/
/* Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                */
/*          Channel = Kanalnummer (0-3)                                    */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void SetMaskBit(Byte DMA, Byte Channel)

{
  Byte Mask;

  Channel = CheckChannel(Channel); /* Kanalnummer prfen, ggf. korrigieren */
  Mask = Channel+dma_SingleMaskReg_SetMaskBit;        /* Bitmaske erzeugen */
  SetSingleMaskReg(DMA, Mask);
}

/*=========================================================================*/
/* ClearMaskBit: Lscht das Maskierungsbit fr den angegebenen Kanal. Dies */
/*               mu gemacht werden, damit die DMA-Anforderung fr diesen  */
/*               Kanal erfolgen kann. Intern wird die Funktion SetSingle-  */
/*               MaskReg aufgerufen.                                       */
/*=========================================================================*/
/* Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                */
/*          Channel = Kanalnummer (0-3)                                    */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void ClearMaskBit(Byte DMA, Byte Channel)

{
  Byte Mask;

  Channel = CheckChannel(Channel); /* Kanalnummer prfen, ggf. korrigieren */
  Mask = Channel+dma_SingleMaskReg_ClearMaskBit;      /* Bitmaske erzeugen */
  SetSingleMaskReg(DMA, Mask);
}

/*=========================================================================*/
/* SetAllMaskBits: Setzt die Maskierungsbits fr alle vier Kanle auf ein- */
/*                 mal. Dadurch wird das Setzen der Maskierungsbits ein-   */
/*                 zeln gespart. Intern wird die Funktion SetMultiMaskReg  */
/*                 aufgerufen.                                             */
/*=========================================================================*/
/* Eingabe: DMA = Nummer des DMA-Controllers (1 oder 2)                    */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void SetAllMaskBits(Byte DMA)

{
  Byte Mask;

  Mask = dma_MultiMaskReg_SetAll;
  SetMultiMaskReg(DMA, Mask);
}

/*=========================================================================*/
/* ClearAllMaskBits: Lscht die Maskierungsbits fr alle vier Kanle auf   */
/*                   einmal. Dadurch wird das mehrmalige Lschen gespart.  */
/*                   Intern wird SetMultiMaskReg aufgerufen.               */
/*=========================================================================*/
/* Eingabe: DMA = Nummer des DMA-Controllers (1 oder 2)                    */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void ClearAllMaskBits(Byte DMA)

{
  Byte Mask;

  Mask = dma_MultiMaskReg_ClearAll;
  SetMultiMaskReg(DMA, Mask);
}

/*=========================================================================*/
/* SetDMAMode: Setzt einen Modus oder mehrere Modi fr den angegebenen     */
/*             DMA-Kanal. Dazu wird die Funktion SetModeReg aufgerufen.    */
/*=========================================================================*/
/* Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                */
/*          Channel = Nummer des Kanals (0-3)                              */
/*          Mode    = Modus bzw. Modi                                      */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void SetDMAMode(Byte DMA, Byte Channel, Byte Mode)

{
  Channel = CheckChannel(Channel); /* Kanalnummer prfen, ggf. korrigieren */
  Mode = (Mode & 0xFC);              /* Fr Modus sind Bits 2-7 zustndig. */
  if ((Mode & 0xFC) == 0x00)     /* Kein Schreiben und Lesen gleichzeitig? */
    SetModeReg(DMA, Channel+Mode);                                 /* nein */
}

/*=========================================================================*/
/* MasterClear: Lscht das Befehlsregister.                                */
/*=========================================================================*/
/* Eingabe: DMA = Nummer des DMA-Controllers (1 oder 2)                    */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void MasterClear(Byte DMA)

{
  if (DMA == 1)                                  /* erster DMA-Controller? */
    outportb(0x0D, 0x00);                   /* ja, Befehlsregister lschen */
  else                             /* nein, zweiter DMA-Controller gewhlt */
    if (Computer == AT)                 /* Handelt es sich um einen PC-AT? */
      outportb(0xDA, 0x00);                 /* ja, Befehlsregister lschen */
}

/*=========================================================================*/
/* ClearFlipFlop: Lscht den First/Last-FlipFlop, um dadurch dem DMA-      */
/*                Controller mitzuteilen, da zunchst Low-Byte und danach */
/*                High-Byte gesetzt wird, wenn ein Kanal programmiert      */
/*                wird.                                                    */
/*=========================================================================*/
/* Eingabe: DMA = Nummer des DMA-Controllers (1 oder 2)                    */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void ClearFlipFlop(Byte DMA)

{
  if (DMA == 1)
    outportb(0x0C, 0x00);
  else
    if (Computer == AT)
      outportb(0xD8, 0x00);    
}

/*=========================================================================*/
/* SetAddrAndSideReg: Setzt das Adre- (untere 16 Bits einer 20 Bit langen */
/*                    Speicheradresse) und Seitenregister (obere 4 Bits    */
/*                    der Speicheradresse) eines bestimmten Kanals. Die    */
/*                    Adresse mu 20 Bit lang sein, wenn ein Adreraum von */
/*                    1 MB adressiert werden soll.                         */
/*=========================================================================*/
/* Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                */
/*          Channel = Kanalnummer (0-3)                                    */
/*          AddrPtr = Speicheradresse in Form eines Zeigers                */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void SetAddrAndSideReg(Byte DMA, Byte Channel, void far *AddrPtr)

{
  Byte Side;
  Byte LoWord;

  Channel = CheckChannel(Channel); /* Kanalnummer prfen, ggf. korrigieren */
  Side = ((HighWord(PtrToLongInt(AddrPtr))) & 0xFF);

  /* Seitenadresse setzen */

  if (DMA == 1)
    switch(Channel)
    {
      case 0 : outportb(0x87, Side); break;
      case 1 : outportb(0x83, Side); break;
      case 2 : outportb(0x81, Side); break;
      case 3 : outportb(0x82, Side); break;
    }
  else
    if (Computer == AT)
      switch(Channel)
      {
	case 0 : outportb(0x8F, Side); break;
	case 1 : outportb(0x8B, Side); break;
	case 2 : outportb(0x89, Side); break;
	case 3 : outportb(0x8A, Side); break;
      }

  ClearFlipFlop(DMA);                                  /* FlipFlop lschen */

  /* Adreregister setzen */

  LoWord = LowWord(PtrToLongInt(AddrPtr));
  if (DMA == 1)
    {
      outportb(2*Channel, (LoWord & 0xFF));
      outportb(2*Channel, ((LoWord >> 8) & 0xFF));
    }
  else
    if (Computer == AT)
      {
	outportb(2*Channel+0xC0, (LoWord & 0xFF));
	outportb(2*Channel+0xC0, ((LoWord >> 8) & 0xFF));
      }
}

/*=========================================================================*/
/* SetBaseCountReg: Setzt das Basiszhlregister eines bestimmten Kanals.   */
/*                  Dadurch wird dem DMA-Controller mitgeteilt, wie viele  */
/*                  Datenbytes bertragen werden. Als Parameter wird die   */
/*                  wirkliche Anzahl der Bytes angegeben, gesetzt wird     */
/*                  Bytes-1.                                               */
/*=========================================================================*/
/* Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                */
/*          Channel = Nummer des DMA-Kanals (0-3)                          */
/*          Bytes   = Anzahl der zu bertragenden Bytes                    */
/* Ausgabe: keine                                                          */
/*-------------------------------------------------------------------------*/

void SetBaseCountReg(Byte DMA, Byte Channel, Word Bytes)

{
  Channel = CheckChannel(Channel); /* Kanalnummer prfen, ggf. korrigieren */
  ClearFlipFlop(DMA);                                  /* FlipFlop lschen */
  if (DMA == 1)
    {
      outportb(2*Channel+1, (Bytes-1) & 0xFF);          /* Low-Byte setzen */
      outportb(2*Channel+1, ((Bytes-1) >> 8) & 0xFF);  /* High-Byte setzen */
    }
  else
    if (Computer == AT)
      {
	outportb(2*Channel+0xC1, (Bytes-1) & 0xFF);
	outportb(2*Channel+0xC1, ((Bytes-1) >> 8) & 0xFF);
      }
}

/*=========================================================================*/
/* GetCurrCountReg: Liefert den Inhalt des aktuellen Zhlregisters zurck, */
/*                  also die Anzahl der noch zu bertragenden Bytes.       */
/*=========================================================================*/
/* Eingabe: DMA     = Nummer des DMA-Controllers (1 oder 2)                */
/*          Channel = Nummer des DMA-Kanals (0-3)                          */
/* Ausgabe: Anzahl der noch zu bertragenden Bytes                         */
/*-------------------------------------------------------------------------*/

Word GetCurrCountReg(Byte DMA, Byte Channel)

{
  Byte LowByte;                                                /* Low-Byte */
  Byte HighByte;                                              /* High-Byte */

  Channel = CheckChannel(Channel); /* Kanalnummer prfen, ggf. korrigieren */
  ClearFlipFlop(DMA);                                  /* FlipFlop lschen */
  if (DMA == 1)
    {
      LowByte = inportb(2*Channel+1);                    /* Low-Byte lesen */
      HighByte = inportb(2*Channel+1);                  /* High-Byte lesen */
      return LowByte+(256*HighByte)+1; /* Anz. Bytes insges. zurckliefern */
    }
  else
    if (Computer == AT)
      {
	LowByte = inportb(2*Channel+0xC1);
	HighByte = inportb(2*Channel+0xC1);
	return LowByte+(256*HighByte)+1;
      }
    else
      return 0;
}
