
{===========================================================================}
{ Konzept        : DATA BECKERs Sound Blaster Superbuch                     }
{ Unit SBFM      : Stellt eine ganze Menge Routinen zur Programmierung der  }
{                  FM-Kanle des Sound Blaster und des Sound Blaster Pro    }
{                  zur Verfgung.                                           }
{===========================================================================}
{ Autor          : Arthur Burda                                             }
{ Dateiname      : SBFM.PAS                                                 }
{ entwickelt am  : 16.07.1993                                               }
{ letztes Update : 01.09.1993                                               }
{ Version        : 1.06                                                     }
{ Compiler       : Turbo Pascal 6.0 und hher                               }
{===========================================================================}

UNIT SBFM;

{---------------------------------------------------------------------------}
{ Compiler-Schalter                                                         }
{---------------------------------------------------------------------------}

{$B-}                         { Kurzschluverfahren fr boolesche Ausdrcke }
{$D-}                                        { keine Debugger-Informationen }
{$F+}                                                { FAR-Aufrufe erlauben }
{$G+}                                                   { 286-Code erzeugen }
{$I-}                                                   { keine I/O-Prfung }
{$O+}                                            { Unit overlayfhig machen }
{$R-}                                               { keine Bereichsprfung }
{$S-}                                                  { keine Stackprfung }
{$X+}                    { Behandlung von Funktionen wie Prozeduren mglich }

INTERFACE

USES CRT, SBDrv;                            { CRT- und SBDrv-Unit einbinden }

CONST

  {-------------------------------------------------------------------------}
  { Offsets der 18 Generatorzellen                                          }
  {-------------------------------------------------------------------------}

  sbfm_GenCellOffs : ARRAY[0..17] OF Byte =
    (0, 1, 2, 3, 4, 5, 8, 9, 10, 11, 12, 13, 16, 17, 18, 19, 20, 21);

  {-------------------------------------------------------------------------}
  { Nummern der Modulatorzellen der 9 verschiedenen Kanle                  }
  {-------------------------------------------------------------------------}

  sbfm_ModCellNum : ARRAY[0..8] OF Byte =
    (0, 1, 2, 6, 7, 8, 12, 13, 14);

  {-------------------------------------------------------------------------}
  { Nummern der Trgerzellen (Carrier) der 9 verschiedenen Kanle           }
  {-------------------------------------------------------------------------}

  sbfm_CarrCellNum : ARRAY[0..8] OF Byte =
    (3, 4, 5, 9, 10, 11, 15, 16, 17);

  {-------------------------------------------------------------------------}
  { Offsets der Modulatorzellen                                             }
  {-------------------------------------------------------------------------}

  sbfm_ModCellOffs : ARRAY[0..8] OF Byte =
    (0, 1, 2, 8, 9, 10, 16, 17, 18);

  {-------------------------------------------------------------------------}
  { Offsets der Trgerzellen                                                }
  {-------------------------------------------------------------------------}

  sbfm_CarrCellOffs : ARRAY[0..8] OF Byte =
    (3, 4, 5, 11, 12, 13, 19, 20, 21);

TYPE

  {=========================================================================}
  { TSides: Mono-Kanal des Sound Blaster und Stereo-Kanle links und rechts }
  {=========================================================================}

  TSides = (Mono, Left, Right);

  {=========================================================================}
  { PInstrument: Zeiger auf TInstrument                                     }
  {=========================================================================}
  { TInstrument: Die 16 Bytes dieses Typs enthalten die Daten eines Instru- }
  {              mentes, fr den Modulator (erster Operator) wie fr den    }
  {              Trger (zweiter Operator). Der Aufbau dieser 16 Bytes be-  }
  {              findet sich in vielen mit dem Sound Blaster mitgelieferten }
  {              Dateiformaten (z.B. IBK) und Programmen.                   }
  {=========================================================================}

  PInstrument = ^TInstrument;
  TInstrument = RECORD

    {=======================================================================}
    { Toneinstellungen des Modulators und des Trgers (Carrier)             }
    {=======================================================================}
    { Bit 0-3 : Frequenzmultiplikator                                       }
    { Bit 4   : Hllkurvenverkrzung (0 = aus, 1 = ein)                     }
    { Bit 5   : Hllkurventyp (0 = abnehmend, 1 = kontinuierlich)           }
    { Bit 6   : Vibrato (Schwankung der Amplitude einer Generatorzelle)     }
    { Bit 7   : Tremolo (Schwankung der Frequenz einer Generatorzelle)      }
    {-----------------------------------------------------------------------}

    Mod_Tone, Carr_Tone : Byte;

    {=======================================================================}
    { Einstellung des Dmpfungsfaktors und der Hochtondmpfung (Lautstrke  }
    { des Signals) des Modulators und des Trgers                           }
    {=======================================================================}
    { Bit 0-5 : Dmpfungsfaktor                                             }
    { Bit 6-7 : Hochtondmpfung (Lautstrke oder Amplitude des Signals)     }
    {-----------------------------------------------------------------------}

    Mod_Ampl, Carr_Ampl : Byte;

    {=======================================================================}
    { Einstellung der Decay- und der Attack-Rate                            }
    {=======================================================================}
    { Bit 0-3 : Decay-Rate (Absinken der Amplitude im zweiten Abschnitt     }
    {           einer Hllkurve)                                            }
    { Bit 4-7 : Attack-Rate (Anstieg der Amplitude im ersten Teil einer     }
    {           Hllkurve)                                                  }
    {-----------------------------------------------------------------------}

    Mod_DecAtt, Carr_DecAtt : Byte;

    {=======================================================================}
    { Einstellung der Release-Rate und der Sustain-Strke                   }
    {=======================================================================}
    { Bit 0-3 : Release-Rate (Absinken der Amplitude im letzten Abschnitt   }
    {           einer Hllkurve)                                            }
    { Bit 4-7 : Sustain-Strke (Niveaupunkt fr den Wechsel vom Decay-Ab-   }
    {           schnitt zu Release-Teil)                                    }
    {-----------------------------------------------------------------------}

    Mod_RelSus, Carr_RelSus : Byte;

    {=======================================================================}
    { Einstellung der Wellenform des Modulators und des Trgers             }
    {=======================================================================}
    { Bit 0-1 : Art des Wellensignals                                       }
    { Bit 2-7 : 0                                                           }
    {-----------------------------------------------------------------------}

    Mod_Wave, Carr_Wave : Byte;

    {=======================================================================}
    { Daten zu der Art der verwendeten Synthese und der Modulations-Rck-   }
    { kopplung (Feedback)                                                   }
    {=======================================================================}
    { Bit 0   : Art der Synthese                                            }
    { Bit 1-3 : Modulations-Rckkopplung                                    }
    { Bit 4-7 : 0                                                           }
    {-----------------------------------------------------------------------}

    SynthFB : Byte;

    {-----------------------------------------------------------------------}
    { Die letzten 5 Bytes sind fr sptere Erweiterungen reserviert.        }
    {-----------------------------------------------------------------------}

    Reserved : ARRAY[0..4] OF Byte;

  END;

  {=========================================================================}
  { TEnvelope: Hllkurventyp. Dimishing bedeutet abnehmend, Continuing kon- }
  {            tinuierlich.                                                 }
  {=========================================================================}

  TEnvelope = (Dimishing, Continuing);

  {=========================================================================}
  { TCellConnection: Typ zur Festlegung der Zellenverknpfung eines Kanals: }
  {                  parallel bzw. seriell (FM-Modulationsmodus).           }
  {=========================================================================}

  TCellConnection = (Parallel, Serial);

  {=========================================================================}
  { TRhythmInstruments: Typ zum Ein- bzw. Ausschalten von rhythmischen      }
  {                     Instrumenten.                                       }
  {=========================================================================}

  TRhythmInstruments = RECORD
    HiHat     : Boolean;                      { "Hi Hat" (Operatorzelle 13) }
    TopCymbal : Boolean;                  { "Top Cymbal" (Operatorzelle 17) }
    TomTom    : Boolean;                     { "Tom Tom" (Operatorzelle 14) }
    SnareDrum : Boolean;                  { "Snare Drum" (Operatorzelle 16) }
    BassDrum  : Boolean;               { "Bass Drum" (Operatoren 12 und 15) }
  END;

{===========================================================================}
{ Prozedur InitSBFM: Initialisiert die Sound-Blaster-Karte und bereitet die }
{                    FM-Register vor. Die Routine prft durch Aufruf der    }
{                    Funktion FMAvailable, ob der FM-Chip vorhanden ist.    }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE InitSBFM;

{===========================================================================}
{ Funktion FMAvailable: Gibt TRUE zurck, wenn der FM-Chip vorhanden ist,   }
{                       sonst FALSE. Die Funktion wird von InitSBFM aufge-  }
{                       rufen.                                              }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION FMAvailable : Boolean;

{===========================================================================}
{ Funktion SBFMIn: Liest den Wert eines FM-Registers aus.                   }
{===========================================================================}
{ Eingabe: Reg  = Nummer des Registers                                      }
{          Side = Mono-Kanal (Mono) oder einer der Stereo-Kanle (Left,     }
{                 Right) bei SB Pro                                         }
{ Ausgabe: ein Datenbyte                                                    }
{---------------------------------------------------------------------------}

FUNCTION SBFMIn(Reg : Byte; Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SBFMOut: Gibt einen Wert an ein internes FM-Register aus.        }
{===========================================================================}
{ Eingabe: Reg  = Nummer des Registers                                      }
{          Side = Mono-Kanal (Mono) oder einer der Stereo-Kanle (Left,     }
{                 Right) bei SB Pro                                         }
{          Data = Datenbyte                                                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SBFMOut(Reg : Byte; Side : TSides; Data : Byte);

{===========================================================================}
{ Prozedur SetMultFactor: Setzt den Frequenz-Multiplikationsfaktor einer    }
{                         Generatorzelle. Dieser Faktor modifiziert die     }
{                         die Frequenzen des Modulators und des Trgers,    }
{                         und zwar so, da die Frequenz, mit der eine Ge-   }
{                         neratorzelle eine Schwingung erzeugt um das       }
{                         "Frequenzfaktor"-fache grer als die eingestell- }
{                         te Tonfrequenz ist (Einstellen der Tonfrequenz    }
{                         bedeutet Einstellen der gleichen Frequenz fr den }
{                         Modulator und den Trger).                        }
{===========================================================================}
{ Eingabe: Cell   = Nummer der Generatorzelle (0-17)                        }
{          Side   = Mono-Kanal (Mono) oder einer der Stereo-Kanle (Left,   }
{                   Right) bei SB Pro                                       }
{          Factor = Multiplikationsfaktor (0-15)                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetMultFactor(Cell : Byte; Side : TSides; Factor : Byte);

{===========================================================================}
{ Funktion GetMultFactor: Liefert den Frequenz-Multiplikationsfaktor einer  }
{                         Operatorzelle zurck.                             }
{===========================================================================}
{ Eingabe: Cell = Nummer der Generatorzelle (0-17)                          }
{          Side = Mono-Kanal (Mono) oder einer der Stereo-Kanle (Left,     }
{                 Right) bei SB Pro                                         }
{ Ausgabe: Multiplikationsfaktor                                            }
{---------------------------------------------------------------------------}

FUNCTION GetMultFactor(Cell : Byte; Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetEnvelopeLength: Schaltet die Hllkurvenverkrzung (oder an-   }
{                             ders Hllkurvendmpfung) ein bzw. aus. Da-    }
{                             durch wird die Modulatorschwingung gedmpft,  }
{                             also in ihrer Amplitude verringert.           }
{===========================================================================}
{ Eingabe: Cell = Nummer der Generatorzelle (0-17)                          }
{          Side = Mono-Kanal (Mono) oder einer der Stereo-Kanle (Left,     }
{                 Right) bei SB Pro                                         }
{          On   = TRUE, wenn die Hllkurvendmpfung eingeschaltet werden    }
{                 soll, FALSE bedeutet Ausschalten                          }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetEnvelopeLength(Cell : Byte; Side : TSides; On : Boolean);

{===========================================================================}
{ Funktion EnvelopeLengthOn: Gibt Auskunft darber, ob die Hllkurvenver-   }
{                            krzung fr eine Generatorzelle ein- oder aus- }
{                            geschaltet ist.                                }
{===========================================================================}
{ Eingabe: Cell = Nummer der Generatorzelle (0-17)                          }
{          Side = Mono-Kanal (Mono) oder einer der Stereo-Kanle (Left,     }
{                 Right) bei SB Pro                                         }
{ Ausgabe: TRUE, wenn die Hllkurvendmpfung eingeschaltet ist, sonst FALSE }
{---------------------------------------------------------------------------}

FUNCTION EnvelopeLengthOn(Cell : Byte; Side : TSides) : Boolean;

{===========================================================================}
{ Prozedur SetEnvelopeType: Setzt die Art der Hllkurve, die durch Schwin-  }
{                           gung einer Generatorzelle entsteht. Zur Auswahl }
{                           stehen zwei Typen: abnehmend (kein Sustain-Be-  }
{                           reich, direkter bergang vom Decay-Teil zum Re- }
{                           lease-Abschnitt) und kontinuierlich (die Sus-   }
{                           tain-Phase endet erst beim Ausschalten des je-  }
{                           weiligen Tons).                                 }
{===========================================================================}
{ Eingabe: Cell     = Nummer der Generatorzelle (0-17)                      }
{          Side     = Mono-Kanal (Mono) oder einer der Stereo-Kanle (Left, }
{                     Right) bei SB Pro                                     }
{          WhatType = Hllkurventyp (Dimishing, Continuing)                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetEnvelopeType(Cell : Byte; Side : TSides; WhatType : TEnvelope);

{===========================================================================}
{ Funktion GetEnvelopeType: Gibt die Art der Hllkurve, die mit Hilfe der   }
{                           Routine SetEnvelopeType gesetzt wurde, zurck.  }
{===========================================================================}
{ Eingabe: Cell = Nummer der Generatorzelle (0-17)                          }
{          Side = Mono-Kanal (Mono) oder einer der Stereo-Kanle (Left,     }
{                 Right) bei SB Pro                                         }
{ Ausgabe: Hllkurventyp (Dimishing, Continuing)                            }
{---------------------------------------------------------------------------}

FUNCTION GetEnvelopeType(Cell : Byte; Side : TSides) : TEnvelope;

{===========================================================================}
{ Prozedur SetVibrato: Schaltet den Vibrato-Effekt einer Generatorzelle ein }
{                      bzw. aus. Der Vibrato-Effekt ist eine Frequenzmodu-  }
{                      lation mit konstanter Modulatorfrequenz (6.4 kHz),   }
{                      d.h. die Frequenz einer Generatorzelle schwankt um   }
{                      diesen Wert. Mit Hilfe dieser Routine wird auch die  }
{                      Strke der Schwankungen in der Frequenz gewhlt.     }
{===========================================================================}
{ Eingabe: Cell  = Nummer der Generatorzelle (0-17)                         }
{          Side  = Mono-Kanal (Mono) oder einer der Stereo-Kanle (Left,    }
{                  Right) bei SB Pro                                        }
{          On    = TRUE, wenn der Vibrato-Effekt eingeschaltet werden soll, }
{                  FALSE bedeutet Ausschalten                               }
{          Level = 1, wenn die Schwankung in der Frequenz maximal 14 Pro-   }
{                  zent betragen soll, 0 fr 7 Prozent                      }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetVibrato(Cell : Byte; Side : TSides; On : Boolean; Level : Byte);

{===========================================================================}
{ Funktion VibratoOn: Liefert TRUE zurck, falls der Vibrato-Effekt einer   }
{                     Zelle aktiv ist, sonst FALSE.                         }
{===========================================================================}
{ Eingabe: Cell = Nummer der Generatorzelle (0-17)                          }
{          Side = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,      }
{                 Right) bei SB Pro                                         }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION VibratoOn(Cell : Byte; Side : TSides) : Boolean;

{===========================================================================}
{ Funktion VibratoLevel: Liefert 1 zurck, wenn die Schwankungen in der     }
{                        Frequenz (Vibrato-Effekt) stark (maximal 4,8 DB)   }
{                        sind und 0, falls diese Schwankungen schwach (ma-  }
{                        ximal 1 DB) sind.                                  }
{                                                                           }
{                        ANMERKUNG: Die Strke des Vibrato-Effekte bezieht  }
{                                   sich auf alle Operatorzellen.           }
{===========================================================================}
{ Eingabe: Side = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,      }
{                 Right) bei SB Pro                                         }
{ Ausgabe: 0 oder 1                                                         }
{---------------------------------------------------------------------------}

FUNCTION VibratoLevel(Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetTremolo: Schaltet den Tremolo-Effekt einer Generatorzelle ein }
{                      bzw. aus. Der Tremolo-Effekt bewirkt eine Schwingung }
{                      der Amplitude der Generatorzelle mit einer konstan-  }
{                      ten Frequenz von 3,7 Hz, was 3,7 Schwingungen pro    }
{                      Sekunde bedeutet. Mit Hilfe dieser Routine wird auch }
{                      die Strke der Signalschwankungen der Operatoren ge- }
{                      whlt.                                               }
{===========================================================================}
{ Eingabe: Cell  = Nummer der Generatorzelle (0-17)                         }
{          Side  = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,     }
{                  Right) bei SB Pro                                        }
{          On    = TRUE, wenn der Tremolo-Effekt eingeschaltet werden soll, }
{                  FALSE bedeutet Ausschalten                               }
{          Level = 1, wenn der Maximalwert der Signalschwankungen der Ope-  }
{                  ratoren 4,8 DB ist, bei 0 ist dieser Wert 1 DB           }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetTremolo(Cell : Byte; Side : TSides; On : Boolean; Level : Byte);

{===========================================================================}
{ Funktion TremoloOn: Liefert TRUE zurck, falls der Tremolo-Effekt einer   }
{                     Zelle aktiv ist, sonst FALSE.                         }
{===========================================================================}
{ Eingabe: Cell = Nummer der Generatorzelle (0-17)                          }
{          Side = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,      }
{                 Right) bei SB Pro                                         }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION TremoloOn(Cell : Byte; Side : TSides) : Boolean;

{===========================================================================}
{ Funktion TremoloLevel: Gibt den Wert 1 zurck, wenn die Schwankungen in   }
{                        der Amplitude maximal 14 Prozent betragen und 0,   }
{                        falls die Schwankungen maximal 7 Prozent betragen. }
{===========================================================================}
{ Eingabe: Side = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,      }
{                 Right) bei SB Pro                                         }
{ Ausgabe: 0 oder 1                                                         }
{---------------------------------------------------------------------------}

FUNCTION TremoloLevel(Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetAttenuation: Legt den Dmpfungsparameter fr eine Zelle fest. }
{                          Der Wertebereich liegt zwischen 0 und 63. Beim   }
{                          Wert 0 wird die Amplitude berhaupt nicht abge-  }
{                          schwcht, bei 63 dagegen ist der Ton praktisch   }
{                          nicht mehr zu hren.                             }
{===========================================================================}
{ Eingabe: Cell   = Nummer der Generatorzelle (0-17)                        }
{          Side   = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,    }
{                   Right) bei SB Pro                                       }
{          Factor = Dmpfungsfaktor (0-63)                                  }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetAttenuation(Cell : Byte; Side : TSides; Factor : Byte);

{===========================================================================}
{ Funktion GetAttenuation: Liefert den Dmpfungsfaktor fr eine Generator-  }
{                          zelle zurck.                                    }
{===========================================================================}
{ Eingabe: Cell = Nummer der Generatorzelle (0-17)                          }
{          Side = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,      }
{                 Right) bei SB Pro                                         }
{ Ausgabe: Dmpfungsfaktor                                                  }
{---------------------------------------------------------------------------}

FUNCTION GetAttenuation(Cell : Byte; Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetHigherPitchLevel: Legt den Faktor fr die Hochtondmpfung     }
{                               fest. Im Klartext bedeutet es, da die      }
{                               Amplitude, also die Lautstrke der hohen    }
{                               Tne verringert wird. Dadurch lassen sich   }
{                               viele Naturklnge besser nachahmen.         }
{===========================================================================}
{ Eingabe: Cell   = Nummer der Generatorzelle (0-17)                        }
{          Side   = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,    }
{                   Right) bei SB Pro                                       }
{          Factor = Dmpfungsfaktor fr Hochtne (0-3)                      }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetHigherPitchLevel(Cell : Byte; Side : TSides; Factor : Byte);

{===========================================================================}
{ Funktion GetHigherPitchLevel: Liefert den Faktor fr die Hochtondmpfung  }
{                               zurck.                                     }
{===========================================================================}
{ Eingabe: Cell = Nummer der Generatorzelle (0-17)                          }
{          Side = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,      }
{                 Right) bei SB Pro                                         }
{ Ausgabe: Faktor fr die Hochtondmpfung                                   }
{---------------------------------------------------------------------------}

FUNCTION GetHigherPitchLevel(Cell : Byte; Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetDecay: Stellt die Decay-Rate fr die Hllkurve einer Zelle    }
{                    ein (Decay-Rate bedeutet Absinken der Amplitude im     }
{                    zweiten Abschnitt der Hllkurve). Der Wertebereich     }
{                    liegt hier zwischen 0 und 15.                          }
{===========================================================================}
{ Eingabe: Cell  = Nummer der Generatorzelle (0-17)                         }
{          Side  = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,     }
{                  Right) bei SB Pro                                        }
{          Param = Decay-Parameter (0-15)                                   }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetDecay(Cell : Byte; Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetDecay: Liefert die mittels SetDecay eingestellte Decay-Rate   }
{                    (genauer den Decay-Parameter) fr die Hllkurve einer  }
{                    Zelle zurck.                                          }
{===========================================================================}
{ Eingabe: Cell = Nummer der Generatorzelle (0-17)                          }
{          Side = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,      }
{                 Right) bei SB Pro                                         }
{ Ausgabe: Decay-Parameter                                                  }
{---------------------------------------------------------------------------}

FUNCTION GetDecay(Cell : Byte; Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetAttack: Stellt den Attack-Parameter fr die Hllkurve einer   }
{                     Generatorzelle ein (dies bedeutet Anstieg der Ampli-  }
{                     tude im ersten Teil der Hllkurve).                   }
{===========================================================================}
{ Eingabe: Cell  = Nummer der Generatorzelle (0-17)                         }
{          Side  = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,     }
{                  Right) bei SB Pro                                        }
{          Param = Attack-Parameter (0-15)                                  }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetAttack(Cell : Byte; Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetAttack: Liefert den mit Hilfe von SetAttack eingestellten     }
{                     Attack-Parameter fr die Hllkurve einer Zelle zu-    }
{                     rck.                                                 }
{===========================================================================}
{ Eingabe: Cell = Nummer der Generatorzelle (0-17)                          }
{          Side = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,      }
{                 Right) bei SB Pro                                         }
{ Ausgabe: Attack-Parameter                                                 }
{---------------------------------------------------------------------------}

FUNCTION GetAttack(Cell : Byte; Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetRelease: Legt den Release-Parameter fr die Hllkurve einer   }
{                      Zelle fest. Release-Parameter bewirkt das Absinken   }
{                      der Amplitude im letzten Abschnitt der Hllkurve.    }
{===========================================================================}
{ Eingabe: Cell  = Nummer der Generatorzelle (0-17)                         }
{          Side  = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,     }
{                  Right) bei SB Pro                                        }
{          Param = Release-Parameter (0-15)                                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetRelease(Cell : Byte; Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetRelease: Liefert den Release-Parameter fr die Hllkurve      }
{                      einer Generatorzelle zurck.                         }
{===========================================================================}
{ Eingabe: Cell = Nummer der Zelle (0-17)                                   }
{          Side = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,      }
{                 Right) bei SB Pro                                         }
{ Ausgabe: Release-Parameter                                                }
{---------------------------------------------------------------------------}

FUNCTION GetRelease(Cell : Byte; Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetSustain: Legt den Sustain-Parameter fr die Hllkurve einer   }
{                      Zelle fest. Als Sustain-Bereich bezeichnet man den   }
{                      Niveaubereich zwischen Decay und Release.            }
{===========================================================================}
{ Eingabe: Cell  = Nummer der Generatorzelle (0-17)                         }
{          Side  = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,     }
{                  Right) bei SB Pro                                        }
{          Param = Sustain-Parameter (0-15)                                 }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetSustain(Cell : Byte; Side : TSides; Param : Byte);

{===========================================================================}
{ Funktion GetSustain: Liefert den Sustain-Parameter fr die Hllkurve      }
{                      einer Generatorzelle zurck.                         }
{===========================================================================}
{ Eingabe: Cell = Nummer der Zelle (0-17)                                   }
{          Side = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,      }
{                 Right) bei SB Pro                                         }
{ Ausgabe: Sustain-Parameter                                                }
{---------------------------------------------------------------------------}

FUNCTION GetSustain(Cell : Byte; Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetWaveForm: Legt die Wellenform einer Zelle fest. Der Wertebe-  }
{                       reich liegt zwischen 0 und 3.                       }
{===========================================================================}
{ Eingabe: Cell = Nummer der Generatorzelle (0-17)                          }
{          Side = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,      }
{                 Right) bei SB Pro                                         }
{          Form = Wellenform (0-3)                                          }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetWaveForm(Cell : Byte; Side : TSides; Form : Byte);

{===========================================================================}
{ Funktion GetWaveForm: Liefert die Wellenform einer Zelle zurck.          }
{===========================================================================}
{ Eingabe: Cell = Nummer der Generatorzelle (0-17)                          }
{          Side = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,      }
{                 Right) bei SB Pro                                         }
{ Ausgabe: Wellenform                                                       }
{---------------------------------------------------------------------------}

FUNCTION GetWaveForm(Cell : Byte; Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetChannelFreq: Legt die Frequenz eines Kanals fest (insgesamt   }
{                          gibt es 9 Kanle). Der Wertebereich fr den      }
{                          Frequenz-Parameter liegt zwischen 0 und 1023.    }
{===========================================================================}
{ Eingabe: Channel   = Nummer des Kanals (0-8)                              }
{          Side      = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left, }
{                      Right) bei SB Pro                                    }
{          FreqParam = Frequenz-Parameter (0-1023)                          }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetChannelFreq(Channel : Byte; Side : TSides; FreqParam : Word);

{===========================================================================}
{ Funktion GetChannelFreq: Liefert den mit Hilfe von SetChannelFreq einge-  }
{                          stellten Frequenz-Parameter eines Kanals zurck. }
{===========================================================================}
{ Eingabe: Channel = Nummer des Kanals (0-8)                                }
{          Side    = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,   }
{                    Right) bei SB Pro                                      }
{ Ausgabe: Frequenz-Parameter                                               }
{---------------------------------------------------------------------------}

FUNCTION GetChannelFreq(Channel : Byte; Side : TSides) : Word;

{===========================================================================}
{ Prozedur SetOctave: Legt die Oktave eines Kanals fest. Der Wertebereich   }
{                     fr die Oktave liegt zwischen 0 und 7.                }
{===========================================================================}
{ Eingabe: Channel = Nummer des Kanals (0-8)                                }
{          Side    = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,   }
{                    Right) bei SB Pro                                      }
{          Octave  = Oktave (0-7)                                           }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetOctave(Channel : Byte; Side : TSides; Octave : Byte);

{===========================================================================}
{ Funktion GetOctave: Liefert die Oktavnummer des angegebenen Kanals zu-    }
{                     rck.                                                 }
{===========================================================================}
{ Eingabe: Channel = Kanalnummer (0-8)                                      }
{          Side    = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,   }
{                    Right) bei SB Pro                                      }
{ Ausgabe: Nummer der Oktave                                                }
{---------------------------------------------------------------------------}

FUNCTION GetOctave(Channel : Byte; Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetTone: Schaltet einen Kanal an bzw. aus.                       }
{===========================================================================}
{ Eingabe: Channel = Nummer des Kanals (0-8)                                }
{          Side    = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,   }
{                    Right) bei SB Pro                                      }
{          On      = TRUE, wenn der Kanal angeschaltet werden soll, FALSE   }
{                    bedeutet Ausschalten                                   }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetTone(Channel : Byte; Side : TSides; On : Boolean);

{===========================================================================}
{ Funktion ToneOn: Liefert TRUE zurck, falls von dem angegebenen Kanal     }
{                  Tne gesendet werden knnen, sonst FALSE.                }
{===========================================================================}
{ Eingabe: Channel = Kanalnummer (0-8)                                      }
{          Side    = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,   }
{                    Right) bei SB Pro                                      }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION ToneOn(Channel : Byte; Side : TSides) : Boolean;

{===========================================================================}
{ Prozedur SetConnection: Legt die Art der Verknpfung der Zellen fr den   }
{                         angegebenen Kanal fest. Die Zellen knnen entwe-  }
{                         der parallel oder seriell (FM-Modulationsmodus)   }
{                         geschaltet werden.                                }
{===========================================================================}
{ Eingabe: Channel  = Nummer des Kanals (0-8)                               }
{          Side     = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,  }
{                     Right) bei SB Pro                                     }
{          WhatType = Zellenverknpfungsart (Parallel, Serial)              }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetConnection(Channel : Byte; Side : TSides;
  WhatType : TCellConnection);

{===========================================================================}
{ Funktion GetConnection: Liefert die Art der Zellenverknpfung fr den an- }
{                         gegebenen Kanal zurck.                           }
{===========================================================================}
{ Eingabe: Channel = Nummer des Kanals (0-8)                                }
{          Side    = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,   }
{                    Right) bei SB Pro                                      }
{ Ausgabe: Typ der Zellenverknpfung (Parallel, Serial)                     }
{---------------------------------------------------------------------------}

FUNCTION GetConnection(Channel : Byte; Side : TSides) : TCellConnection;

{===========================================================================}
{ Prozedur SetFeedback: Legt den Rckkopplungsgrad eines Kanals fest. Der   }
{                       Wert fr einen Kanal kann zwischen 0 und 8 liegen.  }
{                       Es ist zu beachten, da alle anderen Parameter des  }
{                       Kanals unverndert bleiben.                         }
{===========================================================================}
{ Eingabe: Channel = Nummer des Kanals (0-8)                                }
{          Side    = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,   }
{                    Right) bei SB Pro                                      }
{          FBParam = Rckkopplungsparameter (0-7)                           }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetFeedback(Channel : Byte; Side : TSides; FBParam : Byte);

{===========================================================================}
{ Funktion GetFeedback: Liefert den Rckkopplungsparameter eines Kanals zu- }
{                       rck.                                               }
{===========================================================================}
{ Eingabe: Channel = Kanalnummer (0-8)                                      }
{          Side    = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,   }
{                    Right) bei SB Pro                                      }
{ Ausgabe: Rckkopplungsparameter                                           }
{---------------------------------------------------------------------------}

FUNCTION GetFeedback(Channel : Byte; Side : TSides) : Byte;

{===========================================================================}
{ Prozedur SetRhythmInstruments: Schaltet die rhythmischen Instrumente (Hi  }
{                                Hat, Top Cymbal, Tom Tom, Snare Drum, Bass }
{                                Drum) ein bzw. aus. Jedes der Instrumente  }
{                                kann unabhngig von den anderen ein- bzw.  }
{                                ausgeschaltet werden.                      }
{===========================================================================}
{ Eingabe: Instruments = Instrumenten-Record                                }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetRhythmInstruments(Instruments : TRhythmInstruments);

{===========================================================================}
{ Funktion GetRhythmInstruments: Liefert Informationen, welche von den fnf }
{                                rhythmischen Instrumenten eingeschaltet    }
{                                sind und welche nicht.                     }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: Instruments = Instrumenten-Record                                }
{---------------------------------------------------------------------------}

PROCEDURE GetRhythmInstruments(VAR Instruments : TRhythmInstruments);

{===========================================================================}
{ Prozedur SetRhythmusMode: Schaltet den Rhythmus-Modus ein oder aus. Die-  }
{                           ser Modus mu eingeschaltet werden, wenn die    }
{                           rhythmischen Instrumente wie Hi Hat, Top Cymbal }
{                           u.a. verwendet werden sollen. Wird der Modus    }
{                           ausgeschaltet, so wird automatisch der Melodie- }
{                           Modus aktiviert.                                }
{===========================================================================}
{ Eingabe: On = TRUE bedeutet Einschalten, FALSE Ausschalten des Rhythmus-  }
{               Modus (Einschalten des Melodie-Modus)                       }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetRhythmusMode(On : Boolean);

{===========================================================================}
{ Funktion RhythmusModeOn: Liefert TRUE als Ergebnis zurck, falls der sog. }
{                          Rhythmus-Modus aktiv ist, sonst FALSE (Melodie-  }
{                          Modus eingeschaltet).                            }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION RhythmusModeOn : Boolean;

{===========================================================================}
{ Prozedur SetSpeechMode: Setzt den Modus fr zusammengesetzte Sprache oder }
{                         den normalen Synthese-Modus.                      }
{===========================================================================}
{ Eingabe: On = TRUE fr Sprachmodus, FALSE fr Synthese                    }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetSpeechMode(On : Boolean);

{===========================================================================}
{ Funktion SpeechModeOn: Liefert TRUE zurck, wenn der Modus fr zusammen-  }
{                        gesetzte Sprache aktiv ist und FALSE, falls der    }
{                        Synthese-Modus aktiv ist.                          }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: TRUE oder FALSE                                                  }
{---------------------------------------------------------------------------}

FUNCTION SpeechModeOn : Boolean;

{===========================================================================}
{ Prozedur SetInstrumentData: Legt alle Parameter fest, die das Klangbild   }
{                             eines Tones beeinflussen, also alle Daten,    }
{                             die fr die Nachahmung eines Instrumentes ge- }
{                             braucht werden.                               }
{===========================================================================}
{ Eingabe: Channel = Nummer des Kanals (0-8)                                }
{          Side    = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,   }
{                    Right) bei SB Pro                                      }
{          Data    = Zeiger auf die Instrumentdaten                         }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE SetInstrumentData(Channel : Byte; Side : TSides;
  Data : PInstrument);

{===========================================================================}
{ Prozedur PlayNote: Setzt den angegebenen Notenwert in einen Ton auf dem   }
{                    angegebenen Kanal um. Der Kanal wird automatisch akti- }
{                    viert.                                                 }
{===========================================================================}
{ Eingabe: Channel = Nummer des Kanals (0-8)                                }
{          Side    = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,   }
{                    Right) bei SB Pro                                      }
{          Note    = Notenwert (0-127)                                      }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE PlayNote(Channel : Byte; Side : TSides; Note : Byte);

{===========================================================================}
{ Prozedur NoteOff: Schaltet die Note durch Zurcksetzen der Frequenz und   }
{                   der Oktave und Deaktivierung des verwendeten Kanals     }
{                   aus.                                                    }
{===========================================================================}
{ Eingabe: Channel = Nummer des Kanals (0-8)                                }
{          Side    = Mono-Kanal (Mono) oder einer der Stero-Kanle (Left,   }
{                    Right) bei SB Pro                                      }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE NoteOff(Channel : Byte; Side : TSides);

IMPLEMENTATION

CONST

  {-------------------------------------------------------------------------}
  { Tabelle der Frequenz-Parameter fr die Noten                            }
  {-------------------------------------------------------------------------}

  FreqParamList : ARRAY[0..11] OF Word =
    (343, 363, 385, 408, 432, 458, 485, 514, 544, 577, 611, 647);

VAR

  {-------------------------------------------------------------------------}
  { FM-Portadressen des Sound Blaster Pro                                   }
  {-------------------------------------------------------------------------}

  FM_AddrLeft  : Word;   { FM-Register-Portadresse des linken Stereo-Kanals }
  FM_DataLeft  : Word;                     { FM-Datenport des linken Kanals }
  FM_AddrRight : Word;  { FM-Register-Portadresse des rechten Stereo-Kanals }
  FM_DataRight : Word;                    { FM-Datenport des rechten Kanals }

  {-------------------------------------------------------------------------}
  { FM-Portadressen des Sound Blaster                                       }
  {-------------------------------------------------------------------------}

  FM_Addr : Word;                                 { FM-Register-Portadresse }
  FM_Data : Word;                                            { FM-Datenport }

  {-------------------------------------------------------------------------}
  { Feldspeicher fr FM-Register                                            }
  {-------------------------------------------------------------------------}

  SBFMRegs : ARRAY[0..1, 0..255] OF Byte;

PROCEDURE InitSBFM;

VAR
  Count1, Count2 : Word;                                           { Zhler }

BEGIN
  InitSBDrv;                   { Umgebungsvariablen auslesen und berprfen }

  { FM-Portadressen setzen }

  FM_AddrLeft := SB_BaseAddress;
  FM_DataLeft := SB_BaseAddress+1;
  FM_AddrRight := SB_BaseAddress+2;
  FM_DataRight := SB_BaseAddress+3;
  FM_Addr := $388;
  FM_Data := $389;

  Count1 := 0;

  { Interrupts sperren }

  ASM
    CLI
    MOV AL,0FFh
    OUT 0021h,AL
  END;

  FOR Count2 := 1 TO 8 DO
    BEGIN
      SBFMOut($04, Mono, $80);                       { Timer-Reset auslsen }
      SBFMOut($03, Mono, $C0);     { Timer auf 320 Mikrosekunden einstellen }
      SBFMOut($04, Mono, $02);                                { Timer-Start }
      IF (Port[FM_Addr] AND $20) = $20 THEN      { Sound Blaster vorhanden? }
        BEGIN                                                        { nein }
          WriteLn;
          WriteLn('Fehler: Sound Blaster-Karte nicht vorhanden.');
          Halt;
        END;
      WHILE ((Port[FM_Addr] AND $20) = $00) AND (Count1 < $FFFF) DO
        BEGIN
          Inc(Count1);
          Delay(1);                              { eine Millisekunde warten }
        END;
      IF (Count1 = $FFFF) THEN             { Sound-Blaster-Karte vorhanden? }
        BEGIN                                                        { nein }
          WriteLn;
          WriteLn('Fehler: Sound Blaster-Karte nicht vorhanden.');
          Halt;
        END;
      IF Count2 <> 0 THEN
        Count1 := Count1 SHR 1;
    END;

  { Interrupts wieder freigeben }

  ASM
    MOV AL,00h
    OUT 0021h,AL
    STI
  END;

  IF NOT FMAvailable THEN                        { FM-Chip nicht vorhanden? }
    BEGIN                                    { nein, Fehlermeldung ausgeben }
      WriteLn;
      WriteLn('Fehler: FM-Chip nicht gefunden.');
      Halt;
    END;

  { FM-Register initialisieren }

  FOR Count1 := 0 TO 255 DO
    BEGIN
      SBFMOut(Count1, Left, 0);
      SBFMOut(Count1, Right, 0);
    END;
  SBFMOut($01, Mono, $20);                 { FM-Testregister initialisieren }
END;

FUNCTION FMAvailable : Boolean;

VAR
  Status1, Status2 : Byte;                                { Statusvariablen }
  Count            : Word;                                     { ein Zhler }

BEGIN
  SBFMOut($01, Mono, $00);                           { Testregister lschen }
  SBFMOut($04, Mono, $60);    { Zhlerberlauf nicht im Statusreg. anzeigen }
  SBFMOut($04, Mono, $80);              { Flags des Statusregisters lschen }
  Status1 := Port[FM_Addr];                          { Statusregister lesen }
  SBFMOut($02, Mono, $FF);     { sog. "schnellen" Zhler auf 255 einstellen }
  SBFMOut($04, Mono, $21);          { Daten weiterleiten und Zhler starten }
  FOR Count := 1 TO 200 DO                  { etwa 200 Mikrosekunden warten }
    Status2 := Port[FM_Addr];     { Inhalt des Statusreg. jedesmal auslesen }
  Status2 := Port[FM_Addr];                   { Statusregister erneut lesen }
  SBFMOut($04, Mono, $60);    { nichts weiterleiten, kein berlauf anzeigen }
  SBFMOut($04, Mono, $80);              { zum Schlu Statusregister lschen }
  FMAvailable := ((Status1 AND $E0) = $00) AND ((Status2 AND $E0) = $C0);
END;

FUNCTION SBFMIn(Reg : Byte; Side : TSides) : Byte;

BEGIN
  IF Side IN [Mono, Left] THEN       { Mono-Kanal oder linker Stereo-Kanal? }
    SBFMIn := SBFMRegs[0, Reg]             { ja, Inhalt des Registers lesen }
  ELSE                                               { rechter Stereo-Kanal }
    SBFMIn := SBFMRegs[1, Reg];
END;

PROCEDURE SBFMOut(Reg : Byte; Side : TSides; Data : Byte);

VAR
  Count : Word;                                                { ein Zhler }

BEGIN
  CASE Side OF
    Mono :                                                     { Mono-Kanal }
      BEGIN
        Port[FM_Addr] := Reg;  { Register-Nummer zum FM-Registerport senden }
        FOR Count := 0 TO 10 DO                  { 2,3 Mikrosekunden warten }
          BEGIN
          END;
        Port[FM_Data] := Data;     { Datenbyte an den FM-Datenport ausgeben }
        FOR Count := 0 TO 70 DO                 { 23,3 Mikrosekunden warten }
          BEGIN
          END;
        SBFMRegs[0, Reg] := Data;        { Datenbyte im Feldspeicher merken }
      END;
    Left :                                            { linker Stereo-Kanal }
      BEGIN                       { Register-Nummmer an den FM-Registerport }
        Port[FM_AddrLeft] := Reg;            { des linken FM-Chips ausgeben }
        FOR Count := 0 TO 10 DO                  { 2,3 Mikrosekunden warten }
          BEGIN
          END;
        Port[FM_DataLeft] := Data; { Datenbyte an den FM-Datenport ausgeben }
        FOR Count := 0 TO 70 DO                 { 23,3 Mikrosekunden warten }
          BEGIN
          END;
        SBFMRegs[0, Reg] := Data;        { Datenbyte im Feldspeicher merken }
      END;
    Right :                                          { rechter Stereo-Kanal }
      BEGIN                       { Register-Nummmer an den FM-Registerport }
        Port[FM_AddrRight] := Reg;          { des rechten FM-Chips ausgeben }
        FOR Count := 0 TO 10 DO                  { 2,3 Mikrosekunden warten }
          BEGIN
          END;
        Port[FM_DataRight] := Data;   { Datenbyte an den Datenport ausgeben }
        FOR Count := 0 TO 70 DO                 { 23,3 Mikrosekunden warten }
          BEGIN
          END;
        SBFMRegs[1, Reg] := Data;        { Datenbyte im Feldspeicher merken }
      END;
  END;
END;

PROCEDURE SetMultFactor(Cell : Byte; Side : TSides; Factor : Byte);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $20+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  SBFMOut(Reg, Side, (SBFMIn(Reg, Side) AND 240)            { Faktor setzen }
    OR (Factor AND 15));
END;

FUNCTION GetMultFactor(Cell : Byte; Side : TSides) : Byte;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $20+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  GetMultFactor := (SBFMIn(Reg, Side) AND 15);            { Faktor auslesen }
END;

PROCEDURE SetEnvelopeLength(Cell : Byte; Side : TSides; On : Boolean);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $20+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  IF On THEN                            { Hllkurvenverkrzung einschalten? }
    SBFMOut(Reg, Side, SBFMIn(Reg, Side) OR 16)                        { ja }
  ELSE                                           { keine Hllkurvendmpfung }
    SBFMOut(Reg, Side, SBFMIn(Reg, Side) AND NOT 16);
END;

FUNCTION EnvelopeLengthOn(Cell : Byte; Side : TSides) : Boolean;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $20+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  EnvelopeLengthOn := (SBFMIn(Reg, Side) AND 16) = 16;
END;

PROCEDURE SetEnvelopeType(Cell : Byte; Side : TSides; WhatType : TEnvelope);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $20+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  CASE WhatType OF
    Dimishing :                                       { Hllkurve abnehmend }
      SBFMOut(Reg, Side, SBFMIn(Reg, Side) OR 32);
    Continuing :                                 { Hllkurve kontinuierlich }
      SBFMOut(Reg, Side, SBFMIn(Reg, Side) AND NOT 32);
  END;
END;

FUNCTION GetEnvelopeType(Cell : Byte; Side : TSides) : TEnvelope;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $20+sbfm_GenCellOffs[Cell];   { Nr. des zustndigen Reg. bestimmen }
  IF (SBFMIn(Reg, Side) AND 32) = 32 THEN        { Hllkurvenart abnehmend? }
    GetEnvelopeType := Dimishing                                       { ja }
  ELSE                                                               { nein }
    GetEnvelopeType := Continuing;       { Typ der Hllkurve kontinuierlich }
END;

PROCEDURE SetVibrato(Cell : Byte; Side : TSides; On : Boolean; Level : Byte);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $20+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  IF On THEN                                  { Vibrato-Effekt einschalten? }
    SBFMOut(Reg, Side, SBFMIn(Reg, Side) OR 64)                        { ja }
  ELSE                                                               { nein }
    SBFMOut(Reg, Side, SBFMIn(Reg, Side) AND NOT 64);
  Reg := $BD;
  IF Level = 0 THEN                   { Schwankung in der Frequenz schwach? }
    SBFMOut(Reg, Side, SBFMIn(Reg, Side) AND NOT 64)                   { ja }
  ELSE                                                               { nein }
    SBFMOut(Reg, Side, SBFMIn(Reg, Side) OR 64);         { Schwankung stark }
END;

FUNCTION VibratoOn(Cell : Byte; Side : TSides) : Boolean;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $20+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  VibratoOn := (SBFMIn(Reg, Side) AND 64) = 64;
END;

FUNCTION VibratoLevel(Side : TSides) : Byte;

CONST
  Reg : Byte = $BD;

BEGIN
  VibratoLevel := (SBFMIn(Reg, Side) AND 64) DIV 64;
END;

PROCEDURE SetTremolo(Cell : Byte; Side : TSides; On : Boolean; Level : Byte);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $20+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  IF On THEN                                  { Tremolo-Effekt einschalten? }
    SBFMOut(Reg, Side, SBFMIn(Reg, Side) OR 128)                       { ja }
  ELSE                                          { nein, kein Tremolo-Effekt }
    SBFMOut(Reg, Side, SBFMIn(Reg, Side) AND NOT 128);
  Reg := $BD;
  IF Level = 0 THEN                  { Schwankung in der Amplitude schwach? }
    SBFMOut(Reg, Side, SBFMIn(Reg, Side) AND NOT 128)                  { ja }
  ELSE                                                               { nein }
    SBFMOut(Reg, Side, SBFMIn(Reg, Side) OR 128);        { Schwankung stark }
END;

FUNCTION TremoloOn(Cell : Byte; Side : TSides) : Boolean;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $20+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  TremoloOn := (SBFMIn(Reg, Side) AND 128) = 128;
END;

FUNCTION TremoloLevel(Side : TSides) : Byte;

CONST
  Reg : Byte = $BD;

BEGIN
  TremoloLevel := (SBFMIn(Reg, Side) AND 128) DIV 128;
END;

PROCEDURE SetAttenuation(Cell : Byte; Side : TSides; Factor : Byte);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $40+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  SBFMOut(Reg, Side, (SBFMIn(Reg, Side) AND 192)            { Faktor setzen }
    OR (Factor AND 63));
END;

FUNCTION GetAttenuation(Cell : Byte; Side : TSides) : Byte;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $40+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  GetAttenuation := (SBFMIn(Reg, Side) AND 63);
END;

PROCEDURE SetHigherPitchLevel(Cell : Byte; Side : TSides; Factor : Byte);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $40+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  SBFMOut(Reg, Side, (SBFMIn(Reg, Side) AND 63)             { Faktor setzen }
    OR (Factor SHL 6));
END;

FUNCTION GetHigherPitchLevel(Cell : Byte; Side : TSides) : Byte;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $40+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  GetHigherPitchLevel := (SBFMIn(Reg, Side) SHR 6);
END;

PROCEDURE SetDecay(Cell : Byte; Side : TSides; Param : Byte);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $60+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  SBFMOut(Reg, Side, (SBFMIn(Reg, Side) AND 240)         { Parameter setzen }
    OR (Param AND 15));
END;

FUNCTION GetDecay(Cell : Byte; Side : TSides) : Byte;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $60+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  GetDecay := (SBFMIn(Reg, Side) AND 15);           { Decay-Parameter lesen }
END;

PROCEDURE SetAttack(Cell : Byte; Side : TSides; Param : Byte);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $60+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  SBFMOut(Reg, Side, (SBFMIn(Reg, Side) AND 15)          { Parameter setzen }
    OR (Param SHL 4));
END;

FUNCTION GetAttack(Cell : Byte; Side : TSides) : Byte;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $60+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  GetAttack := (SBFMIn(Reg, Side) SHR 4);       { Attack-Parameter abfragen }
END;

PROCEDURE SetRelease(Cell : Byte; Side : TSides; Param : Byte);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $80+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  SBFMOut(Reg, Side, (SBFMIn(Reg, Side) AND 240)         { Parameter setzen }
    OR (Param AND 15));
END;

FUNCTION GetRelease(Cell : Byte; Side : TSides) : Byte;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $80+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  GetRelease := (SBFMIn(Reg, Side) AND 15);       { Release-Parameter lesen }
END;

PROCEDURE SetSustain(Cell : Byte; Side : TSides; Param : Byte);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $80+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  SBFMOut(Reg, Side, (SBFMIn(Reg, Side) AND 15)          { Parameter setzen }
    OR (Param SHL 4));
END;

FUNCTION GetSustain(Cell : Byte; Side : TSides) : Byte;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $80+sbfm_GenCellOffs[Cell];       { Nummer des Registers bestimmen }
  GetSustain := (SBFMIn(Reg, Side) SHR 4);        { Sustain-Parameter lesen }
END;

PROCEDURE SetWaveForm(Cell : Byte; Side : TSides; Form : Byte);

CONST
  Reg : Byte = $E0;                                       { Register-Nummer }

BEGIN
  SBFMOut(Reg+sbfm_GenCellOffs[Cell], Side, Form AND 3); { Wellentyp setzen }
END;

FUNCTION GetWaveForm(Cell : Byte; Side : TSides) : Byte;

CONST
  Reg : Byte = $E0;                                       { Register-Nummer }

BEGIN
  GetWaveForm := (SBFMIn(Reg+sbfm_GenCellOffs[Cell], Side) AND 3);
END;

PROCEDURE SetChannelFreq(Channel : Byte; Side : TSides; FreqParam : Word);

VAR
  Reg1, Reg2 : Byte;                                     { Register-Nummern }

BEGIN

  { Register-Nummern bestimmen }

  Reg1 := $A0+Channel;
  Reg2 := $B0+Channel;

  { Daten an FM-Ports senden }

  SBFMOut(Reg1, Side, FreqParam AND 255);
  SBFMOut(Reg2, Side, (SBFMIn(Reg2, Side) AND 252)
    OR ((FreqParam SHR 8) AND 3));
END;

FUNCTION GetChannelFreq(Channel : Byte; Side : TSides) : Word;

VAR
  Reg1, Reg2 : Byte;                                     { Register-Nummern }

BEGIN

  { Register-Nummern bestimmen }

  Reg1 := $A0+Channel;
  Reg2 := $B0+Channel;

  { Frequenz-Parameter auslesen }

  GetChannelFreq := SBFMIn(Reg1, Side)+(256*(SBFMIn(Reg2, Side) AND 3));
END;

PROCEDURE SetOctave(Channel : Byte; Side : TSides; Octave : Byte);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $B0+Channel;                      { Nummer des Registers ermitteln }
  SBFMOut(Reg, Side, (SBFMIn(Reg, Side) AND 227) OR ((Octave AND 7) SHL 2));
END;

FUNCTION GetOctave(Channel : Byte; Side : TSides) : Byte;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $B0+Channel;                      { Nummer des Registers ermitteln }
  GetOctave := (((SBFMIn(Reg, Side) AND NOT 3) AND 31) SHR 2);
END;

PROCEDURE SetTone(Channel : Byte; Side : TSides; On : Boolean);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $B0+Channel;                           { Register-Nummer bestimmen }
  IF On THEN                                            { Kanal anschalten? }
    SBFMOut(Reg, Side, SBFMIn(Reg, Side) OR 32)                        { ja }
  ELSE                                                  { nein, ausschalten }
    SBFMOut(Reg, Side, SBFMIn(Reg, Side) AND NOT 32);
END;

FUNCTION ToneOn(Channel : Byte; Side : TSides) : Boolean;

VAR
  Reg : Byte;

BEGIN
  Reg := $B0+Channel;
  ToneOn := (SBFMIn(Reg, Side) AND 32) = 32;
END;

PROCEDURE SetConnection(Channel : Byte; Side : TSides;
  WhatType : TCellConnection);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $C0+Channel;   { Bestimmen, welches Reg. angesprochen werden soll? }
  CASE WhatType OF
    Parallel :
      SBFMOut(Reg, Side, SBFMIn(Reg, Side) OR 1);
    Serial :
      SBFMOut(Reg, Side, SBFMIn(Reg, Side) AND NOT 1);
  END;
END;

FUNCTION GetConnection(Channel : Byte; Side : TSides) : TCellConnection;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $C0+Channel;   { Bestimmen, welches Reg. angesprochen werden soll? }
  IF (SBFMIn(Reg, Side) AND 1) = 1 THEN        { Zellen parallel verknpft? }
    GetConnection := Parallel                                          { ja }
  ELSE                                                               { nein }
    GetConnection := Serial;                     { serielle Verknpfungsart }
END;

PROCEDURE SetFeedback(Channel : Byte; Side : TSides; FBParam : Byte);

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $C0+Channel;                           { Register-Nummer bestimmen }
  SBFMOut(Reg, Side, (SBFMIn(Reg, Side) AND 241) OR ((FBParam AND 7) SHL 1));
END;

FUNCTION GetFeedback(Channel : Byte; Side : TSides) : Byte;

VAR
  Reg : Byte;                                             { Register-Nummer }

BEGIN
  Reg := $C0+Channel;                           { Register-Nummer bestimmen }
  GetFeedback := (((SBFMIn(Reg, Side) AND NOT 1) AND 15) SHR 1);
END;

PROCEDURE SetRhythmInstruments(Instruments : TRhythmInstruments);

CONST
  Reg : Byte = $BD;                                       { Register-Nummer }

BEGIN
  WITH Instruments DO
    BEGIN
      IF HiHat THEN                                 { "Hi Hat" einschalten? }
        SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) OR 1)                     { ja }
      ELSE                                              { nein, ausschalten }
        SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) AND NOT 1);
      IF TopCymbal THEN                         { "Top Cymbal" einschalten? }
        SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) OR 2)                     { ja }
      ELSE                                                           { nein }
        SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) AND NOT 2);
      IF TomTom THEN                                { "Tom Tom" anschalten? }
        SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) OR 4)                                 { ja }
      ELSE                                                           { nein }
        SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) AND NOT 4);
      IF SnareDrum THEN           { Soll "Snare Drum" eingeschaltet werden? }
        SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) OR 8)                     { ja }
      ELSE                                                           { nein }
        SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) AND NOT 8);
      IF BassDrum THEN                           { "Bass Drum" einschalten? }
        SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) OR 16)                    { ja }
      ELSE                                                           { nein }
        SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) AND NOT 16);
    END;
END;

PROCEDURE GetRhythmInstruments(VAR Instruments : TRhythmInstruments);

CONST
  Reg : Byte = $BD;                                       { Register-Nummer }

BEGIN
  WITH Instruments DO
    BEGIN
      IF (SBFMIn(Reg, Mono) AND 1) = 1 THEN       { "Hi Hat" eingeschaltet? }
        HiHat := TRUE                                                  { ja }
      ELSE                                                           { nein }
        HiHat := FALSE;
      IF (SBFMIn(Reg, Mono) AND 2) = 2 THEN           { "Top Cymbal" aktiv? }
        TopCymbal := TRUE                                              { ja }
      ELSE                                                           { nein }
        TopCymbal := FALSE;
      IF (SBFMIn(Reg, Mono) AND 4) = 4 THEN                 { "Tom Tom" an? }
        TomTom := TRUE                                                 { ja }
      ELSE                                                           { nein }
        TomTom := FALSE;
      IF (SBFMIn(Reg, Mono) AND 8) = 8 THEN           { "Snare Drum" aktiv? }
        SnareDrum := TRUE                                              { ja }
      ELSE                                                           { nein }
        SnareDrum := FALSE;
      IF (SBFMIn(Reg, Mono) AND 16) = 16 THEN  { "Bass Drum" eingeschaltet? }
        BassDrum := TRUE                                               { ja }
      ELSE                                                           { nein }
        BassDrum := FALSE;
    END;
END;

PROCEDURE SetRhythmusMode(On : Boolean);

CONST
  Reg : Byte = $BD;                                       { Register-Nummer }

BEGIN
  IF On THEN                                  { Rhythmus-Modus einschalten? }
    SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) OR 32)                        { ja }
  ELSE                                                { nein, Melodie-Modus }
    SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) AND NOT 32);
END;

FUNCTION RhythmusModeOn : Boolean;

CONST
  Reg : Byte = $BD;                                       { Register-Nummer }

BEGIN
  RhythmusModeOn := (SBFMIn(Reg, Mono) AND 32) = 32;
END;

PROCEDURE SetSpeechMode(On : Boolean);

CONST
  Reg : Byte = $08;

BEGIN
  IF On THEN                                      { Sprachmodus aktivieren? }
    SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) OR 128)                       { ja }
  ELSE                                                               { nein }
    SBFMOut(Reg, Mono, SBFMIn(Reg, Mono) AND NOT 128); { Synthese-Modus ein }
END;

FUNCTION SpeechModeOn : Boolean;

CONST
  Reg : Byte = $08;

BEGIN
  SpeechModeOn := (SBFMIn(Reg, Mono) AND 128) = 128;
END;

PROCEDURE SetInstrumentData(Channel : Byte; Side : TSides;
  Data : PInstrument);

VAR
  Modulator, Carrier : Byte;        { Offset der Modulator- und Trgerzelle }

BEGIN

  { Offsets bestimmen (Modulator und Trger }

  Modulator := sbfm_ModCellOffs[Channel];
  Carrier := sbfm_CarrCellOffs[Channel];

  { Daten des Instrumentes setzen }

  IF Data <> NIL THEN
    BEGIN
      SBFMOut($20+Modulator, Side, Data^.Mod_Tone);
      SBFMOut($20+Carrier, Side, Data^.Carr_Tone);
      SBFMOut($40+Modulator, Side, Data^.Mod_Ampl);
      SBFMOut($40+Carrier, Side, Data^.Carr_Ampl);
      SBFMOut($60+Modulator, Side, Data^.Mod_DecAtt);
      SBFMOut($60+Carrier, Side, Data^.Carr_DecAtt);
      SBFMOut($80+Modulator, Side, Data^.Mod_RelSus);
      SBFMOut($80+Carrier, Side, Data^.Carr_RelSus);
      SBFMOut($E0+Modulator, Side, Data^.Mod_Wave);
      SBFMOut($E0+Carrier, Side, Data^.Carr_Wave);
      SBFMOut($C0+Channel, Side, Data^.SynthFB);
    END;
END;

PROCEDURE PlayNote(Channel : Byte; Side : TSides; Note : Byte);

VAR
  FreqParam         : Word;                   { Frequenz-Parameter der Note }
  Octave            : Byte;                                        { Oktave }
  RhythmInstruments : TRhythmInstruments;  { Aktivitten der rhythm. Instr. }

BEGIN
  FreqParam := FreqParamList[Note MOD 12];   { Frequenz-Parameter bestimmen }
  Octave := Note DIV 12;                                 { Oktave bestimmen }
  IF NOT RhythmusModeOn THEN { Melodie-Modus (maximal 9 Instrumente) aktiv? }
    BEGIN                                                              { ja }
      SetChannelFreq(Channel, Side, FreqParam);       { Frequenz einstellen }
      SetOctave(Channel, Side, Octave);                 { Oktave einstellen }
      SetTone(Channel, Side, TRUE);                      { Kanal aktivieren }
    END
  ELSE                                               { Rhythmus-Modus aktiv }
    BEGIN
      IF Channel = 8 THEN                            { "Tom Tom" (Kanal 8)? }
        BEGIN                                                          { ja }
          SetChannelFreq(Channel, Side, FreqParam);   { Frequenz einstellen }
          SetOctave(Channel, Side, Octave);             { Oktave einstellen }
        END;
      IF Channel < 6 THEN      { normales Instrument im rhythmischen Modus? }
        BEGIN                                                          { ja }
          SetChannelFreq(Channel, Side, FreqParam);   { Frequenz einstellen }
          SetOctave(Channel, Side, Octave);             { Oktave einstellen }
          SetTone(Channel, Side, TRUE);                  { Kanal aktivieren }
        END
      ELSE                              { nein, ein rhythmisches Instrument }
        BEGIN                                { Aktivitten der rhythmischen }
          GetRhythmInstruments(RhythmInstruments);      { Instrumente lesen }

          { rhythm. Instrument in Abhngigkeit }
          { von dem verwendeten Kanal setzen   }

          WITH RhythmInstruments DO
            CASE Channel OF
              6  : BassDrum := TRUE;
              7  : SnareDrum := TRUE;
              8  : TomTom := TRUE;
              9  : TopCymbal := TRUE;
              10 : HiHat := TRUE;
            END;
          SetRhythmInstruments(RhythmInstruments);
        END;
    END
END;

PROCEDURE NoteOff(Channel : Byte; Side : TSides);

VAR
  RhythmInstruments : TRhythmInstruments;  { Aktivitten der rhythm. Instr. }

BEGIN
  IF NOT RhythmusModeOn THEN                         { Melodie-Modus aktiv? }
    BEGIN                                                              { ja }
      SetChannelFreq(Channel, Side, 0);             { Frequenz zurcksetzen }
      SetOctave(Channel, Side, 0);                    { Oktave zurcksetzen }
      SetTone(Channel, Side, FALSE);                    { Kanal ausschalten }
    END
  ELSE                                         { nein, Rhythmus-Modus aktiv }
    BEGIN
      IF Channel = 8 THEN                            { "Tom Tom" (Kanal 8)? }
        BEGIN
          SetChannelFreq(Channel, Side, 0);         { Frequenz zurcksetzen }
          SetOctave(Channel, Side, 0);                { Oktave zurcksetzen }
        END;
      IF Channel < 6 THEN      { normales Instrument im rhythmischen Modus? }
        BEGIN                                                          { ja }
          SetChannelFreq(Channel, Side, 0);         { Frequenz zurcksetzen }
          SetOctave(Channel, Side, 0);                { Oktave zurcksetzen }
          SetTone(Channel, Side, FALSE);                { Kanal ausschalten }
        END
      ELSE                              { nein, ein rhythmisches Instrument }
        BEGIN                                { Aktivitten der rhythmischen }
          GetRhythmInstruments(RhythmInstruments);      { Instrumente lesen }

          { rhythm. Instrument in Abhngigkeit    }
          { von dem verwendeten Kanal ausschalten }

          WITH RhythmInstruments DO
            CASE Channel OF
              6  : BassDrum := FALSE;
              7  : SnareDrum := FALSE;
              8  : TomTom := FALSE;
              9  : TopCymbal := FALSE;
              10 : HiHat := FALSE;
            END;
          SetRhythmInstruments(RhythmInstruments);
        END;
    END;
END;

{---------------------------------------------------------------------------}
{ Startcode der Unit                                                        }
{---------------------------------------------------------------------------}

BEGIN
  InitSBFM;
END.
