/*****************************************************************************/
/*                              AUTOCAL.C                                    */
/*****************************************************************************/
/*                                                                           */
/* Designed by: Michel Pelletier                                             */
/* Date: 	April 30, 1993                                               */
/*                                                                           */
/* Synopsis:    This file contains the AutomaticCalibration() function       */
/*              for automatically calibrating analog references as well as   */
/*              input amplifier gain for the IM-LC and the IM-ASD.           */
/*                                                                           */
/*****************************************************************************/
/* Inclusions: */

#include <grabtest.h>

/*****************************************************************************/
/* Variables: */

unsigned short	gain_settings[4] = {  I_GAIN3 ,  I_GAIN2 ,  I_GAIN1 ,  I_GAIN0  };
unsigned char	*gain_strings[4] = { "I_GAIN3", "I_GAIN2", "I_GAIN1", "I_GAIN0" };
char		wait_states[4] = { '|', '/', '-', '\\' };

extern signed long	blackref, whiteref, gain;
extern short 		interlace;
extern int		grab_source, grab_destination;
extern short		roix, roiy, roi_posx, roi_posy;
extern int		wleft, wtop, wright, wbottom, wwidth, wheight;
extern	int		font_xsize, font_ysize;

/*****************************************************************************/
/* Prototypes: */

static int  GetLevels( unsigned long fraction, unsigned short grb_dst, 
			unsigned short *h_level, unsigned short *l_level );

/*****************************************************************************/

int	AutomaticCalibration( void )
{
int		status, i, step;
unsigned short	best_gain, best_diff, high_level, low_level;
unsigned short	roi_xsize, roi_ysize, roi_xstart, roi_ystart;
unsigned long	roi_product, roi_fraction;
signed long	sav_blackref, sav_whiteref, sav_gain;

Message("Automatic Calibration:  Please wait...   ");
ShowMessage( WITHOUT_PAUSE );

/*===================================*/
/* Initialise variables and pointers */
/*===================================*/
status = E_OK;
best_diff = 0;
sav_gain = gain;
sav_whiteref = whiteref;
sav_blackref = blackref;
roi_xsize = roix - 16;		/* We will be processing the */
roi_ysize = roiy - 16;		/* whole image, except the   */
roi_xstart = roi_posx + 8;	/* first 8 pixels on every   */
roi_ystart = roi_posy + 8;	/* side                      */

/* Calculate the total number of pixels in the ROI */
roi_product = ((unsigned long)roi_xsize) * ((unsigned long)roi_ysize);

/* Calculate the fraction of pixels that will be used for finding    */
/* the lower and higher acceptable levels of the image.  In this     */
/* case, acceptability is set at 1% of the total number of pixels.   */
roi_fraction = (unsigned long)( (((double)roi_product) * ((double)0.01)) + 
							((double)0.5) );
/*=======================*/
/* Set up ROI parameters */
/*=======================*/
pcstroisiz( roi_xsize, roi_ysize );
pcstroipos( I_ROI1, 8, 8 );
pcstroipos( I_ROI2, roi_xstart, roi_ystart );

/*========================================================*/
/* Let's make sure the input amplifier is at it's optimal */
/* setting.                                               */
/*========================================================*/
whiteref = 0;
blackref = 63;
for( i = 0; i < 4; i++ )
	{
	OVR_Print( wleft + (font_xsize * 41), wtop + font_ysize,
		1, 1, wleft + (font_xsize * 41), wtop + font_ysize,
		OVR_BLUE, OVR_BRIGHTWHITE, "%c", wait_states[(step++) % 3] );
	gain = gain_settings[i];
	GrabImages( 1 );
	GetLevels( roi_fraction, grab_destination, &high_level, &low_level );
	if( ((high_level - low_level) > best_diff) &&
	     (high_level < WHITE_LEVEL) &&
	     (low_level > BLACK_LEVEL) )
		{
		best_gain = gain_settings[i];
		best_diff = high_level - low_level;
		}
	}
if( best_diff == 0 )
	{
	status = E_CANT_SET_PREAMP;
	}
else
	{
	gain = best_gain;
	}

if( status == E_OK )
	{
	/*============================================*/
	/* Let's calibrate black and white references */
	/*============================================*/
	whiteref = 0;
	blackref = 63;
	do	{
		OVR_Print( wleft + (font_xsize * 41), wtop + font_ysize,
			1, 1, wleft + (font_xsize * 41), wtop + font_ysize,
			OVR_BLUE, OVR_BRIGHTWHITE, "%c", wait_states[(step++) % 3] );
		GrabImages( 1 );
		GetLevels( roi_fraction, grab_destination, &high_level,
							&low_level );
		if( high_level < WHITE_LEVEL )
			{
			whiteref++;
			}
		if( whiteref == 64 )
			{
			status = E_CANT_SET_WHITE;
			}
		if( low_level > BLACK_LEVEL )
			{
			blackref--;
			}
		if( blackref == -1 )
			{
			status = E_CANT_SET_BLACK;
			}
		} while( (status == E_OK) &&
			((high_level < WHITE_LEVEL) ||
			(low_level > BLACK_LEVEL)) );
	}
RemoveMessage();
if( status == E_OK )
	{
	Message( "dislgain( %s );", gain_strings[gain] );
	Message( "distrefwhite( %2d );", whiteref );
	Message( "distrefblack( %2d );", blackref );
	ShowMessage( WITH_PAUSE );
	}
else
	{
	/*========================================*/
	/* Restore parameters if an error occured */
	/*========================================*/
	gain = sav_gain;
	whiteref = sav_whiteref;
	blackref = sav_blackref;
	}

/*========================*/
/* Restore ROI parameters */
/*========================*/
pcstroisiz( roix, roiy );
pcstroipos( I_ROI1, 0, 0 );
pcstroipos( I_ROI2, roi_posx, roi_posy );
HostWait4Proc;

return( status );
}

/*****************************************************************************/

static int GetLevels( unsigned long fraction, unsigned short grb_dst, 
	unsigned short *h_level, unsigned short *l_level )
{
int	i;
unsigned long	histo_buf[256];
unsigned long	*histo_ptr;
unsigned long	accumulator;

pcslsync( I_PROC_ALONE );
pcstcond( 0, I_FOR, 1 );

/*=========================*/
/* Calculate the histogram */
/*=========================*/
stcchist( grb_dst, I_PAL_IS_256, I_LUT_CLEAR );
iosditag( I_PROC );
iowtitag( I_PROC );

/*=================*/
/* Read the result */
/*=================*/
lslmrdlut( I_LUT_STAT_24B, 512L );
iogetbuf( (unsigned short *)histo_buf, 512 );

/*======================*/
/* Find the lower level */
/*======================*/
accumulator = 0L;
histo_ptr = &histo_buf[0];
for( i=0; (i<256) && (accumulator < fraction); i++ )
	{
	accumulator += *histo_ptr;
	histo_ptr++;
	}
*l_level = i - 1;

/*=======================*/
/* Find the higher level */
/*=======================*/
accumulator = 0L;
histo_ptr = &histo_buf[255];
for( i=0; (i<256) && (accumulator < fraction); i++ )
	{
	accumulator += *histo_ptr;
	histo_ptr--;
	}
*h_level = 256 - i;

return( E_OK );
}

/*****************************************************************************/
