/*********************************************************************/
/*                          WINALIGN.C                               */
/*********************************************************************/
/*
*	By:			Michel Pelletier
*				Matrox Electronic Systems ltd.
*
*	Date:		November 3, 1993.
*
*	Synopsys:	MIL Windows 3.1 application for the Image-LC,
*				that is used for adjusting VGA to Image-LC screen
*				alignement.
*/
/*********************************************************************/

/*======================*/
/* Inclusion statements */
/*======================*/
#include	<windows.h>
#include	<commdlg.h>
#include	<stdlib.h>
#include	<stdio.h>
#include	<string.h>
#include	<time.h>
#include	<mil.h>
#include	<imseries.h>
#include	<i_head.h>
#include	<proto.h>
#include	"winalign.h"
#include	"video.h"

/*===================*/
/* Local definitions */
/*===================*/
#define	E_OK						0
#define	E_MIL_INIT_ERROR			-1
#define E_FILE_IO_NOT_COMPLETED		-2
#define E_FILE_IO_IMPROPER_FORMAT   -3
#define	DEF_KEYING_INDEX			247
#define	DEF_KEYING_MASK				0xff
#define DEF_DRAWING_COLOR			(RGB(160,160,164))
#define MAX_SIZE_OF_A_LINE			128
#define O_INTERLACED				1
#define	O_NON_INTERLACED			2
#define	IMAGE_LC					3

/*=====================*/
/* Function prototypes */
/*=====================*/
int PASCAL WinMain( HANDLE hInstance, HANDLE hPrevInstance, LPSTR Cmdtl, int nCmdShow);
long FAR PASCAL _export MainWndProc( HWND hWnd, UINT message, UINT wParam, LONG lParam);
BOOL FAR PASCAL _export About(HWND hDlg, UINT message, UINT wParam, LONG lParam);
BOOL FAR PASCAL _export Instructions(HWND hDlg, UINT message, UINT wParam, LONG lParam);
BOOL FAR PASCAL _export KeyingControlHelp( HWND hDlg, UINT message, UINT wParam, LONG lParam);
BOOL FAR PASCAL _export KeyingControl( HWND hDlg, UINT message, UINT wParam, LONG lParam);

static int	InitMilSystem( void );
static void	ReleaseMilSystem( void );
static int	VALfileInput( char *filename_str );
static int	VALfileOutput( char *filename_str );
static WORD GetOutHTotalDisplayed( void );
static WORD GetOutHFrontPorch( void );
static WORD GetOutHSyncPulse( void );
static WORD GetOutHBackPorch( void );
static void SetOutHParameters( WORD ohfp, WORD ohsync, WORD ohbp, WORD ohdisp );
static WORD GetOutVTotalDisplayed( void );
static WORD GetOutVFrontPorch( void );
static WORD GetOutVSyncPulse( void );
static WORD GetOutVBackPorch( void );
static void SetOutVParameters( WORD ovfp, WORD ovsync, WORD ovbp, WORD ovdisp );
static BYTE GetOutScanMode( void );
static BYTE GetGspRatio( void );

/*==================*/
/* Global variables */
/*==================*/
MIL_ID			MilImage;			/* Image buffer identifier */
MIL_ID			MilSystem;			/* System identifier       */
HANDLE			hInst;      		/* Handle to this instance */
VIDPARMS		video;
BOOL			modif_parms = FALSE;
char			ver_string[] = "WinAlign VGA to Image-LC screen alignement utility V1.0";
char			text_buffer[100];
unsigned char	keying_index = DEF_KEYING_INDEX;
unsigned char	keying_mask = DEF_KEYING_MASK;
COLORREF		drawing_color = DEF_DRAWING_COLOR;

/*---------------------------------------------------------------------------*/
/*
*	Function:	WinMain()
*
*	By:			Michel Pelletier
*				Matrox Electronic Systems ltd.
*
*	Date:		October 22, 1993.
*
*	Synopsys:	Calls initialization function, processes message loop
*/
/*--------------------------------------------------------------------------*/

int PASCAL WinMain( HANDLE hInstance, HANDLE hPrevInstance, LPSTR Cmdtl, 
		int nCmdShow)
{
static MSG			msg;
static WNDCLASS		wc;
static HWND			hwnd;
static HACCEL		hAcc;
short				board_type;

hInst = hInstance;

/* Only allow one instance of program */
if( hPrevInstance )
	{
	MessageBox( NULL,	"WinAlign already active.\n"
                     	"This version of WinAlign does not allow "
                      	"sharing of the Image-LC card.",
                      	"WinAlign - Error", 
                      	MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( FALSE );
	}

wc.style          = CS_HREDRAW | CS_VREDRAW;
wc.lpfnWndProc    = (WNDPROC)MainWndProc;
wc.cbClsExtra     = 0;
wc.cbWndExtra     = 0;
wc.hInstance      = hInstance;
wc.hIcon          = LoadIcon( hInstance, MAKEINTRESOURCE( IDI_ICON1 ) );
wc.hCursor        = LoadCursor( NULL, IDC_ARROW );
wc.hbrBackground  = NULL;
wc.lpszMenuName   = MAKEINTRESOURCE( IDR_MENU1 );
wc.lpszClassName  = "WinAlign";
if( !RegisterClass( &wc ) )
	{
	return( FALSE );
	}

/* Initialize MIL library environment */
if( InitMilSystem() != E_OK )
	{
	MessageBox( NULL,	"MIL Library Initialization Failed!\n",
                      	"WinAlign - Error", 
                      	MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( FALSE );
	}

/* Make sure we have an Image-LC */
MnatEnterNativeMode( M_NAT_NULL );
pciqsys( I_IQ_BOARD_TYPE, &board_type );
MnatLeaveNativeMode( M_NAT_NULL );
if( (board_type & 0x00ff) != IMAGE_LC )
	{
	MessageBox( NULL,	"This utility is for the Image-LC only.\n",
                      	"WinAlign - Error", 
                      	MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( FALSE );
	}

hwnd = CreateWindow(	"WinAlign",		/* window class name       */
			"WinAlign Utility",			/* window caption          */
			WS_OVERLAPPEDWINDOW |
			WS_HSCROLL | WS_VSCROLL,	/* window style            */
			CW_USEDEFAULT,				/* initial x position      */
			CW_USEDEFAULT,				/* initial y position      */
			CW_USEDEFAULT,				/* initial x size          */
			CW_USEDEFAULT,				/* initial y size          */
			NULL,						/* parent window handle    */
			NULL,						/* window menu handle      */
			hInstance,					/* program instance handle */
			NULL );						/* creation parameters     */

if( !hwnd )
	{
	ReleaseMilSystem();
	return( FALSE );
	}

/* Ok.. display window.. */
ShowWindow( hwnd, nCmdShow );
UpdateWindow( hwnd );

/* Message processing loop */
hAcc = LoadAccelerators( hInstance, MAKEINTRESOURCE( IDR_ACCELERATOR1 ) );
while( GetMessage( &msg, NULL, 0, 0 ) )
	{
	if( !TranslateAccelerator( hwnd, hAcc, &msg ) )
		{
		TranslateMessage( &msg );
		DispatchMessage( &msg );
		}
	}
return( msg.wParam );
}

/*---------------------------------------------------------------------------*/
/*
*	Function:	MainWndProc()
*
*	By:			Michel Pelletier
*				Matrox Electronic Systems ltd.
*
*	Date:		October 22, 1993.
*
*	Synopsys:	Processes messages.
*/
/*---------------------------------------------------------------------------*/

long FAR PASCAL _export MainWndProc( HWND hWnd, UINT message, UINT wParam, 
				LONG lParam)
{
FARPROC			lpProcDialog;
HPEN			hPen, hBkPen, hOldPen;
HBRUSH			hBkBrush, hOldBrush;
OPENFILENAME	ofn;
PAINTSTRUCT		ps;
TEXTMETRIC		tm;
RECT			client_rect;
POINT			screen_points[4];
HDC				hDC;
static char		filename[_MAX_PATH];
static HMENU	hmenu;
static short	nVscrollPos, nVscrollMax, nHscrollPos, nHscrollMax;
static int		sav_vga_switch, sav_vga_comp;
short			nVscrollInc, nHscrollInc;

switch( message )
	{
	case WM_CREATE:
	hmenu = GetMenu( hWnd );

	/* Make sure scroll bars are invisible */
	SetScrollRange (hWnd, SB_VERT, 0, 0, TRUE );
	SetScrollRange (hWnd, SB_HORZ, 0, 0, TRUE );
	return( 0 );

	case WM_PAINT:
	hDC = BeginPaint( hWnd, &ps );

	/* Select brush for drawing background color */
	hBkBrush  = CreateSolidBrush( drawing_color );
	hBkPen    = CreatePen( PS_SOLID, 1, drawing_color );
    hOldBrush = SelectObject( hDC, hBkBrush );
    hOldPen   = SelectObject( hDC, hBkPen );

	/* Draw the client area background */
	Rectangle( hDC, ps.rcPaint.left,  ps.rcPaint.top,
		ps.rcPaint.right, ps.rcPaint.bottom );

	/* Select solid pen */
	/* Drawing with white color  */
	hPen = CreatePen( PS_SOLID, 1, RGB(255, 255, 255) );
	SelectObject( hDC, hPen );

	/* Draw crosshair pattern */
	GetClientRect( hWnd, &client_rect );
	MoveTo( hDC, client_rect.right / 2, 0 );
	LineTo( hDC, client_rect.right / 2, client_rect.bottom );
	MoveTo( hDC, 0,                     client_rect.bottom / 2 );
	LineTo( hDC, client_rect.right,     client_rect.bottom / 2 );

	/* Draw ID patterns */
	GetTextMetrics( hDC, &tm );

	MoveTo( hDC, (client_rect.right / 4) + (tm.tmMaxCharWidth / 2), 0 );
	LineTo( hDC, (client_rect.right / 4) + (tm.tmMaxCharWidth / 2), tm.tmHeight );
	MoveTo( hDC, client_rect.right / 4, tm.tmHeight / 2 );
	LineTo( hDC, (client_rect.right / 4) + tm.tmMaxCharWidth, tm.tmHeight / 2 );

	MoveTo( hDC, client_rect.right / 4, tm.tmHeight + tm.tmExternalLeading );
	LineTo( hDC, (client_rect.right / 4) + tm.tmMaxCharWidth, (2 * tm.tmHeight) + tm.tmExternalLeading );
	MoveTo( hDC, client_rect.right / 4, (2 * tm.tmHeight) + tm.tmExternalLeading );
	LineTo( hDC, (client_rect.right / 4) + tm.tmMaxCharWidth, tm.tmHeight + tm.tmExternalLeading );

	SetTextColor( hDC, RGB(255, 255, 255) );
	SetBkMode( hDC, TRANSPARENT );
	TextOut( hDC, (client_rect.right / 4) + tm.tmMaxCharWidth + tm.tmAveCharWidth, 
		0, text_buffer, 
		wsprintf( text_buffer, "= VGA Buffer" ) );	
	TextOut( hDC, (client_rect.right / 4) + tm.tmMaxCharWidth + tm.tmAveCharWidth, 
		tm.tmHeight + tm.tmExternalLeading, text_buffer,
		wsprintf( text_buffer, "= IM-LC Buffer" ) );	
	SetBkMode( hDC, OPAQUE );

	/* Restore pen and brush */
	SelectObject( hDC, hOldPen );
	SelectObject( hDC, hOldBrush );
	DeleteObject( hPen );
	DeleteObject( hBkPen );
	DeleteObject( hBkBrush );

	EndPaint(hWnd, &ps);
	return( 0 );

	case WM_SIZE:
	case WM_MOVE:
	/* Determine where the client area is located over */
	/* the Image-LC display buffer                     */
	GetClientRect( hWnd, &client_rect );
	screen_points[0].x = client_rect.left;
	screen_points[0].y = client_rect.top;
	screen_points[1].x = client_rect.right;
	screen_points[1].y = client_rect.top;
	screen_points[2].x = client_rect.left;
	screen_points[2].y = client_rect.bottom;
	screen_points[3].x = client_rect.right;
	screen_points[3].y = client_rect.bottom;
	ClientToScreen( hWnd, &screen_points[0] );
	ClientToScreen( hWnd, &screen_points[1] );
	ClientToScreen( hWnd, &screen_points[2] );
	ClientToScreen( hWnd, &screen_points[3] );

	/* Draw Image-LC X pattern */
	MgraClear( MilImage, 0L );
	MgraLine( MilImage, screen_points[0].x, screen_points[0].y,
	                    screen_points[3].x, screen_points[3].y );
	MgraLine( MilImage, screen_points[1].x, screen_points[1].y,
	                    screen_points[2].x, screen_points[2].y );

	/* Make sure VGA pattern is redrawn */
	InvalidateRect( hWnd, NULL, TRUE );
	return( 0 );

	case WM_COMMAND:
	switch( wParam )
		{
		case IDM_ABOUT:
		lpProcDialog = MakeProcInstance( (FARPROC)About, hInst );
		DialogBox( hInst, MAKEINTRESOURCE( IDD_ABOUT ), hWnd, lpProcDialog );
		FreeProcInstance( lpProcDialog );
		return( 0 );

		case IDM_INSTRUCTIONS:
		lpProcDialog = MakeProcInstance( (FARPROC)Instructions, hInst );
		DialogBox( hInst, MAKEINTRESOURCE( IDD_INSTRUCTIONS ), hWnd, lpProcDialog );
		FreeProcInstance( lpProcDialog );
		return( 0 );

		case IDM_KEYINGCONTROLHELP:
		lpProcDialog = MakeProcInstance( (FARPROC)KeyingControlHelp, hInst );
		DialogBox( hInst, MAKEINTRESOURCE( IDD_KEYINGCONTROLHELP ), hWnd, lpProcDialog );
		FreeProcInstance( lpProcDialog );
		return( 0 );

		case IDM_KEYINGCONTROL:
		lpProcDialog = MakeProcInstance( (FARPROC)KeyingControl, hInst );
		DialogBox( hInst, MAKEINTRESOURCE( IDD_KEYINGCONTROL ), hWnd, lpProcDialog );
		FreeProcInstance( lpProcDialog );
		return( 0 );
		
		case IDM_EXIT:
		PostMessage( hWnd, WM_CLOSE, 0, 0L );
		return( 0 );

		case IDM_LOADVALFILE:
		filename[0] = '\000';
		ofn.lStructSize       = sizeof( OPENFILENAME );
		ofn.hwndOwner         = hWnd;
		ofn.hInstance         = NULL;
		ofn.lpstrFilter       = "VAL Files (*.VAL)\000*.val\000All Files (*.*)\000*.*\000\000";
		ofn.lpstrCustomFilter = NULL;
		ofn.nMaxCustFilter    = 0;
		ofn.nFilterIndex      = 0;
		ofn.lpstrFile         = filename;
		ofn.nMaxFile          = _MAX_PATH;
		ofn.lpstrFileTitle    = NULL;
		ofn.nMaxFileTitle     = 0;
		ofn.lpstrInitialDir   = NULL;
		ofn.lpstrTitle        = "Load VAL File";
		ofn.Flags             = OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
		ofn.nFileOffset       = 0;
		ofn.nFileExtension    = 0;
		ofn.lpstrDefExt       = "val";
		ofn.lCustData         = 0L;
		ofn.lpfnHook          = NULL;
		ofn.lpTemplateName    = NULL;
		if( GetOpenFileName (&ofn) )
			{
			if( VALfileInput( filename ) == E_OK )
				{
				/* Transfer the video parameters to the board */
				/* through the fifo                           */
				/* Making sure the VGA switch is properly set */
				sav_vga_switch = video.dpytap.im_lc.vga_switch;
				video.dpytap.im_lc.vga_switch = 2;
				sav_vga_comp = video.dpytap.im_lc.vga_comp;
				video.dpytap.im_lc.vga_comp = 0;
				MnatEnterNativeMode( M_NAT_NULL );
				lsdcldvideo( (long *)&video );
				MnatLeaveNativeMode( M_NAT_NULL );

				/* Enable scroll bars */
				nVscrollMax = GetOutVBackPorch() + GetOutVFrontPorch();
				nVscrollPos = GetOutVBackPorch();
				SetScrollRange (hWnd, SB_VERT, 0, nVscrollMax, FALSE) ;
				SetScrollPos   (hWnd, SB_VERT, nVscrollPos, TRUE) ;

				nHscrollMax = GetOutHBackPorch() + GetOutHFrontPorch();
				nHscrollPos = GetOutHBackPorch();
				SetScrollRange (hWnd, SB_HORZ, 0, nHscrollMax, FALSE) ;
				SetScrollPos   (hWnd, SB_HORZ, nHscrollPos, TRUE) ;

				/* Enable 'Save' menu item */
				EnableMenuItem( hmenu, (UINT)IDM_SAVEVALFILE, MF_ENABLED | MF_BYCOMMAND );

				/* Enable VGA keying */
				MnatDispSetVgaKeying( M_VGA_KEY, M_EQUAL, keying_mask, keying_index );
				EnableMenuItem( hmenu, (UINT)IDM_KEYINGCONTROL, MF_ENABLED | MF_BYCOMMAND );
				}
			}
		return( 0 );

		case IDM_SAVEVALFILE:
		filename[0] = '\000';
		ofn.lStructSize       = sizeof( OPENFILENAME );
		ofn.hwndOwner         = hWnd;
		ofn.hInstance         = NULL;
		ofn.lpstrFilter       = "VAL Files (*.VAL)\000*.val\000All Files (*.*)\000*.*\000\000";
		ofn.lpstrCustomFilter = NULL;
		ofn.nMaxCustFilter    = 0;
		ofn.nFilterIndex      = 0;
		ofn.lpstrFile         = filename;
		ofn.nMaxFile          = _MAX_PATH;
		ofn.lpstrFileTitle    = NULL;
		ofn.nMaxFileTitle     = 0;
		ofn.lpstrInitialDir   = NULL;
		ofn.lpstrTitle        = "Save VAL File";
		ofn.Flags             = OFN_NOREADONLYRETURN | OFN_HIDEREADONLY | 
								OFN_OVERWRITEPROMPT | OFN_PATHMUSTEXIST;
		ofn.nFileOffset       = 0;
		ofn.nFileExtension    = 0;
		ofn.lpstrDefExt       = "val";
		ofn.lCustData         = 0L;
		ofn.lpfnHook          = NULL;
		ofn.lpTemplateName    = NULL;
		if( GetSaveFileName (&ofn) )
			{
			/* Restore state of VGA switch */
			/* and then save VAL file      */
			video.dpytap.im_lc.vga_switch = sav_vga_switch;
			video.dpytap.im_lc.vga_comp = sav_vga_comp;

			if( VALfileOutput( filename ) == E_OK )
				{
				modif_parms = FALSE;
				}
			}
		return( 0 );
		}
	return( 0 );

	case WM_VSCROLL:
	switch (wParam)
		{
		case SB_TOP:
		nVscrollInc = -nVscrollPos ;
		break ;

		case SB_BOTTOM:
		nVscrollInc = nVscrollMax - nVscrollPos ;
		break ;

		case SB_LINEUP:
		nVscrollInc = -1 ;
		break ;

		case SB_LINEDOWN:
		nVscrollInc = 1 ;
		break ;

		case SB_PAGEUP:
		nVscrollInc = -8;
		break ;

		case SB_PAGEDOWN:
		nVscrollInc = 8;
		break ;

		case SB_THUMBPOSITION:
		nVscrollInc = LOWORD (lParam) - nVscrollPos ;
		break ;

		default:
		nVscrollInc = 0 ;
		}
	nVscrollInc = max (-nVscrollPos, min (nVscrollInc, nVscrollMax - nVscrollPos)) ;
	if (nVscrollInc != 0)
		{
		nVscrollPos += nVscrollInc ;
		SetScrollPos (hWnd, SB_VERT, nVscrollPos, TRUE) ;
		SetOutVParameters(	GetOutVFrontPorch() - nVscrollInc, 
							GetOutVSyncPulse(), 
							GetOutVBackPorch() + nVscrollInc, 
							GetOutVTotalDisplayed() );
		/* Transfer the video parameters to the board */
		/* through the fifo                           */
		MnatEnterNativeMode( M_NAT_NULL );
		lsdcldvideo( (long *)&video );
		MnatLeaveNativeMode( M_NAT_NULL );
		}
	return( 0 );

	case WM_HSCROLL:
	switch (wParam)
		{
		case SB_LINEUP:
		nHscrollInc = -1 ;
		break;                         

		case SB_LINEDOWN:
		nHscrollInc = 1 ;
		break ;

		case SB_PAGEUP:
		nHscrollInc = -8 ;
		break ;

		case SB_PAGEDOWN:
		nHscrollInc = 8 ;
		break ;

		case SB_THUMBPOSITION:
		nHscrollInc = LOWORD (lParam) - nHscrollPos ;
		break ;

		default:
		nHscrollInc = 0 ;
		}
	nHscrollInc = max (-nHscrollPos, min (nHscrollInc, nHscrollMax - nHscrollPos)) ;
	if (nHscrollInc != 0)
		{
		nHscrollPos += nHscrollInc ;
		SetScrollPos (hWnd, SB_HORZ, nHscrollPos, TRUE) ;
		SetOutHParameters(	GetOutHFrontPorch() - nHscrollInc, 
							GetOutHSyncPulse(), 
							GetOutHBackPorch() + nHscrollInc, 
							GetOutHTotalDisplayed() );
		/* Transfer the video parameters to the board */
		/* through the fifo                           */
		MnatEnterNativeMode( M_NAT_NULL );
		lsdcldvideo( (long *)&video );
		MnatLeaveNativeMode( M_NAT_NULL );
		}
	return( 0 );

	case WM_KEYDOWN:
	switch (wParam)
		{
		case VK_HOME:
		SendMessage (hWnd, WM_VSCROLL, SB_TOP, 0L) ;
		break ;

		case VK_END:
		SendMessage (hWnd, WM_VSCROLL, SB_BOTTOM, 0L) ;
		break ;

		case VK_PRIOR:
		SendMessage (hWnd, WM_VSCROLL, SB_PAGEUP, 0L) ;
		break ;

		case VK_NEXT:
		SendMessage (hWnd, WM_VSCROLL, SB_PAGEDOWN, 0L) ;
		break ;

		case VK_UP:
		SendMessage (hWnd, WM_VSCROLL, SB_LINEUP, 0L) ;
		break ;

		case VK_DOWN:
		SendMessage (hWnd, WM_VSCROLL, SB_LINEDOWN, 0L) ;
		break ;

		case VK_LEFT:
		SendMessage (hWnd, WM_HSCROLL, SB_PAGEUP, 0L) ;
		break ;

		case VK_RIGHT:
		SendMessage (hWnd, WM_HSCROLL, SB_PAGEDOWN, 0L) ;
		break;
		}
	return( 0 );

	case WM_CLOSE:
	if( modif_parms == TRUE )
		{
		switch( MessageBox( NULL,
						"Video parameters have not been\n"
						"saved to a file.  Do you wish\n"
                     	"to save them?",
                     	"WinAlign", 
                      	MB_YESNOCANCEL | MB_ICONEXCLAMATION | MB_TASKMODAL ) )
			{
			case IDYES:
			PostMessage( hWnd, WM_COMMAND, (WPARAM)IDM_SAVEVALFILE, 0L );
			return( 0 );
			
			case IDNO:
			break;
			
			case IDCANCEL:
			return( 0 );
			}
		}
	DestroyWindow( hWnd );
	return( 0 );
	
	case WM_DESTROY:
	ReleaseMilSystem();
	PostQuitMessage( 0 );
	return( 0 );

	case WM_ENDSESSION:
	if( wParam == TRUE )
		{
		ReleaseMilSystem();
		}
	return( 0 );
	}
return( DefWindowProc( hWnd, message, wParam, lParam ) );
}

/*---------------------------------------------------------------------------*/
/*
*	Function:	About()
*
*	By:			Michel Pelletier
*				Matrox Electronic Systems ltd.
*
*	Date:		October 22, 1993.
*
*	Synopsys:	Processes messages for "About" dialog box
*/
/*---------------------------------------------------------------------------*/

BOOL FAR PASCAL _export About(HWND hDlg, UINT message, UINT wParam, LONG lParam)
{
switch( message )
	{
	case WM_INITDIALOG:
	return( TRUE );

	case WM_COMMAND:
	if( wParam == IDOK )
		{
		EndDialog( hDlg, TRUE );
		return( TRUE );
		}
	break;
	}
return( FALSE );
}

/*---------------------------------------------------------------------------*/
/*
*	Function:	Instructions()
*
*	By:			Michel Pelletier
*				Matrox Electronic Systems ltd.
*
*	Date:		October 22, 1993.
*
*	Synopsys:	Processes messages for "Instructions" dialog box
*/
/*---------------------------------------------------------------------------*/

BOOL FAR PASCAL _export Instructions(HWND hDlg, UINT message, UINT wParam, LONG lParam)
{
switch( message )
	{
	case WM_INITDIALOG:
	return( TRUE );

	case WM_COMMAND:
	if( wParam == IDOK )
		{
		EndDialog( hDlg, TRUE );
		return( TRUE );
		}
	break;
	}
return( FALSE );
}

/*---------------------------------------------------------------------------*/
/*
*	Function:	KeyingControlHelp()
*
*	By:			Michel Pelletier
*				Matrox Electronic Systems ltd.
*
*	Date:		November 3, 1993.
*
*	Synopsys:	Processes messages for "KeyingControlHelp" dialog box
*/
/*---------------------------------------------------------------------------*/

BOOL FAR PASCAL _export KeyingControlHelp(HWND hDlg, UINT message, UINT wParam, LONG lParam)
{
switch( message )
	{
	case WM_INITDIALOG:
	return( TRUE );

	case WM_COMMAND:
	if( wParam == IDOK )
		{
		EndDialog( hDlg, TRUE );
		return( TRUE );
		}
	break;
	}
return( FALSE );
}

/*---------------------------------------------------------------------------*/
/*
*	Function:	KeyingControl()
*
*	By:			Michel Pelletier
*				Matrox Electronic Systems ltd.
*
*	Date:		November 3, 1993.
*
*	Synopsys:	Processes messages for "KeyingControl" dialog box
*/
/*---------------------------------------------------------------------------*/

BOOL FAR PASCAL _export KeyingControl(HWND hDlg, UINT message, UINT wParam, LONG lParam)
{
static unsigned char	sav_k_index, sav_k_mask;
static COLORREF			sav_d_color;

switch( message )
	{
	case WM_INITDIALOG:
	sav_k_index = keying_index;
	sav_k_mask = keying_mask;
	sav_d_color = drawing_color;
	sprintf( text_buffer, "%.2X", keying_index );
	SetDlgItemText( hDlg, IDC_KEYINGINDEXVALUE, text_buffer );
	sprintf( text_buffer, "%.2X", keying_mask );
	SetDlgItemText( hDlg, IDC_KEYINGMASKVALUE,  text_buffer );
	sprintf( text_buffer, "%.6lX", drawing_color );
	SetDlgItemText( hDlg, IDC_DRAWINGCOLORVALUE, text_buffer );
	return( TRUE );

	case WM_COMMAND:
	switch( wParam )
		{
		case IDCANCEL:
		keying_mask = sav_k_mask;
		keying_index = sav_k_index;
		drawing_color = sav_d_color;
		MnatDispSetVgaKeying( M_VGA_KEY, M_EQUAL, keying_mask, keying_index );
		InvalidateRect( GetParent( hDlg ), NULL, TRUE );
		UpdateWindow( GetParent( hDlg ) );
		EndDialog( hDlg, TRUE );
		return( TRUE );

		case IDOK:
		EndDialog( hDlg, TRUE );
		return( TRUE );

		case IDC_KEYINGINDEXVALUE:
		if( HIWORD( lParam ) == EN_CHANGE )
			{
			GetDlgItemText( hDlg, IDC_KEYINGINDEXVALUE, text_buffer, 4 );
			keying_index = (unsigned char)(strtoul( text_buffer, NULL, 16 ) & 0xff);
			MnatDispSetVgaKeying( M_VGA_KEY, M_EQUAL, keying_mask, keying_index );
			}
		return( TRUE );

		case IDC_KEYINGMASKVALUE:
		if( HIWORD( lParam ) == EN_CHANGE )
			{
			GetDlgItemText( hDlg, IDC_KEYINGMASKVALUE, text_buffer, 4 );
			keying_mask = (unsigned char)(strtoul( text_buffer, NULL, 16 ) & 0xff);
			MnatDispSetVgaKeying( M_VGA_KEY, M_EQUAL, keying_mask, keying_index );
			}
		return( TRUE );

		case IDC_DRAWINGCOLORVALUE:
		if( HIWORD( lParam ) == EN_CHANGE )
			{
			GetDlgItemText( hDlg, IDC_DRAWINGCOLORVALUE, text_buffer, 10 );
			drawing_color = (COLORREF)(strtoul( text_buffer, NULL, 16 ) & 0x00ffffffL);
			InvalidateRect( GetParent( hDlg ), NULL, TRUE );
			UpdateWindow( GetParent( hDlg ) );
			}
		return( TRUE );
		}
	break;
	}
return( FALSE );
}

/*---------------------------------------------------------------------------*/
/*
*	Function:	InitMilSystem()
*
*	By:			Michel Pelletier
*				Matrox Electronic Systems ltd.
*
*	Date:		October 22, 1993.
*
*	Synopsys:	This function is for initializing the MIL library.
*				It also performs allocation of general purpose MIL
*				objects and initializes appropriate variables.
*/
/*---------------------------------------------------------------------------*/

static int	InitMilSystem( void )
{
/* Initialize the MIL library and allocate objects */
MsysInit( M_DEFAULT );
MdevAlloc( M_IMAGE_SYSTEM, M_DEFAULT, M_DEFAULT, M_WINDOWS, &MilSystem );
MbufAlloc2d( GetSystemMetrics( SM_CXSCREEN ), GetSystemMetrics( SM_CYSCREEN ),
             8 + M_UNSIGNED, M_IMAGE + M_GRAPH + M_DISP, &MilImage );
MdispSelect( MilImage );

/* Check for errors */
if( MsysGetError( M_GLOBAL, M_NULL ) != M_NULL_ERROR )
	{
	return( E_MIL_INIT_ERROR );
	}
return( E_OK );
}

/*---------------------------------------------------------------------------*/
/*
*	Function:	ReleaseMilSystem()
*
*	By:			Michel Pelletier
*				Matrox Electronic Systems ltd.
*
*	Date:		October 22, 1993.
*
*	Synopsys:	This function is for releasing the MIL library 
*				resources.  It should be called once, prior to
*				leaving this program.
*/
/*---------------------------------------------------------------------------*/

static void	ReleaseMilSystem( void )
{
MdispDeselect( MilImage );
MbufFree( MilImage );
MdevFree( MilSystem );
}

/*---------------------------------------------------------------------------*/
/*
*	Function:	VALfileInput()
*
*	By:			Michel Pelletier
*				Matrox Electronic Systems ltd.
*
*	Date:		October 22, 1993.
*
*	Synopsys:	This function is for reading a VAL file from disk.
*/
/*---------------------------------------------------------------------------*/

static int VALfileInput( char *filename_str )
{
FILE 	*val_file;
int 	i, found_it;
char 	buffer[MAX_SIZE_OF_A_LINE + 1];
char 	name[20];
char 	**ref_name;
int  	*ref_len;
long 	value;
int  	flags[34];
void 	*vid_ptr;

for( i=0; i<34; i++ )		/* Initialise flags to value FALSE. */
	{
	flags[i] = FALSE;
	}

/* Open the VAL file */
if( (val_file = fopen( filename_str, "rb" )) == NULL )
	{
	MessageBox( NULL,
		"Unable to open VAL file.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_NOT_COMPLETED );
	}

/* Read the .VAL file */
while( fgets( buffer, MAX_SIZE_OF_A_LINE, val_file ) != NULL )
	{
	name[0] = '\0';

	/* If line is empty, or does not contain both 'name' and 'value'     */
    /* fields, skip it */
	if( sscanf( buffer, "%15s = 0x%lx\n", name, &value ) != 2 )
		continue;

	/* Comments start with a ";" in first column, next line if the case */
	if( name[0] == ';' )
		continue;

	/* Identify variable */
	vid_ptr = (void *)&video;
	ref_name = sysdef_name;
	ref_len = sysdef_len;
	found_it = FALSE;

	for( i=0; (i<34) && (found_it == FALSE); i++ )
		{
		if( i == 17 )
			{
			ref_name = vidparm_name;
			ref_len = vidparm_len;
			}
		if( strnicmp( name, ref_name[i%17], strlen( ref_name[i%17] ) ) == 0 )
			{
			flags[i] = TRUE;
			found_it = TRUE;
			}
		switch( ref_len[i%17] )
			{
			case sizeof( DWORD ):
			if( found_it == TRUE )
				{
				*((DWORD *)vid_ptr) = *((DWORD *)&value);
				}
			else
				{
				((DWORD *)vid_ptr)++;
				}
			break;

			case sizeof( WORD ):
			if( found_it == TRUE )
				{
				*((WORD *)vid_ptr) = *((WORD *)&value);
				}
			else
				{
				((WORD *)vid_ptr)++;
				}
			break;

			case sizeof( BYTE ):
			if( found_it == TRUE )
				{
				*((BYTE *)vid_ptr) = *((BYTE *)&value);
				}
			else
				{
				((BYTE *)vid_ptr)++;
				}
			break;
			}
		}
	}

if( feof( val_file ) == 0 )	/* Check for errors */
	{
	fclose( val_file );
	MessageBox( NULL,
		"Error reading VAL file.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_NOT_COMPLETED );
	}

for( i=0; i<34; i++ )		/* Make sure all 34 values have been read */
	{
	if( flags[i] != TRUE )
		{
		fclose( val_file );
		MessageBox( NULL,
			"VAL file is improper format.\n"
			"Bad or missing parameter(s).\n",
			"WinAlign - Error", 
			MB_OK | MB_ICONSTOP | MB_TASKMODAL );
		return( E_FILE_IO_IMPROPER_FORMAT );
		}
	}

if( fclose( val_file ) != 0 )
	{
	MessageBox( NULL,
		"Unable to close VAL file.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_NOT_COMPLETED );
	}

/* Do some validation */
switch( video.hosttype )
	{
	case 0x00:
	case 0x01:
	case 0x02:
	case 0x03:
	case 0xff:
	break;

	default:
	MessageBox( NULL,
		"VAL file is improper format.\n"
		"Incompatible \"hosttype\" parameter.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_IMPROPER_FORMAT );
	break;
	}
switch( video.boardtype )
	{
	case 0x03:
	case 0xff:
	break;

	default:
	MessageBox( NULL,
		"VAL file is improper format.\n"
		"Incompatible \"boardtype\" parameter.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_IMPROPER_FORMAT );
	}
switch( video.dpytap.im_lc.clk_div )
	{
	case 0x00:
	case 0x01:
	case 0x02:
	break;

	default:
	MessageBox( NULL,
		"VAL file is improper format.\n"
		"Incompatible \"dpytap\" parameter.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_IMPROPER_FORMAT );
	}
switch( video.dpytap.im_lc.vga_comp )
	{
	case 0x00:
	case 0x01:
	break;

	default:
	MessageBox( NULL,
		"VAL file is improper format.\n"
		"Incompatible \"dpytap\" parameter.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_IMPROPER_FORMAT );
	}
if( !(video.sysctl.imlc.edclk) )
	{
	MessageBox( NULL,
		"VAL file is improper format.\n"
		"Incompatible \"sysctl\" parameter.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_IMPROPER_FORMAT );
	}
if( !(video.sysctl.imlc.esync) )
	{
	MessageBox( NULL,
		"VAL file is improper format.\n"
		"Incompatible \"sysctl\" parameter.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_IMPROPER_FORMAT );
	}
if( video.scrw != ((WORD)GetSystemMetrics( SM_CXSCREEN )) )
	{
	MessageBox( NULL,
		"VAL file is improper format.\n"
		"Incompatible \"scrw\" parameter.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_IMPROPER_FORMAT );
	}
if( video.scrh != ((WORD)GetSystemMetrics( SM_CYSCREEN )) )
	{
	MessageBox( NULL,
		"VAL file is improper format.\n"
		"Incompatible \"scrh\" parameter.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_IMPROPER_FORMAT );
	}
return( E_OK );
}

/*---------------------------------------------------------------------------*/
/*
*	Function:	VALfileOutput()
*
*	By:			Michel Pelletier
*				Matrox Electronic Systems ltd.
*
*	Date:		October 22, 1993.
*
*	Synopsys:	This function is for writing a VAL file to disk.
*/
/*---------------------------------------------------------------------------*/

static int VALfileOutput( char *filename_str )
{
FILE *val_file;
char drive[_MAX_DRIVE];
char dir[_MAX_DIR];
char fname[_MAX_FNAME];
char ext[_MAX_EXT];
char *board;
char *scan_mode;
char *time_string;
char *syncloc;
register i;
void *vid_ptr;
char **name;
char **expl;
int  *len;
int factor;
time_t the_time;

if( (val_file = fopen( filename_str, "wb" )) == NULL )
	{
	MessageBox( NULL,
		"Unable to open VAL file.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_NOT_COMPLETED );
	}

_splitpath( filename_str, drive, dir, fname, ext );

board = "IMAGE-LC";

switch( GetOutScanMode() )
	{
	case O_INTERLACED:
	scan_mode = "interlaced";
	factor = 2;
	break;

	case O_NON_INTERLACED:
	scan_mode = "non-interlaced";
	factor = 1;
	break;
	}

switch( video.dpytap.im_lc.sync_format )
	{
	case 0:
	case 1:
	syncloc = "Composite Sync On Green";
	break;

	case 2:
	syncloc = "No Composite Sync On Red, Green or Blue";
	break;

	case 3:
	syncloc = "Composite Sync On Red, Green And Blue";
	break;
	}

time( &the_time );				/* Get the current date and  */
time_string = ctime( &the_time );		/* time.                     */

fprintf( val_file, ";\r\n" );
fprintf( val_file, ";  %s%s\r\n", fname, ext );
fprintf( val_file, ";\r\n" );
fprintf( val_file, ";  %s\r\n", ver_string );
fprintf( val_file, ";  Copyright (C) 1993.  Matrox Electronic Systems Ltd.\r\n" );
fprintf( val_file, ";\r\n" );
fprintf( val_file, ";  Date:   %.11s%.4s\r\n", time_string, &time_string[20] );
fprintf( val_file, ";  Time:   %.8s\r\n", &time_string[11] );
fprintf( val_file, ";\r\n" );
fprintf( val_file, ";  %s configuration file for %d x %d %s display.\r\n",
          board, GetOutHTotalDisplayed() * GetGspRatio(), 
	  GetOutVTotalDisplayed(), scan_mode );
fprintf( val_file, ";\r\n" );
fprintf( val_file, ";     - Display size: %d x %d %s\r\n", 
          GetOutHTotalDisplayed() * GetGspRatio(), 
	  GetOutVTotalDisplayed(), scan_mode );
fprintf( val_file, ";\r\n" );
fprintf( val_file, ";       Width of horizontal SYNC pulse .......... %4d clock cycles.\r\n",
	GetOutHSyncPulse() );
fprintf( val_file, ";       Length of horizontal front porch ........ %4d clock cycles.\r\n",
	GetOutHFrontPorch() );
fprintf( val_file, ";       Length of horizontal back porch ......... %4d clock cycles.\r\n",
	GetOutHBackPorch() );
fprintf( val_file, ";\r\n" );
fprintf( val_file, ";       Width of vertical SYNC pulse ........... %5.1lf lines.\r\n",
	((double)GetOutVSyncPulse()) / ((double)factor) );
fprintf( val_file, ";       Length of vertical front porch ......... %5.1lf lines.\r\n",
	((double)GetOutVFrontPorch()) / ((double)factor) );
fprintf( val_file, ";       Length of vertical back porch .......... %5.1lf lines.\r\n",
	((double)GetOutVBackPorch()) / ((double)factor) );
fprintf( val_file, ";\r\n" );
fprintf( val_file, ";     - SYNC output:\r\n" );
fprintf( val_file, ";       %s.\r\n", syncloc );
fprintf( val_file, ";\r\n" );
fprintf( val_file, ";\r\n" );

vid_ptr = (void *)&video;
name = sysdef_name;
expl = sysdef_expl;
len = sysdef_len;

for( i=0; i<34; i++ )
	{
	if( i == 17 )
		{
		name = vidparm_name;
		expl = vidparm_expl;
		len = vidparm_len;
		}
	fprintf( val_file, "%-15s = ", name[i%17] );
	switch( len[i%17] )
		{
		case sizeof( DWORD ):
		fprintf( val_file, "0x%.8lX      ;", *((DWORD *)vid_ptr) );
		((DWORD *)vid_ptr)++;
		break;

		case sizeof( WORD ):
		fprintf( val_file, "0x%.4X          ;", *((WORD *)vid_ptr) );
		((WORD *)vid_ptr)++;
		break;

		case sizeof( BYTE ):
		fprintf( val_file, "0x%.2X            ;", *((BYTE *)vid_ptr) );
		((BYTE *)vid_ptr)++;
		break;
		}
	fprintf( val_file, " %s\r\n", expl[i%17] );
	}

if( fclose( val_file ) != 0 )
	{
	MessageBox( NULL,
		"Unable to close VAL file.\n",
		"WinAlign - Error", 
		MB_OK | MB_ICONSTOP | MB_TASKMODAL );
	return( E_FILE_IO_NOT_COMPLETED );
	}

return( E_OK );
}

/*****************************************************************************/

static WORD GetOutHTotalDisplayed( void )
{
WORD	tmp_hsblnk, tmp_heblnk;

tmp_hsblnk = video.hsblnk + 1;
tmp_heblnk = video.heblnk + 1;

switch( GetGspRatio() )
	{
	case 1:
	tmp_heblnk = tmp_heblnk + 4;
	break;

	case 2:
	tmp_heblnk = tmp_heblnk + 2;
	break;

	case 4:
	tmp_heblnk = tmp_heblnk + 1;
	break;
	}

return( tmp_hsblnk - tmp_heblnk );
}

/*****************************************************************************/

static WORD GetOutHFrontPorch( void )
{
WORD	tmp_htotal, tmp_hsblnk;

tmp_htotal = video.htotal + 1;
tmp_hsblnk = video.hsblnk + 1;

return( tmp_htotal - tmp_hsblnk );
}

/*****************************************************************************/

static WORD GetOutHSyncPulse( void )
{
WORD	tmp_hesync;

tmp_hesync = video.hesync + 1;

return( tmp_hesync );
}

/*****************************************************************************/

static WORD GetOutHBackPorch( void )
{
WORD	tmp_heblnk, tmp_hesync;

tmp_heblnk = video.heblnk + 1;
tmp_hesync = video.hesync + 1;

switch( GetGspRatio() )
	{
	case 1:
	tmp_heblnk = tmp_heblnk + 4;
	break;

	case 2:
	tmp_heblnk = tmp_heblnk + 2;
	break;

	case 4:
	tmp_heblnk = tmp_heblnk + 1;
	break;
		}
return( tmp_heblnk - tmp_hesync );
}

/*****************************************************************************/

void SetOutHParameters( WORD ohfp, WORD ohsync, WORD ohbp, WORD ohdisp )
{
WORD	tmp_htotal, tmp_hsblnk, tmp_heblnk, tmp_hesync;

tmp_htotal = ohfp + ohsync + ohbp + ohdisp - 1;
tmp_hsblnk = ohsync + ohbp + ohdisp - 1;
tmp_heblnk = ohsync + ohbp - 1;
tmp_hesync = ohsync - 1;

switch( GetGspRatio() )
	{
	case 1:
	tmp_heblnk = tmp_heblnk - 4;
	break;

	case 2:
	tmp_heblnk = tmp_heblnk - 2;
	break;

	case 4:
	tmp_heblnk = tmp_heblnk - 1;
	break;
	}
video.scrw = ohdisp * GetGspRatio();
video.htotal = tmp_htotal;
video.hsblnk = tmp_hsblnk;
video.heblnk = tmp_heblnk;
video.hesync = tmp_hesync;

modif_parms = TRUE;
}

/*****************************************************************************/

static WORD GetOutVTotalDisplayed( void )
{
if( GetOutScanMode() == O_INTERLACED )
	{
	return( (video.vsblnk - video.veblnk - 1) * 2 ); 
	}
else
	{
	return( video.vsblnk - video.veblnk );
	}
}

/*****************************************************************************/

static WORD GetOutVFrontPorch( void )
{
return( video.vtotal - video.vsblnk );
}

/*****************************************************************************/

static WORD GetOutVSyncPulse( void )
{
return( (video.vesync + 1) / 2 );
}

/*****************************************************************************/

static WORD GetOutVBackPorch( void )
{
if( GetOutScanMode() == O_INTERLACED )
	{
	return( (2 * video.veblnk) - (3 * GetOutVSyncPulse()) + 2 );
	}
else
	{
	return( video.veblnk - GetOutVSyncPulse() + 1 );
	}
}

/*****************************************************************************/

void SetOutVParameters( WORD ovfp, WORD ovsync, WORD ovbp, WORD ovdisp )
{
if( GetOutScanMode() == O_INTERLACED )
	{
	video.scrh =   (WORD)ovdisp;
	video.vesync = (WORD)((ovsync * 2) - 1);
	video.veblnk = (WORD)((((double)ovsync) * ((double)3)
			/ ((double)2)) + (((double)ovbp) / 
			((double)2)) - ((double)1));
	/* Watch the order of the two following lines.  They must not be */
	/* inverted because the second line uses the value obtained by   */
	/* the first line                                                */
	video.vtotal = (WORD)((((double)ovbp) / ((double)2)) +
			(((double)ovdisp) / ((double)2)) +
			(((double)ovsync) * ((double)3) /
			((double)2)) + ((double)ovfp));
	video.vsblnk = (WORD)(video.vtotal - ovfp);
	}
else
	{
	video.scrh = (WORD)ovdisp;
	video.vesync = (WORD)((ovsync * 2) - 1);
	video.veblnk = (WORD)(ovsync + ovbp - 1);
	video.vtotal = (WORD)(ovsync + ovbp + ovdisp + ovfp - 1);
	video.vsblnk = (WORD)(ovsync + ovbp + ovdisp - 1);
	}
modif_parms = TRUE;
}

/*****************************************************************************/

static BYTE GetOutScanMode( void )
{
if( video.dpytap.im_lc.interlaced )
	{
	return( O_INTERLACED );
	}
else
	{
	return( O_NON_INTERLACED );
	}
}

/*****************************************************************************/

static BYTE GetGspRatio( void )
{
switch( video.dpytap.im_lc.clk_div )
	{
	case 0x00:
	return( 1 );

	case 0x01:
	return( 2 );

	case 0x02:
	return( 4 );
	}
}
