package org.apache.log4j.appender;

import java.io.BufferedInputStream;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;

import junit.framework.Assert;

final class FileTestHelper {

  private final String testDir = System.getProperty("user.dir")
      + System.getProperty("file.separator") + "test"
      + System.getProperty("file.separator") + "composite_roll";

  FileTestHelper setUp() {
    // clear out the contents of the test directory
    File testDir = new File(this.testDir);
    if (testDir.exists()) {
      this.tearDown();
    } else {
      Assert.assertTrue("Cannot create directory " + testDir.getAbsolutePath()
          + " used by " + this.getClass().getName(), testDir.mkdirs());
    }
    return this;
  }

  /**
   * @return 258 character String
   */
  String generateBlah() {
    StringBuffer buffer = new StringBuffer();
    for (int i = 0; i < 64; i++) {
      buffer.append("blah");
    }
    buffer.append(System.getProperty("line.separator"));
    return buffer.toString();
  }

  InputStream readFile(final File file) {
    Assert.assertTrue("File " + file + " does not exist", file.exists());
    InputStream is = null;
    ByteArrayOutputStream baos = null;
    try {
      is = new BufferedInputStream(new FileInputStream(file));
    } catch (FileNotFoundException e) {
      e.printStackTrace();
      Assert.fail(e.getMessage());
    }
    baos = new ByteArrayOutputStream();
    try {
      while (is.available() > 0) {
        int data = is.read();
        if (data == -1) {
          break;
        }
        baos.write(data);
      }
    } catch (IOException e) {
      e.printStackTrace();
      Assert.fail(e.getMessage());
    } finally {
      if (is != null) {
        try {
          is.close();
        } catch (IOException e) {
          // ignore
        }
      }
    }
    return new ByteArrayInputStream(baos.toByteArray());
  }

  FileTestHelper writeFile(final String filename) {
    this.writeFile(filename, 1);
    return this;
  }

  FileTestHelper writeFile(final String filename, final int numRandomStrings) {
    String fullName = this.testDir + System.getProperty("file.separator")
        + filename;
    File file = new File(fullName);
    if (!file.exists()) {
      new File(this.testDir).mkdirs();
    }
    try {
      Writer writer = new BufferedWriter(new FileWriter(file));
      for (int i = 0; i < numRandomStrings; i++) {
        writer.write("Generated by test suite");
        writer.write(System.getProperty("line.separator"));
        writer.write(this.generateBlah());
      }
      writer.flush();
      writer.close();
    } catch (IOException e) {
      Assert.fail(e.getMessage());
    }
    return this;
  }

  FileTestHelper tearDown() {
    // clear out the contents of the test directory
    File testDir = new File(this.testDir);
    if (testDir.exists()) {
      // empty the directory
      File[] logFiles = testDir.listFiles();
      for (int i = logFiles.length; i-- > 0;) {
        logFiles[i].delete(); // if a file couldn't be deleted, leave it behind
      }
      // try to delete the directories
      testDir.delete();
    }
    return this;
  }

  String getTestDir() {
    return testDir;
  }

  String filenameForTest(String filename) {
    return this.testDir + System.getProperty("file.separator") + filename;
  }

  File fileForTest(String filename) {
    return new File(this.testDir + System.getProperty("file.separator")
        + filename);
  }
}
