/* 
 * Copyright 1999,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.log4j.appender;

import java.io.File;
import java.io.FilenameFilter;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.apache.log4j.helpers.FileHelper;

/**
 * Responsible for listing, sorting, and filtering log file lists.
 * 
 * @author <a href="mailto:simon_park_mail AT yahoo DOT co DOT uk">Simon Park</a>
 * @version 1.12
 */
final class LogFileList extends AbstractList implements List {

  private final File logFile;

  private final List fileList;

  private final FilenameFilter filenameFilter;

  /**
   * 
   */
  LogFileList(final File file, final FilenameFilter filenameFilter) {
    super();
    this.logFile = file;
    this.fileList = new ArrayList();
    this.filenameFilter = filenameFilter;
    this.init();
  }

  /*
   * (non-Javadoc)
   * 
   * @see java.util.AbstractList#get(int)
   */
  public final Object get(int index) {
    return this.fileList.get(index);
  }

  /*
   * (non-Javadoc)
   * 
   * @see java.util.AbstractList#remove(int)
   */
  public final Object remove(int index) {
    return this.fileList.remove(index);
  }

  /*
   * (non-Javadoc)
   * 
   * @see java.util.AbstractCollection#size()
   */
  public final int size() {
    return this.fileList.size();
  }

  /**
   * @return The first {@link File} in the list, or null if the list is empty.
   */
  final File firstFile() {
    return (this.size() > 0) ? ((File) this.get(0)) : null;
  }

  /**
   * @return The last {@link File} in the list, or null if the list is empty.
   */
  final File lastFile() {
    return (this.size() > 0) ? ((File) this.get(this.size() - 1)) : null;
  }

  private void init() {
    File dir = null;
    if (this.logFile.isDirectory()) {
      dir = this.logFile;
    } else {
      dir = new FileHelper().parentDirOf(this.logFile);
    }
    if (dir != null) {
      this.readFileListFrom(dir);
      this.sortFileList();

    }
  }

  private void readFileListFrom(final File dir) {
    if (dir.exists() && dir.canRead()) {
      final File[] files = (this.filenameFilter != null) ? dir
          .listFiles(this.filenameFilter) : dir.listFiles();
      if (files != null) {
        this.fileList.addAll(Arrays.asList(files));
      }
    }
  }

  /**
   * Sort by file modification date.
   */
  private void sortFileList() {
    if (this.size() > 1) {
      Collections.sort(this.fileList, new Comparator() {

        public final int compare(final Object o1, final Object o2) {
          final File f1 = (File) o1;
          final File f2 = (File) o2;
          if (f1.lastModified() < f2.lastModified()) {
            return -1;
          }
          if (f1.lastModified() > f2.lastModified()) {
            return 1;
          }
          return 0;
        }
      });
    }
  }
}
