/*******************************************************************************
* 
* "This software program is licensed subject to the GNU General Public License 
* (GPL). Version 2, June 1991, available at 
* <http://www.fsf.org/copyleft/gpl.html>"
* 
* GNU General Public License 
* 
* Version 2, June 1991
* 
* Copyright (C) 1989, 1991 Free Software Foundation, Inc.  
* 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
* 
* Everyone is permitted to copy and distribute verbatim copies of this license
* document, but changing it is not allowed.
* 
* Preamble
* 
* The licenses for most software are designed to take away your freedom to 
* share and change it. By contrast, the GNU General Public License is intended
* to guarantee your freedom to share and change free software--to make sure 
* the software is free for all its users. This General Public License applies 
* to most of the Free Software Foundation's software and to any other program 
* whose authors commit to using it. (Some other Free Software Foundation 
* software is covered by the GNU Library General Public License instead.) You 
* can apply it to your programs, too.
* 
* When we speak of free software, we are referring to freedom, not price. Our
* General Public Licenses are designed to make sure that you have the freedom 
* to distribute copies of free software (and charge for this service if you 
* wish), that you receive source code or can get it if you want it, that you 
* can change the software or use pieces of it in new free programs; and that 
* you know you can do these things.
* 
* To protect your rights, we need to make restrictions that forbid anyone to 
* deny you these rights or to ask you to surrender the rights. These 
* restrictions translate to certain responsibilities for you if you distribute
* copies of the software, or if you modify it.
* 
* For example, if you distribute copies of such a program, whether gratis or 
* for a fee, you must give the recipients all the rights that you have. You 
* must make sure that they, too, receive or can get the source code. And you 
* must show them these terms so they know their rights.
*  
* We protect your rights with two steps: (1) copyright the software, and (2) 
* offer you this license which gives you legal permission to copy, distribute 
* and/or modify the software. 
* 
* Also, for each author's protection and ours, we want to make certain that 
* everyone understands that there is no warranty for this free software. If 
* the software is modified by someone else and passed on, we want its 
* recipients to know that what they have is not the original, so that any 
* problems introduced by others will not reflect on the original authors' 
* reputations. 
* 
* Finally, any free program is threatened constantly by software patents. We 
* wish to avoid the danger that redistributors of a free program will 
* individually obtain patent licenses, in effect making the program 
* proprietary. To prevent this, we have made it clear that any patent must be 
* licensed for everyone's free use or not licensed at all. 
* 
* The precise terms and conditions for copying, distribution and modification 
* follow. 
* 
* TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION
* 
* 0. This License applies to any program or other work which contains a notice
*    placed by the copyright holder saying it may be distributed under the 
*    terms of this General Public License. The "Program", below, refers to any
*    such program or work, and a "work based on the Program" means either the 
*    Program or any derivative work under copyright law: that is to say, a 
*    work containing the Program or a portion of it, either verbatim or with 
*    modifications and/or translated into another language. (Hereinafter, 
*    translation is included without limitation in the term "modification".) 
*    Each licensee is addressed as "you". 
* 
*    Activities other than copying, distribution and modification are not 
*    covered by this License; they are outside its scope. The act of running 
*    the Program is not restricted, and the output from the Program is covered 
*    only if its contents constitute a work based on the Program (independent 
*    of having been made by running the Program). Whether that is true depends
*    on what the Program does. 
* 
* 1. You may copy and distribute verbatim copies of the Program's source code 
*    as you receive it, in any medium, provided that you conspicuously and 
*    appropriately publish on each copy an appropriate copyright notice and 
*    disclaimer of warranty; keep intact all the notices that refer to this 
*    License and to the absence of any warranty; and give any other recipients 
*    of the Program a copy of this License along with the Program. 
* 
*    You may charge a fee for the physical act of transferring a copy, and you 
*    may at your option offer warranty protection in exchange for a fee. 
* 
* 2. You may modify your copy or copies of the Program or any portion of it, 
*    thus forming a work based on the Program, and copy and distribute such 
*    modifications or work under the terms of Section 1 above, provided that 
*    you also meet all of these conditions: 
* 
*    * a) You must cause the modified files to carry prominent notices stating 
*         that you changed the files and the date of any change. 
* 
*    * b) You must cause any work that you distribute or publish, that in 
*         whole or in part contains or is derived from the Program or any part 
*         thereof, to be licensed as a whole at no charge to all third parties
*         under the terms of this License. 
* 
*    * c) If the modified program normally reads commands interactively when 
*         run, you must cause it, when started running for such interactive 
*         use in the most ordinary way, to print or display an announcement 
*         including an appropriate copyright notice and a notice that there is
*         no warranty (or else, saying that you provide a warranty) and that 
*         users may redistribute the program under these conditions, and 
*         telling the user how to view a copy of this License. (Exception: if 
*         the Program itself is interactive but does not normally print such 
*         an announcement, your work based on the Program is not required to 
*         print an announcement.) 
* 
*    These requirements apply to the modified work as a whole. If identifiable 
*    sections of that work are not derived from the Program, and can be 
*    reasonably considered independent and separate works in themselves, then 
*    this License, and its terms, do not apply to those sections when you 
*    distribute them as separate works. But when you distribute the same 
*    sections as part of a whole which is a work based on the Program, the 
*    distribution of the whole must be on the terms of this License, whose 
*    permissions for other licensees extend to the entire whole, and thus to 
*    each and every part regardless of who wrote it. 
* 
*    Thus, it is not the intent of this section to claim rights or contest 
*    your rights to work written entirely by you; rather, the intent is to 
*    exercise the right to control the distribution of derivative or 
*    collective works based on the Program. 
* 
*    In addition, mere aggregation of another work not based on the Program 
*    with the Program (or with a work based on the Program) on a volume of a 
*    storage or distribution medium does not bring the other work under the 
*    scope of this License. 
* 
* 3. You may copy and distribute the Program (or a work based on it, under 
*    Section 2) in object code or executable form under the terms of Sections 
*    1 and 2 above provided that you also do one of the following: 
* 
*    * a) Accompany it with the complete corresponding machine-readable source 
*         code, which must be distributed under the terms of Sections 1 and 2 
*         above on a medium customarily used for software interchange; or, 
* 
*    * b) Accompany it with a written offer, valid for at least three years, 
*         to give any third party, for a charge no more than your cost of 
*         physically performing source distribution, a complete machine-
*         readable copy of the corresponding source code, to be distributed 
*         under the terms of Sections 1 and 2 above on a medium customarily 
*         used for software interchange; or, 
* 
*    * c) Accompany it with the information you received as to the offer to 
*         distribute corresponding source code. (This alternative is allowed 
*         only for noncommercial distribution and only if you received the 
*         program in object code or executable form with such an offer, in 
*         accord with Subsection b above.) 
* 
*    The source code for a work means the preferred form of the work for 
*    making modifications to it. For an executable work, complete source code 
*    means all the source code for all modules it contains, plus any 
*    associated interface definition files, plus the scripts used to control 
*    compilation and installation of the executable. However, as a special 
*    exception, the source code distributed need not include anything that is 
*    normally distributed (in either source or binary form) with the major 
*    components (compiler, kernel, and so on) of the operating system on which
*    the executable runs, unless that component itself accompanies the 
*    executable. 
* 
*    If distribution of executable or object code is made by offering access 
*    to copy from a designated place, then offering equivalent access to copy 
*    the source code from the same place counts as distribution of the source 
*    code, even though third parties are not compelled to copy the source 
*    along with the object code. 
* 
* 4. You may not copy, modify, sublicense, or distribute the Program except as
*    expressly provided under this License. Any attempt otherwise to copy, 
*    modify, sublicense or distribute the Program is void, and will 
*    automatically terminate your rights under this License. However, parties 
*    who have received copies, or rights, from you under this License will not
*    have their licenses terminated so long as such parties remain in full 
*    compliance. 
* 
* 5. You are not required to accept this License, since you have not signed 
*    it. However, nothing else grants you permission to modify or distribute 
*    the Program or its derivative works. These actions are prohibited by law 
*    if you do not accept this License. Therefore, by modifying or 
*    distributing the Program (or any work based on the Program), you 
*    indicate your acceptance of this License to do so, and all its terms and
*    conditions for copying, distributing or modifying the Program or works 
*    based on it. 
* 
* 6. Each time you redistribute the Program (or any work based on the 
*    Program), the recipient automatically receives a license from the 
*    original licensor to copy, distribute or modify the Program subject to 
*    these terms and conditions. You may not impose any further restrictions 
*    on the recipients' exercise of the rights granted herein. You are not 
*    responsible for enforcing compliance by third parties to this License. 
* 
* 7. If, as a consequence of a court judgment or allegation of patent 
*    infringement or for any other reason (not limited to patent issues), 
*    conditions are imposed on you (whether by court order, agreement or 
*    otherwise) that contradict the conditions of this License, they do not 
*    excuse you from the conditions of this License. If you cannot distribute 
*    so as to satisfy simultaneously your obligations under this License and 
*    any other pertinent obligations, then as a consequence you may not 
*    distribute the Program at all. For example, if a patent license would 
*    not permit royalty-free redistribution of the Program by all those who 
*    receive copies directly or indirectly through you, then the only way you 
*    could satisfy both it and this License would be to refrain entirely from 
*    distribution of the Program. 
* 
*    If any portion of this section is held invalid or unenforceable under any
*    particular circumstance, the balance of the section is intended to apply
*    and the section as a whole is intended to apply in other circumstances. 
* 
*    It is not the purpose of this section to induce you to infringe any 
*    patents or other property right claims or to contest validity of any 
*    such claims; this section has the sole purpose of protecting the 
*    integrity of the free software distribution system, which is implemented 
*    by public license practices. Many people have made generous contributions
*    to the wide range of software distributed through that system in 
*    reliance on consistent application of that system; it is up to the 
*    author/donor to decide if he or she is willing to distribute software 
*    through any other system and a licensee cannot impose that choice. 
* 
*    This section is intended to make thoroughly clear what is believed to be 
*    a consequence of the rest of this License. 
* 
* 8. If the distribution and/or use of the Program is restricted in certain 
*    countries either by patents or by copyrighted interfaces, the original 
*    copyright holder who places the Program under this License may add an 
*    explicit geographical distribution limitation excluding those countries, 
*    so that distribution is permitted only in or among countries not thus 
*    excluded. In such case, this License incorporates the limitation as if 
*    written in the body of this License. 
* 
* 9. The Free Software Foundation may publish revised and/or new versions of 
*    the General Public License from time to time. Such new versions will be 
*    similar in spirit to the present version, but may differ in detail to 
*    address new problems or concerns. 
* 
*    Each version is given a distinguishing version number. If the Program 
*    specifies a version number of this License which applies to it and "any 
*    later version", you have the option of following the terms and 
*    conditions either of that version or of any later version published by 
*    the Free Software Foundation. If the Program does not specify a version 
*    number of this License, you may choose any version ever published by the 
*    Free Software Foundation. 
* 
* 10. If you wish to incorporate parts of the Program into other free programs
*     whose distribution conditions are different, write to the author to ask 
*     for permission. For software which is copyrighted by the Free Software 
*     Foundation, write to the Free Software Foundation; we sometimes make 
*     exceptions for this. Our decision will be guided by the two goals of 
*     preserving the free status of all derivatives of our free software and 
*     of promoting the sharing and reuse of software generally. 
* 
*    NO WARRANTY
* 
* 11. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY 
*     FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN 
*     OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES 
*     PROVIDE THE PROGRAM "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER 
*     EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
*     WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE 
*     ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH 
*     YOU. SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL 
*     NECESSARY SERVICING, REPAIR OR CORRECTION. 
* 
* 12. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING 
*     WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR 
*     REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR 
*     DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL 
*     DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE THE PROGRAM 
*     (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED 
*     INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE OF 
*     THE PROGRAM TO OPERATE WITH ANY OTHER PROGRAMS), EVEN IF SUCH HOLDER OR 
*     OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES. 
* 
* END OF TERMS AND CONDITIONS
* 
* How to Apply These Terms to Your New Programs
* 
* If you develop a new program, and you want it to be of the greatest 
* possible use to the public, the best way to achieve this is to make it free 
* software which everyone can redistribute and change under these terms. 
* 
* To do so, attach the following notices to the program. It is safest to 
* attach them to the start of each source file to most effectively convey the
* exclusion of warranty; and each file should have at least the "copyright" 
* line and a pointer to where the full notice is found. 
* 
* one line to give the program's name and an idea of what it does.
* Copyright (C) yyyy  name of author
* 
* This program is free software; you can redistribute it and/or modify it 
* under the terms of the GNU General Public License as published by the Free 
* Software Foundation; either version 2 of the License, or (at your option) 
* any later version.
* 
* This program is distributed in the hope that it will be useful, but WITHOUT 
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
* more details.
* 
* You should have received a copy of the GNU General Public License along with
* this program; if not, write to the Free Software Foundation, Inc., 59 
* Temple Place - Suite 330, Boston, MA  02111-1307, USA.
* 
* Also add information on how to contact you by electronic and paper mail. 
* 
* If the program is interactive, make it output a short notice like this when 
* it starts in an interactive mode: 
* 
* Gnomovision version 69, Copyright (C) year name of author Gnomovision comes 
* with ABSOLUTELY NO WARRANTY; for details type 'show w'.  This is free 
* software, and you are welcome to redistribute it under certain conditions; 
* type 'show c' for details.
* 
* The hypothetical commands 'show w' and 'show c' should show the appropriate 
* parts of the General Public License. Of course, the commands you use may be 
* called something other than 'show w' and 'show c'; they could even be 
* mouse-clicks or menu items--whatever suits your program. 
* 
* You should also get your employer (if you work as a programmer) or your 
* school, if any, to sign a "copyright disclaimer" for the program, if 
* necessary. Here is a sample; alter the names: 
* 
* Yoyodyne, Inc., hereby disclaims all copyright interest in the program 
* 'Gnomovision' (which makes passes at compilers) written by James Hacker.
* 
* signature of Ty Coon, 1 April 1989
* Ty Coon, President of Vice
* 
* This General Public License does not permit incorporating your program into 
* proprietary programs. If your program is a subroutine library, you may 
* consider it more useful to permit linking proprietary applications with the 
* library. If this is what you want to do, use the GNU Library General Public 
* License instead of this License.
*******************************************************************************/

/*****************************************************************************
******************************************************************************
**                                                                          **
** INTEL CORPORATION                                                        **
**                                                                          **
** This software is supplied under the terms of the license included        **
** above.  All use of this software must be in accordance with the terms    **
** of that license.                                                         **
**                                                                          **
** THIS FILE IS USED WHEN CREATING ians_core.o. HENCE, IT SHOULD NOT BE     **
** MODIFIED!                                                                **
**                                                                          **
**  Module Name:    ians_kernel.c                                           **
**                                                                          **
**  Abstract:       This module contains all the functions needed for the   **
**                  interface of the iANS module with the kernel.           **
**                                                                          **
**  Environment:    Kernel Mode (Linux 2.4.x)                               **
**                                                                          **
**                  Written using Tab Size = 4, Indent Size = 4             **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
******************************************************************************
*****************************************************************************/

#define _IANS_MAIN_MODULE_C_

#include <linux/init.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/inetdevice.h>
#include <linux/etherdevice.h>
#include <linux/notifier.h>

#include "ans_version.h"
#include "ians_kcompat.h"
#include "ians_status.h"
#include "ans_interface.h"
#include "lib/incg_dev.h"
#include "lib/incg_net.h"
#include "lib/incg_defs.h"
#include "lib/incg_flow.h"
#include "lib/incg_locks.h"
#include "lib/incg_kthread.h"
#include "lib/incg_message.h"
#include "lib/incg_gp_mem.h"
#include "lib/incg_open_utils.h"


MODULE_AUTHOR("Intel Corporation, <linux.nics@intel.com>");
MODULE_DESCRIPTION("Intel(R) Advanced Network Services");
MODULE_LICENSE("Proprietary");

/*****************************************************************************
**                  F o r w a r d  D e c l e r a t i o n s                  **
*****************************************************************************/

/*---------------------- VADAPTER DEVICE ENTRY POINTS ----------------------*/
static int iansVadapterOpen(struct net_device *dev);
static int iansVadapterStop(struct net_device *dev);
static int iansVadapterSend(struct sk_buff *skb, struct net_device *dev);
static struct net_device_stats *iansVadapterGetStats(struct net_device *dev);
static void iansVadapterSetMulticastList(struct net_device *dev);
static int iansVadapterSetMacAddress(struct net_device *dev, void *addr);
static int iansVadapterDoIoctl(struct net_device *dev, struct ifreq *rq, int cmd);
static int iansVadapterChangeMtu(struct net_device *dev, int new_mtu);

/*----------------------- CONTROL DEVICE ENTRY POINTS ----------------------*/
static int iansControlOpen(struct net_device *dev);
static int iansControlStop(struct net_device *dev);
static int iansControlSend(struct sk_buff *skb, struct net_device *dev);
static struct net_device_stats *iansControlGetStats(struct net_device *dev);
static void iansControlSetMCList(struct net_device *dev);
static int iansControlSetMacAddress(struct net_device *dev, void *addr);
static int iansControlDoIoctl(struct net_device *dev, struct ifreq *rq, int cmd);
static int iansControlChangeMtu(struct net_device *dev, int new_mtu);

/*------------------------ REGISTERED FUNCTIONS ----------------------------*/
static int ians_netdev_event(struct notifier_block *nb, unsigned long notification, void *ptr);
static int ians_inetaddr_event(struct notifier_block *nb, unsigned long notification, void *ptr);

/*-------------------------- SERVICE FUNCTIONS -----------------------------*/
static struct net_device* _iansDevAlloc(const char *name);
static struct net_device* _iansInitEtherdev(const char *name);
static inline IANS_STATUS _convertOsMCListToAnsMCList(struct dev_mc_list *osList, int osCount, MULTICAST_LIST *ansList);
static inline void _convertOsPFToAnsPF(USHORT osFlags, PACKET_FILTER *packetFilter);
static inline IANS_STATUS _ConvertNetDevEventToAnsEvent(unsigned long notification, INCG_NETDEV_NOTIFICATIONS *event);

/*****************************************************************************
******************************************************************************
**                  G L O B A L  P A R A M E T E R S                        **
******************************************************************************
*****************************************************************************/
static struct net_device *control_g = NULL;

static struct notifier_block ians_netdev_notifier = {
    notifier_call: ians_netdev_event,
};

static struct notifier_block ians_inetaddr_notifier = {
    notifier_call: ians_inetaddr_event,
};

/*--------------------------------------------------------------------------*/

char * iansGetANSVersion(void) {
    
    return ANS_VERSION;
}

/*---------------------- VADAPTER DEVICE FUNCTIONS -------------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
IANS_STATUS iansInitVadapterDev(const char *name, device_t **ansVadapterDev)
{
    struct net_device *osVadapterDev = NULL;

    ASSERT(name);

    //check if a device by this name already exists
    if ((osVadapterDev = dev_get_by_name(name)) != NULL) {
        dev_put(osVadapterDev);
        return IANS_DEVICE_ALREADY_EXISTS;
    }

    if((osVadapterDev = _iansInitEtherdev(name)) == NULL) {
        return IANS_VADAPTER_INITIALIZE_FAIL;
    }

    osVadapterDev->open = iansVadapterOpen;
    osVadapterDev->stop = iansVadapterStop;
    osVadapterDev->hard_start_xmit = iansVadapterSend;
    osVadapterDev->get_stats = iansVadapterGetStats;
    osVadapterDev->set_multicast_list = iansVadapterSetMulticastList;
    osVadapterDev->set_mac_address = iansVadapterSetMacAddress;
    osVadapterDev->do_ioctl = iansVadapterDoIoctl;
    osVadapterDev->change_mtu = iansVadapterChangeMtu;
    osVadapterDev->tx_queue_len = 0;
    
    //if queue len is changed to > 0 then it will be better to free packets and
    //return 0 in case that vadapterSend fails so that the skbs are not stuck
    //in the vadapter transmit queue.                                

    iNCGGPSizedMemAlloc((void **)(ansVadapterDev), sizeof(device_t));
    if(*ansVadapterDev == NULL) {
        IANS_PRINT_ERROR("Failed to allocate an IANS device structure\n");
        iNCGGPMemFree((void **)(&osVadapterDev));
        return IANS_MEM_ERROR;
    }

    (*ansVadapterDev)->pOsDev = osVadapterDev;
    (*ansVadapterDev)->name = osVadapterDev->name;
    (*ansVadapterDev)->mtu = &(osVadapterDev->mtu);
    (*ansVadapterDev)->mac = (MAC_ADDR *)(osVadapterDev->dev_addr);

    if (iNCGDevVlanInit(*ansVadapterDev) != IANS_OK) {
        IANS_PRINT_ERROR("Failed to allocate the vlan part of the device\n");
        iNCGGPMemFree((void **)(&osVadapterDev));
        iNCGGPMemFree((void **)(ansVadapterDev));
        return IANS_MEM_ERROR;
    }

    return IANS_OK;
}


/*---------------------- VADAPTER DEVICE ENTRY POINTS ----------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansVadapterOpen(struct net_device *dev)
{
    device_t *ansDev = NULL;
    int res = 0;
    IANS_STATUS ansRes = IANS_OK;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return -EAGAIN;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return -EAGAIN;
    }

    ANSConfigLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigUnlockBH();
        return -EAGAIN;
    }

    ansRes = ansDev->open(ansDev);
    if(ansRes == IANS_OK) {
        MOD_INC_USE_COUNT;
    }
    else {
        res = -EPERM; //operation not permitted
    }

    ANSConfigUnlockBH();

    return res;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansVadapterStop(struct net_device *dev)
{
    device_t *ansDev = NULL;
    IANS_STATUS ansRes = IANS_OK;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return -EAGAIN;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return -EAGAIN;
    }

    ANSConfigLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigUnlockBH();
        return -EAGAIN;
    }

    ansRes = ansDev->close(ansDev);
    if(ansRes == IANS_OK) {
        MOD_DEC_USE_COUNT;
    }

    ANSConfigUnlockBH();

    return 0;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansVadapterSend(struct sk_buff *skb, struct net_device *dev)
{
    device_t *ansDev;
    message_t msg; //automatic variable to save allocation time.
    int ansRes = IANS_ERROR;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        goto out;
    }

    if (skb == NULL) {
        IANS_PRINT_ERROR("skb == NULL\n");
        goto out;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        goto out;
    }

    ANSConfigLock();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        goto unlock_out;
    }

    iAnsMemSet(&msg, 0, sizeof(message_t));

    iansConvertMessageToAnsMessage(skb, &msg, TRANSMIT_FLOW);

    ansRes = ansDev->send(&msg, ansDev);

unlock_out:    
    ANSConfigUnlock();
    
out:    
    if(ansRes != IANS_OK) {
        //ANS failed to transmit the packet
        //free the packet and return 0 like all ethernet drivers do
        kfree_skb(skb); 
    }

    return 0;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static struct net_device_stats *iansVadapterGetStats(struct net_device *dev)
{
    device_t *ansDev = NULL;
    struct net_device_stats *res = NULL;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return NULL;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return NULL;
    }

    ANSConfigLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigUnlockBH();
        return NULL;
    }

    res = (struct net_device_stats *) ansDev->getStats(ansDev);

    ANSConfigUnlockBH();

    return res;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static void iansVadapterSetMulticastList(struct net_device *dev)
{
    MULTICAST_LIST multicastList;
    PACKET_FILTER packetFilter;
    device_t *ansDev = NULL;
    IANS_STATUS res;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return;
    }

    memset(&multicastList, 0, sizeof(MULTICAST_LIST));

    if(dev->mc_list) {
        ASSERT(dev->mc_count);
        res = _convertOsMCListToAnsMCList(dev->mc_list, dev->mc_count, &multicastList);
        if(res != IANS_OK) {
            return;
        }
    }

    _convertOsPFToAnsPF(dev->flags, &packetFilter);

    ANSConfigChangeLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigChangeUnlockBH();
        return;
    }

    res = ansDev->setMulticastList(ansDev, &multicastList, packetFilter);
    if(res != IANS_OK) {
        IANS_PRINT_ERROR("Set multicast list failed!\n");
        ANSConfigChangeUnlockBH();
        return;
    }

    ANSConfigChangeUnlockBH();
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansVadapterSetMacAddress(struct net_device *dev, void *addr)
{
    device_t *ansDev = NULL;
    sockAddr_t *sockaddr;
    IANS_STATUS res;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return -EAGAIN;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return -EAGAIN;
    }

    if (addr == NULL) {
        IANS_PRINT_ERROR("addr == NULL\n");
        return -EAGAIN;
    }

    sockaddr = (sockAddr_t*)addr;
    if (!is_valid_ether_addr(sockaddr->saData)) {
        return -EADDRNOTAVAIL;
    }

    ANSConfigChangeLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigChangeUnlockBH();
        return -EAGAIN;
    }

    res = ansDev->setMacAddr(ansDev, sockaddr);
    if(res != IANS_OK) {
        IANS_PRINT_ERROR("Set MAC address failed!\n");
        ANSConfigChangeUnlockBH();
        return -1;
    }

    ANSConfigChangeUnlockBH();

    return 0;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansVadapterDoIoctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
    device_t *ansDev = NULL;
    IANS_STATUS res;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return -EAGAIN;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return -EAGAIN;
    }

    if (rq == NULL) {
        IANS_PRINT_ERROR("rq == NULL\n");
        return -EAGAIN;
    }

    if((cmd >= SIOCDEVPRIVATE) && (cmd <(SIOCDEVPRIVATE+16))) {
        // Private IOCTL's handling is undefined. reject them.
        IANS_PRINT_DEBUG("vadapter %s got a private IOCTL 0x%04X\n", dev->name, cmd);
        return 0;
    }

    ANSConfigLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        ANSConfigUnlockBH();
        return -EAGAIN;
    }

    res = ansDev->doIoctl(ansDev, rq, cmd);

    ANSConfigUnlockBH();

    return res;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansVadapterChangeMtu(struct net_device *dev, int new_mtu)
{
    device_t *ansDev = NULL;
    int res;

    if (dev == NULL) {
        IANS_PRINT_ERROR("dev == NULL\n");
        return -EAGAIN;
    }

    if (dev->name == NULL) {
        IANS_PRINT_ERROR("dev->name == NULL\n");
        return -EAGAIN;
    }

    if(new_mtu == dev->mtu) {
        //nothing to do
        return 0;
    }

    if(new_mtu < 0) {
        //must be positive
        return -EINVAL;
    }

    //ANSConfigLockBH();

    ansDev = ANSGetVadapterAnsDevice(dev->name);
    if(ansDev == NULL) {
        IANS_PRINT_ERROR("there is no such ans device.\n");
        //ANSConfigUnlockBH();
        return -EAGAIN;
    }

    res = ansDev->changeMtu(ansDev, new_mtu);
    if(res != IANS_OK) {
        IANS_PRINT_ERROR("Change MTU failed!\n");
        //ANSConfigUnlockBH();
        return -EINVAL;
    }

    //ANSConfigUnlockBH();
    return 0;
}


/*---------------------- CONTROL DEVICE FUNCTIONS --------------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansControlDoIoctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
    IANS_STATUS res = IANS_OK;
    char *data;

    if(dev != control_g) {
        IANS_PRINT_ERROR("Got control IOCTL on non control device: dev->name %s\n", dev->name);
        return IANS_OS_ERROR;
    }

    if (rq == NULL) {
        IANS_PRINT_ERROR("rq == NULL\n");
        return IANS_OS_ERROR;
    }

    switch(cmd) {
        case IANSIOCCONFIG:
	    iNCGGPSizedMemAlloc((void **)(&data), ANSGetConfigIoctlSize());
            if(data == NULL) {
                IANS_PRINT_ERROR("Got null data in control IOCTL\n");
                res = IANS_OS_ERROR;
                break;
            }
	    if (iansCopyFromUser(data, rq->ifr_ifru.ifru_data, ANSGetConfigIoctlSize())) {
		iNCGGPMemFree((void **)(&data));
                return IANS_OS_ERROR;
	    }
            res = ANSDoConfigIoctl(data);
	    if (iansCopyToUser(rq->ifr_ifru.ifru_data, data, ANSGetConfigIoctlSize())) {
                res = IANS_OS_ERROR;
	    }
	    iNCGGPMemFree((void **)(&data));
            break;

        case IANSIOCSERVICE:
            iNCGGPSizedMemAlloc((void **)(&data), ANSGetServiceIoctlSize());
            if(data == NULL) {
                IANS_PRINT_ERROR("Got null data in service IOCTL\n");
                res = IANS_OS_ERROR;
                break;
            }
            if (iansCopyFromUser(data, rq->ifr_ifru.ifru_data, ANSGetServiceIoctlSize())) {
                iNCGGPMemFree((void **)(&data));
                return IANS_OS_ERROR;
	    }
            res = ANSDoServiceIoctl(data);
	    if (iansCopyToUser(rq->ifr_ifru.ifru_data, data, ANSGetServiceIoctlSize())) {
                res = IANS_OS_ERROR;
	    }
	    iNCGGPMemFree((void **)(&data));
	    break;

        default:
            IANS_PRINT_DEBUG("Ioctl not supported\n");
            res = IANS_OS_ERROR;
            break;
    }

    return res;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansControlOpen(struct net_device *dev)
{
    IANS_PRINT_DEBUG("Got open on control device\n");
    return -EPERM;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansControlStop(struct net_device *dev)
{
    IANS_PRINT_DEBUG("Got stop on control device\n");
    return -EPERM;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansControlSend(struct sk_buff *skb, struct net_device *dev)
{
    IANS_PRINT_DEBUG("Got hard_start_xmit for control device\n");
    return 0;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static struct net_device_stats *iansControlGetStats(struct net_device *dev)
{
    static struct net_device_stats stats;
    //IANS_PRINT_DEBUG("Got get_statistics for control device\n");

    memset(&stats, 0, sizeof(struct net_device_stats));

    return &stats;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static void iansControlSetMCList(struct net_device *dev)
{
    IANS_PRINT_DEBUG("Got set_multicast_list for control device\n");
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansControlSetMacAddress(struct net_device *dev, void *addr)
{
    IANS_PRINT_DEBUG("Got set_mac_address for control device\n");
    return -EPERM;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int iansControlChangeMtu(struct net_device *dev, int new_mtu)
{
    IANS_PRINT_DEBUG("Got change_mtu for control device\n");
    return -EPERM;
}


/*------------------------ REGISTERED FUNCTIONS ----------------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int ians_netdev_event(struct notifier_block *nb, unsigned long notification, void *ptr)
{
    IANS_STATUS res = IANS_OK;
    INCG_NETDEV_NOTIFICATIONS event;
    struct net_device *dev = NULL;

    if (ptr == NULL) {
       IANS_PRINT_ERROR("bad notification parameters\n");
       return NOTIFY_BAD;
    }

    dev = (struct net_device*)ptr;
    if (!dev) {
        IANS_PRINT_ERROR("pointer to net_device is NULL!\n");
        return NOTIFY_DONE;
    }

    res = _ConvertNetDevEventToAnsEvent(notification, &event);
    if (res != IANS_OK) {
        IANS_PRINT_DEBUG("unknown notification\n");
        return NOTIFY_DONE;
    }

    res = ANSHandleNetDevEvent(dev->name, event);
    if (res != IANS_OK) {
        IANS_PRINT_ERROR("couldn't handle notification\n");
    }

    return NOTIFY_DONE;
}

/*****************************************************************************
**                                                                          **
**  Function Name: ians_inetaddr_event                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static int ians_inetaddr_event(struct notifier_block *nb, unsigned long notification, void *ptr)
{
    IANS_STATUS res = IANS_OK;
    INCG_NETDEV_NOTIFICATIONS event;
    struct in_ifaddr *ifa = (struct in_ifaddr *)ptr;
    struct net_device *dev = NULL;

    if (ifa){
        dev = ifa->ifa_dev->dev;
    }
    else {
        IANS_PRINT_ERROR("bad notification parameters\n");
        return NOTIFY_BAD;
    }

    res = _ConvertNetDevEventToAnsEvent(notification, &event);
    if(res != IANS_OK) {
        IANS_PRINT_ERROR("unknown notification\n");
    }

    res = ANSHandleInetAddrEvent(dev->name, event,ifa->ifa_address);
    if(res != IANS_OK) {
        IANS_PRINT_ERROR("couldn't handle notification\n");
    }

    return NOTIFY_DONE;
}


/*--------------------------- MODULE FUNCTIONS -----------------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
int ians_init_module(void)
{
    IANS_STATUS res = IANS_OK;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
    IANS_PRINT_ERROR("Cannot load IANS module on 2.2.x kernels.\n");
    return -1;
#endif

    res = ANSConfigCreate();
    if(res!=IANS_OK) {
        return -1;
    }

    if((control_g = _iansInitEtherdev("ians")) == NULL) {
        IANS_PRINT_ERROR("Init etherdev failed - cannot load the IANS module.\n");
        return -1;
    }

    control_g->open = iansControlOpen;
    control_g->stop = iansControlStop;
    control_g->hard_start_xmit = iansControlSend;
    control_g->get_stats = iansControlGetStats;
    control_g->set_multicast_list = iansControlSetMCList;
    control_g->set_mac_address = iansControlSetMacAddress;
    control_g->do_ioctl = iansControlDoIoctl;
    control_g->change_mtu = iansControlChangeMtu;
    control_g->flags = 0;
    control_g->mtu = 0;
    control_g->tx_queue_len = 0;

    SET_MODULE_OWNER(control_g);

    register_netdev(control_g);

    register_netdevice_notifier(&ians_netdev_notifier);

    register_inetaddr_notifier(&ians_inetaddr_notifier);

    iNCGCreateKernelThreads();

    return 0;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
void ians_cleanup_module(void)
{
    iNCGDestroyKernelThreads();

    unregister_inetaddr_notifier(&ians_inetaddr_notifier);

    unregister_netdevice_notifier(&ians_netdev_notifier);

    unregister_netdev(control_g);

    iNCGGPMemFree((void**)(&control_g));

    ANSConfigFree();

#ifdef MEM_DEBUG
    do {
        int i = iNCGGPMemGetAllocCount();
        if(i != 0) {
            IANS_PRINT_DEBUG("Allocation count = %d\n", i);
        }
    } while(0);
#endif //MEM_DEBUG
}


/*-------------------------- SERVICE FUNCTIONS -----------------------------*/

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static struct net_device* _iansDevAlloc(const char *name)
{
    struct net_device *dev;
    struct net_device *temp;

    if((temp = dev_get_by_name(name)) != NULL) {
        dev_put(temp);
        IANS_PRINT_ERROR("A device with this name(%s) already exists\n", name);
        return NULL;
    }

    iNCGGPSizedMemAlloc((void **)(&dev), sizeof(struct net_device));
    if(dev == NULL) {
        IANS_PRINT_ERROR("kmalloc failed.\n");
        return NULL;
    }

    iAnsMemSet(dev, 0, sizeof(struct net_device));

    iAnsStrCpy(dev->name, name);

    return dev;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static struct net_device* _iansInitEtherdev(const char *name)
{
    struct net_device *dev = NULL;

    if((dev = _iansDevAlloc(name)) == NULL) {
        return NULL;
    }

    ether_setup(dev);

    return dev;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static inline IANS_STATUS _convertOsMCListToAnsMCList(struct dev_mc_list *osList, int osCount, MULTICAST_LIST *ansList)
{
    struct dev_mc_list *currList;
    IANS_FLOW_STATUS res;

    ASSERT(osList && ansList);

    for(currList = osList; currList != NULL; currList = currList->next) {
        res = ANSAddMulticastAddress(ansList,
                                     (MAC_ADDR*)currList->dmi_addr,
                                     currList->dmi_addrlen,
                                     currList->dmi_users,
                                     currList->dmi_gusers);

        if(res != FLOW_OK) {
            if(res == FLOW_GP_RESOURCES) {
                IANS_PRINT_ERROR("kmalloc failed.\n");
            }
            IANS_PRINT_ERROR("Failed in adding a multicast address.\n");
            return IANS_ERROR;
        }
    }

    ASSERT(osCount == ansList->NumAddresses);

    return IANS_OK;
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static inline void _convertOsPFToAnsPF(USHORT osFlags, PACKET_FILTER *packetFilter)
{
    if(osFlags & IFF_PROMISC) {
        *packetFilter |= PF_SYS_PROMISC_BIT;
    }
    else {
        *packetFilter &= ~(PF_SYS_PROMISC_BIT);
    }

    if(osFlags & IFF_ALLMULTI) {
        *packetFilter |= PF_SYS_ALLMCA_BIT;
    }
    else {
        *packetFilter &= ~(PF_SYS_ALLMCA_BIT);
    }
}

/*****************************************************************************
**                                                                          **
**  Function Name:                                                          **
**                                                                          **
**  Abstract:                                                               **
**                                                                          **
**  Parameters:                                                             **
**                                                                          **
**  Assumptions:                                                            **
**                                                                          **
**  Return Value:                                                           **
**                                                                          **
**  Revision History:                                                       **
**                                                                          **
*****************************************************************************/
static inline IANS_STATUS _ConvertNetDevEventToAnsEvent(unsigned long notification, INCG_NETDEV_NOTIFICATIONS *event)
{
    switch (notification) {
        case NETDEV_UP:
            *event = IANS_NETDEV_UP;
            break;
        case NETDEV_DOWN:
            *event = IANS_NETDEV_DOWN;
            break;
        case NETDEV_UNREGISTER:
            *event = IANS_NETDEV_UNREGISTER;
            break;
        case NETDEV_GOING_DOWN:
            *event = IANS_NETDEV_GOING_DOWN;
            break;
        default:
            return IANS_ERROR;
    }

    return IANS_OK;
}

/*-------------------------- MODULE ENTRY POINTS ---------------------------*/
module_init(ians_init_module);
module_exit(ians_cleanup_module);

