
/****************************************************************************
 ENSONIQ Soundscape Digital Audio Driver
 Protected-Mode Version
 Created 09/08/95, Andrew P. Weir
****************************************************************************/


#include "sndscape.h"


/****************************************************************************
 EXTERNAL GLOBAL VARS - hardware config info ...
****************************************************************************/

WORD			BasePort;	/* Gate Array/MPU-401 base port */
short			MidiIrq;	/* the MPU-401 IRQ */
WORD			WavePort;	/* the AD-1848 base port */
short			WaveIrq;	/* the PCM IRQ */
short			DmaChan;	/* the PCM DMA channel */


/****************************************************************************
 INTERNAL GLOBAL VARS - all kinds of stuff ...
****************************************************************************/

short			Windx;		/* Wave IRQ index - for reg writes */
short			Mindx;		/* MIDI IRQ index - for reg writes */

char *			DmaBuffer;	/* dma buffer ptr */
WORD			DmaBuffSel;	/* dma buffer selector */
SHARED volatile	*	SharedP;	/* dos shared mem struct ptr */
WORD			SharedSel;	/* dos shared mem selector */

short			IcType;		/* the Ensoniq chip type */
short			CdCfgSav;	/* gate array register save area */
short			DmaCfgSav;	/* gate array register save area */
short			IntCfgSav;	/* gate array register save area */
WORD			PicMaskReg;	/* the PIC mask register */
short			PicMaskSav;	/* save area for mask state */
short			Detected;	/* suuccessful detection flag */
short			Vector;		/* the Wave interrupt vector num */
void (interrupt far    *PmVecSav)();	/* prot-mode vector save */
DWORD			RmVecSav;	/* real-mode vector save */

short			DacSavL;	/* DAC left volume save */
short			DacSavR;	/* DAC right volume save */
short			CdxSavL;	/* CD/Aux left volume save */
short			CdxSavR;	/* CD/Aux right volume save */
short			AdcSavL;	/* ADC left volume save */
short			AdcSavR;	/* ADC right volume save */

short const		SsIrqs[4] = { 9, 5, 7, 10 };  /* Soundscape IRQs */
short const		RsIrqs[4] = { 9, 7, 5, 15 };  /* an older IRQ set */
short const *		Irqs;           /* pointer to one of the IRQ sets */

DMAC_REGS const		DmacRegs[4] = {	/* the DMAC regs for chans 0-3 ... */
				{ 0x00, 0x01, 0x08, 0x0a, 0x0b, 0x0c, 0x87 },
				{ 0x02, 0x03, 0x08, 0x0a, 0x0b, 0x0c, 0x83 },
				{ 0x04, 0x05, 0x08, 0x0a, 0x0b, 0x0c, 0x81 },
				{ 0x06, 0x07, 0x08, 0x0a, 0x0b, 0x0c, 0x82 }
};
DMAC_REGS const *	DmacRegP;	/* a pointer to a DMAC reg struct */


/****************************************************************************
 EXTERNAL DRIVER FUNCTIONS
****************************************************************************/

/*
  short DetectSoundscape(void);

  This function is used to detect the presence of a Soundscape card in a
  system. It will read the hardware config info from the SNDSCAPE.INI file,
  the path to which is indicated by the SNDSCAPE environment variable. This
  config info will be stored in global variable space for the other driver
  functions to reference. Once the config settings have been determined, a
  hardware test will be performed to see if the Soundscape card is actually
  present. If this function is not explicitly called by the application, it
  it will be called by the OpenSoundscape function.

  INPUTS:
  	None

  RETURNS:
  	0   - if Soundscape detection was successful
	-1  - if
		The SNDSCAPE environment variable was not found
		The SNDSCAPE.INI file cannot be opened
		The SNDSCAPE.INI file is missing information or is corrupted
		The Soundscape hardware is not detected
*/

short DetectSoundscape(void) {
	char	 static str[78];
	short    tmp, status;
	char    *ep;
	FILE    *fp;

	/* initialize some local vars ... */
	status = 0;
	fp = 0;

	/* get the environment var and build the filename; then open it */
	if( !(ep = getenv("SNDSCAPE")) )
		return -1;

	strcpy(str, ep);
	if( str[strlen(str) - 1] == '\\' )
		str[strlen(str) - 1] = '\0';
	strcat(str, "\\SNDSCAPE.INI");
	if( !(fp = fopen(str, "r")) )
		return -1;

	/* read all of the necessary config info ... */
	if( GetConfigEntry("Product", str, fp) ) {
		status = -1;
		goto exit;
	}
	/* if an old product name is read, set the IRQs accordingly */
	strupr(str);
	if( strstr(str, "SOUNDFX") || strstr(str, "MEDIA_FX") )
		Irqs = RsIrqs;
	else
		Irqs = SsIrqs;

	if( GetConfigEntry("Port", str, fp) ) {
		status = -1;
		goto exit;
	}
	BasePort = (WORD) strtol(str, NULL, 16);

	if( GetConfigEntry("WavePort", str, fp) ) {
		status = -1;
		goto exit;
	}
	WavePort = (WORD) strtol(str, NULL, 16);

	if( GetConfigEntry("IRQ", str, fp) ) {
		status = -1;
		goto exit;
	}
	MidiIrq = (short) strtol(str, NULL, 10);
	if( MidiIrq == 2)
		MidiIrq = 9;

	if( GetConfigEntry("SBIRQ", str, fp) ) {
		status = -1;
		goto exit;
	}
	WaveIrq = (short) strtol(str, NULL, 10);
	if( WaveIrq == 2 )
		WaveIrq = 9;

	if( GetConfigEntry("DMA", str, fp) ) {
		status = -1;
		goto exit;
	}
	DmaChan = (short) strtol(str, NULL, 10);

	/* see if Soundscape is there by reading HW ... */
	if( (inp(BasePort + GA_HOSTCTL_OFF) & 0x78) != 0x00 ) {
		status = -1;
		goto exit;
	}
	if( (inp(BasePort + GA_ADDR_OFF) & 0xf0) == 0xf0 ) {
		status = -1;
		goto exit;
	}
	outp(BasePort + GA_ADDR_OFF, 0xf5);
	tmp = inp(BasePort + GA_ADDR_OFF);
	if( (tmp & 0xf0) == 0xf0) {
		status = -1;
		goto exit;
	}
	if( (tmp & 0x0f) != 0x05) {
		status = -1;
		goto exit;
	}

	/* formulate the chip ID */
	if( (tmp & 0x80) != 0x00 )
		IcType = MMIC;
	else if((tmp & 0x70) != 0x00)
		IcType = OPUS;
	else
		IcType = ODIE;

	/* now do a quick check to make sure the CoDec is there too */
	if( (inp(WavePort) & 0x80) != 0x00 ) {
		status = -1;
		goto exit;
	}

	Detected = 0xed;
exit:
	if( fp != 0 )
		fclose(fp);

	return status;
}


/*
  short OpenSoundscape(void);

  This function opens the Soundscape driver. It will setup the Soundscape
  hardware for Native PCM mode (the default mode is Sound Blaster emulation).
  It will first call the DetectSoundscape function it it determines that it
  has not already been called.

  INPUTS:
  	None.

  RETURNS:
  	0   - if successful
	-1  - if the DetectSoundscape function needed to be called and
	      returned unnsuccessful, or if DOS mem could not be allocated
*/

short OpenSoundscape(void) {

	union REGS     r;
	DWORD          tmp;

	/* see if we need to detect first */
	if( Detected != 0xed )
		if( DetectSoundscape() )
			return -1;

	/* In case the CoDec is running, stop it */
	StopCoDec();

	/* Clear possible CoDec and SoundBlaster emulation interrupts */
	outp(WavePort + CD_STATUS_OFF, 0x00);
	inp(0x22e);

	/* Get DOS memory for 2x DMA buff size, align the buffer within */
	if( !(tmp = DosMalloc(2*BUFFERSIZE)) )
		return -1;
	DmaBuffSel = (WORD) (tmp >> 16);
	DmaBuffer = (char *) ((tmp & 0x0000ffff) << 4);
	if( (DWORD) DmaBuffer >> 16 != ((DWORD) DmaBuffer + BUFFERSIZE-1) >> 16 )
		DmaBuffer += BUFFERSIZE;

	/* Get DOS memory for the real-mode ISR and shared globals */
	if( !(tmp = DosMalloc(256)) )
		return -1;
	SharedP = (SHARED *) ((tmp & 0x0000ffff) << 4);

	/* Copy the real-mode isr into the shared memory */
	memcpy(&SharedP->rmisr, RmWaveIsr, sizeof(SharedP->rmisr));

	/* Init config entries in shared memory */
	SharedP->dmabuffer = DmaBuffer;
	SharedP->waveport = WavePort;
	SharedP->waveirq = WaveIrq;

	/* Build the interrupt vector number */
	Vector = (WaveIrq < 8 ? 0x08 : 0x70) + (WaveIrq & 0x07);

	/* Save and install protected-mode vector first */
	PmVecSav = _dos_getvect(Vector);
	_dos_setvect(Vector, PmWaveIsr);

	/* Save real-mode vector */
	r.h.bl = Vector;
	r.w.ax = 0x0200;
	int386(0x31, &r, &r);
	RmVecSav = ((DWORD) r.w.cx << 16) | r.w.dx;

	/* Install real-mode vector */
	r.h.bl = Vector;
	r.w.cx = (WORD) ((DWORD) SharedP >> 4);
	r.w.dx = 0;
	r.w.ax = 0x0201;
	int386(0x31, &r, &r);

	/* Set DMA controller register set pointer based on channel */
	DmacRegP = &DmacRegs[DmaChan];

	/* If necessary, save some regs, do some resource re-routing */
	if( IcType != MMIC) {

		/* derive the MIDI and Wave IRQ indices (0-3) for reg writes */
		for( Mindx = 0; Mindx < 4; ++Mindx )
			if( MidiIrq == *(Irqs + Mindx) )
				break;
		for( Windx = 0; Windx < 4; ++Windx )
			if( WaveIrq == *(Irqs + Windx) )
				break;

		/* setup the CoDec DMA polarity */
		GaWrite(GA_DMACFG_REG, 0x50);

		/* give the CoDec control of the DMA and Wave IRQ resources */
	    	CdCfgSav = GaRead(GA_CDCFG_REG);
		GaWrite(GA_CDCFG_REG, 0x89 | (DmaChan << 4) | (Windx << 1));
	
		/* pull the Sound Blaster emulation off of those resources */
	    	DmaCfgSav = GaRead(GA_DMAB_REG);
		GaWrite(GA_DMAB_REG, 0x20);
		IntCfgSav = GaRead(GA_INTCFG_REG);
		GaWrite(GA_INTCFG_REG, 0xf0 | (Mindx << 2) | Mindx);
	}

	/* Save all volumes that we might use, init some levels */
	DacSavL = CdRead(CD_DACL_REG);
	DacSavR = CdRead(CD_DACR_REG);
	CdxSavL = CdRead(CD_CDAUXL_REG);
	CdxSavR = CdRead(CD_CDAUXL_REG);
	AdcSavL = CdRead(CD_ADCL_REG);
	AdcSavR = CdRead(CD_ADCR_REG);

	SetDacVol(127, 127);
	SetAdcVol(96, 96);

	/* Select the mic/line input to the record mux; */
	/* if not ODIE, set the mic gain bit too */
	CdWrite(CD_ADCL_REG, (CdRead(CD_ADCL_REG) & 0x3f) |
		(IcType == ODIE ? 0x80 : 0xa0));
	CdWrite(CD_ADCR_REG, (CdRead(CD_ADCR_REG) & 0x3f) |
		(IcType == ODIE ? 0x80 : 0xa0));

	/* Put the CoDec into mode change state */
	outp(WavePort + CD_ADDR_OFF, 0x40);

	/* Setup CoDec mode - single DMA chan, AutoCal on */
	CdWrite(CD_CONFIG_REG, 0x0c);

	/* enable the CoDec interrupt pin */
	CdWrite(CD_PINCTL_REG, CdRead(CD_PINCTL_REG) | 0x02);

	/* Setup PIC register ptr, unmask our IRQ */
	PicMaskReg = WaveIrq < 8 ? 0x21 : 0xa1;
	PicMaskSav = inp(PicMaskReg);
	outp(PicMaskReg, PicMaskSav & ~(1 << (WaveIrq & 0x07)));

	return 0;
}


/*
  short StartCoDec(WORD srate, short stereo, short size16bit, short direction);

  This function will start the CoDec auto-restart DMA process.

  INPUTS:
  	srate      - the audio sample rate in Hertz
	stereo     - 0 for mono data, 1 for L/R interleaved stereo data
	size16bit  - 0 for 8-bit unsigned data, 1 for 16-bit signed data
	direction  - 0 for playback, 1 for record

  RETURNS:
  	0          - if the CoDec was started successfully
  	-1         - if the sample rate is unacceptable
*/

short StartCoDec(WORD srate, short stereo, short size16bit, short direction) {
	WORD    i, tmp;

	/* make sure the AD-1848 is not already running */
	StopCoDec();

	/* set format */
	if( SetFormat(srate, stereo, size16bit, direction) )
		return -1;

	/* pre-fill the audio buffer with silence */
	tmp = size16bit ? 0x00 : 0x80;
	for( i = 0; i < BUFFERSIZE; ++i )
		*(DmaBuffer + i) = (char) tmp;

	/* initialize the int flip-flop and "next buffer to fill" ptr */
	SharedP->flipflop = 0;
	SharedP->nextbuffer = 0;

	/* Write the CoDec interrupt count - sample frames per half-buffer. */
	/* If not using ODIE and recording, use extended count regs */
	tmp = (BUFFERSIZE/2 >> (stereo + size16bit)) - 1;
	if( IcType != ODIE && direction ) {
		CdWrite(CD_XLCOUNT_REG, tmp);
		CdWrite(CD_XUCOUNT_REG, tmp >> 8);
	}
	else {
		CdWrite(CD_LCOUNT_REG, tmp);
		CdWrite(CD_UCOUNT_REG, tmp >> 8);
	}

	/* Set up the PC DMA Controller */
	outp(DmacRegP->mask, 0x04 | DmaChan);
	outp(DmacRegP->mode,
		(direction ? DMAC_AUTO_IN : DMAC_AUTO_OUT) | DmaChan);
	outp(DmacRegP->page, (short) ((DWORD) DmaBuffer >> 16));

	/* Disable around 2-byte programming sequences */
	_disable();
	outp(DmacRegP->clrff, 0x00);
	outp(DmacRegP->addr, (WORD) DmaBuffer);
	outp(DmacRegP->addr, (WORD) DmaBuffer >> 8);
	outp(DmacRegP->count, BUFFERSIZE - 1);
	outp(DmacRegP->count, (BUFFERSIZE - 1) >> 8);
	_enable();

	/* clear status, unmask the PC DMA Controller */
	inp(DmacRegP->status);
	outp(DmacRegP->mask, DmaChan);

	/* disable mode change state and start the CoDec */
	outp(WavePort + CD_ADDR_OFF, 0x00);
	CdWrite(CD_CONFIG_REG, direction ? 0x02 : 0x01);

	return 0;
}


/*
  char *GetBuffer(void);

  This function is used by the application, normally in a polling fashion,
  to get a pointer to the next buffer half to be filled/stored.

  INPUTS:
  	None

  RETURNS:
  	0      - if the CoDec is not ready for audio data / has no audio data
	non-0  - a pointer to a BUFFERSIZE/2 byte buffer will be returned if
	         the CoDec is ready for data / just filled a buffer
*/

char *GetBuffer(void) {
	BYTE    *p;

	p = SharedP->nextbuffer;
	if( p )
		SharedP->nextbuffer = 0;

	return p;
}


/*
  void PauseCoDec(void);

  This function will pause the CoDec auto-restart DMA process.
  Don't use this to stop the CoDec - use the StopCoDec function to do
  that; it will cleanup DRQs.

  INPUTS:
  	None

  RETURNS:
  	Nothing
*/

void PauseCoDec() {

	CdWrite(CD_CONFIG_REG, CdRead(CD_CONFIG_REG) & 0xfc);

	return;
}


/*
  void ResumeCoDec(short direction);

  This function will pause the CoDec auto-restart DMA process.

  INPUTS:
	direction  - 0 for playback, 1 for record

  RETURNS:
  	Nothing
*/

void ResumeCoDec(short direction) {

	CdWrite(CD_CONFIG_REG, direction ? 0x02 : 0x01);

	return;
}


/*
  void StopCoDec(void);

  This function will stop the CoDec auto-restart DMA process.

  INPUTS:
  	None

  RETURNS:
  	Nothing
*/

void StopCoDec(void) {
	WORD	i;

	CdWrite(CD_CONFIG_REG, CdRead(CD_CONFIG_REG) & 0xfc);

	/* Let the CoDec receive its last DACK(s). The DMAC must not be */
	/*  masked while the CoDec has DRQs pending. */
	for( i = 0; i < 256; ++i )
		if( !(inp(DmacRegP->status) & (0x10 << DmaChan)) )
			break;

	return;
}


/*
  void CloseSoundscape(void);

  This function will make sure the CoDec is stopped, return the Soundscape
  hardware to it's default Sound Blaster emulation mode, de-install the PCM
  interrupt, resore the PIC mask, and do other cleanup.

  INPUTS:
  	None

  RETURNS:
  	None
*/

void CloseSoundscape(void) {

	union REGS     r;

	/* in case the CoDec is running, stop it */
	StopCoDec();

	/* mask the PC DMA Controller */
	outp(DmacRegP->mask, 0x04 | DmaChan);

	/* restore pic state */
	outp(PicMaskReg, PicMaskSav);

	/* disable the CoDec interrupt pin */
	CdWrite(CD_PINCTL_REG, CdRead(CD_PINCTL_REG) & 0xfd);

	/* restore all volumes ... */
	CdWrite(CD_DACL_REG, DacSavL);
	CdWrite(CD_DACR_REG, DacSavR);
	CdWrite(CD_CDAUXL_REG, CdxSavL);
	CdWrite(CD_CDAUXL_REG, CdxSavR);
	CdWrite(CD_ADCL_REG, AdcSavL);
	CdWrite(CD_ADCR_REG, AdcSavR);

	/* if necessary, restore gate array resource registers */
	if ( IcType != MMIC ) {
		GaWrite(GA_INTCFG_REG, IntCfgSav);
		GaWrite(GA_DMAB_REG, DmaCfgSav);
		GaWrite(GA_CDCFG_REG, CdCfgSav);
	}

	/* restore rm vector */
	r.h.bl = Vector;
	r.x.ecx = (WORD) (RmVecSav >> 16);
	r.x.edx = (WORD) RmVecSav;
	r.x.eax = 0x0201;
	int386(0x31, &r, &r);

	/* restore pm vector */
	_dos_setvect(Vector, PmVecSav);

	/* free previously allocated DOS memory */
	DosFree(SharedSel);
	DosFree(DmaBuffSel);

	return;
}


/*
  void SetDacVol(short lvol, short rvol);

  This function sets the left and right DAC output level in the CoDec.

  INPUTS:
  	lvol  - left volume, 0-127
	rvol  - right volume, 0-127

  RETURNS:
  	Nothing

*/

void SetDacVol(short lvol, short rvol) {

	CdWrite(CD_DACL_REG, ~(lvol >> 1) & 0x3f);
	CdWrite(CD_DACR_REG, ~(rvol >> 1) & 0x3f);

	return;
}


/*
  void SetCdRomVol(short lvol, short rvol);

  This function sets the left and right CD-ROM output level in the CoDec.

  INPUTS:
  	lvol  - left volume, 0-127
	rvol  - right volume, 0-127

  RETURNS:
  	Nothing

*/

void SetCdRomVol(short lvol, short rvol) {

	CdWrite(CD_CDAUXL_REG, ~(lvol >> 2) & 0x1f);
	CdWrite(CD_CDAUXR_REG, ~(rvol >> 2) & 0x1f);

	return;
}


/*
  void SetAdcVol(short lvol, short rvol);

  This function sets the left and right ADC input level in the CoDec.

  INPUTS:
  	lvol  - left volume, 0-127
	rvol  - right volume, 0-127

  RETURNS:
  	Nothing

*/

void SetAdcVol(short lvol, short rvol) {

	CdWrite(CD_ADCL_REG, (CdRead(CD_ADCL_REG) & 0xf0) | (lvol & 0x7f) >> 3);
	CdWrite(CD_ADCR_REG, (CdRead(CD_ADCR_REG) & 0xf0) | (rvol & 0x7f) >> 3);

	return;
}


/****************************************************************************
 INTERNAL DRIVER FUNCTIONS
****************************************************************************/


/*
  short GaRead(WORD rnum);

  This function is used to read the indirect addressed registers in the
  Ensoniq Soundscape gate array.

  INPUTS:
  	rnum  - the numner of the indirect register to be read

  RETURNS:
  	the contents of the indirect register are returned

*/

short GaRead(WORD rnum) {
	outp(BasePort + GA_ADDR_OFF, rnum);
	return inp(BasePort + GA_DATA_OFF);
}


/*
  short GaWrite(WORD rnum, short value);

  This function is used to write the indirect addressed registers in the
  Ensoniq Soundscape gate array.

  INPUTS:
  	rnum   - the numner of the indirect register to be read
	value  - the byte value to be written to the indirect register

  RETURNS:
  	Nothing
*/

void GaWrite(WORD rnum, short value) {
	outp(BasePort + GA_ADDR_OFF, rnum);
	outp(BasePort + GA_DATA_OFF, value);

	return;
}


/*
  short CdRead(WORD rnum);

  This function is used to read the indirect addressed registers in the
  AD-1848 or compatible CoDec. It will preserve the special function bits
  in the upper-nibble of the indirect address register.

  INPUTS:
  	rnum  - the numner of the indirect register to be read

  RETURNS:
  	the contents of the indirect register are returned

*/

short CdRead(WORD rnum) {
	outp(WavePort + CD_ADDR_OFF,
		(inp(WavePort + CD_ADDR_OFF) & 0xf0) | rnum);
	return inp(WavePort + CD_DATA_OFF);
}


/*
  short CdWrite(WORD rnum, short value);

  This function is used to write the indirect addressed registers in the
  Ad-1848 or compatible CoDec. It will preserve the special function bits
  in the upper-nibble of the indirect address register.

  INPUTS:
  	rnum   - the numner of the indirect register to be read
	value  - the byte value to be written to the indirect register

  RETURNS:
  	Nothing
*/

void CdWrite(WORD rnum, short value) {
	outp(WavePort + CD_ADDR_OFF,
		(inp(WavePort + CD_ADDR_OFF) & 0xf0) | rnum);
	outp(WavePort + CD_DATA_OFF, value);

	return;
}


/*
  short GetConfigEntry(char *entry, char *dst, FILE *fp);

  This function parses a file (SNDSCAPE.INI) for a left-hand string and,
  if found, writes its associated right-hand value to a destination buffer.
  This function is case-insensitive.

  INPUTS:
	fp  - a file pointer to the open SNDSCAPE.INI config file
	dst - the destination buffer pointer
	lhp - a pointer to the right-hand string

  RETURNS:
	0   - if successful
	-1  - if the right-hand string is not found or has no equate
*/

short GetConfigEntry(char *entry, char *dest, FILE *fp) {
	char	 static str[83];
	char	 static tokstr[33];
	char	*p;

	/* make a local copy of the entry, upper-case it */
	strcpy(tokstr, entry);
	strupr(tokstr);

	/* rewind the file and try to find it ... */
	rewind(fp);
	for( ;; ) {
	    	/* get the next string from the file */
		fgets(str, 83, fp);
		if( feof(fp) )
			return -1;

		/* properly terminate the string */
		for( p = str; *p != '\0'; ++p ) {
			if( *p == ' ' || *p == '\t' || *p == 0x0a || *p == 0x0d ) {
				*p = '\0';
				break;
			}
		}

		/* see if it's an 'equate' string; if so, zero the '=' */
		if( !(p = strchr(str, '=')) )
			continue;
		*p = '\0';

		/* upper-case the current string and test it */
		strupr(str);
		if( strcmp(str, tokstr) )
			continue;

		/* it's our string - copy the right-hand value to buffer */
		for( p = str + strlen(str) + 1; (*dest++ = *p++) != '\0'; )
			;
		break;
	}

	return 0;
}


/*
  SetFormat(WORD srate, short stereo, short size16bit, short direction);

  This function sets the CoDec audio data format for record or playback.

  INPUTS:
  	srate      - the audio sample rate in Hertz
	stereo     - 0 for mono data, 1 for L/R interleaved stereo data
	size16bit  - 0 for 8-bit unsigned data, 1 for 16-bit signed data
	direction  - 0 for playback, 1 for record

  RETURNS:
	0          - if successful
	-1         - if the sample rate is unacceptable
*/

short SetFormat(WORD srate, short stereo, short size16bit, short direction) {
	short   format;
	DWORD   i;

	/* init the format register value */
	format = 0;

	/* first, find the sample rate ... */
	switch(srate) {
	 case  5512U:
		format = 0x01;
		break;
	 case  6615U:
		format = 0x0f;
		break;
	 case  8000U:
		format = 0x00;
		break;
	 case  9600U:
		format = 0x0e;
		break;
	 case 11025U:
		format = 0x03;
		break;
	 case 16000U:
		format = 0x02;
		break;
	 case 18900U:
		format = 0x05;
		break;
	 case 22050U:
		format = 0x07;
		break;
	 case 27428U:
		format = 0x04;
		break;
	 case 32000U:
		format = 0x06;
		break;
	 case 33075U:
		format = 0x0d;
		break;
	 case 37800U:
		format = 0x09;
		break;
	 case 44100U:
		format = 0x0b;
		break;
	 case 48000U:
		format = 0x0c;
		break;
	 default:
		return -1;
	}

	/* set other format bits ... */
	if( stereo )
		format |= 0x10;
	if( size16bit )
		format |= 0x40;

	/* make sure the the CoDec is in mode change state */
	outp(WavePort + CD_ADDR_OFF, 0x40);

	/* and write the format register */
	CdWrite(CD_FORMAT_REG, format);

	/* if not using ODIE and recording, setup extended format register */
	if( IcType != ODIE && direction )
		CdWrite(CD_XFORMAT_REG, format & 0x70);

	/* delay for internal re-synch */
	for( i = 0; i < 200000UL; ++i )
		inp(BasePort + GA_ADDR_OFF);

	return 0;
}


/*
  DWORD DosMalloc(DWORD size);

  This function allocates DOS memory.

  INPUTS:
        size - size, in bytes, of memory required

  RETURNS:
	0        - if the requested amount could not be allocated
	SEL:SEG  - if the allocation was successful
	           SEL is a 16-bit protected-mode selector value
	           SEG is a 16-bit DOS segment value
*/

DWORD DosMalloc(DWORD size) {

	union REGS   r;

	r.w.bx = (WORD) ((size + 15) >> 4);
	r.w.ax = 0x0100;
	int386(0x31, &r, &r);

	if( r.w.cflag )
		return (void *) 0;
	return (r.x.edx << 16) | r.w.ax;
}


/*
  void DosFree(WORD selector);

  This function frees previously allocated DOS memory.

  INPUTS:
        selector  - the 16-bit selector for the allocated memory

  RETURNS:
*/

void DosFree(WORD selector) {

	union REGS   r;

	r.w.dx = selector;
	r.w.ax = 0x0101;
	int386(0x31, &r, &r);

	return;
}
