
//------------------------------------------------------------------------------
// File: Bdasampl.cpp
//
// Desc: Sample code implementing BDA DVB-S graph building.
//
// 
// Copyright (c) 2003, Conexant Sytems, Inc. All rights reserved.
//------------------------------------------------------------------------------
#include "bdasampl.h"
#include "graph.h"
#include "resource.h"

#include <initguid.h>
#include <objbase.h>


// 
// NOTE: In this sample, text strings are hard-coded for 
// simplicity and for readability.  For product code, you should
// use string tables and LoadString().
//

// Global data
HWND      hwndMain=0;
HWND      g_hwndDlg=0;
HWND      g_hwndDlg_Setting=0;
HINSTANCE hInst=0;
TCHAR     szAppName[]  = TEXT("BDASampl\0");
TCHAR     szAppTitle[] = TEXT("DVB-S BDA CI Sample\0");

CBDAFilterGraph* g_pfg = NULL;

// Signal Status Thread
DWORD  dwThreadID;
HANDLE hThread = INVALID_HANDLE_VALUE;

INT WINAPI
WinMain(
    HINSTANCE hInstance,
    HINSTANCE hPrevInstance,
    LPSTR lpszCmdLine,
    INT nCmdShow)
{
    MSG      msg={0};
    WNDCLASS wndclass={0};
    HWND     hwnd=0;
    HACCEL   hAccel=0;

    // Save application instance handle for later use
    hInst = hInstance;

    // Initialize COM library
    HRESULT hr = CoInitializeEx (NULL, COINIT_APARTMENTTHREADED);
    if (FAILED (hr))
    {
        ErrorMessageBox(TEXT("Failed to initialize COM library!\0"));
        return 0;
    }

    // Register window class
    wndclass.style         = 0;
    wndclass.lpfnWndProc   = WndProc;
    wndclass.cbClsExtra    = 0;
    wndclass.cbWndExtra    = 0;
    wndclass.hInstance     = hInst;
    wndclass.hIcon         = LoadIcon(hInst, TEXT("BDASAMPLICON"));
    wndclass.hCursor       = LoadCursor(NULL, IDC_ARROW);
    wndclass.hbrBackground = (HBRUSH)(COLOR_WINDOW + 2);
    wndclass.lpszMenuName  = szAppName;
    wndclass.lpszClassName = szAppName;
    RegisterClass(&wndclass);

    hwnd = CreateWindow(szAppName, szAppTitle, 
                WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN, 
                200, 200, 640, 480, 
                NULL, NULL, hInst, NULL);
    assert(hwnd);

    // Create the BDA filter graph and initialize its components
    if (!g_pfg)
    {
        g_pfg = new CBDAFilterGraph();
        // If the graph failed to build, don't go any further.
        if (!g_pfg)
        {
            ErrorMessageBox(TEXT("Failed to create the filter graph!"));
            return 0;
        }
    }

    // Display main window and configure accelerators    
    ShowWindow(hwnd, nCmdShow);
    hwndMain = hwnd;

    hAccel = LoadAccelerators(hInst, MAKEINTRESOURCE(ACC_GRAPH));

    // Main message loop
    while(GetMessage(&msg, NULL, 0, 0) > 0)
    {
        if(!TranslateAccelerator(hwnd, hAccel, &msg))
        {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
    }

    // Release the BDA components and clean up
    if ( g_pfg )
    {
        delete g_pfg;
        g_pfg = NULL;
    }

    CoUninitialize ();
    
    return msg.wParam;
}


// WndProc                                                                    
LRESULT CALLBACK
WndProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch(message)
    {
        case WM_CREATE:
        {
            DbgInitialise (hInst);
            break;
        }

        case WM_SIZE:
        {
            if(g_pfg && g_pfg->m_fGraphBuilt)
            {
                g_pfg->SetVideoWindow(hwndMain);
            }
            break;
        }
        
        case WM_COMMAND:
            switch(LOWORD(wParam))
            {
                case IDM_BUILD_DVBT:
                {
                    if (!g_pfg)
                        break;
                    
                    // this may take a while, put up hourglass cursor 
                    SetCursor(LoadCursor(NULL, IDC_WAIT));

                    if (FAILED(g_pfg->BuildGraph(L"MYDVB-S"))) {
                        ErrorMessageBox(TEXT("Could not Build the DVB-S BDA FilterGraph."));
                    }
                    else {
                        g_pfg->SetVideoWindow(hwndMain);
                    }
        
                    HMENU hMenu = GetSubMenu(GetMenu(hwnd), 1);

                    EnableMenuItem(hMenu, IDM_BUILD_DVBT, MF_GRAYED | MF_BYCOMMAND);

                    EnableMenuItem(hMenu, IDM_STOP_GRAPH, 
                        (g_pfg->m_fGraphRunning) ? (MF_BYCOMMAND|MF_ENABLED) : (MF_BYCOMMAND|MF_GRAYED));

                    EnableMenuItem(hMenu, IDM_SELECT_CHANNEL, 
                        (g_pfg->m_fGraphBuilt) ? (MF_BYCOMMAND|MF_ENABLED) : (MF_BYCOMMAND|MF_GRAYED));

                    // set cursor back
                    SetCursor(LoadCursor(NULL, IDC_ARROW));

                    break;
                }
            
                case IDM_RUN_GRAPH:
                {
                    if (!g_pfg)
                        break;
                    
                    if(g_pfg->m_fGraphBuilt)
                    {   
                        if(!g_pfg->m_fGraphRunning)
                        {
                            // this may take a while, put up hourglass cursor 
                            SetCursor(LoadCursor(NULL, IDC_WAIT));

                            if(FAILED(g_pfg->RunGraph()))
                            {
                                ErrorMessageBox(TEXT("Could not play the FilterGraph."));
                            }

                            // set cursor back
                            SetCursor(LoadCursor(NULL, IDC_ARROW));
                        }
                    }
                    else
                    {
                        ErrorMessageBox(TEXT("The FilterGraph is not yet built."));
                    }

					HMENU hMenu = GetSubMenu (GetMenu (hwnd), 1);
					EnableMenuItem(hMenu, IDM_SETTING, 
                        (g_pfg->m_fGraphRunning) ? (MF_BYCOMMAND|MF_GRAYED) : (MF_BYCOMMAND|MF_ENABLED));

					
                    break;
                }
            
                case IDM_STOP_GRAPH:
                {
                    if (!g_pfg)
                        break;
                    
                    if(g_pfg->m_fGraphBuilt)
                    {
                        if(g_pfg->m_fGraphRunning)
                        {
                            if(FAILED(g_pfg->StopGraph()))
                            {
                                ErrorMessageBox(TEXT("Could not stop the FilterGraph,"));
                            }
                        }
                    }
                    else
                    {
                        ErrorMessageBox(TEXT("The FilterGraph is not yet built."));
                    }

                    HMENU hMenu = GetSubMenu (GetMenu (hwnd), 1);
                    EnableMenuItem (hMenu, IDM_SELECT_CHANNEL, MF_BYCOMMAND | MF_GRAYED);
					EnableMenuItem(hMenu, IDM_SETTING, 
                        (g_pfg->m_fGraphRunning) ? (MF_BYCOMMAND|MF_GRAYED) : (MF_BYCOMMAND|MF_ENABLED));
                    break;
                }
            
                case IDM_SELECT_CHANNEL:
                {
                    if (!g_pfg)
                        break;
                    
                    if(g_pfg->m_fGraphBuilt)
                    {
                        g_hwndDlg = reinterpret_cast <HWND> ( DialogBox(
                            hInst, 
                            MAKEINTRESOURCE(IDD_TERRESTRIAL_PARAMS),
                            hwnd, 
                            reinterpret_cast<DLGPROC>(SelectChannelDlgProc)) );
                    }
                    else
                    {
                        ErrorMessageBox(TEXT("The FilterGraph is not yet built."));
                    }

                    g_pfg->SetVideoWindow(hwndMain);

                    break;
                }
            
				case IDM_SETTING:
				{
					if (!g_pfg)
                        break;
                    
                    if (g_pfg->m_fGraphBuilt)
                    {
                        if(!g_pfg->m_fGraphRunning)
                        {
							g_hwndDlg_Setting = reinterpret_cast <HWND> ( DialogBox(
								hInst, 
								MAKEINTRESOURCE(IDD_SETTING_PARAMS),
								hwnd, 
								reinterpret_cast<DLGPROC>(SettingDlgProc)) );                            
                        }
                    }
                    else
                    {
                        ErrorMessageBox(TEXT("The FilterGraph is not yet built."));
					}

					break;
				}

                case IDM_ABOUT:
                {
                    DialogBox(hInst, MAKEINTRESOURCE(IDD_ABOUT), hwnd, 
                              reinterpret_cast<DLGPROC>(AboutDlgProc));

                    g_pfg->SetVideoWindow(hwndMain);

                    break;
                }
            
                case IDM_EXIT:
                {
                    PostMessage(hwnd, WM_CLOSE, 0, 0);
                    break;
                }
            
            default:
                break;
        }

    case WM_INITMENU:
        if (!g_pfg)
            break;
                    
        if(g_pfg->m_fGraphFailure)
        {
            EnableMenuItem((HMENU)wParam, IDM_BUILD_DVBT, MF_BYCOMMAND| MF_GRAYED);
            EnableMenuItem((HMENU)wParam, IDM_RUN_GRAPH, MF_BYCOMMAND| MF_GRAYED);
            EnableMenuItem((HMENU)wParam, IDM_STOP_GRAPH, MF_BYCOMMAND| MF_GRAYED);
        }
        else
        {
            EnableMenuItem((HMENU)wParam, IDM_RUN_GRAPH, 
                g_pfg->m_fGraphRunning ? MF_BYCOMMAND|MF_GRAYED : MF_BYCOMMAND|MF_ENABLED);

            EnableMenuItem((HMENU)wParam, IDM_BUILD_DVBT, 
                g_pfg->m_fGraphBuilt ? MF_BYCOMMAND|MF_GRAYED : MF_BYCOMMAND|MF_ENABLED);
        
            // we can stop viewing if it's currently viewing
            EnableMenuItem((HMENU)wParam, IDM_STOP_GRAPH, 
                (g_pfg->m_fGraphRunning) ? MF_BYCOMMAND|MF_ENABLED : MF_BYCOMMAND|MF_GRAYED);

        }

        EnableMenuItem((HMENU)wParam, IDM_SELECT_CHANNEL, 
            g_pfg->m_fGraphBuilt ? MF_BYCOMMAND|MF_ENABLED : MF_BYCOMMAND|MF_GRAYED);

        break;

    case WM_CLOSE:
        DbgTerminate();
        PostQuitMessage(0);
        break;

    default:
        break;
    }

    return DefWindowProc(hwnd, message, wParam, lParam);
}


DWORD WINAPI
SignalStrengthAndQualityThread(LPVOID pParam)
{
    HWND hDlg = (HWND)pParam;

	CoInitialize(NULL);

    BOOL bResult;

    for ( ;; )
    {
        LONG lSignalStrength;
        bResult = g_pfg->GetSignalStrength(&lSignalStrength);
        if ( bResult == FALSE )
        {
			SetDlgItemText(hDlg, IDC_EDIT_SIGNAL_STRENGTH, "error");
        }
		else
		{
			SetDlgItemInt(hDlg, IDC_EDIT_SIGNAL_STRENGTH, lSignalStrength, TRUE);
		}

        LONG lSignalQuality;
        bResult = g_pfg->GetSignalQuality(&lSignalQuality);
        if ( bResult == FALSE )
        {
			SetDlgItemText(hDlg, IDC_EDIT_SIGNAL_QUALITY, "error");
        }
		else
		{
	        SetDlgItemInt(hDlg, IDC_EDIT_SIGNAL_QUALITY, lSignalQuality, TRUE);
		}

        Sleep(250); // 250 milli seconds
    }

	CoUninitialize();

    return 0;
}

// SelectChannelDlgProc
// Dialog Procedure for the "Select Channel" dialog box.
//                                                                              
BOOL CALLBACK
SelectChannelDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    if (!g_pfg)
        return FALSE;;    

    ULONG ulCarrierFrequency = g_pfg->GetCarrierFrequency();
    //ULONG ONID               = g_pfg->GetONID();
    //ULONG TSID               = g_pfg->GetTSID();
    //ULONG SID                = g_pfg->GetSID();  

    BOOL bRet = FALSE;

    switch(message)
    {
        case WM_INITDIALOG:
        {
            //refresh the controls
            SetDlgItemInt(hDlg, IDC_EDIT_CARRIER_FREQ, ulCarrierFrequency, TRUE);

            // setup the Carrier freq. spin button control
            HWND hEdit = GetDlgItem( hDlg, IDC_EDIT_CARRIER_FREQ );
            HWND hSpin = GetDlgItem( hDlg, IDC_SPIN_CARRIER_FREQ );
            SendMessage( hSpin, UDM_SETBUDDY, (WPARAM) hEdit, 0 );
            SendMessage( hSpin, UDM_SETRANGE32, SMALLEST_FREQ, LARGEST_FREQ );
            SendMessage( hSpin, UDM_SETPOS, 0, 0 );
            SendMessage( hEdit, EM_LIMITTEXT, 8, 0 );
            SetDlgItemInt(hDlg, IDC_EDIT_CARRIER_FREQ, ulCarrierFrequency, TRUE);

            LONG lSignalStrength;
            BOOL bResult = g_pfg->GetSignalStrength(&lSignalStrength);
            if ( bResult )
            {
                SetDlgItemInt(hDlg, IDC_EDIT_SIGNAL_STRENGTH, lSignalStrength, TRUE);
            }

            LONG lSignalQuality;
            bResult = g_pfg->GetSignalQuality(&lSignalQuality);
            if ( bResult )
            {
                SetDlgItemInt(hDlg, IDC_EDIT_SIGNAL_QUALITY, lSignalQuality, TRUE);
            }
            
            break;
        }

        case WM_DESTROY:
        {
            if (hThread && hThread != INVALID_HANDLE_VALUE)
            {
			    TerminateThread(hThread, 0);
                CloseHandle(hThread);
                hThread = NULL;
            }

            EndDialog (hDlg, 0);
            return TRUE;
        }

        case WM_COMMAND:
        {
            switch(LOWORD(wParam))
            {
                case IDOK:
                {
                    // set the carrier frequency from the user selection
                    SetCarrierFrequency(hDlg);

                    // set the new channel/frequency
                    g_pfg->ChangeChannel ();

                    break;
                }

                case IDC_CHECK_MAP_PIDS:
                {
                    if (!g_pfg)
                        break;

                    if(g_pfg->m_fGraphRunning)
                    {
                        DialogBox(hInst, MAKEINTRESOURCE(IDD_DEMUX_PIDS), hDlg, 
                              reinterpret_cast<DLGPROC>(DemuxPIDDlgProc));
                    }
                    else
                    {
                        ErrorMessageBox(TEXT("The FilterGraph is not yet running."));
                    }

                    CheckDlgButton(hDlg, IDC_CHECK_MAP_PIDS, BST_UNCHECKED);

                    break;
                }

                case IDC_UPDATE:
                {
                    hThread = NULL;
                    hThread = CreateThread(NULL, 0, SignalStrengthAndQualityThread, (LPVOID)(hDlg), 0, &dwThreadID);

                    break;
                }

                case IDC_ENTER:
                {
                    // set the new channel/frequency
                    g_pfg->ChangeChannel ();

                    break;
                }

                case IDCANCEL:
                {
                    if (hThread)
                    {
		        	    TerminateThread(hThread, 0);
                        CloseHandle(hThread);
                        hThread = NULL;
                    }

                    EndDialog (hDlg, 0);

                    break;
                }
            }

            break;
        }
    }
    return FALSE;
}

                               
BOOL CALLBACK
SettingDlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    if (!g_pfg)
        return FALSE;;    

    ULONG ulCarrierFrequency = g_pfg->GetCarrierFrequency();
    //ULONG ONID               = g_pfg->GetONID();
    //ULONG TSID               = g_pfg->GetTSID();
    //ULONG SID                = g_pfg->GetSID();  

    BOOL bRet = FALSE;

    switch(message)
    {
        case WM_INITDIALOG:
        {
            //refresh the controls			
			HWND hDiSEqC = GetDlgItem( hDlg, IDC_LIST_DiSEqC );
			SendMessage( hDiSEqC, LB_ADDSTRING, 0, (LPARAM) TEXT("None"));
			SendMessage( hDiSEqC, LB_ADDSTRING, 0, (LPARAM) TEXT("A"));
			SendMessage( hDiSEqC, LB_ADDSTRING, 0, (LPARAM) TEXT("B"));
			SendMessage( hDiSEqC, LB_ADDSTRING, 0, (LPARAM) TEXT("C"));
			SendMessage( hDiSEqC, LB_ADDSTRING, 0, (LPARAM) TEXT("D"));
			SendMessage( hDiSEqC, LB_SETCURSEL, (WPARAM)4, 0);

			HWND hLNB_POWER = GetDlgItem( hDlg, IDC_LIST_LNB_POWER );
			SendMessage( hLNB_POWER, LB_ADDSTRING, 0, (LPARAM) TEXT("OFF"));
			SendMessage( hLNB_POWER, LB_ADDSTRING, 0, (LPARAM) TEXT("ON"));			
			SendMessage( hLNB_POWER, LB_SETCURSEL, (WPARAM)1, 0);

			HWND hLNB_TYPE = GetDlgItem( hDlg, IDC_LIST_LNB_TYPE );
			SendMessage( hLNB_TYPE, LB_ADDSTRING, 0, (LPARAM) TEXT("Set by AP"));
			SendMessage( hLNB_TYPE, LB_ADDSTRING, 0, (LPARAM) TEXT("5150_5750"));
			SendMessage( hLNB_TYPE, LB_ADDSTRING, 0, (LPARAM) TEXT("9750_10600"));
			SendMessage( hLNB_TYPE, LB_ADDSTRING, 0, (LPARAM) TEXT("Custom"));
			SendMessage( hLNB_TYPE, LB_SETCURSEL, (WPARAM)3, 0);

			SetDlgItemInt(hDlg, IDC_EDIT_LNB_LF, 0, TRUE);
			SetDlgItemInt(hDlg, IDC_EDIT_LNB_HF, 11300, TRUE);

			HWND h22KTONE = GetDlgItem( hDlg, IDC_LIST_22KTONE );
			SendMessage( h22KTONE, LB_ADDSTRING, 0, (LPARAM) TEXT("Set by AP"));
			SendMessage( h22KTONE, LB_ADDSTRING, 0, (LPARAM) TEXT("OFF"));
			SendMessage( h22KTONE, LB_ADDSTRING, 0, (LPARAM) TEXT("ON"));			
			SendMessage( h22KTONE, LB_SETCURSEL, (WPARAM)1, 0);

			HWND hPolarisation = GetDlgItem( hDlg, IDC_LIST_Polarisation );			
			SendMessage( hPolarisation, LB_ADDSTRING, 0, (LPARAM) TEXT("H"));
			SendMessage( hPolarisation, LB_ADDSTRING, 0, (LPARAM) TEXT("V"));			
			SendMessage( hPolarisation, LB_SETCURSEL, (WPARAM)1, 0);

			SetDlgItemInt(hDlg, IDC_EDIT_FREQ, 12615000, TRUE);
			SetDlgItemInt(hDlg, IDC_EDIT_SAMPLE_RATE, 22425, TRUE);       
            
            break;
        }

        case WM_DESTROY:
        {
            if (hThread && hThread != INVALID_HANDLE_VALUE)
            {
			    TerminateThread(hThread, 0);
                CloseHandle(hThread);
                hThread = NULL;
            }

            EndDialog (hDlg, 0);
            return TRUE;
        }

        case WM_COMMAND:
        {
            switch(LOWORD(wParam))
            {
                case IDOK:
                {                   
					TCHAR tachTempBuffer[20];
					TCHAR tachBuffer[20];
					INT	  iIndex;

					if (!GetDlgItemText(hDlg, IDC_EDIT_FREQ, tachTempBuffer, sizeof(tachTempBuffer)/sizeof(tachTempBuffer[0])))
					{
						ErrorMessageBox(TEXT("Get IDC_EDIT_FREQ error !!!!"));
						EndDialog (hDlg, 0);
						break;
					}
					else
					{
						//RemoveCommas(tachTempBuffer, tachBuffer);
						g_pfg->m_ulCarrierFrequency = _ttol(tachTempBuffer);
					}

					if (!GetDlgItemText(hDlg, IDC_EDIT_SAMPLE_RATE, tachTempBuffer, sizeof(tachTempBuffer)/sizeof(tachTempBuffer[0])))
					{
						ErrorMessageBox(TEXT("Get IDC_EDIT_SAMPLE_RATE error !!!!"));
						EndDialog (hDlg, 0);
						break;
					}
					else
					{
						//RemoveCommas(tachTempBuffer, tachBuffer);
						g_pfg->m_ulSymbolRate = _ttoi(tachTempBuffer);
					}

					if (!GetDlgItemText(hDlg, IDC_EDIT_LNB_LF, tachTempBuffer, sizeof(tachTempBuffer)/sizeof(tachTempBuffer[0])))
					{
						ErrorMessageBox(TEXT("Get IDC_EDIT_LNB_LF error !!!!"));
						EndDialog (hDlg, 0);
						break;
					}
					else
					{
						RemoveCommas(tachTempBuffer, tachBuffer);
						g_pfg->m_LNB_LOF = _ttol(tachTempBuffer);
					}

					if (!GetDlgItemText(hDlg, IDC_EDIT_LNB_HF, tachTempBuffer, sizeof(tachTempBuffer)/sizeof(tachTempBuffer[0])))
					{
						ErrorMessageBox(TEXT("Get IDC_EDIT_LNB_HF error !!!!"));
						EndDialog (hDlg, 0);
						break;
					}
					else
					{
						RemoveCommas(tachTempBuffer, tachBuffer);
						g_pfg->m_LNB_HOF = _ttol(tachTempBuffer);
					}

					HWND hDiSEqC = GetDlgItem( hDlg, IDC_LIST_DiSEqC );
					iIndex = SendMessage(hDiSEqC, LB_GETCURSEL, 0, 0);					
					g_pfg->m_DISEQC = (DISEQC)iIndex;

					HWND hLNB_POWER = GetDlgItem( hDlg, IDC_LIST_LNB_POWER );
					iIndex = SendMessage(hLNB_POWER, LB_GETCURSEL, 0, 0);					
					g_pfg->m_LNB_POWER = (LNB_POWER)iIndex;

					HWND hLNB_TYPE = GetDlgItem( hDlg, IDC_LIST_LNB_TYPE );
					iIndex = SendMessage(hLNB_TYPE, LB_GETCURSEL, 0, 0);					
					g_pfg->m_LNB_TYPE = (LNB_TYPE)iIndex;

					HWND h22KTONE = GetDlgItem( hDlg, IDC_LIST_22KTONE );
					iIndex = SendMessage(h22KTONE, LB_GETCURSEL, 0, 0);					
					g_pfg->m_K22_Tone = (K22_Tone)iIndex;

					HWND hPolarisation = GetDlgItem( hDlg, IDC_LIST_Polarisation );
					iIndex = SendMessage(hPolarisation, LB_GETCURSEL, 0, 0);					
					if (0 == iIndex) {
						g_pfg->m_SignalPolarisation = BDA_POLARISATION_LINEAR_H;
					}
					else {
						g_pfg->m_SignalPolarisation = BDA_POLARISATION_LINEAR_V;
					}
										
                    // set the new settings
                    g_pfg->ChangeSetting();
					
					EndDialog (hDlg, 0);
                    break;
                }

                case IDCANCEL:
                {
                    if (hThread)
                    {
		        	    TerminateThread(hThread, 0);
                        CloseHandle(hThread);
                        hThread = NULL;
                    }

                    EndDialog (hDlg, 0);

                    break;
                }
				
                break;
            }

            break;
        }

		break;
    }
    return FALSE;
}

static VOID
SetCarrierFrequency(HWND hDlg)
{
    LONG lCarrierFreq;

    TCHAR tachTempBuffer[20];
    TCHAR tachBuffer[20];

    if (!GetDlgItemText(hDlg, IDC_EDIT_CARRIER_FREQ, tachTempBuffer, sizeof(tachTempBuffer)/sizeof(tachTempBuffer[0])))
    {
        ErrorMessageBox(TEXT("Carrier Frequency error !!!!"));
        return;
    }
    else
    {
        RemoveCommas(tachTempBuffer, tachBuffer);

        lCarrierFreq = _ttol(tachBuffer);
    }

    if ( lCarrierFreq < SMALLEST_FREQ )
    {
        ErrorMessageBox(TEXT("Carrier Frequency must be larger than 0 !!!!"));
    }
    else if ( lCarrierFreq > LARGEST_FREQ )
    {
        ErrorMessageBox(TEXT("Carrier Frequency must be less than 999999 !!!!"));
    }
    else
    {
        g_pfg->SetCarrierFrequency(lCarrierFreq);
    }

}

static VOID
SetVideoPID(HWND hDlg)
{
    LONG lPID;

    TCHAR tachBuffer[10]={0};
    TCHAR tachTempBuffer[10]={0};
    
    if (!GetDlgItemText(hDlg, IDC_EDIT_VIDEO_PID, tachTempBuffer, sizeof(tachTempBuffer)/sizeof(tachTempBuffer[0])))
    {
        ErrorMessageBox(TEXT("Video PID value error !!!!"));
        return;
    }
    else
    {
        RemoveCommas(tachTempBuffer, tachBuffer);

        lPID = _ttol(tachBuffer);
    }
    
    if ( lPID < SMALLEST_MPEG_PID || lPID > LARGEST_MPEG_PID )
    {
        ErrorMessageBox(TEXT("Video PID value error, must be between 0 - 8191 !!!!"));
    }
    else
    {
        g_pfg->SetVideoPID(lPID);
    }

}

static VOID
SetAudioPID(HWND hDlg)
{
    LONG lPID;
    
    TCHAR tachBuffer[10]={0};
    TCHAR tachTempBuffer[10]={0};
    
    if (!GetDlgItemText(hDlg, IDC_EDIT_AUDIO_PID, tachTempBuffer, sizeof(tachTempBuffer)/sizeof(tachTempBuffer[0])))
    {
        ErrorMessageBox(TEXT("Audio PID value error !!!!"));
        return;
    }
    else
    {
        RemoveCommas(tachTempBuffer, tachBuffer);

        lPID = _ttol(tachBuffer);
    }
    
    if ( lPID < SMALLEST_MPEG_PID || lPID > LARGEST_MPEG_PID )
    {
        ErrorMessageBox(TEXT("Audio PID value error, must be between 0 - 8191 !!!!"));
    }
    else
    {
        g_pfg->SetAudioPID(lPID);
    }

}

static VOID
SetPMTPID(HWND hDlg)
{
    LONG lPID;
    
    TCHAR tachBuffer[10]={0};
    TCHAR tachTempBuffer[10]={0};
    
    if (!GetDlgItemText(hDlg, IDC_EDIT_PMT_PID, tachTempBuffer, sizeof(tachTempBuffer)/sizeof(tachTempBuffer[0])))
    {
        ErrorMessageBox(TEXT("PMT PID value error !!!!"));
        return;
    }
    else
    {
        RemoveCommas(tachTempBuffer, tachBuffer);

        lPID = _ttol(tachBuffer);
    }
    
    if ( lPID < SMALLEST_MPEG_PID || lPID > LARGEST_MPEG_PID )
    {
        ErrorMessageBox(TEXT("Audio PID value error, must be between 0 - 8191 !!!!"));
    }
    else
    {
        g_pfg->SetTHParserPID(lPID);
    }

}

//
// RemoveCommas
// Takes a buffer with commas and creates new comma free buffer.  When
// you use spin button control in a dialog the value returned by
// GetDlgItemText has commas in it.  So this routine removes the commas.
//
static VOID
RemoveCommas(TCHAR* pCommaBuff, TCHAR* pCommaFreeBuff)
{
    INT i, j;

    i = j = 0;

    while ( *pCommaBuff != NULL )
    {
        if ( *pCommaBuff != ',' )
        {
            *pCommaFreeBuff++ = *pCommaBuff;
        }

        pCommaBuff++;
    }

}



// DemuxPIDDlgProc
//
// Dialog Procedure for the "Demux PID Selection" dialog box.
//
BOOL CALLBACK 
DemuxPIDDlgProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
    if (!g_pfg)
        return FALSE;

    LONG lVideoPID = g_pfg->GetVideoPID();
    LONG lAudioPID = g_pfg->GetAudioPID();
	LONG lTHParserPID = g_pfg->GetTHParserPID();

    switch(msg) 
    {
        case WM_DESTROY:
        {
            EndDialog (hDlg, 0);
            return TRUE;
        }

        case WM_INITDIALOG:
        {
            // setup the Video PID spin button control
            HWND hEdit = GetDlgItem( hDlg, IDC_EDIT_VIDEO_PID );
            HWND hSpin = GetDlgItem( hDlg, IDC_SPIN_VIDEO_PID );
            SendMessage( hSpin, UDM_SETBUDDY, (WPARAM) hEdit, 0 );
            SendMessage( hSpin, UDM_SETRANGE32, 0, 8191 ); // 0x1fff
            SendMessage( hSpin, UDM_SETPOS, 0, 0 );
            SendMessage( hEdit, EM_LIMITTEXT, 8, 0 );
            SetDlgItemInt(hDlg, IDC_EDIT_VIDEO_PID, lVideoPID, TRUE);

            // setup the Audio PID spin button control
            hEdit = GetDlgItem( hDlg, IDC_EDIT_AUDIO_PID );
            hSpin = GetDlgItem( hDlg, IDC_SPIN_AUDIO_PID );
            SendMessage( hSpin, UDM_SETBUDDY, (WPARAM) hEdit, 0 );
            SendMessage( hSpin, UDM_SETRANGE32, 0, 8191 ); // 0x1fff
            SendMessage( hSpin, UDM_SETPOS, 0, 0 );
            SendMessage( hEdit, EM_LIMITTEXT, 8, 0 );
            SetDlgItemInt(hDlg, IDC_EDIT_AUDIO_PID, lAudioPID, TRUE);

			// setup the CAPMT PID spin button control
            hEdit = GetDlgItem( hDlg, IDC_EDIT_PMT_PID );
            hSpin = GetDlgItem( hDlg, IDC_SPIN_PMT_PID );
            SendMessage( hSpin, UDM_SETBUDDY, (WPARAM) hEdit, 0 );
            SendMessage( hSpin, UDM_SETRANGE32, 0, 8191 ); // 0x1fff
            SendMessage( hSpin, UDM_SETPOS, 0, 0 );
            SendMessage( hEdit, EM_LIMITTEXT, 8, 0 );
            SetDlgItemInt(hDlg, IDC_EDIT_PMT_PID, lTHParserPID, TRUE);

            //refresh the controls
            EnableWindow(GetDlgItem(hDlg,IDC_EDIT_VIDEO_PID), TRUE);
            EnableWindow(GetDlgItem(hDlg,IDC_EDIT_AUDIO_PID), TRUE);
			EnableWindow(GetDlgItem(hDlg,IDC_EDIT_PMT_PID), TRUE);

            break;
        }

        case WM_COMMAND:
        {
            switch(LOWORD(wParam))
            {
                case IDOK:
                {
                    if (!g_pfg)
                        break;

                    // get the currently mapped PIDs
                    LONG lCurrVideoPID = g_pfg->GetVideoPID();
                    LONG lCurrAudioPID = g_pfg->GetAudioPID();
					LONG lCurrPMTPID = g_pfg->GetTHParserPID();

                    // format the new Video PID and set member var.
                    SetVideoPID(hDlg);

                    // format the new Audio PID and set member var.
                    SetAudioPID(hDlg);

					// format the new Audio PID and set member var.
                    SetPMTPID(hDlg);
                    
                    // get the new PIDs to map
                    LONG lNewVideoPID = g_pfg->GetVideoPID();
                    LONG lNewAudioPID = g_pfg->GetAudioPID();
					LONG lNewPMTPID = g_pfg->GetTHParserPID();

                    if ( lCurrVideoPID != lNewVideoPID || lCurrAudioPID != lNewAudioPID )
                    {
                        // un-map the old PIDS
                        g_pfg->UnmapVideoAndAudioPIDs(lCurrPMTPID, lCurrVideoPID, lCurrAudioPID);  
                    
                    }

                    //map the new PID values                    
					g_pfg->SetVideoAndAudioPIDs(lNewPMTPID, lNewVideoPID, lNewAudioPID);

                    break;
                }

                case IDCANCEL:
                {
                    EndDialog (hDlg, 0);
                    break;
                }

                default:
                    break;

            }
        }

        break;
        
    }

    return FALSE;

}


// AboutDlgProc
//
// Dialog Procedure for the "about" dialog box.
//
BOOL CALLBACK 
AboutDlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    switch(msg) 
    {
        case WM_COMMAND:
            EndDialog(hwnd, TRUE);
            return TRUE;
    
        case WM_INITDIALOG:
            return TRUE;
    }
    return FALSE;
}


// ErrorMessageBox
//
// Opens a Message box with a error message in it.  The user can     
// select the OK button to continue.
//
VOID
ErrorMessageBox(LPTSTR szFormat, ...)
{
    static TCHAR szBuffer[2048];  // Large buffer
    const size_t NUMCHARS = sizeof(szBuffer) / sizeof(szBuffer[0]);
    const int LASTCHAR = NUMCHARS - 1;

    // Format the input string
    va_list pArgs;
    va_start(pArgs, szFormat);

    // Use a bounded buffer size to prevent buffer overruns.  Limit count to
    // character size minus one to allow for a NULL terminating character.
    _vsntprintf(szBuffer, NUMCHARS - 1, szFormat, pArgs);
    va_end(pArgs);

    // Ensure that the formatted string is NULL-terminated
    szBuffer[LASTCHAR] = TEXT('\0');

    // Display a message box with the formatted string
    MessageBox(hwndMain, szBuffer, TEXT("Error!\0"), MB_OK|MB_ICONEXCLAMATION|MB_TASKMODAL);
}

