/*----------------------------------------------------------------------
 *
 *  curses.c
 *
 *  copyright (c) 1987,88,89,90 J. Alan Eldridge
 *
 *  main module for Curses! v. 3.X
 *  
 *----------------------------------------------------------------------
 */

#include <stdarg.h>

#include "curses.h"

#ifndef __GNUC__
#define GET_TTY_FROM_ENV    1   /* read env. var for video options */
#else
#define GET_TTY_FROM_ENV    0   /* don't read env. var for video options */
#endif

/* GLOBAL VARS */

WINDOW  *stdscr             =   NULL,   /* default window */
        *curscr             =   NULL;   /* screen image window */

int     LINES               =   25,     /* # rows on physical screen */
        COLS                =   80;     /* # cols on physical screen */

int     __USE_BIOS          =   0;      /* don't use BIOS */

int     __DEFNORM           =   VID_DEFNORM,  /* default "normal" attribute */
        __DEFSTAND          =   VID_DEFSTAND, /* default "standout" attribute */
        __DEFBRIGHT         =   VID_DEFBRIGHT,/* default "bright" attribute */
        __VIDMODE           =   VID_COLOR_80x25;    /* BIOS video mode */

MACHINE_ID      __MACHID    =   { 0xff, 0x00 };

FAR_PTR_UNION  __VIDADDR    =   { 0xb8000000L };

/* STATIC VARS */

static int  nlflag      =   1,  /* newline mapping toggle (don't turn off!) */
            echoflag    =   0,  /* echo toggle (don't turn on!) */
            rawflag     =   0,  /* raw i/o toggle (don't turn on!) */
            machid_set  =   0;  /* did user set machine type explicitly? */
            
/*  TTY CONTROL FUNCTIONS */

void
set_machine_id(id_ptr)
MACHINE_ID  *id_ptr;
{
    machid_set = 1;
    __MACHID = *id_ptr;
}

echo()
{
    echoflag = 1;
}

noecho()
{
    echoflag = 0;
}

raw()
{
    rawflag = 1;
}

noraw()
{
    rawflag = 0;
}

crmode()
{
}

nocrmode()
{
}

nl()
{
    nlflag = 1;
}

nonl()
{
    nlflag = 0;
}

/* CHARACTER INPUT */

wgetch(win)
WINDOW  *win;
{
    int c;

    c = kbgetc();

    if (!rawflag) {
        if (c == '\r' && nlflag)
            c = '\n';
        if (echoflag) {
            if (waddch(win, c) == ERR)
                return ERR;
            wrefresh(win);
        }
    }

    return c;
}

/* CHARACTER OUTPUT */

waddch(win, ch)
WINDOW  *win;
int     ch;
{
    int lry, lrx, cury, curx;

    getmaxrc(win, lry, lrx);
    getyx(win, cury, curx);

    if (ch == '\b') {
        if (curx > 0)
            curx--;
        else if (cury > 0 && (win->flags & _WWRAP)) {
            curx = lrx;
            cury--;
        } else
            return ERR;
    } else if (ch == '\n') {
        markwin(win);
        wclrtoeol(win);
        if (cury < lry) {
            if (nlflag)
                curx = 0;
            cury++;
        } else if (win->flags & _WSCROLL) {
            if (nlflag)
                curx = 0;
            scroll(win);
        } else
            return ERR;
    } else if (ch == '\r')
        curx = 0;
    else {
        markwin(win);
        if (curx <= lrx) {
            win->buf[cury][curx].chr = ch;
            win->buf[cury][curx].att = win->attrib;
            curx++;
            if ((win->flags & _WWRAP) && curx > lrx) {
                if (cury < lry) {
                    curx = 0;
                    cury++;
                } else if (win->flags & _WSCROLL) {
                    curx = 0;
                    scroll(win);
                }
            }
        } else
            return ERR;
    }

    win->cury = cury;
    win->curx = curx;
    if (ch != '\b')
        markwin(win);

    return (ch & 0xff);
}

/* POSITIONING WITHIN WINDOW */

wmove(win, r, c)
WINDOW  *win;
int     r, c;
{
    /* 
        WARNING: this function intentionally does
        no range checking on its arguments
    */

    win->cury = r;
    win->curx = c;
    return OK;
}

/* DISPLAY FATAL ERROR MESSAGE AND DIE */

void
curses_fatal_error(char *fmt, ...)
{
    char    cbuf[150];
    char    ubuf[100];
    va_list ap;
    
    sprintf(cbuf, "Curses! v.%d.%d: ", CURSES_MAJOR_VERSION,
        CURSES_MINOR_VERSION);

    va_start(ap, fmt);
    vsprintf(ubuf, fmt, ap);
    va_end(ap);

    strcat(cbuf, ubuf);
    strcat(cbuf, "\n");

    fputs(cbuf, stderr);
    exit(1);
}

/* STARTUP FUNCTION */

static  VID_SCR_BUFF    *scr_save_ptr   =   NULL;

int
initscr(void)
{
#if GET_TTY_FROM_ENV
    char *termval;
#endif	/* GET_TTY_FROM_ENV */
    
    __VIDADDR.longval = vid_buf_addr(__VIDMODE = vid_get_mode());
    if (__VIDADDR.longval == 0L)
        curses_fatal_error("can't use video mode %d", __VIDMODE);
 
    curscr = newwin(VID_MAX_ROWS, VID_MAX_COLS, 0, 0);

    if (curscr)
        stdscr = newwin(0, 0, 0, 0);
    else
        return ERR;

    if (!stdscr) {
        delwin(curscr);
        return ERR;
    }
    
#ifndef __GNUC__
    if (!machid_set)
        get_machine_id(&__MACHID);
#endif

    _kb_init();
	
#if GET_TTY_FROM_ENV
    termval = getenv("CURSES_TERM");
    if (termval && !stricmp(termval, "BIOS"))
        __USE_BIOS = 1;
#endif  /* GET_TTY_FROM_ENV */
    
    if (scr_save_ptr = malloc(sizeof(VID_SCR_BUFF)))
        vid_save_scr(scr_save_ptr);

    hidecursor();
    undcursor();
    showcursor();
    refresh();

    return OK;
}

/* SHUTDOWN FUNCTION */

endwin()
{
    delwin(stdscr);
    delwin(curscr);
    if (scr_save_ptr) {
        vid_rest_scr(scr_save_ptr);
        free(scr_save_ptr);
    }
    return OK;
}
