/* IMGJPEG.c
 *
 * Original file by the Independent JPEG Group's software (see copyright
 * bellow
 *
 * File adapted by Cedric JOULAIN, the SWORD Group
 * Copyright(C) 1995, The SWORD Group
 */

/*
 * cjpeg.c
 *
 * Copyright (C) 1991-1994, Thomas G. Lane.
 * This file is part of the Independent JPEG Group's software.
 * For conditions of distribution and use, see the accompanying README file.
 *
 * This file contains a command-line user interface for the JPEG compressor.
 * It should work on any system with Unix- or MS-DOS-style command lines.
 *
 * Two different command line styles are permitted, depending on the
 * compile-time switch TWO_FILE_COMMANDLINE:
 *	cjpeg [options]  inputfile outputfile
 *	cjpeg [options]  [inputfile]
 * In the second style, output is always to standard output, which you'd
 * normally redirect to a file or pipe to some other program.  Input is
 * either from a named file or from standard input (typically redirected).
 * The second style is convenient on Unix but is unhelpful on systems that
 * don't support pipes.  Also, you MUST use the first style if your system
 * doesn't do binary I/O to stdin/stdout.
 * To simplify script writing, the "-outfile" switch is provided.  The syntax
 *	cjpeg [options]  -outfile outputfile  inputfile
 * works regardless of which command line style is used.
 */

#include "cdjpeg.h"		/* Common decls for cjpeg/djpeg applications */
#define JMAKE_MSG_TABLE
#include "cderror.h"		/* create message string table */
#include "jversion.h"		/* for version message */

#include <ctype.h>		/* to declare isupper(), tolower() */
#ifdef USE_SETMODE
#include <fcntl.h>		/* to declare setmode()'s parameter macros */
/* If you have setmode() but not <io.h>, just delete this line: */
#include <io.h>			/* to declare setmode() */
#endif

#ifdef USE_CCOMMAND		/* command-line reader for Macintosh */
#ifdef __MWERKS__
#include <SIOUX.h>              /* Metrowerks declares it here */
#endif
#ifdef THINK_C
#include <console.h>		/* Think declares it here */
#endif
#endif

#include <setjmp.h>

#ifdef DONT_USE_B_MODE		/* define mode parameters for fopen() */
#define READ_BINARY	"r"
#define WRITE_BINARY	"w"
#else
#define READ_BINARY	"rb"
#define WRITE_BINARY	"wb"
#endif

#ifndef EXIT_FAILURE		/* define exit() codes if not provided */
#define EXIT_FAILURE  1
#endif
#ifndef EXIT_SUCCESS
#ifdef VMS
#define EXIT_SUCCESS  1		/* VMS is very nonstandard */
#else
#define EXIT_SUCCESS  0
#endif
#endif
#ifndef EXIT_WARNING
#ifdef VMS
#define EXIT_WARNING  1		/* VMS is very nonstandard */
#else
#define EXIT_WARNING  2
#endif
#endif

/* JC */

#define BYTE unsigned char

/*
 * ERROR HANDLING:
 *
 * The JPEG library's standard error handler (jerror.c) is divided into
 * several "methods" which you can override individually.  This lets you
 * adjust the behavior without duplicating a lot of code, which you might
 * have to update with each future release.
 *
 * Our example here shows how to override the "error_exit" method so that
 * control is returned to the library's caller when a fatal error occurs,
 * rather than calling exit() as the standard error_exit method does.
 *
 * We use C's setjmp/longjmp facility to return control.  This means that the
 * routine which calls the JPEG library must first execute a setjmp() call to
 * establish the return point.  We want the replacement error_exit to do a
 * longjmp().  But we need to make the setjmp buffer accessible to the
 * error_exit routine.  To do this, we make a private extension of the
 * standard JPEG error handler object.  (If we were using C++, we'd say we
 * were making a subclass of the regular error handler.)
 *
 * Here's the extended error handler struct:
 */

struct my_error_mgr {
  struct jpeg_error_mgr pub;	/* "public" fields */

  jmp_buf setjmp_buffer;	/* for return to caller */
};

typedef struct my_error_mgr * my_error_ptr;

/*
 * This routine determines what format the input file is,
 * and selects the appropriate input-reading module.
 *
 * To determine which family of input formats the file belongs to,
 * we may look only at the first byte of the file, since C does not
 * guarantee that more than one character can be pushed back with ungetc.
 * Looking at additional bytes would require one of these approaches:
 *     1) assume we can fseek() the input file (fails for piped input);
 *     2) assume we can push back more than one character (works in
 *        some C implementations, but unportable);
 *     3) provide our own buffering (breaks input readers that want to use
 *        stdio directly, such as the RLE library);
 * or  4) don't put back the data, and modify the input_init methods to assume
 *        they start reading after the start of file (also breaks RLE library).
 * #1 is attractive for MS-DOS but is untenable on Unix.
 *
 * The most portable solution for file types that can't be identified by their
 * first byte is to make the user tell us what they are.  This is also the
 * only approach for "raw" file types that contain only arbitrary values.
 * We presently apply this method for Targa files.  Most of the time Targa
 * files start with 0x00, so we recognize that case.  Potentially, however,
 * a Targa file could start with any byte value (byte 0 is the length of the
 * seldom-used ID field), so we provide a switch to force Targa input mode.
 */

static boolean is_targa;	/* records user -targa switch */


LOCAL cjpeg_source_ptr
select_file_type (j_compress_ptr cinfo, FILE * infile)
{
  int c;

  if (is_targa) {
#ifdef TARGA_SUPPORTED
	 return jinit_read_targa(cinfo);
#else
	 ERREXIT(cinfo, JERR_TGA_NOTCOMP);
#endif
  }

  if ((c = getc(infile)) == EOF)
	 ERREXIT(cinfo, JERR_INPUT_EMPTY);
  if (ungetc(c, infile) == EOF)
	 ERREXIT(cinfo, JERR_UNGETC_FAILED);

  switch (c) {
#ifdef BMP_SUPPORTED
  case 'B':
    return jinit_read_bmp(cinfo);
#endif
#ifdef GIF_SUPPORTED
  case 'G':
    return jinit_read_gif(cinfo);
#endif
#ifdef PPM_SUPPORTED
  case 'P':
    return jinit_read_ppm(cinfo);
#endif
#ifdef RLE_SUPPORTED
  case 'R':
	 return jinit_read_rle(cinfo);
#endif
#ifdef TARGA_SUPPORTED
  case 0x00:
    return jinit_read_targa(cinfo);
#endif
  default:
    ERREXIT(cinfo, JERR_UNKNOWN_FORMAT);
    break;
  }

  return NULL;			/* suppress compiler warnings */
}

/*
 * Argument-parsing code.
 * The switch parser is designed to be useful with DOS-style command line
 * syntax, ie, intermixed switches and file names, where only the switches
 * to the left of a given file name affect processing of that file.
 * The main program in this file doesn't actually use this capability...
 */


static const char * progname;	/* program name for error messages */
static char * outfilename;	/* for -outfile switch */


LOCAL void
usage (void)
/* complain about bad command line */
{
  fprintf(stderr, "usage: %s [switches] ", progname);
#ifdef TWO_FILE_COMMANDLINE
  fprintf(stderr, "inputfile outputfile\n");
#else
  fprintf(stderr, "[inputfile]\n");
#endif

  fprintf(stderr, "Switches (names may be abbreviated):\n");
  fprintf(stderr, "  -quality N     Compression quality (0..100; 5-95 is useful range)\n");
  fprintf(stderr, "  -grayscale     Create monochrome JPEG file\n");
#ifdef ENTROPY_OPT_SUPPORTED
  fprintf(stderr, "  -optimize      Optimize Huffman table (smaller file, but slow compression)\n");
#endif
#ifdef TARGA_SUPPORTED
  fprintf(stderr, "  -targa         Input file is Targa format (usually not needed)\n");
#endif
  fprintf(stderr, "Switches for advanced users:\n");
#ifdef DCT_ISLOW_SUPPORTED
  fprintf(stderr, "  -dct int       Use integer DCT method%s\n",
	  (JDCT_DEFAULT == JDCT_ISLOW ? " (default)" : ""));
#endif
#ifdef DCT_IFAST_SUPPORTED
  fprintf(stderr, "  -dct fast      Use fast integer DCT (less accurate)%s\n",
	  (JDCT_DEFAULT == JDCT_IFAST ? " (default)" : ""));
#endif
#ifdef DCT_FLOAT_SUPPORTED
  fprintf(stderr, "  -dct float     Use floating-point DCT method%s\n",
	  (JDCT_DEFAULT == JDCT_FLOAT ? " (default)" : ""));
#endif
  fprintf(stderr, "  -restart N     Set restart interval in rows, or in blocks with B\n");
#ifdef INPUT_SMOOTHING_SUPPORTED
  fprintf(stderr, "  -smooth N      Smooth dithered input (N=1..100 is strength)\n");
#endif
  fprintf(stderr, "  -maxmemory N   Maximum memory to use (in kbytes)\n");
  fprintf(stderr, "  -outfile name  Specify name for output file\n");
  fprintf(stderr, "  -verbose  or  -debug   Emit debug output\n");
  fprintf(stderr, "Switches for wizards:\n");
#ifdef C_ARITH_CODING_SUPPORTED
  fprintf(stderr, "  -arithmetic    Use arithmetic coding\n");
#endif
  fprintf(stderr, "  -baseline      Force baseline output\n");
#ifdef C_MULTISCAN_FILES_SUPPORTED
  fprintf(stderr, "  -nointerleave  Create noninterleaved JPEG file\n");
#endif
  fprintf(stderr, "  -qtables file  Use quantization tables given in file\n");
  fprintf(stderr, "  -qslots N[,...]    Set component quantization tables\n");
  fprintf(stderr, "  -sample HxV[,...]  Set component sampling factors\n");
  exit(EXIT_FAILURE);
}


LOCAL boolean
keymatch (char * arg, const char * keyword, int minchars)
/* Case-insensitive matching of (possibly abbreviated) keyword switches. */
/* keyword is the constant keyword (must be lower case already), */
/* minchars is length of minimum legal abbreviation. */
{
  register int ca, ck;
  register int nmatched = 0;

  while ((ca = *arg++) != '\0') {
	 if ((ck = *keyword++) == '\0')
		return FALSE;		/* arg longer than keyword, no good */
	 if (isupper(ca))		/* force arg to lcase (assume ck is already) */
		ca = tolower(ca);
	 if (ca != ck)
		return FALSE;		/* no good */
	 nmatched++;			/* count matched characters */
  }
  /* reached end of argument; fail if it's too short for unique abbrev */
  if (nmatched < minchars)
	 return FALSE;
  return TRUE;			/* A-OK */
}


LOCAL int
qt_getc (FILE * file)
/* Read next char, skipping over any comments (# to end of line) */
/* A comment/newline sequence is returned as a newline */
{
  register int ch;

  ch = getc(file);
  if (ch == '#') {
	 do {
		ch = getc(file);
	 } while (ch != '\n' && ch != EOF);
  }
  return ch;
}


LOCAL long
read_qt_integer (FILE * file)
/* Read an unsigned decimal integer from a quantization-table file */
/* Swallows one trailing character after the integer */
{
  register int ch;
  register long val;

  /* Skip any leading whitespace, detect EOF */
  do {
	 ch = qt_getc(file);
	 if (ch == EOF)
		return EOF;
  } while (isspace(ch));

  if (! isdigit(ch)) {
	 fprintf(stderr, "%s: bogus data in quantization file\n", progname);
	 exit(EXIT_FAILURE);
  }

  val = ch - '0';
  while (ch = qt_getc(file), isdigit(ch)) {
	 val *= 10;
	 val += ch - '0';
  }
  return val;
}


LOCAL void
read_quant_tables (j_compress_ptr cinfo, char * filename, int scale_factor,
			boolean force_baseline)
/* Read a set of quantization tables from the specified file.
 * The file is plain ASCII text: decimal numbers with whitespace between.
 * Comments preceded by '#' may be included in the file.
 * There may be one to NUM_QUANT_TBLS tables in the file, each of 64 values.
 * The tables are implicitly numbered 0,1,etc.
 * NOTE: does not affect the qslots mapping, which will default to selecting
 * table 0 for luminance (or primary) components, 1 for chrominance components.
 * You must use -qslots if you want a different component->table mapping.
 */
{
  /* ZIG[i] is the zigzag-order position of the i'th element of a DCT block */
  /* read in natural order (left to right, top to bottom). */
  static const int ZIG[DCTSIZE2] = {
	  0,  1,  5,  6, 14, 15, 27, 28,
	  2,  4,  7, 13, 16, 26, 29, 42,
	  3,  8, 12, 17, 25, 30, 41, 43,
	  9, 11, 18, 24, 31, 40, 44, 53,
	 10, 19, 23, 32, 39, 45, 52, 54,
	 20, 22, 33, 38, 46, 51, 55, 60,
	 21, 34, 37, 47, 50, 56, 59, 61,
	 35, 36, 48, 49, 57, 58, 62, 63
	 };
  FILE * fp;
  int tblno, i;
  long val;
  unsigned int table[DCTSIZE2];

  if ((fp = fopen(filename, "r")) == NULL) {
	 fprintf(stderr, "%s: can't open %s\n", progname, filename);
	 exit(EXIT_FAILURE);
  }
  tblno = 0;

  while ((val = read_qt_integer(fp)) != EOF) { /* read 1st element of table */
	 if (tblno >= NUM_QUANT_TBLS) {
		fprintf(stderr, "%s: too many tables in file %s\n", progname, filename);
		exit(EXIT_FAILURE);
	 }
	 table[0] = (unsigned int) val;
	 for (i = 1; i < DCTSIZE2; i++) {
		if ((val = read_qt_integer(fp)) == EOF) {
	fprintf(stderr, "%s: incomplete table in file %s\n", progname, filename);
	exit(EXIT_FAILURE);
		}
		table[ZIG[i]] = (unsigned int) val;
	 }
	 jpeg_add_quant_table(cinfo, tblno, table, scale_factor, force_baseline);
	 tblno++;
  }

  fclose(fp);
}


LOCAL void
set_quant_slots (j_compress_ptr cinfo, char *arg)
/* Process a quantization-table-selectors parameter string, of the form
 *     N[,N,...]
 * If there are more components than parameters, the last value is replicated.
 */
{
  int val = 0;			/* default table # */
  int ci;
  char ch;

  for (ci = 0; ci < MAX_COMPONENTS; ci++) {
	 if (*arg) {
		ch = ',';			/* if not set by sscanf, will be ',' */
		if (sscanf(arg, "%d%c", &val, &ch) < 1)
	usage();
		if (ch != ',')
	usage();		/* syntax check */
		if (val < 0 || val >= NUM_QUANT_TBLS) {
	fprintf(stderr, "JPEG quantization tables are numbered 0..%d\n",
		NUM_QUANT_TBLS-1);
	exit(EXIT_FAILURE);
		}
		cinfo->comp_info[ci].quant_tbl_no = val;
		while (*arg && *arg++ != ',') /* advance to next segment of arg string */
	;
	 } else {
		/* reached end of parameter, set remaining components to last table */
		cinfo->comp_info[ci].quant_tbl_no = val;
	 }
  }
}


LOCAL void
set_sample_factors (j_compress_ptr cinfo, char *arg)
/* Process a sample-factors parameter string, of the form
 *     HxV[,HxV,...]
 * If there are more components than parameters, "1x1" is assumed.
 */
{
  int ci, val1, val2;
  char ch1, ch2;

  for (ci = 0; ci < MAX_COMPONENTS; ci++) {
	 if (*arg) {
		ch2 = ',';		/* if not set by sscanf, will be ',' */
		if (sscanf(arg, "%d%c%d%c", &val1, &ch1, &val2, &ch2) < 3)
	usage();
		if ((ch1 != 'x' && ch1 != 'X') || ch2 != ',')
	usage();		/* syntax check */
		if (val1 <= 0 || val1 > 4 || val2 <= 0 || val2 > 4) {
	fprintf(stderr, "JPEG sampling factors must be 1..4\n");
	exit(EXIT_FAILURE);
		}
		cinfo->comp_info[ci].h_samp_factor = val1;
		cinfo->comp_info[ci].v_samp_factor = val2;
		while (*arg && *arg++ != ',') /* advance to next segment of arg string */
	;
	 } else {
		/* reached end of parameter, set remaining components to 1x1 sampling */
		cinfo->comp_info[ci].h_samp_factor = 1;
		cinfo->comp_info[ci].v_samp_factor = 1;
	 }
  }
}


LOCAL int
parse_switches (j_compress_ptr cinfo, int argc, char **argv,
		int last_file_arg_seen, boolean for_real)
/* Parse optional switches.
 * Returns argv[] index of first file-name argument (== argc if none).
 * Any file names with indexes <= last_file_arg_seen are ignored;
 * they have presumably been processed in a previous iteration.
 * (Pass 0 for last_file_arg_seen on the first or only iteration.)
 * for_real is FALSE on the first (dummy) pass; we may skip any expensive
 * processing.
 */
{
  int argn;
  char * arg;
  int quality;			/* -quality parameter */
  int q_scale_factor;		/* scaling percentage for -qtables */
  boolean force_baseline;
  char * qtablefile = NULL;	/* saves -qtables filename if any */
  char * qslotsarg = NULL;	/* saves -qslots parm if any */
  char * samplearg = NULL;	/* saves -sample parm if any */

  /* Set up default JPEG parameters. */
  /* Note that default -quality level need not, and does not,
	* match the default scaling for an explicit -qtables argument.
	*/
  quality = 75;			/* default -quality value */
  q_scale_factor = 100;		/* default to no scaling for -qtables */
  force_baseline = FALSE;	/* by default, allow 16-bit quantizers */
  is_targa = FALSE;
  outfilename = NULL;
  cinfo->err->trace_level = 0;

  /* Scan command line options, adjust parameters */

  for (argn = 1; argn < argc; argn++) {
	 arg = argv[argn];
	 if (*arg != '-') {
		/* Not a switch, must be a file name argument */
		if (argn <= last_file_arg_seen) {
	outfilename = NULL;	/* -outfile applies to just one input file */
	continue;		/* ignore this name if previously processed */
		}
		break;			/* else done parsing switches */
	 }
	 arg++;			/* advance past switch marker character */

	 if (keymatch(arg, "arithmetic", 1)) {
		/* Use arithmetic coding. */
#ifdef C_ARITH_CODING_SUPPORTED
		cinfo->arith_code = TRUE;
#else
		fprintf(stderr, "%s: sorry, arithmetic coding not supported\n",
			progname);
		exit(EXIT_FAILURE);
#endif

	 } else if (keymatch(arg, "baseline", 1)) {
		/* Force baseline output (8-bit quantizer values). */
		force_baseline = TRUE;

	 } else if (keymatch(arg, "dct", 2)) {
		/* Select DCT algorithm. */
		if (++argn >= argc)	/* advance to next argument */
	usage();
		if (keymatch(argv[argn], "int", 1)) {
	cinfo->dct_method = JDCT_ISLOW;
		} else if (keymatch(argv[argn], "fast", 2)) {
	cinfo->dct_method = JDCT_IFAST;
		} else if (keymatch(argv[argn], "float", 2)) {
	cinfo->dct_method = JDCT_FLOAT;
		} else
	usage();

	 } else if (keymatch(arg, "debug", 1) || keymatch(arg, "verbose", 1)) {
		/* Enable debug printouts. */
		/* On first -d, print version identification */
		static boolean printed_version = FALSE;

		if (! printed_version) {
	fprintf(stderr, "Independent JPEG Group's CJPEG, version %s\n%s\n",
		JVERSION, JCOPYRIGHT);
	printed_version = TRUE;
		}
		cinfo->err->trace_level++;

	 } else if (keymatch(arg, "grayscale", 2) || keymatch(arg, "greyscale",2)) {
		/* Force a monochrome JPEG file to be generated. */
		jpeg_set_colorspace(cinfo, JCS_GRAYSCALE);

	 } else if (keymatch(arg, "maxmemory", 3)) {
		/* Maximum memory in Kb (or Mb with 'm'). */
		long lval;
		char ch = 'x';

		if (++argn >= argc)	/* advance to next argument */
	usage();
		if (sscanf(argv[argn], "%ld%c", &lval, &ch) < 1)
	usage();
		if (ch == 'm' || ch == 'M')
	lval *= 1000L;
		cinfo->mem->max_memory_to_use = lval * 1000L;

	 } else if (keymatch(arg, "nointerleave", 3)) {
		/* Create noninterleaved file. */
#ifdef C_MULTISCAN_FILES_SUPPORTED
		cinfo->interleave = FALSE;
#else
		fprintf(stderr, "%s: sorry, multiple-scan support was not compiled\n",
			progname);
		exit(EXIT_FAILURE);
#endif

	 } else if (keymatch(arg, "optimize", 1) || keymatch(arg, "optimise", 1)) {
		/* Enable entropy parm optimization. */
#ifdef ENTROPY_OPT_SUPPORTED
		cinfo->optimize_coding = TRUE;
#else
		fprintf(stderr, "%s: sorry, entropy optimization was not compiled\n",
			progname);
		exit(EXIT_FAILURE);
#endif

	 } else if (keymatch(arg, "outfile", 4)) {
		/* Set output file name. */
		if (++argn >= argc)	/* advance to next argument */
	usage();
		outfilename = argv[argn];	/* save it away for later use */

	 } else if (keymatch(arg, "quality", 1)) {
		/* Quality factor (quantization table scaling factor). */
		if (++argn >= argc)	/* advance to next argument */
	usage();
		if (sscanf(argv[argn], "%d", &quality) != 1)
	usage();
		/* Change scale factor in case -qtables is present. */
		q_scale_factor = jpeg_quality_scaling(quality);

	 } else if (keymatch(arg, "qslots", 2)) {
		/* Quantization table slot numbers. */
		if (++argn >= argc)	/* advance to next argument */
	usage();
		qslotsarg = argv[argn];
		/* Must delay setting qslots until after we have processed any
		 * colorspace-determining switches, since jpeg_set_colorspace sets
		 * default quant table numbers.
		 */

	 } else if (keymatch(arg, "qtables", 2)) {
		/* Quantization tables fetched from file. */
		if (++argn >= argc)	/* advance to next argument */
	usage();
		qtablefile = argv[argn];
		/* We postpone actually reading the file in case -quality comes later. */

	 } else if (keymatch(arg, "restart", 1)) {
		/* Restart interval in MCU rows (or in MCUs with 'b'). */
		long lval;
		char ch = 'x';

		if (++argn >= argc)	/* advance to next argument */
	usage();
		if (sscanf(argv[argn], "%ld%c", &lval, &ch) < 1)
	usage();
		if (lval < 0 || lval > 65535L)
	usage();
		if (ch == 'b' || ch == 'B') {
	cinfo->restart_interval = (unsigned int) lval;
	cinfo->restart_in_rows = 0; /* else prior '-restart n' overrides me */
		} else {
	cinfo->restart_in_rows = (int) lval;
	/* restart_interval will be computed during startup */
		}

	 } else if (keymatch(arg, "sample", 2)) {
		/* Set sampling factors. */
		if (++argn >= argc)	/* advance to next argument */
	usage();
		samplearg = argv[argn];
		/* Must delay setting sample factors until after we have processed any
		 * colorspace-determining switches, since jpeg_set_colorspace sets
		 * default sampling factors.
		 */

	 } else if (keymatch(arg, "smooth", 2)) {
		/* Set input smoothing factor. */
		int val;

		if (++argn >= argc)	/* advance to next argument */
	usage();
		if (sscanf(argv[argn], "%d", &val) != 1)
	usage();
		if (val < 0 || val > 100)
	usage();
		cinfo->smoothing_factor = val;

	 } else if (keymatch(arg, "targa", 1)) {
		/* Input file is Targa format. */
		is_targa = TRUE;

	 } else {
		usage();			/* bogus switch */
	 }
  }

  /* Post-switch-scanning cleanup */

  if (for_real) {

	 /* Set quantization tables for selected quality. */
	 /* Some or all may be overridden if -qtables is present. */
	 jpeg_set_quality(cinfo, quality, force_baseline);

	 if (qtablefile != NULL)	/* process -qtables if it was present */
		read_quant_tables(cinfo, qtablefile, q_scale_factor, force_baseline);

	 if (qslotsarg != NULL)	/* process -qslots if it was present */
		set_quant_slots(cinfo, qslotsarg);

	 if (samplearg != NULL)	/* process -sample if it was present */
		set_sample_factors(cinfo, samplearg);

  }

  return argn;			/* return index of next arg (file name) */
}


int Load_C (char *Name, int *W, int *H, BYTE **R, BYTE **G, BYTE **B)
{
  struct jpeg_compress_struct cinfo;
  struct jpeg_error_mgr jerr;
  int file_index;
  int I, Pos;
  cjpeg_source_ptr src_mgr;
  FILE * input_file;
  FILE * output_file;
  JDIMENSION num_scanlines;

  progname = "cjpeg";		/* in case C library doesn't provide it */

  /* Initialize the JPEG compression object with default error handling. */
  cinfo.err = jpeg_std_error(&jerr);
  jpeg_create_compress(&cinfo);
  /* Add some application-specific error messages (from cderror.h) */
  jerr.addon_message_table = addon_message_table;
  jerr.first_addon_message = JMSG_FIRSTADDONCODE;
  jerr.last_addon_message = JMSG_LASTADDONCODE;

  /* Initialize JPEG parameters.
	* Much of this may be overridden later.
	* In particular, we don't yet know the input file's color space,
	* but we need to provide some value for jpeg_set_defaults() to work.
	*/

  cinfo.in_color_space = JCS_RGB; /* arbitrary guess */
  jpeg_set_defaults(&cinfo);

  /* Scan command line to find file names.
	* It is convenient to use just one switch-parsing routine, but the switch
	* values read here are ignored; we will rescan the switches after opening
	* the input file.
	*/

/*  file_index = parse_switches(&cinfo, argc, argv, 0, FALSE);*/

  /* Open the input file. */
	 if ((input_file = fopen(Name, READ_BINARY)) == NULL) {
		fprintf(stderr, "%s: can't open %s\n", progname, Name);
		exit(EXIT_FAILURE);
	 }

  output_file = stdout;

  /* Figure out the input file format, and set up to read it. */
  src_mgr = select_file_type(&cinfo, input_file);
  src_mgr->input_file = input_file;

  /* Read the input file header to obtain file size & colorspace. */
  (*src_mgr->start_input) (&cinfo, src_mgr);

  /* Now that we know input colorspace, fix colorspace-dependent defaults */
  jpeg_default_colorspace(&cinfo);

  /* Adjust default compression parameters by re-parsing the options */
/*  file_index = parse_switches(&cinfo, argc, argv, 0, TRUE);*/

  /* Specify data destination for compression */
  jpeg_stdio_dest(&cinfo, NULL);

  /* Start compressor */
  jpeg_start_compress(&cinfo, TRUE);

  /* JC */
  *W = cinfo.image_width;
  *H = cinfo.image_height;
  *R = (BYTE *) malloc((*W) * (*H));
  *G = (BYTE *) malloc((*W) * (*H));
  *B = (BYTE *) malloc((*W) * (*H));
  Pos = 0;

  /* Process data */
  while (cinfo.next_scanline < cinfo.image_height) {
	 num_scanlines = (*src_mgr->get_pixel_rows) (&cinfo, src_mgr);
	 for(I=0; I<cinfo.image_width; I++, Pos++)
		{
		(*R)[Pos] = (src_mgr->buffer)[0][I*3];
		(*G)[Pos] = (src_mgr->buffer)[0][I*3+1];
		(*B)[Pos] = (src_mgr->buffer)[0][I*3+2];
		}
	 cinfo.next_scanline += num_scanlines;
  }

  /* Finish compression and release memory */
  (*src_mgr->finish_input) (&cinfo, src_mgr);
/*  jpeg_finish_compress(&cinfo);*/
  jpeg_destroy_compress(&cinfo);

  /* All done. */
  /* exit(jerr.num_warnings ? EXIT_WARNING : EXIT_SUCCESS); */
  return 0;			/* suppress no-return-value warnings */
}

/*
 * Here's the routine that will replace the standard error_exit method:
 */

METHODDEF void
my_error_exit (j_common_ptr cinfo)
{
  /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
  my_error_ptr myerr = (my_error_ptr) cinfo->err;

  /* Always display the message. */
  /* We could postpone this until after returning, if we chose. */
  (*cinfo->err->output_message) (cinfo);

  /* Return control to the setjmp point */
  longjmp(myerr->setjmp_buffer, 1);
}

/*
 * Sample routine for JPEG decompression.  We assume that the source file name
 * is passed in.  We want to return 1 on success, 0 on error.
 */


int read_JPEG_file (char *Name, int *W, int *H, BYTE **R, BYTE **G, BYTE **B)
{
  /* This struct contains the JPEG decompression parameters and pointers to
	* working space (which is allocated as needed by the JPEG library).
	*/
  struct jpeg_decompress_struct cinfo;
  /* We use our private extension JPEG error handler. */
  struct my_error_mgr jerr;
  /* More stuff */
  FILE * infile;		/* source file */
  JSAMPARRAY buffer;		/* Output row buffer */
  int row_stride;		/* physical row width in output buffer */
  int I, Pos;

  /* In this example we want to open the input file before doing anything else,
	* so that the setjmp() error recovery below can assume the file is open.
	* VERY IMPORTANT: use "b" option to fopen() if you are on a machine that
	* requires it in order to read binary files.
	*/

  if ((infile = fopen(Name, "rb")) == NULL) {
	 fprintf(stderr, "can't open %s\n", Name);
	 return 0;
  }

  /* Step 1: allocate and initialize JPEG decompression object */

  /* We set up the normal JPEG error routines, then override error_exit. */
  cinfo.err = jpeg_std_error(&jerr.pub);
  jerr.pub.error_exit = my_error_exit;
  /* Establish the setjmp return context for my_error_exit to use. */
  if (setjmp(jerr.setjmp_buffer)) {
	 /* If we get here, the JPEG code has signaled an error.
	  * We need to clean up the JPEG object, close the input file, and return.
	  */
	 jpeg_destroy_decompress(&cinfo);
	 fclose(infile);
	 return 0;
  }
  /* Now we can initialize the JPEG decompression object. */
  jpeg_create_decompress(&cinfo);

  /* Step 2: specify data source (eg, a file) */

  jpeg_stdio_src(&cinfo, infile);

  /* Step 3: read file parameters with jpeg_read_header() */

  (void) jpeg_read_header(&cinfo, TRUE);
  /* We can ignore the return value from jpeg_read_header since
	*   (a) suspension is not possible with the stdio data source, and
	*   (b) we passed TRUE to reject a tables-only JPEG file as an error.
	* See libjpeg.doc for more info.
	*/

  /* Step 4: set parameters for decompression */

  /* In this example, we don't need to change any of the defaults set by
	* jpeg_read_header(), so we do nothing here.
	*/

  /* Step 5: Start decompressor */

  jpeg_start_decompress(&cinfo);

  /* We may need to do some setup of our own at this point before reading
	* the data.  After jpeg_start_decompress() we have the correct scaled
	* output image dimensions available, as well as the output colormap
	* if we asked for color quantization.
	* In this example, we need to make an output work buffer of the right size.
	*/
  /* JSAMPLEs per row in output buffer */
  row_stride = cinfo.output_width * cinfo.output_components;
  /* Make a one-row-high sample array that will go away when done with image */
  buffer = (*cinfo.mem->alloc_sarray)
		((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);

  /* JC */
  *W = cinfo.output_width;
  *H = cinfo.output_height;
  *R = (BYTE *) malloc((*W) * (*H));
  *G = (BYTE *) malloc((*W) * (*H));
  *B = (BYTE *) malloc((*W) * (*H));
  Pos = 0;

  /* Step 6: while (scan lines remain to be read) */
  /*           jpeg_read_scanlines(...); */

  /* Here we use the library's state variable cinfo.output_scanline as the
	* loop counter, so that we don't have to keep track ourselves.
	*/
  while (cinfo.output_scanline < cinfo.output_height) {
	 (void) jpeg_read_scanlines(&cinfo, buffer, 1);
	 /* Assume put_scanline_someplace wants a pointer and sample count. */
	 if(cinfo.output_components == 1) /* JPEG in GrayScale */
	 {
		for(I=0; I<cinfo.output_width; I++, Pos++)
		  {
		  (*R)[Pos] = buffer[0][I];
		  (*G)[Pos] = buffer[0][I];
		  (*B)[Pos] = buffer[0][I];
		  }
	 }
	 else
	 {
		for(I=0; I<cinfo.output_width; I++, Pos++)
		  {
		  (*R)[Pos] = buffer[0][I*3];
		  (*G)[Pos] = buffer[0][I*3+1];
		  (*B)[Pos] = buffer[0][I*3+2];
		  }
	 }

  }

  /* Step 7: Finish decompression */

  (void) jpeg_finish_decompress(&cinfo);
  /* We can ignore the return value since suspension is not possible
	* with the stdio data source.
	*/

  /* Step 8: Release JPEG decompression object */

  /* This is an important step since it will release a good deal of memory. */
  jpeg_destroy_decompress(&cinfo);

  /* After finish_decompress, we can close the input file.
   * Here we postpone it until after no more JPEG errors are possible,
   * so as to simplify the setjmp error logic above.  (Actually, I don't
	* think that jpeg_destroy can do an error exit, but why assume anything...)
   */
  fclose(infile);

  /* At this point you may want to check to see whether any corrupt-data
   * warnings occurred (test whether jerr.pub.num_warnings is nonzero).
   */

  /* And we're done! */
  return 1;
}
