/* $Id: log.c,v 1.6 2001/07/29 17:58:36 richdawe Exp $ */

/*
 *  log.c - Logging functions for zippo
 *  Copyright (C) 1999-2001 by Richard Dawe
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "common.h"

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

/* libzippo includes */
#include <libzippo/log.h>
#include <libzippo/package.h>

/* Operation numeric <-> string mapping table. */
typedef struct {
  int         op;
  const char *str;
} log_op_mapping;

static log_op_mapping log_op_map[] = {
  { LOG_OP_INSTALL,   "install"   },
  { LOG_OP_UNINSTALL, "uninstall" },
  { LOG_OP_SYNCDB,    "syncdb"    },
  { LOG_OP_NONE,      "none"      },
  { LOG_OP_NONE,      NULL        } /* Terminating entry. */
};

/* -------------
 * - op_to_str -
 * ------------- */

/* This is an internal routine to convert the numeric operation code to a text
 * string. */

static
const char *op_to_str (int op)
{
  int i = 0;

  for (i = 0; log_op_map[i].str != NULL; i++) {
    /* Return match */
    if (log_op_map[i].op == op)
      return(log_op_map[i].str);
  }

  /* Fail */
  return(NULL);
}

/* -------------
 * - str_to_op -
 * ------------- */

/* This is an internal function to convert from a string operation to a numeric
 * code. */

static int __attribute__((unused))
     str_to_op (const char *str)
{
  int i = 0;

  for (i = 0; log_op_map[i].str != NULL; i++) {
    /* Return match */
    if (strcmp(log_op_map[i].str, str) == 0)
      return(log_op_map[i].op);
  }

  /* Fail */
  return(LOG_OP_NONE);
}

/* --------------
 * - lz_loginit -
 * -------------- */

int
lz_loginit (LZ_LOG *log)
{
  log->fp = NULL;
  return(1);
}

/* --------------
 * - lz_logopen -
 * -------------- */

int
lz_logopen (LZ_LOG *log, const char *name)
{
  /* Close the log file, if it's open. */
  if (log->fp != NULL) {
    if (!lz_logclose(log)) {
      /* Failed */
      return(0);
    }
  }

  /* Open the log file for append. */
  log->fp = fopen(name, "at");
  return(log->fp != NULL);
}

/* ---------------
 * - lz_logclose -
 * --------------- */

int lz_logclose (LZ_LOG *log)
{
  if (fclose(log->fp) != 0) {
    /* Blimey, fclose() failed! */
    return(0);
  }
	
  log->fp = NULL;
  return(1);
}

/* -----------
 * - lz_logf -
 * ----------- */

int __attribute__((format (printf, 4, 5)))
     lz_logf (LZ_LOG *log, int op, PACKAGE_INFO *package, const char *fmt, ...)
{
  time_t t;
  char *ct = NULL;
  const char *op_str = NULL;
  int n1, n2;
  va_list arg;

  /* Get the local time in string form. */
  time(&t);
  ct = ctime((const time_t *) &t);

  /* Convert the operation to a string. */
  op_str = op_to_str(op);

  /* Log the event */
  n1 = fprintf(log->fp, "%s: %s: %s %s (%s): ", ct, op_str,
	       package->name,
	       package_version_string(&package->version),
	       package_type_string(package));

  if (n1 < 0)
    return(-1);

  va_start(arg, fmt);
  n2 = vfprintf(log->fp, fmt, arg);
  va_end(arg);

  if (n2 < 0)
    return(-1);
	
  return(n1 + n2);
}
