/* $Id: rcfile.c,v 1.7 2001/07/29 18:00:17 richdawe Exp $ */

/*
 *  rcfile.c - Configuration file functions for zippo
 *  Copyright (C) 1999-2001 by Richard Dawe
 *      
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "common.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <limits.h>

/* libzippo includes */
#include <libzippo/util.h>

#include "rcfile.h"

static char rcfilename[PATH_MAX];
static int rcfile_inited = 0;

/* These are used for substitutions, */
static char zippo_root[PATH_MAX];
static char zippo_prefix[PATH_MAX];

/* These store paths to resources, parsed from the configuration file(s). */
static char **p_installed_dsm = NULL;
static char **p_installed_mft = NULL;
static char **p_available_dsm = NULL;
static char **p_zip           = NULL;
static char **p_tar_gzip      = NULL;
static char **p_ftp_proxies   = NULL;
static char **p_http_proxies  = NULL;

/* --------------
 * - add_string -
 * -------------- */

/* Convenience function for adding items to NULL-terminated string arrays. */

static int inline
add_string (char ***strings, char *s)
{
  char **x = *strings;
  int i = 0;

  /* Need to allocate space for entry plus NULL terminator. */
  if (x == NULL) {
    x = (char **) malloc(sizeof(char *) * 2);
  } else {
    for (i = 0; x[i] != NULL; i++) {;} /* NULL not included in count */
    x = (char **) realloc(x, sizeof(char *) * (i + 2));    
  }

  if (x == NULL)
    return(0);

  *strings = x;
  x[i] = s;
  x[i + 1] = NULL;

  return(1);
}

/* ----------------
 * - free_strings -
 * ---------------- */

/* Convenience function for tidying up NULL-terminated string arrays. */

static int inline
free_strings (char ***strings)
{
  char **x = *strings;
  int i;

  if (x != NULL) {
    for (i = 0; (x)[i] != NULL; i++) {
      free((x)[i]);
    }
    free(x);
    *strings = NULL;
  }
  return(1);
}

/* ------------
 * - res2list -
 * ------------ */

/* Convenience function for mapping rc_resource_t value to a list pointer. */

static char ***
res2list (const rc_resource_t res)
{
  switch(res) {
  case RC_INSTALLED_MFT: return(&p_installed_mft);
  case RC_INSTALLED_DSM: return(&p_installed_dsm);
  case RC_AVAILABLE_DSM: return(&p_available_dsm);
  case RC_ZIP:           return(&p_zip);
  case RC_TAR_GZIP:      return(&p_tar_gzip);

  case RC_NONE:
  default:
    return(NULL);
  }
}

/* ---------------
 * - rcfile_init -
 * --------------- */

int
rcfile_init (const char *filename, const char *root, const char *prefix)
{
  /* Allow this to be called multiple times */
  if (rcfile_inited)
    rcfile_uninit();

  if (access(filename, R_OK) != 0)
    return(0);
  /* Assume root, prefix OK */

  strcpy(rcfilename, filename);
  strcpy(zippo_root, root);
  strcpy(zippo_prefix, prefix);

  rcfile_inited = 1;
  return(1);
}

/* -----------------
 * - rcfile_uninit -
 * ----------------- */

int
rcfile_uninit (void)
{
  if (!rcfile_inited)
    return(1);

  /* Free parsed paths */
  free_strings(&p_installed_dsm);
  free_strings(&p_installed_mft);
  free_strings(&p_available_dsm);
  free_strings(&p_zip);
  free_strings(&p_tar_gzip);
  free_strings(&p_ftp_proxies);
  free_strings(&p_http_proxies);

  rcfile_inited = 0;
  return(1);
}

/* ----------------
 * - rcfile_parse -
 * ---------------- */

int
rcfile_parse (void)
{  
  FILE *fp  = NULL;
  int   ret = 0;

  if (!rcfile_inited)
    return(0);

  fp = fopen(rcfilename, "rt");
  if (fp == NULL)
    return(0);

  /* Now run parser */
  rcfile_yyin = fp;
  ret = !rcfile_yyparse(); /* rcfile_yyparse() returns 0 on success */

  fclose(fp);

  return(ret);
}

/* ----------------------
 * - rcfile_get_proxies -
 * ---------------------- */

/*
 * TODO: Update lexer to understand this format:
 *
 * Parse a line of the following form:
 *
 * hostspec := <host>[(:<port>)|( [[on ]port ]<port>)]
 *
 * proxy (ftp|http) [using|with] <hostspec>[( or|,) <hostspec> [...]]
 *
 * This looks like a mouthful, but ends up being near to an English sentence,
 * e.g. "proxy ftp with myproxy on port 8080". The list of addresses is
 * returned as "<host>:<port>" strings, dynamically allocated (i.e. free()
 * it!).
 */

char **
rcfile_get_proxies (const char *protocol)
{
  if (strcmp(protocol, "ftp") == 0)
    return(p_ftp_proxies);
  else if (strcmp(protocol, "http") == 0)
    return(p_http_proxies);

  return(NULL);
}

/* -----------------------
 * - rcfile_add_resource -
 * ----------------------- */

int
rcfile_add_resource (const rc_resource_t res, const char *path)
{
  char    buf[PATH_MAX];
  char   *p = NULL;
  char ***list = NULL;

  /* Work out which list to use */
  list = res2list(res);
  if (list == NULL)
    return(0);

  /* Make substitutions on the path, if there are any. */
  strcpy(buf, path);

  /* TODO: Other substitutions, including environment. */
  for (p = strstr(buf, RCFILE_SUBST_ZIPPO_ROOT);
       p != NULL; p = strstr(buf, RCFILE_SUBST_ZIPPO_ROOT)) {
    if (   (*(p - 1) == '@')
	   && (*(p + strlen(RCFILE_SUBST_ZIPPO_ROOT)) == '@')) {
      /*    <blah>@ZIPPO-ROOT@<blah2>
	    -> <blah><-------gap--------><blah2>
	    -> <blah><---substitution---><blah2> */

      /* Chop the middle part out */
      *(p - 1) = '\0';
      p += strlen(RCFILE_SUBST_ZIPPO_ROOT);
      *p = '\0';
      p++;

      /* strcpy() can't copy overlapping areas, so use
	 memmove(). */
      memmove(  p + strlen(zippo_root)
		- strlen(RCFILE_SUBST_ZIPPO_ROOT) - 2,
		p, strlen(p) + 1);
      memcpy(buf + strlen(buf), zippo_root,
	     strlen(zippo_root)); /* Don't copy nul! */
    }
  }

  /* Now add to the list */
  p = strdup(buf);
  if (p == NULL)
    return(0);
  
  /* TODO: Make substitutions */
  /*strcpy(buf, p);
  if (!rcfile_substitute_special(buf, sizeof(buf))) return(NULL);
  if (!rcfile_substitute_environment(buf, sizeof(buf))) return(NULL);*/

  if (!add_string(list, p)) {
    free(p);
    return(0);
  }

  return(1);
}

/* ------------------------
 * - rcfile_get_resources -
 * ------------------------ */

char **
rcfile_get_resources (rc_resource_t res)
{
  char ***list = NULL;

  list = res2list(res);
  if (list == NULL)
    return(NULL);  

  return(*list);
}

/* ------------------
 * - rcfile_yyerror -
 * ------------------ */

/* Parser support routine */

void
rcfile_yyerror (char *s)
{
  /* TODO: Fix out-by-one errors */
  fprintf(stderr, "%s:%d: %s\n", rcfilename, rcfile_yylineno, s);
}
