/* `ln' program to create links between files.
   Copyright (C) 86, 89, 90, 91, 95, 96, 97, 1998 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* Written by Mike Parker and David MacKenzie. */

#ifdef _AIX
 #pragma alloca
#endif

#include <config.h>
#include <stdio.h>
#include <sys/types.h>
#include <getopt.h>

#include "system.h"
#include "backupfile.h"
#include "closeout.h"
#include "error.h"

#ifdef  __DJGPP__

/* Versions of DJGPP > 2.03 support symlinks (currently only DJGPP CVS). If
 * symlinks are supported, we can rely on DJGPP to DTRT. */
#if !(defined(HAVE_LSTAT) && defined(S_ISLNK))

/* DJGPP doesn't have S_ISLNK defined, because MS-DOS doesn't support links.
 * However, DJGPP's `link' simulates hard links by copying, and we can
 * have ``symbolic links'' for DJGPP .EXE programs.  Therefore, just
 * for the purpose of this program, it makes sense to define S_ISLNK
 * so we could have working `ln'.  */

#define S_ISLNK(x)  (0)     /* defined, but no file is a link */

#endif /* !(HAVE_LSTAT && S_ISLNK) */

#endif /* __DJGPP__ */

int link ();			/* Some systems don't declare this anywhere. */

#ifdef S_ISLNK
int symlink ();
#endif

/* Construct a string NEW_DEST by concatenating DEST, a slash, and
   basename(SOURCE) in alloca'd memory.  Don't modify DEST or SOURCE.  */

#define PATH_BASENAME_CONCAT(new_dest, dest, source)			\
    do									\
      {									\
	char *source_base;						\
	char *tmp_source;						\
									\
	tmp_source = (char *) alloca (strlen ((source)) + 1);		\
	strcpy (tmp_source, (source));					\
	strip_trailing_slashes (tmp_source);				\
	source_base = base_name (tmp_source);				\
									\
	(new_dest) = (char *) alloca (strlen ((dest)) + 1		\
				      + strlen (source_base) + 1);	\
	stpcpy (stpcpy (stpcpy ((new_dest), (dest)), "/"), source_base);\
      }									\
    while (0)

char *base_name ();
char *dirname ();
enum backup_type get_version ();
int isdir ();
int yesno ();
void strip_trailing_slashes ();

/* The name by which the program was run, for error messages.  */
char *program_name;

/* FIXME: document */
enum backup_type backup_type;

/* A pointer to the function used to make links.  This will point to either
   `link' or `symlink'. */
static int (*linkfunc) ();

/* If nonzero, make symbolic links; otherwise, make hard links.  */
static int symbolic_link;

/* A string describing type of link to make.  For use in verbose
   diagnostics and in error messages.  */
static const char *link_type_string;

/* If nonzero, ask the user before removing existing files.  */
static int interactive;

/* If nonzero, remove existing files unconditionally.  */
static int remove_existing_files;

/* If nonzero, list each file as it is moved. */
static int verbose;

/* If nonzero, allow the superuser to make hard links to directories. */
static int hard_dir_link;

/* If nonzero, and the specified destination is a symbolic link to a
   directory, treat it just as if it were a directory.  Otherwise, the
   command `ln --force --no-dereference file symlink-to-dir' deletes
   symlink-to-dir before creating the new link.  */
static int dereference_dest_dir_symlinks = 1;

/* If nonzero, display usage information and exit.  */
static int show_help;

/* If nonzero, print the version on standard output and exit.  */
static int show_version;

static struct option const long_options[] =
{
  {"backup", no_argument, NULL, 'b'},
  {"directory", no_argument, NULL, 'F'},
  {"no-dereference", no_argument, NULL, 'n'},
  {"force", no_argument, NULL, 'f'},
  {"interactive", no_argument, NULL, 'i'},
  {"suffix", required_argument, NULL, 'S'},
  {"symbolic", no_argument, NULL, 's'},
  {"verbose", no_argument, NULL, 'v'},
  {"version-control", required_argument, NULL, 'V'},
  {"help", no_argument, &show_help, 1},
  {"version", no_argument, &show_version, 1},
  {NULL, 0, NULL, 0}
};

/* Return nonzero if SOURCE and DEST point to the same name in the same
   directory.  */

static int
same_name (const char *source, const char *dest)
{
  struct stat source_dir_stats;
  struct stat dest_dir_stats;
  char *source_dirname, *dest_dirname;

  source_dirname = dirname (source);
  dest_dirname = dirname (dest);
  if (source_dirname == NULL || dest_dirname == NULL)
    error (1, 0, _("virtual memory exhausted"));

  if (stat (source_dirname, &source_dir_stats))
    {
      /* Shouldn't happen.  */
      error (1, errno, "%s", source_dirname);
    }

  if (stat (dest_dirname, &dest_dir_stats))
    {
      /* Shouldn't happen.  */
      error (1, errno, "%s", dest_dirname);
    }

  free (source_dirname);
  free (dest_dirname);

  return (source_dir_stats.st_dev == dest_dir_stats.st_dev
	  && source_dir_stats.st_ino == dest_dir_stats.st_ino
	  && STREQ (base_name (source), base_name (dest)));
}

/* Make a link DEST to the (usually) existing file SOURCE.
   Symbolic links to nonexistent files are allowed.
   If DEST is a directory, put the link to SOURCE in that directory.
   Return 1 if there is an error, otherwise 0.  */

static int
do_link (const char *source, const char *dest)
{
  struct stat source_stats;
  struct stat dest_stats;
  char *dest_backup = NULL;
  int lstat_status;

  /* Use stat here instead of lstat.
     On SVR4, link does not follow symlinks, so this check disallows
     making hard links to symlinks that point to directories.  Big deal.
     On other systems, link follows symlinks, so this check is right.  */
  if (!symbolic_link)
    {
      if (stat (source, &source_stats) != 0)
	{
	  error (0, errno, "%s", source);
	  return 1;
	}
      if (!hard_dir_link && S_ISDIR (source_stats.st_mode))
	{
	  error (0, 0, _("%s: hard link not allowed for directory"), source);
	  return 1;
	}
    }

  lstat_status = lstat (dest, &dest_stats);
  if (lstat_status != 0 && errno != ENOENT)
    {
      error (0, errno, "%s", dest);
      return 1;
    }

  /* If the destination is a directory or (it is a symlink to a directory
     and the user has not specified --no-dereference), then form the
     actual destination name by appending base_name (source) to the
     specified destination directory.  */
  if ((lstat_status == 0
       && S_ISDIR (dest_stats.st_mode))
#ifdef S_ISLNK
      || (dereference_dest_dir_symlinks
	  && (lstat_status == 0
	      && S_ISLNK (dest_stats.st_mode)
	      && isdir (dest)))
#endif
     )
    {
      /* Target is a directory; build the full filename. */
      char *new_dest;
      PATH_BASENAME_CONCAT (new_dest, dest, source);
      dest = new_dest;

      /* Get stats for new DEST.  */
      lstat_status = lstat (dest, &dest_stats);
      if (lstat_status != 0 && errno != ENOENT)
	{
	  error (0, errno, "%s", dest);
	  return 1;
	}
    }

  /* If --force (-f) has been specified without --backup, then before
     making a link ln must remove the destination file if it exists.
     (with --backup, it just renames any existing destination file)
     But if the source and destination are the same, don't remove
     anything and fail right here.  */
  if (remove_existing_files
      && lstat_status == 0
      /* Allow `ln -sf --backup k k' to succeed in creating the
	 self-referential symlink, but don't allow the hard-linking
	 equivalent: `ln -f k k' (with or without --backup) to get
	 beyond this point, because the error message you'd get is
	 misleading.  */
      && (backup_type == none || !symlink)
      && (!symlink || stat (source, &source_stats) == 0)
      && source_stats.st_dev == dest_stats.st_dev
      && source_stats.st_ino == dest_stats.st_ino
      /* The following detects whether removing DEST will also remove
 	 SOURCE.  If the file has only one link then both are surely
 	 the same link.  Otherwise check whether they point to the same
	 name in the same directory.  */
      && (source_stats.st_nlink == 1 || same_name (source, dest)))
    {
      error (0, 0, _("`%s' and `%s' are the same file"), source, dest);
      return 1;
    }

  if (lstat_status == 0 || lstat (dest, &dest_stats) == 0)
    {
      if (S_ISDIR (dest_stats.st_mode))
	{
	  error (0, 0, _("%s: cannot overwrite directory"), dest);
	  return 1;
	}
      if (interactive)
	{
	  fprintf (stderr, _("%s: replace `%s'? "), program_name, dest);
	  if (!yesno ())
	    return 0;
	}
      else if (!remove_existing_files && backup_type == none)
	{
	  error (0, 0, _("%s: File exists"), dest);
	  return 1;
	}

      if (backup_type != none)
	{
	  char *tmp_backup = find_backup_file_name (dest, backup_type);
	  if (tmp_backup == NULL)
	    error (1, 0, _("virtual memory exhausted"));
	  dest_backup = (char *) alloca (strlen (tmp_backup) + 1);
	  strcpy (dest_backup, tmp_backup);
	  free (tmp_backup);
	  if (rename (dest, dest_backup))
	    {
	      if (errno != ENOENT)
		{
		  error (0, errno, _("cannot backup `%s'"), dest);
		  return 1;
		}
	      else
		dest_backup = NULL;
	    }
	}
      else if (unlink (dest) && errno != ENOENT)
	{
	  error (0, errno, _("cannot remove `%s'"), dest);
	  return 1;
	}
    }
  else if (errno != ENOENT)
    {
      error (0, errno, "%s", dest);
      return 1;
    }

  if (verbose)
    printf (_("create %s %s to %s\n"), link_type_string, dest, source);

  if ((*linkfunc) (source, dest) == 0)
    {
      return 0;
    }

  error (0, errno, _("cannot create %s `%s' to `%s'"), link_type_string,
	 dest, source);

  if (dest_backup)
    {
      if (rename (dest_backup, dest))
	error (0, errno, _("cannot un-backup `%s'"), dest);
    }
  return 1;
}

static void
usage (int status)
{
  if (status != 0)
    fprintf (stderr, _("Try `%s --help' for more information.\n"),
	     program_name);
  else
    {
      printf (_("\
Usage: %s [OPTION]... TARGET [LINK_NAME]\n\
  or:  %s [OPTION]... TARGET... DIRECTORY\n\
"),
	      program_name, program_name);
      printf (_("\
Create a link to the specified TARGET with optional LINK_NAME.  If there is\n\
more than one TARGET, the last argument must be a directory;  create links\n\
in DIRECTORY to each TARGET.  Create hard links by default, symbolic links\n\
with --symbolic.  When creating hard links, each TARGET must exist.\n\
\n\
  -b, --backup                make a backup of each existing destination file\n\
  -d, -F, --directory         hard link directories (super-user only)\n\
  -f, --force                 remove existing destination files\n\
  -n, --no-dereference        treat destination that is a symlink to a\n\
                                directory as if it were a normal file\n\
  -i, --interactive           prompt whether to remove destinations\n\
  -s, --symbolic              make symbolic links instead of hard links\n\
  -S, --suffix=SUFFIX         override the usual backup suffix\n\
  -v, --verbose               print name of each file before linking\n\
  -V, --version-control=WORD  override the usual version control\n\
      --help                  display this help and exit\n\
      --version               output version information and exit\n\
\n\
"));
      printf (_("\
The backup suffix is ~, unless set with SIMPLE_BACKUP_SUFFIX.  The\n\
version control may be set with VERSION_CONTROL, values are:\n\
\n\
  t, numbered     make numbered backups\n\
  nil, existing   numbered if numbered backups exist, simple otherwise\n\
  never, simple   always make simple backups\n\
"));
      puts (_("\nReport bugs to <bug-fileutils@gnu.org>."));
      close_stdout ();
    }
  exit (status);
}

int
main (int argc, char **argv)
{
  int c;
  int errors;
  int make_backups = 0;
  char *version;

  program_name = argv[0];
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  version = getenv ("SIMPLE_BACKUP_SUFFIX");
  if (version)
    simple_backup_suffix = version;
  version = getenv ("VERSION_CONTROL");

  symbolic_link = remove_existing_files = interactive = verbose
    = hard_dir_link = 0;
  errors = 0;

  while ((c = getopt_long (argc, argv, "bdfinsvFS:V:", long_options, NULL))
	 != -1)
    {
      switch (c)
	{
	case 0:			/* Long-named option. */
 	  break;
	case 'b':
	  make_backups = 1;
	  break;
	case 'd':
	case 'F':
	  hard_dir_link = 1;
	  break;
	case 'f':
	  remove_existing_files = 1;
	  interactive = 0;
	  break;
	case 'i':
	  remove_existing_files = 0;
	  interactive = 1;
	  break;
	case 'n':
	  dereference_dest_dir_symlinks = 0;
	  break;
	case 's':
#ifdef S_ISLNK
	  symbolic_link = 1;
#else
	  error (1, 0, _("symbolic links are not supported on this system"));
#endif
	  break;
	case 'v':
	  verbose = 1;
	  break;
	case 'S':
	  simple_backup_suffix = optarg;
	  break;
	case 'V':
	  version = optarg;
	  break;
	default:
	  usage (1);
	  break;
	}
    }

  if (show_version)
    {
      printf ("ln (%s) %s\n", GNU_PACKAGE, VERSION);
      close_stdout ();
      exit (0);
    }

  if (show_help)
    usage (0);

  if (optind == argc)
    {
      error (0, 0, _("missing file argument"));
      usage (1);
    }

  backup_type = (make_backups ? get_version (version) : none);

#ifdef S_ISLNK
  if (symbolic_link)
    {
      linkfunc = symlink;
      link_type_string = _("symbolic link");
    }
  else
    {
      linkfunc = link;
      link_type_string = _("hard link");
    }
#else
  link_type_string = _("link");
#endif

  if (optind == argc - 1)
    errors = do_link (argv[optind], ".");
  else if (optind == argc - 2)
    {
      struct stat source_stats;
      char *source;
      char *dest;
      char *new_dest;

      source = argv[optind];
      dest = argv[optind + 1];

      /* When the destination is specified with a trailing slash and the
	 source exists but is not a directory, convert the user's command
	 `ln source dest/' to `ln source dest/basename(source)'.  */

      if (dest[strlen (dest) - 1] == '/'
	  && lstat (source, &source_stats) == 0
	  && !S_ISDIR (source_stats.st_mode))
	{
	  PATH_BASENAME_CONCAT (new_dest, dest, source);
	}
      else
	{
	  new_dest = dest;
	}

      errors = do_link (source, new_dest);
    }
  else
    {
      char *to;

      to = argv[argc - 1];
      if (!isdir (to))
	error (1, 0, _("when making multiple links, last argument must be a directory"));
      for (; optind < argc - 1; ++optind)
	errors += do_link (argv[optind], to);
    }

  if (verbose)
    close_stdout ();
  exit (errors != 0);
}
