/*Routines derived from the subroutines shared by all languages that are variants of C.
  Copyright (C) 1992-2002 Free Software Foundation, Inc.

  Authors: Jukka Virtanen <jtv@hut.fi>
           Peter Gerwinski <peter@gerwinski.de>
           Frank Heckenbach <frank@pascal.gnu.de>

  This file is derived from GCC's `c-common.c'.

  This file is part of GNU Pascal.

  GNU Pascal is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published
  by the Free Software Foundation; either version 2, or (at your
  option) any later version.

  GNU Pascal is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with GNU Pascal; see the file COPYING. If not, write to the
  Free Software Foundation, 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA. */

#include "gbe.h"

#include "gpc-defs.h"
#include "types.h"
#include "parse.h"
#include "rtscall.h"
#include "util.h"

#ifndef WCHAR_TYPE_SIZE
#ifdef INT_TYPE_SIZE
#define WCHAR_TYPE_SIZE INT_TYPE_SIZE
#else
#define WCHAR_TYPE_SIZE BITS_PER_WORD
#endif
#endif

static int is_pascal_loop_check PARAMS ((tree));

/* Nonzero means the expression being parsed will never be evaluated.
   This is a count, since unevaluated expressions can nest.  */
int skip_evaluation;

enum attrs {A_PACKED, A_NOCOMMON, A_COMMON, A_NORETURN, A_CONST, A_T_UNION,
            A_CONSTRUCTOR, A_DESTRUCTOR, A_MODE, A_SECTION, A_ALIGNED,
            A_UNUSED, A_WEAK, A_ALIAS};

#ifdef EGCS97
static void declare_hidden_char_array   PARAMS ((const char *, const char *));
#else
static void declare_hidden_char_array   PARAMS ((char *, char *));
#endif
static void add_attribute               PARAMS ((enum attrs, char *,
                                               int, int, int));
static void init_attributes             PARAMS ((void));

/* Make bindings for __FUNCTION__ and __PRETTY_FUNCTION__.  */

void
declare_function_name ()
{
#ifdef EGCS97
  const char *name, *printable_name;
#else
  char *name, *printable_name;
#endif

  if (current_function_decl == NULL)
    {
      name = "";
      printable_name = "top level";
    }
  else
    {
      /* Allow functions to be nameless (such as artificial ones).  */
      if (DECL_NAME (current_function_decl))
        name = IDENTIFIER_POINTER (DECL_NAME (current_function_decl));
      else
        name = "";
      printable_name = (*decl_printable_name) (current_function_decl, 2);
    }

  declare_hidden_char_array ("__FUNCTION__", name);
  declare_hidden_char_array ("__PRETTY_FUNCTION__", printable_name);
}

static void
declare_hidden_char_array (name, value)
#ifdef EGCS97
     const char *name, *value;
#else
     char *name, *value;
#endif
{
  tree decl, type, init;
  int vlen;

  /* If the default size of char arrays isn't big enough for the name,
     or if we want to give warnings for large objects, make a bigger one.  */
  vlen = strlen (value) + 1;
  type = char_array_type_node;
  if (TREE_INT_CST_LOW (TYPE_MAX_VALUE (TREE_TYPE (type))) < vlen
      || warn_larger_than)
    type = build_array_type (char_type_node,
                             build_index_type (build_int_2 (vlen, 0)));
#ifndef EGCS97
  push_obstacks_nochange ();
#endif
  decl = build_decl (VAR_DECL, get_identifier (name), type);
  TREE_STATIC (decl) = 1;
  TREE_READONLY (decl) = 1;
  TREE_ASM_WRITTEN (decl) = 1;
  DECL_SOURCE_LINE (decl) = 0;
  DECL_ARTIFICIAL (decl) = 1;
  DECL_IN_SYSTEM_HEADER (decl) = 1;
  DECL_IGNORED_P (decl) = 1;
  init = build_string (vlen, value);
  TREE_TYPE (init) = type;
  DECL_INITIAL (decl) = init;
  finish_decl (pushdecl (decl), init, NULL_TREE);
}

/* Given a chain of STRING_CST nodes,
   concatenate them into one STRING_CST
   and give it a suitable array-of-chars data type.  */

tree
combine_strings (strings, implicit)
     tree strings;
     int implicit;
{
  register tree value, t;
  register int length = 1;
  int wide_length = 0;
  int wide_flag = 0;
  int wchar_bytes = TYPE_PRECISION (wchar_type_node) / BITS_PER_UNIT;
  int nchars;
  tree indices;

  if (TREE_CHAIN (strings))
    {
      /* More than one in the chain, so concatenate.  */
      register char *p, *q;

      if (implicit)
        {
          if (PEDANTIC (B_D_PASCAL))
            error ("string catenation without `Concat' or `+' is a GNU Pascal extension");
        }
      else
        {
          if (PEDANTIC (NOT_CLASSIC_PASCAL))
            error ("string catenation with `+' is an ISO 10206 Extended Pascal extension");
        }

      /* Don't include the \0 at the end of each substring,
         except for the last one.
         Count wide strings and ordinary strings separately.  */
      for (t = strings; t; t = TREE_CHAIN (t))
        {
          if (TREE_TYPE (t) == wchar_array_type_node)
            {
              wide_length += (TREE_STRING_LENGTH (t) - wchar_bytes);
              wide_flag = 1;
            }
          else
            length += (TREE_STRING_LENGTH (t) - 1);
        }

      /* If anything is wide, the non-wides will be converted,
         which makes them take more space.  */
      if (wide_flag)
        length = length * wchar_bytes + wide_length;

#ifdef EGCS97
      p = xmalloc (length);
#else
      p = savealloc (length);
#endif

      /* Copy the individual strings into the new combined string.
         If the combined string is wide, convert the chars to ints
         for any individual strings that are not wide.  */

      q = p;
      for (t = strings; t; t = TREE_CHAIN (t))
        {
          int len = (TREE_STRING_LENGTH (t)
                     - ((TREE_TYPE (t) == wchar_array_type_node)
                        ? wchar_bytes : 1));
          if ((TREE_TYPE (t) == wchar_array_type_node) == wide_flag)
            {
              bcopy (TREE_STRING_POINTER (t), q, len);
              q += len;
            }
          else
            {
              int i;
              for (i = 0; i < len; i++)
                {
                  if (WCHAR_TYPE_SIZE == HOST_BITS_PER_SHORT)
                    ((short *) q)[i] = TREE_STRING_POINTER (t)[i];
                  else
                    ((int *) q)[i] = TREE_STRING_POINTER (t)[i];
                }
              q += len * wchar_bytes;
            }
        }
      if (wide_flag)
        {
          int i;
          for (i = 0; i < wchar_bytes; i++)
            *q++ = 0;
        }
      else
        *q = 0;

      value = make_node (STRING_CST);
      TREE_STRING_POINTER (value) = p;
      TREE_STRING_LENGTH (value) = length;
      TREE_CONSTANT (value) = 1;
    }
  else
    {
      value = strings;
      length = TREE_STRING_LENGTH (value);
      if (TREE_TYPE (value) == wchar_array_type_node)
        wide_flag = 1;
    }

  /* Compute the number of elements, for the array type.  */
  nchars = wide_flag ? length / wchar_bytes : length;

  /* In Pascal, strings start at index 1.
   *
   * Also, take out the trailing NULL from the upper boundary
   * (except for the empty string --> kludge)-:.
   *
   * Now: TREE_STRING_LENGTH includes the NULL domain indices don't.
   * Why? To output C-compatible strings and to avoid changing
   *      the files that output assembler code.
   * How: The only place where the trailing null is meaningful
   *      is when we ASSIGN the string constant somewhere.
   *
   *      This is handled in expand_assignment() in expr.c
   */
  if (nchars <= 1)
    indices = build_index_2_type (build_int_2 (1, 0), build_int_2 (1, 0));
  else
    indices = build_index_2_type (build_int_2 (1, 0),
                  build_int_2 (nchars - (wide_flag ? wchar_bytes : 1), 0));

  /* Create the array type for the string constant.
     -Wwrite-strings says make the string constant an array of const char
     so that copying it to a non-const pointer will get a warning.  */
  if (warn_write_strings && (! flag_traditional && ! flag_writable_strings))
    {
      tree elements
        = p_build_type_variant (wide_flag ? wchar_type_node : char_type_node,
                                1, 0);
      TREE_TYPE (value) = build_array_type (elements, indices);
    }
  else
    TREE_TYPE (value)
      = build_array_type (wide_flag ? wchar_type_node : char_type_node,
                          indices);
  /* In addition, the string (array) constant is `packed' */
  PASCAL_TYPE_PACKED (TREE_TYPE (value)) = 1;
  TREE_CONSTANT (value) = 1;
  TREE_STATIC (value) = 1;
  return value;
}

/* Same as above, but insert "\n\t" between the strings. */
tree asm_combine_strings (strings)
     tree strings;
{
  tree scan = strings;
  while (TREE_CHAIN (scan))
    {
      tree newline_tab = build_string (3, "\n\t");
                                    /* Does the trailing NUL count here??? */
      TREE_CHAIN (newline_tab) = TREE_CHAIN (scan);
      TREE_CHAIN (scan) = newline_tab;
      scan = TREE_CHAIN (newline_tab);
    }
  return combine_strings (strings, 1);
}

/* To speed up processing of attributes, we maintain an array of
   IDENTIFIER_NODES and the corresponding attribute types.  */

/* Array to hold attribute information.  */

static struct {enum attrs id; tree name; int min, max, decl_req;} attrtab[50];

static int attrtab_idx = 0;

/* Add an entry to the attribute table above.  */

static void
add_attribute (id, string, min_len, max_len, decl_req)
     enum attrs id;
     char *string;
     int min_len, max_len;
     int decl_req;
{
  char buf[100];

  attrtab[attrtab_idx].id = id;
  attrtab[attrtab_idx].name = get_identifier (string);
  attrtab[attrtab_idx].min = min_len;
  attrtab[attrtab_idx].max = max_len;
  attrtab[attrtab_idx++].decl_req = decl_req;

  sprintf (buf, "__%s__", string);

  attrtab[attrtab_idx].id = id;
  attrtab[attrtab_idx].name = get_identifier (buf);
  attrtab[attrtab_idx].min = min_len;
  attrtab[attrtab_idx].max = max_len;
  attrtab[attrtab_idx++].decl_req = decl_req;
}

/* Initialize attribute table.  */

static void
init_attributes ()
{
  add_attribute (A_PACKED, "packed", 0, 0, 0);
  add_attribute (A_NOCOMMON, "nocommon", 0, 0, 1);
  add_attribute (A_COMMON, "common", 0, 0, 1);
  add_attribute (A_NORETURN, "noreturn", 0, 0, 1);
  add_attribute (A_NORETURN, "volatile", 0, 0, 1);
  add_attribute (A_UNUSED, "unused", 0, 0, 0);
  add_attribute (A_CONST, "const", 0, 0, 1);
  add_attribute (A_T_UNION, "transparent_union", 0, 0, 0);
  add_attribute (A_CONSTRUCTOR, "constructor", 0, 0, 1);
  add_attribute (A_DESTRUCTOR, "destructor", 0, 0, 1);
  add_attribute (A_MODE, "mode", 1, 1, 1);
  add_attribute (A_SECTION, "section", 1, 1, 1);
  add_attribute (A_ALIGNED, "aligned", 0, 1, 0);
  add_attribute (A_WEAK, "weak", 0, 0, 1);
  add_attribute (A_ALIAS, "alias", 1, 1, 1);
}

/* Process the attributes listed in ATTRIBUTES and PREFIX_ATTRIBUTES
   and install them in NODE, which is either a DECL (including a TYPE_DECL)
   or a TYPE.  PREFIX_ATTRIBUTES can appear after the declaration specifiers
   and declaration modifiers but before the declaration proper.  */

void
decl_attributes (node, attributes, prefix_attributes)
     tree node, attributes, prefix_attributes;
{
  tree decl = 0, type = 0;
  int is_type = 0;
  tree a;

  if (attrtab_idx == 0)
    init_attributes ();

  if (TREE_CODE_CLASS (TREE_CODE (node)) == 'd')
    {
      decl = node;
      type = TREE_TYPE (decl);
      is_type = TREE_CODE (node) == TYPE_DECL;
    }
  else if (TREE_CODE_CLASS (TREE_CODE (node)) == 't')
    type = node, is_type = 1;

  attributes = chainon (prefix_attributes, attributes);

  for (a = attributes; a; a = TREE_CHAIN (a))
    {
      tree name = TREE_PURPOSE (a);
      tree args = TREE_VALUE (a);
      int i;
      enum attrs id;

      for (i = 0; i < attrtab_idx; i++)
        if (attrtab[i].name == name)
          break;

      if (i == attrtab_idx)
        {
          if (! valid_machine_attribute (name, args, decl, type))
            warning ("`%s' attribute directive ignored",
                     IDENTIFIER_POINTER (name));
          else if (decl != 0)
            type = TREE_TYPE (decl);
          continue;
        }
      else if (attrtab[i].decl_req && decl == 0)
        {
          warning ("`%s' attribute does not apply to types",
                   IDENTIFIER_POINTER (name));
          continue;
        }
      else if (list_length (args) < attrtab[i].min
               || list_length (args) > attrtab[i].max)
        {
          error ("wrong number of arguments specified for `%s' attribute",
                 IDENTIFIER_POINTER (name));
          continue;
        }

      id = attrtab[i].id;
      switch (id)
        {
        case A_PACKED:
          if (is_type)
            TYPE_PACKED (type) = 1;
          else if (TREE_CODE (decl) == FIELD_DECL)
            DECL_PACKED (decl) = 1;
          /* We can't set DECL_PACKED for a VAR_DECL, because the bit is
             used for DECL_REGISTER.  It wouldn't mean anything anyway.  */
          else
            warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
          break;

        case A_NOCOMMON:
          if (TREE_CODE (decl) == VAR_DECL)
            DECL_COMMON (decl) = 0;
          else
            warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
          break;

        case A_COMMON:
          if (TREE_CODE (decl) == VAR_DECL)
            DECL_COMMON (decl) = 1;
          else
            warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
          break;

        case A_NORETURN:
          if (TREE_CODE (decl) == FUNCTION_DECL)
            TREE_THIS_VOLATILE (decl) = 1;
          else if (TREE_CODE (type) == POINTER_TYPE
                   && TREE_CODE (TREE_TYPE (type)) == FUNCTION_TYPE)
            TREE_TYPE (decl) = type
              = build_pointer_type
                (p_build_type_variant (TREE_TYPE (type),
                                       TREE_READONLY (TREE_TYPE (type)), 1));
          else
            warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
          break;

        case A_UNUSED:
          if (is_type)
            TREE_USED (type) = 1;
          else if (TREE_CODE (decl) == PARM_DECL
                   || TREE_CODE (decl) == VAR_DECL
                   || TREE_CODE (decl) == FUNCTION_DECL)
            TREE_USED (decl) = 1;
          else
            warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
          break;

        case A_CONST:
          if (TREE_CODE (decl) == FUNCTION_DECL)
            TREE_READONLY (decl) = 1;
          else if (TREE_CODE (type) == POINTER_TYPE
                   && TREE_CODE (TREE_TYPE (type)) == FUNCTION_TYPE)
            TREE_TYPE (decl) = type
              = build_pointer_type
                (p_build_type_variant (TREE_TYPE (type), 1,
                                       TREE_THIS_VOLATILE (TREE_TYPE (type))));
          else
            warning ( "`%s' attribute ignored", IDENTIFIER_POINTER (name));
          break;

        case A_T_UNION:
          if (is_type
              && TREE_CODE (type) == UNION_TYPE
              && (decl == 0
                  || (TYPE_FIELDS (type) != 0
                      && TYPE_MODE (type) == DECL_MODE (TYPE_FIELDS (type)))))
            TYPE_TRANSPARENT_UNION (type) = 1;
          else if (decl != 0 && TREE_CODE (decl) == PARM_DECL
                   && TREE_CODE (type) == UNION_TYPE
                   && TYPE_MODE (type) == DECL_MODE (TYPE_FIELDS (type)))
            DECL_TRANSPARENT_UNION (decl) = 1;
          else
            warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
          break;

        case A_CONSTRUCTOR:
          if (TREE_CODE (decl) == FUNCTION_DECL
              && TREE_CODE (type) == FUNCTION_TYPE
              && decl_function_context (decl) == 0)
            {
              DECL_STATIC_CONSTRUCTOR (decl) = 1;
              TREE_USED (decl) = 1;
            }
          else
            warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
          break;

        case A_DESTRUCTOR:
          if (TREE_CODE (decl) == FUNCTION_DECL
              && TREE_CODE (type) == FUNCTION_TYPE
              && decl_function_context (decl) == 0)
            {
              DECL_STATIC_DESTRUCTOR (decl) = 1;
              TREE_USED (decl) = 1;
            }
          else
            warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
          break;

        case A_MODE:
          if (TREE_CODE (TREE_VALUE (args)) != IDENTIFIER_NODE)
            warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
          else
            {
              int j;
              char *p = IDENTIFIER_POINTER (TREE_VALUE (args));
              int len = strlen (p);
              enum machine_mode mode = VOIDmode;
              tree typefm;

              if (len > 4 && p[0] == '_' && p[1] == '_'
                  && p[len - 1] == '_' && p[len - 2] == '_')
                {
                  char *newp = (char *) alloca (len - 1);

                  strcpy (newp, &p[2]);
                  newp[len - 4] = '\0';
                  p = newp;
                }

              /* Give this decl a type with the specified mode.
                 First check for the special modes.  */
              if (! strcmp (p, "byte"))
                mode = byte_mode;
              else if (!strcmp (p, "word"))
                mode = word_mode;
              else if (! strcmp (p, "pointer"))
                mode = ptr_mode;
              else
                for (j = 0; j < NUM_MACHINE_MODES; j++)
                  if (!strcmp (p, GET_MODE_NAME (j)))
                    mode = (enum machine_mode) j;

              if (mode == VOIDmode)
                error ("unknown machine mode `%s'", p);
              else if (0 == (typefm = type_for_mode (mode,
                                                     TREE_UNSIGNED (type))))
                error ("no data type for mode `%s'", p);
              else
                {
                  TREE_TYPE (decl) = type = typefm;
                  DECL_SIZE (decl) = 0;
                  layout_decl (decl, 0);
                }
            }
          break;

        case A_SECTION:
#ifdef ASM_OUTPUT_SECTION_NAME
          if ((TREE_CODE (decl) == FUNCTION_DECL
               || TREE_CODE (decl) == VAR_DECL)
              && TREE_CODE (TREE_VALUE (args)) == STRING_CST)
            {
              if (TREE_CODE (decl) == VAR_DECL
                  && current_function_decl != NULL_TREE
                  && ! TREE_STATIC (decl))
                error_with_decl (decl,
                  "section attribute cannot be specified for local variables");
              /* The decl may have already been given a section attribute from
                 a previous declaration.  Ensure they match.  */
              else if (DECL_SECTION_NAME (decl) != NULL_TREE
                       && strcmp (TREE_STRING_POINTER (DECL_SECTION_NAME (decl)),
                                  TREE_STRING_POINTER (TREE_VALUE (args))) != 0)
                error_with_decl (node,
                                 "section of `%s' conflicts with previous declaration");
              else
                DECL_SECTION_NAME (decl) = TREE_VALUE (args);
            }
          else
            error_with_decl (node,
                           "section attribute not allowed for `%s'");
#else
          error_with_decl (node,
                  "section attributes are not supported for this target");
#endif
          break;

        case A_ALIGNED:
          {
            tree align_expr
              = (args ? TREE_VALUE (args)
                 : size_int (BIGGEST_ALIGNMENT / BITS_PER_UNIT));
            int align;

            /* Strip any NOPs of any kind.  */
            while (TREE_CODE (align_expr) == NOP_EXPR
                   || TREE_CODE (align_expr) == CONVERT_EXPR
                   || TREE_CODE (align_expr) == NON_LVALUE_EXPR)
              align_expr = TREE_OPERAND (align_expr, 0);

            if (TREE_CODE (align_expr) != INTEGER_CST)
              {
                error ("requested alignment is not a constant");
                continue;
              }

            align = TREE_INT_CST_LOW (align_expr) * BITS_PER_UNIT;

            if (exact_log2 (align) == -1)
              error ("requested alignment is not a power of 2");
            else if (is_type)
              TYPE_ALIGN (type) = align;
            else if (TREE_CODE (decl) != VAR_DECL
                     && TREE_CODE (decl) != FIELD_DECL)
              error_with_decl (decl,
                               "alignment may not be specified for `%s'");
            else
              DECL_ALIGN (decl) = align;
          }
          break;

        case A_WEAK:
          declare_weak (decl);
          break;

        case A_ALIAS:
          if ((TREE_CODE (decl) == FUNCTION_DECL && DECL_INITIAL (decl))
              || (TREE_CODE (decl) != FUNCTION_DECL && ! DECL_EXTERNAL (decl)))
            error_with_decl (decl, "`%s' defined both normally and as an alias");
          else if (decl_function_context (decl) == 0)
            {
              tree id = get_identifier (TREE_STRING_POINTER
                                        (TREE_VALUE (args)));
              if (TREE_CODE (decl) == FUNCTION_DECL)
                DECL_INITIAL (decl) = error_mark_node;
              else
                DECL_EXTERNAL (decl) = 0;
              assemble_alias (decl, id);
            }
          else
            warning ("`%s' attribute ignored", IDENTIFIER_POINTER (name));
          break;
        }
    }
}

/* Print a warning if a constant expression had overflow in folding.
   Invoke this function on every expression that the language
   requires to be a constant expression.
   Note the ANSI C standard says it is erroneous for a
   constant expression to overflow. */
void
constant_expression_warning (value)
     tree value;
{
  if ((TREE_CODE (value) == INTEGER_CST || TREE_CODE (value) == REAL_CST
       || TREE_CODE (value) == COMPLEX_CST)
      && TREE_CONSTANT_OVERFLOW (value) && pedantic)
    pedwarn ("overflow in constant expression");
}

/* Print a warning if an expression had overflow in folding.
   Invoke this function on every expression that
   (1) appears in the source code, and
   (2) might be a constant expression that overflowed, and
   (3) is not already checked by convert_and_check;
   however, do not invoke this function on operands of explicit casts. */
void
overflow_warning (value)
     tree value;
{
  if ((TREE_CODE (value) == INTEGER_CST
       || (TREE_CODE (value) == COMPLEX_CST
           && TREE_CODE (TREE_REALPART (value)) == INTEGER_CST))
      && TREE_OVERFLOW (value))
    {
      TREE_OVERFLOW (value) = 0;
      if (skip_evaluation == 0)
        warning ("integer overflow in expression");
    }
  else if ((TREE_CODE (value) == REAL_CST
            || (TREE_CODE (value) == COMPLEX_CST
                && TREE_CODE (TREE_REALPART (value)) == REAL_CST))
           && TREE_OVERFLOW (value))
    {
      TREE_OVERFLOW (value) = 0;
      if (skip_evaluation == 0)
        warning ("floating point overflow in expression");
    }
}

/* Print a warning if a large constant is truncated to unsigned,
   or if -Wconversion is used and a constant < 0 is converted to unsigned.
   Invoke this function on every expression that might be implicitly
   converted to an unsigned type.  */
void
unsigned_conversion_warning (result, operand)
     tree result, operand;
{
  if (TREE_CODE (operand) == INTEGER_CST
      && TREE_CODE (TREE_TYPE (result)) == INTEGER_TYPE
      && TREE_UNSIGNED (TREE_TYPE (result))
      && skip_evaluation == 0
      && !int_fits_type_p (operand, TREE_TYPE (result)))
    {
      if (!int_fits_type_p (operand, signed_type (TREE_TYPE (result))))
        /* This detects cases like converting -129 or 256 to unsigned char.  */
        warning ("large integer implicitly truncated to unsigned type");
      else if (warn_conversion)
        warning ("negative integer implicitly converted to unsigned type");
    }
}

/* Convert EXPR to TYPE, warning about conversion problems with constants.
   Invoke this function on every expression that is converted implicitly,
   i.e. because of language rules and not because of an explicit cast.  */
tree
convert_and_check (type, expr)
     tree type, expr;
{
  tree t = convert (type, expr);
  if (TREE_CODE (t) == INTEGER_CST)
    {
      /* Do some trivial range checks. */
      if (TREE_CODE (expr) == INTEGER_CST
          && ((TYPE_MIN_VALUE (type)
               && TREE_CODE (TYPE_MIN_VALUE (type)) == INTEGER_CST
               && int_cst_lt_always (expr, TYPE_MIN_VALUE (type)))
              || (TYPE_MAX_VALUE (type)
                  && TREE_CODE (TYPE_MAX_VALUE (type)) == INTEGER_CST
                  && int_cst_lt_always (TYPE_MAX_VALUE (type), expr))))
        {
          error ("constant out of range");
          t = integer_zero_node;
        }
      if (TREE_OVERFLOW (t))
        {
          TREE_OVERFLOW (t) = 0;

          /* Do not diagnose overflow in a constant expression merely
             because a conversion overflowed.  */
          TREE_CONSTANT_OVERFLOW (t) = TREE_CONSTANT_OVERFLOW (expr);

          /* No warning for converting 0x80000000 to int.  */
          if (!(TREE_UNSIGNED (type) < TREE_UNSIGNED (TREE_TYPE (expr))
                && TREE_CODE (TREE_TYPE (expr)) == INTEGER_TYPE
                && TYPE_PRECISION (type) == TYPE_PRECISION (TREE_TYPE (expr))))
            /* If EXPR fits in the unsigned version of TYPE,
               don't warn unless pedantic.  */
            if ((pedantic
                 || TREE_UNSIGNED (type)
                 || ! int_fits_type_p (expr, unsigned_type (type)))
                && skip_evaluation == 0)
                warning ("overflow in implicit constant conversion");
        }
      else
        unsigned_conversion_warning (t, expr);
    }
  return t;
}

#if 0  /* Not (yet?) used */
/* Validate the expression in `case' elements and apply default promotions.  */
tree
check_case_value (value)
     tree value;
{
  if (value == NULL_TREE)
    return value;
  /* Strip NON_LVALUE_EXPRs since we aren't using as an lvalue.  */
  STRIP_TYPE_NOPS (value);
  if (TREE_CODE (value) != INTEGER_CST
      && value != error_mark_node)
    {
      error ("case label does not reduce to an ordinal constant");
      value = error_mark_node;
    }
  else
    /* Promote char or short to int. */
    value = default_conversion (value);
  constant_expression_warning (value);
  return value;
}
#endif

/* Return an integer type with BITS bits of precision,
   that is unsigned if UNSIGNEDP is nonzero, otherwise signed. */
tree
type_for_size (bits, unsignedp)
     unsigned bits;
     int unsignedp;
{
  if (bits == TYPE_PRECISION (integer_type_node))
    return unsignedp ? unsigned_type_node : integer_type_node;

  if (bits == TYPE_PRECISION (signed_char_type_node))
    return unsignedp ? unsigned_char_type_node : signed_char_type_node;

  if (bits == TYPE_PRECISION (short_integer_type_node))
    return unsignedp ? short_unsigned_type_node : short_integer_type_node;

  if (bits == TYPE_PRECISION (long_integer_type_node))
    return unsignedp ? long_unsigned_type_node : long_integer_type_node;

  if (bits == TYPE_PRECISION (long_long_integer_type_node))
    return (unsignedp ? long_long_unsigned_type_node
            : long_long_integer_type_node);

  if (bits <= TYPE_PRECISION (intQI_type_node))
    return unsignedp ? unsigned_intQI_type_node : intQI_type_node;

  if (bits <= TYPE_PRECISION (intHI_type_node))
    return unsignedp ? unsigned_intHI_type_node : intHI_type_node;

  if (bits <= TYPE_PRECISION (intSI_type_node))
    return unsignedp ? unsigned_intSI_type_node : intSI_type_node;

  if (bits <= TYPE_PRECISION (intDI_type_node))
    return unsignedp ? unsigned_intDI_type_node : intDI_type_node;

  return 0;
}

/* Return a data type that has machine mode MODE.
   If the mode is an integer,
   then UNSIGNEDP selects between signed and unsigned types.  */
tree
type_for_mode (mode, unsignedp)
     enum machine_mode mode;
     int unsignedp;
{
  if (mode == TYPE_MODE (integer_type_node))
    return unsignedp ? unsigned_type_node : integer_type_node;

  if (mode == TYPE_MODE (signed_char_type_node))
    return unsignedp ? unsigned_char_type_node : signed_char_type_node;

  if (mode == TYPE_MODE (short_integer_type_node))
    return unsignedp ? short_unsigned_type_node : short_integer_type_node;

  if (mode == TYPE_MODE (long_integer_type_node))
    return unsignedp ? long_unsigned_type_node : long_integer_type_node;

  if (mode == TYPE_MODE (long_long_integer_type_node))
    return unsignedp ? long_long_unsigned_type_node : long_long_integer_type_node;

  if (mode == TYPE_MODE (intQI_type_node))
    return unsignedp ? unsigned_intQI_type_node : intQI_type_node;

  if (mode == TYPE_MODE (intHI_type_node))
    return unsignedp ? unsigned_intHI_type_node : intHI_type_node;

  if (mode == TYPE_MODE (intSI_type_node))
    return unsignedp ? unsigned_intSI_type_node : intSI_type_node;

  if (mode == TYPE_MODE (intDI_type_node))
    return unsignedp ? unsigned_intDI_type_node : intDI_type_node;

  if (mode == TYPE_MODE (float_type_node))
    return float_type_node;

  if (mode == TYPE_MODE (double_type_node))
    return double_type_node;

  if (mode == TYPE_MODE (long_double_type_node))
    return long_double_type_node;

  if (mode == TYPE_MODE (build_pointer_type (char_type_node)))
    return build_pointer_type (char_type_node);

  if (mode == TYPE_MODE (build_pointer_type (integer_type_node)))
    return build_pointer_type (integer_type_node);

  return 0;
}

/* Return the minimum number of bits needed to represent VALUE in a
   signed or unsigned type, UNSIGNEDP says which.  */
#ifdef EGCS97
unsigned int
#else
int
#endif
min_precision (value, unsignedp)
     tree value;
     int unsignedp;
{
  int log;

  /* If the value is negative, compute its negative minus 1. The latter
     adjustment is because the absolute value of the largest negative value
     is one larger than the largest positive value. This is equivalent to
     a bit-wise negation, so use that operation instead. */
  if (tree_int_cst_sgn (value) < 0)
    value = fold (build1 (BIT_NOT_EXPR, TREE_TYPE (value), value));

  /* Return the number of bits needed, taking into account the fact
     that we need one more bit for a signed than unsigned type.  */
  if (integer_zerop (value))
    log = 0;
  else if (TREE_INT_CST_HIGH (value) != 0)
    log = HOST_BITS_PER_WIDE_INT + floor_log2 (TREE_INT_CST_HIGH (value));
  else
    log = floor_log2 (TREE_INT_CST_LOW (value));
  return log + 1 + ! unsignedp;
}

/* Print an error message for invalid operands to arith operation CODE.
   NOP_EXPR is used as a special case (see truthvalue_conversion).  */
void
binary_op_error (code)
     enum tree_code code;
{
  register char *opname;
  switch (code)
    {
    case NOP_EXPR:         error ("invalid type conversion"); return;
    case PLUS_EXPR:        opname = "+"; break;
    case MINUS_EXPR:       opname = "-"; break;
    case MULT_EXPR:        opname = "*"; break;
    case MAX_EXPR:         opname = "`Max'"; break;
    case MIN_EXPR:         opname = "`Min'"; break;
    case LE_EXPR:          opname = "<="; break;
    case GE_EXPR:          opname = ">="; break;
    case LT_EXPR:          opname = "<"; break;
    case GT_EXPR:          opname = ">"; break;
#if 0  /* not yet */
    case LROTATE_EXPR:     opname = "`RotateLeft'"; break;
    case RROTATE_EXPR:     opname = "`RotateRight'"; break;
#endif
    case EQ_EXPR:          opname = "="; break;
    case NE_EXPR:          opname = "<>"; break;
    case TRUNC_MOD_EXPR:
    case FLOOR_MOD_EXPR:   opname = "`mod'"; break;
    case TRUNC_DIV_EXPR:
    case FLOOR_DIV_EXPR:   opname = "`div'"; break;
    case BIT_AND_EXPR:     opname = "bitwise `and' or set intersection"; break;
    case BIT_IOR_EXPR:     opname = "bitwise `or' or set union"; break;
    case TRUTH_ANDIF_EXPR: opname = "`and_then'"; break;
    case TRUTH_ORIF_EXPR:  opname = "`or_else'"; break;
    case BIT_XOR_EXPR:     opname = "bitwise `xor' or symmetric set difference"; break;
    case BIT_ANDTC_EXPR:   opname = "set difference"; break;
    case IN_EXPR:          opname = "`in'"; break;
    case TRUTH_AND_EXPR:   opname = "`and'"; break;
    case TRUTH_OR_EXPR:    opname = "`or'"; break;
    case RDIV_EXPR:        opname = "/"; break;
    case LSHIFT_EXPR:      opname = "`shl'"; break;
    case RSHIFT_EXPR:      opname = "`shr'"; break;
    default:               opname = "unknown"; break;
    }
  error ("invalid operands to binary %s", opname);
}

/* Subroutine of get_operator_identifier():
 *
 * Return a TREE_LIST of IDENTIFIER_NODEs representing the names of
 * the type TYPE, if some can be figured out anyhow,
 * otherwise return NULL_TREE.
 */
static tree
get_type_name (type)
     tree type;
{
  tree res = NULL_TREE;
  if (! type || TREE_CODE (type) == ERROR_MARK)
    return res;
  if (TREE_CODE_CLASS (TREE_CODE (type)) == 't' && TYPE_NAME (type))
    type = TYPE_NAME (type);
  if (TREE_CODE (type) == ERROR_MARK)
    return res;
  if (TREE_CODE (type) == IDENTIFIER_NODE)
    return build_tree_list (type, NULL_TREE);
  if (TREE_CODE (type) == TYPE_DECL && DECL_NAME (type) && TREE_CODE (DECL_NAME (type)) == IDENTIFIER_NODE)
    res = chainon (res, build_tree_list (NULL_TREE, DECL_NAME (type)));
  if (TREE_CODE (type) == TYPE_DECL)
    type = TREE_TYPE (type);
  if (((long int) type < 0x1000 && (long int) type > -0x1000) || TREE_CODE (type) == ERROR_MARK)
    return res;

  /* Let's do the same for TYPE_MAIN_VARIANT (type). */
  if (TREE_CODE_CLASS (TREE_CODE (type)) != 't')
    return res;
  type = TYPE_MAIN_VARIANT (type);
  /* @@@@ That's certainly not correct, but this function is
     one of the main sources of gpc1 crashes, and I don't
     understand what's really going on at all. Anyway, on
     Solaris/gcc-2.95.x (fjf536g.pas) type becomes 8 here.
     That's obviously wrong, so kludge it for now ... -- Frank */
  if ((long int) type < 0x1000 && (long int) type > -0x1000)
    return res;
  if (! type || TREE_CODE (type) == ERROR_MARK)
    return res;
  if (TREE_CODE_CLASS (TREE_CODE (type)) == 't' && TYPE_NAME (type))
    type = TYPE_NAME (type);
  else if (TREE_CODE (type) == REFERENCE_TYPE && TYPE_NAME (TREE_TYPE (type)))
    type = TYPE_NAME (TREE_TYPE (type));
  if (TREE_CODE (type) == ERROR_MARK)
    return res;
  if (TREE_CODE (type) == IDENTIFIER_NODE)
    return chainon (res, build_tree_list (NULL_TREE, type));
  if (TREE_CODE (type) == TYPE_DECL && DECL_NAME (type) && TREE_CODE (DECL_NAME (type)) == IDENTIFIER_NODE)
    res = chainon (res, build_tree_list (NULL_TREE, DECL_NAME (type)));
  if (TREE_CODE (type) == TYPE_DECL)
    type = TREE_TYPE (type);

  /* See if we get some names when scanning all variants of this type. */
  type = TYPE_NEXT_VARIANT (TYPE_MAIN_VARIANT (type));
  while (type && TREE_CODE (type) != ERROR_MARK)
    {
      tree t = type;
      if (TREE_CODE_CLASS (TREE_CODE (t)) == 't' && TYPE_NAME (t))
        t = TYPE_NAME (t);
      else if (TREE_CODE (type) == REFERENCE_TYPE && TYPE_NAME (TREE_TYPE (t)))
        t = TYPE_NAME (TREE_TYPE (t));
      if (TREE_CODE (t) == ERROR_MARK)
        break;

      if (t && TREE_CODE (t) == IDENTIFIER_NODE)
        res = chainon (res, build_tree_list (NULL_TREE, t));
      else if (t && TREE_CODE (t) == TYPE_DECL && DECL_NAME (t) && TREE_CODE (DECL_NAME (t)) == IDENTIFIER_NODE)
        res = chainon (res, build_tree_list (NULL_TREE, DECL_NAME (t)));
      if (TREE_CODE_CLASS (TREE_CODE (type)) != 't')
        break;
      type = TYPE_NEXT_VARIANT (type);

      /* @@@@ That's certainly not correct, but this function is
         one of the main sources of gpc1 crashes, and I don't
         understand what's really going on at all. Anyway, on
         Linux/gcc-2.8.1 (tui.pas) type becomes 0x100 here.
         That's obviously wrong, so kludge it for now ... -- Frank */
      if ((long int) type < 0x1000 && (long int) type > -0x1000)
        break;
    }
  return res;
}

/* Return an identifier_node for the assembler-name
 * of a user-defined operator.  Both args may be
 * expression, type or type_decl nodes.
 */
tree
get_operator_identifier (op_name, arg1, arg2, new)
     char *op_name;
     tree arg1, arg2;
     int new;
{
  tree t1, t2;

#if 1
  /* If no user operators have been (or are being) defined,
     don't waste time looking for any. */
  static int operators_defined = 0;
  if (new) operators_defined = 1;
  if (!operators_defined) return NULL_TREE;
#endif

  if (arg1 == error_mark_node || arg2 == error_mark_node)
    return NULL_TREE;

  if (TREE_CODE_CLASS (TREE_CODE (arg1)) != 't')
    {
      if (TREE_CODE (arg1) == TREE_LIST)
        arg1 = TREE_VALUE (arg1);
      else
        arg1 = TREE_TYPE (arg1);
    }

  if (TREE_CODE_CLASS (TREE_CODE (arg2)) != 't')
    {
      if (TREE_CODE (arg2) == TREE_LIST)
        arg2 = TREE_VALUE (arg2);
      else
        arg2 = TREE_TYPE (arg2);
    }

  arg1 = get_type_name (arg1);
  arg2 = get_type_name (arg2);

  for (t1 = arg1; t1; t1 = TREE_CHAIN (t1))
    for (t2 = arg2; t2; t2 = TREE_CHAIN (t2))
      {
        /* @@@@ That's certainly not correct, but this function is
           one of the main sources of gpc1 crashes, and I don't
           understand what's really going on at all. Anyway, on
           Linux/gcc-2.95.x (fjf536g.pas) TREE_VALUE (t1) and
           TREE_VALUE (t2) become 4 here.
           That's obviously wrong, so kludge it for now ... -- Frank */
        if (   !((long int) TREE_VALUE (t1) < 0x1000 && (long int) TREE_VALUE (t1) > -0x1000)
            && !((long int) TREE_VALUE (t2) < 0x1000 && (long int) TREE_VALUE (t2) > -0x1000))
          {
            char *temp = concat (op_name, "_", IDENTIFIER_POINTER (TREE_VALUE (t1)),
                                          "_", IDENTIFIER_POINTER (TREE_VALUE (t2)), NULL_PTR);
            tree result = get_identifier (temp);
            free (temp);
            if (new || lookup_name (result))
              return result;
          }
      }
  return NULL_TREE;
}

/* Convert the operations for sets to their corresponding bitwise
 * boolean operators.  Other that sets, do some checks and call
 * build_binary_op with the supplied args (no conversion).
 *
 * @@@ Here is a great chance to optimize constant sets (later).
 *
 * If the arguments are constructors, this is a good place
 * to convert them to sets since we know how big the other
 * set is we are operating with.
 */
tree
build_pascal_binary_op (code, exp1, exp2)
     enum tree_code code;
     tree exp1;
     tree exp2;
{
  enum tree_code t1 = TREE_CODE (exp1);
  enum tree_code t2 = TREE_CODE (exp2);
  char *op_name;
  tree result;

  if (PASCAL_TYPE_RESTRICTED (TREE_TYPE (exp1))
      || PASCAL_TYPE_RESTRICTED (TREE_TYPE (exp2)))
    error ("invalid operation with a restricted object");

  /* Check for string constants being chars. */
  if (t1 == STRING_CST && TREE_STRING_LENGTH (exp1) == 2)
    {
      exp1 = string_may_be_char (exp1);
      t1 = TREE_CODE (exp1);
    }
  if (t2 == STRING_CST && TREE_STRING_LENGTH (exp2) == 2)
    {
      exp2 = string_may_be_char (exp2);
      t2 = TREE_CODE (exp2);
    }

  /* INTEGER_CSTs are of type `LongestCard', usually.
   * Convert them to something smaller here.
   *
   * @@@@ Overloaded operators must be re-implemented!
   * We must keep a list of operators, try to apply them using
   * usual parameter conversions, trigger an error if it is
   * ambiguous, use the default operator if there is no
   * overloaded one. The same mechanism will work for
   * overloaded functions.
   */
  if (TREE_CODE (TREE_TYPE (exp1)) == INTEGER_TYPE
      && TREE_CODE (TREE_TYPE (exp2)) == INTEGER_TYPE
      && code != TRUNC_MOD_EXPR
      && code != FLOOR_MOD_EXPR)
    {
      tree common_ordinal_type = select_integer_type (exp1, exp2, code);
      exp1 = convert (common_ordinal_type, exp1);
      exp2 = convert (common_ordinal_type, exp2);
    }

  /* Look if this is a user-defined operator
   * which must be converted to a function call
   */
  switch (code)
    {
      case PLUS_EXPR:        op_name = "plus";    break;
      case MINUS_EXPR:       op_name = "minus";   break;
      case MULT_EXPR:        op_name = "mult";    break;
      case RDIV_EXPR:        op_name = "rdiv";    break;
      case TRUNC_DIV_EXPR:   op_name = "div";     break;
      case TRUNC_MOD_EXPR:
      case FLOOR_MOD_EXPR:   op_name = "mod";     break;
      case EXPON_EXPR:       op_name = "power";   break;
      case LEX_POW:          op_name = "pow";     break;
      case IN_EXPR:          op_name = "in";      break;
      case LT_EXPR:          op_name = "lt";      break;
      case EQ_EXPR:          op_name = "eq";      break;
      case GT_EXPR:          op_name = "gt";      break;
      case NE_EXPR:          op_name = "neq";     break;
      case GE_EXPR:          op_name = "gte";     break;
      case LE_EXPR:          op_name = "lte";     break;
      case TRUTH_OR_EXPR:
      case TRUTH_ORIF_EXPR:
      case BIT_IOR_EXPR:     op_name = "or";      break;
      case TRUTH_AND_EXPR:
      case TRUTH_ANDIF_EXPR:
      case BIT_AND_EXPR:     op_name = "and";     break;
      case TRUTH_XOR_EXPR:
      case BIT_XOR_EXPR:     op_name = "xor";     break;
      case LEX_SYMDIFF:      op_name = "symdiff"; break;
      case LSHIFT_EXPR:      op_name = "shl";     break;
      case RSHIFT_EXPR:      op_name = "shr";     break;
      default:               op_name = NULL;
    }
  if (op_name)
    {
      tree func, e1 = exp1, e2 = exp2;

      /* For "fresh" constants use the most basic type,
       * e.g. `Real', not `LongReal'.
       * @@@@ This used to convert *all* real-type expressions to
       * `double', but I don't remember why.  I have changed this,
       * hoping that this does not break anything.  PG
       */
      if (TREE_CODE (e1) == INTEGER_CST
          && PASCAL_TREE_FRESH_INT_CST (e1))
        e1 = integer_type_node;
      else if (TREE_CODE (e1) == REAL_CST)
        e1 = double_type_node;
      if (TREE_CODE (e2) == INTEGER_CST
          && PASCAL_TREE_FRESH_INT_CST (e2))
        e2 = integer_type_node;
      else if (TREE_CODE (e2) == REAL_CST)
        e2 = double_type_node;

      func = get_operator_identifier (op_name, e1, e2, 0);
      if (func)
        func = lookup_name (func);
      if (func && TREE_CODE (func) == FUNCTION_DECL)
        {
          tree arg = build_tree_list (NULL_TREE, exp1);
          arg = chainon (arg, build_tree_list (NULL_TREE, exp2));
          return build_function_call (func, arg);
        }
      else if (TREE_CODE (exp1) == INTEGER_CST
               || TREE_CODE (exp1) == REAL_CST
               || TREE_CODE (exp2) == INTEGER_CST
               || TREE_CODE (exp2) == REAL_CST)
        {
          /* Retry with the "longest" versions of the types. */
          e1 = exp1;
          e2 = exp2;
          if (TREE_CODE (e1) == INTEGER_CST
              && PASCAL_TREE_FRESH_INT_CST (e1))
            e1 = long_long_integer_type_node;
          else if (TREE_CODE (e1) == REAL_CST)
            e1 = long_double_type_node;
          if (TREE_CODE (e2) == INTEGER_CST
              && PASCAL_TREE_FRESH_INT_CST (e2))
            e2 = long_long_integer_type_node;
          else if (TREE_CODE (e2) == REAL_CST)
            e2 = long_double_type_node;

          func = get_operator_identifier (op_name, e1, e2, 0);
          if (func)
            func = lookup_name (func);
          if (func && TREE_CODE (func) == FUNCTION_DECL)
            {
              tree arg = build_tree_list (NULL_TREE, exp1);
              arg = chainon (arg, build_tree_list (NULL_TREE, exp2));
              return build_function_call (func, arg);
            }
        }
    }

  /* @@ This was in the parser before, but for overloaded operators both
        the type checking and the joining of `pow' and `**' is too early
        there (maur7.pas). It's probably not the best place here, but
        operator overloading will have to be rewritten, anyway, etc. ...
        @@ LEX_POW is not really an `enum tree_code', I hope noone will
           notice (in particular not the compiler ;-). -- Frank */
  if (code == EXPON_EXPR)
    {
      if (TREE_CODE (TREE_TYPE (exp2)) == INTEGER_TYPE)
        exp2 = convert (TREE_TYPE (real_zero_node), exp2);
      if (TREE_CODE (TREE_TYPE (exp2)) != REAL_TYPE)
        {
          error ("`**' exponent is not of real or integer type");
          return error_mark_node;
        }
    }
  else if (code == LEX_POW)
    {
      code = EXPON_EXPR;
      if (TREE_CODE (TREE_TYPE (exp2)) != INTEGER_TYPE)
        {
          error ("`pow' exponent is not of integer type");
          return error_mark_node;
        }
    }
  /* @@ Similarly as the previous case */
  else if (code == LEX_SYMDIFF)
    {
      code = BIT_XOR_EXPR;
      if (TREE_CODE (TREE_TYPE (exp1)) != SET_TYPE || TREE_CODE (TREE_TYPE (exp2)) != SET_TYPE)
        {
          error ("operands of `><' must be sets");
          return error_mark_node;
        }
    }

  /* Call functions. */
  if (TREE_CODE (exp1) == FUNCTION_DECL)
    exp1 = probably_call_function (exp1);
  if (TREE_CODE (exp2) == FUNCTION_DECL)
    exp2 = probably_call_function (exp2);

  /* Convert set constructors to sets. */
  if (t1 == CONSTRUCTOR)
    exp1 = construct_set (exp1, NULL_TREE, 1);
  t1 = TREE_CODE (TREE_TYPE (exp1));
  if (t2 == CONSTRUCTOR)
    exp2 = construct_set (exp2, NULL_TREE, 1);
  t2 = TREE_CODE (TREE_TYPE (exp2));

  if (t1 == ERROR_MARK || t2 == ERROR_MARK)
    return error_mark_node;

  if (code == EXPON_EXPR)
    return build_rts_call ((t2 == INTEGER_TYPE) ? r_POW : r_EXPON,
                           chainon (build_tree_list (NULL_TREE, exp1),
                                    build_tree_list (NULL_TREE, exp2)));

  /* Handle IN_EXPR, reverse the arguments for expr.c. */
  if (code == IN_EXPR)
    {
      if (TREE_CODE (TREE_TYPE (TREE_TYPE (exp2))) == VOID_TYPE)
        {
          warning ("`... in []' (empty set) is always `False'");
          return build (COMPOUND_EXPR, boolean_type_node, exp1, boolean_false_node);  /* evaluate exp1! */
        }

      if (comptypes (TYPE_MAIN_VARIANT (TREE_TYPE (TREE_TYPE (exp2))),
                     TYPE_MAIN_VARIANT (TREE_TYPE (exp1))) == 0)
        error ("incompatible operands to `in'");

      if (TREE_CODE (exp2) == CONSTRUCTOR)
        {
          /* Optimize `foo in [a, b .. c]' to become
             `(foo = a) or ((foo >= b) and (foo <= c))'. */
          tree elem, exp, result = NULL_TREE;
          exp = save_expr (exp1);
          for (elem = CONSTRUCTOR_ELTS (exp2); elem; elem = TREE_CHAIN (elem))
            {
              tree min_c = TREE_PURPOSE (elem);
              tree max_c = TREE_VALUE (elem);
              tree condition;

              STRIP_NOPS (min_c);
              STRIP_NOPS (max_c);

              /* !max_c: old-style set constructor, should not happen */
              assert (max_c);

              if (comptypes (TYPE_MAIN_VARIANT (TREE_TYPE (exp1)), TYPE_MAIN_VARIANT (TREE_TYPE (min_c))) == 0
                  || comptypes (TYPE_MAIN_VARIANT (TREE_TYPE (exp1)), TYPE_MAIN_VARIANT (TREE_TYPE (max_c))) == 0)
                binary_op_error (code);

              if (max_c == min_c)
                {
                  /* This is a single element, not a range. */
                  condition = build_pascal_binary_op (EQ_EXPR, exp, min_c);
                }
              else
                {
                  /* This is a range. Avoid warning in the case
                     `<unsigned value> in [0 .. <whatever>]'. */
                  if (integer_zerop (min_c)
                      && TREE_UNSIGNED (TREE_TYPE (exp)))
                    condition = build_pascal_binary_op (LE_EXPR, exp, max_c);
                  else
                    {
                      condition = build_pascal_binary_op (GE_EXPR, exp, min_c);
                      condition = build_pascal_binary_op (TRUTH_ANDIF_EXPR,
                                      condition,
                                      build_pascal_binary_op (LE_EXPR, exp, max_c));
                    }
                }
              if (result)
                result = build_pascal_binary_op (TRUTH_ORIF_EXPR,
                                                 result, condition);
              else
                result = condition;
            }
#if 0
          if (result)
            return result;
          else
            {
              /* @@ Is this still needed (see above)? */
              warning ("`... in []' (empty set) is always `False'");
              return build (COMPOUND_EXPR, boolean_type_node, exp1, boolean_false_node);  /* evaluate exp1! */
            }
#else
          assert (result);
          return result;
#endif
        }
      else
        /* No optimization possible. Create an RTS call. */
        return rts_call (set_in, boolean_type_node, ptype_set (1, ptype_int),
                         chainon (actual_set_parameters (exp2),
                                  build_tree_list (NULL_TREE, convert (integer_type_node, exp1))));
    }

  /* All string and char types are compatible in Extended Pascal. */
  if (is_string_compatible_type (exp1, 1)
      && is_string_compatible_type (exp2, 1)
      && (!(t1 == CHAR_TYPE && t2 == CHAR_TYPE)
          || code == PLUS_EXPR
          || is_variable_string_type (TREE_TYPE (exp1))
          || is_variable_string_type (TREE_TYPE (exp2))))
    {
      int rts_code = 0;
      switch (code)
        {
          case EQ_EXPR: rts_code = '=';    break;
          case NE_EXPR: rts_code = LEX_NE; break;
          case LT_EXPR: rts_code = '<';    break;
          case LE_EXPR: rts_code = LEX_LE; break;
          case GT_EXPR: rts_code = '>';    break;
          case GE_EXPR: rts_code = LEX_GE; break;
          default: /* NOTHING */;
        }
      if (rts_code)
        {
          if (t1 != t2 && PEDANTIC (NOT_CLASSIC_PASCAL))
            {
              pedwarn ("ISO 7185 Pascal does not allow comparison of");
              pedwarn (" different string and char types");
            }
          return build_rts_call (rts_code,
                                 chainon (build_tree_list (NULL_TREE, exp1),
                                          build_tree_list (NULL_TREE, exp2)));
        }

      /* All string catenations are handled here. */
      if (code == PLUS_EXPR)
        {
          if ((TREE_CODE (exp1) == STRING_CST
               || TREE_CODE (exp1) == INTEGER_CST)
              && (TREE_CODE (exp2) == STRING_CST
                  || TREE_CODE (exp2) == INTEGER_CST))
            {
              /* Concatenation of string constants.
               * Treat it in a special way to preserve constantness.
               *
               * But first transform char constants (back)-: to
               * string constants.
               */
              if (TREE_CODE (exp1) == INTEGER_CST)
                {
                  char buffer[2];
                  buffer[0] = TREE_INT_CST_LOW (exp1);
                  buffer[1] = 0;
                  exp1 = build_string (2, buffer);
                }
              if (TREE_CODE (exp2) == INTEGER_CST)
                {
                  char buffer[2];
                  buffer[0] = TREE_INT_CST_LOW (exp2);
                  buffer[1] = 0;
                  exp2 = build_string (2, buffer);
                }
              TREE_CHAIN (exp1) = exp2;
              return combine_strings (exp1, 0);
            }
          else
            {
              /* Length of the combined strings */
              tree len1 = PASCAL_STRING_LENGTH (exp1);
              tree len2 = PASCAL_STRING_LENGTH (exp2);
              tree length = build_binary_op (PLUS_EXPR, len1, len2, 0);

              /* Create a new string object */
              tree nstr = alloca_string (length);
              tree sval = PASCAL_STRING_VALUE (nstr);
              tree str_addr = build_unary_op (ADDR_EXPR, sval, 0);

              /* Assign the first string to the new object */
              if (t1 == CHAR_TYPE)
                expand_expr_stmt (
                  build_modify_expr (
                    build_array_ref (sval, integer_one_node),
                    NOP_EXPR,
                    exp1));
              else
                expand_expr_stmt (
                  build_string_move (
                    str_addr, build1 (ADDR_EXPR,
                                      string_type_node,
                                      PASCAL_STRING_VALUE (exp1)),
                    len1));

              /* Catenate the second string to the first */
              if (t2 == CHAR_TYPE)
                expand_expr_stmt (
                  build_modify_expr (
                    build_array_ref (sval,
                                     build_binary_op (PLUS_EXPR, len1,
                                                      integer_one_node, 0)),
                                     NOP_EXPR, exp2));
              else
                expand_expr_stmt (
                  build_string_move (
                    build (PLUS_EXPR, string_type_node, str_addr, len1, 0),
                    build1 (ADDR_EXPR, string_type_node,
                            PASCAL_STRING_VALUE (exp2)), len2));

              /* Save the combined length of strings */
              expand_expr_stmt (build_modify_expr (PASCAL_STRING_LENGTH (nstr),
                                                   NOP_EXPR, length));
              return non_lvalue (nstr);
            }
        }
    }

  if ((t1 == SET_TYPE || t1 == CONSTRUCTOR)
      && (t2 == SET_TYPE || t2 == CONSTRUCTOR))
    {
      int rts_id = 0;
      int negate = 0, empty2;
      tree params, result_type = NULL_TREE, temp;
      if (! comptypes (TYPE_MAIN_VARIANT (TREE_TYPE (exp1)), TYPE_MAIN_VARIANT (TREE_TYPE (exp2))))
        error ("operation on incompatible sets");
      if (code == BIT_XOR_EXPR && PEDANTIC (E_O_PASCAL))
        error ("symmetric set difference is an ISO 10206 Extended Pascal extension");

      empty2 = TREE_CODE (TREE_TYPE (TREE_TYPE (exp2))) == VOID_TYPE;

      /* In some cases a constant ([], False or True) is returned,
         but one expression must be evaluated for side-effects.
         Note that for [], the COMPOUND_EXPR must have the type of
         the non-empty set, otherwise routines that process the
         expression would regard it as the empty set constant
         because of its type, and would not evaluate the expression. */
      if (TREE_CODE (TREE_TYPE (TREE_TYPE (exp1))) == VOID_TYPE)
        switch (code)
          {
            case PLUS_EXPR:
            case BIT_XOR_EXPR:
              return exp2;
            case MINUS_EXPR:
            case MULT_EXPR:
              return build (COMPOUND_EXPR, TREE_TYPE (exp2), exp2,
                       construct_set (build_set_constructor (NULL_TREE), TREE_TYPE (exp2), 1));
            case GT_EXPR:
              warning ("`>' comparison of the empty set is always false");
              return build (COMPOUND_EXPR, boolean_type_node, exp2, boolean_false_node);
            case LE_EXPR:
              warning ("`<=' comparison of the empty set is always true");
              return build (COMPOUND_EXPR, boolean_type_node, exp2, boolean_true_node);
            case EQ_EXPR:
            case GE_EXPR:
              if (empty2)
                return boolean_true_node;
              rts_id = set_isempty;
              exp1 = exp2;
              break;
            case NE_EXPR:
            case LT_EXPR:
              if (empty2)
                return boolean_false_node;
              rts_id = set_isempty;
              exp1 = exp2;
              negate = 1;
              break;
            default: /* NOTHING */;
          }
      if (!rts_id && empty2)
        switch (code)
          {
            case PLUS_EXPR:
            case MINUS_EXPR:
            case BIT_XOR_EXPR:
              return exp1;
            case MULT_EXPR:
              return build (COMPOUND_EXPR, TREE_TYPE (exp1), exp1,
                            construct_set (build_set_constructor (NULL_TREE), TREE_TYPE (exp1), 1));
            case LT_EXPR:
              warning ("`<' comparison against the empty set is always false");
              return build (COMPOUND_EXPR, boolean_type_node, exp1, boolean_false_node);
            case GE_EXPR:
              warning ("`>=' comparison against the empty set is always true");
              return build (COMPOUND_EXPR, boolean_type_node, exp1, boolean_true_node);
            case EQ_EXPR:
            case LE_EXPR:
              rts_id = set_isempty;
              break;
            case NE_EXPR:
            case GT_EXPR:
              rts_id = set_isempty;
              negate = 1;
              break;
            default: /* NOTHING */;
          }
      if (!rts_id)
        switch (code)
          {
            case PLUS_EXPR:
              rts_id = set_union;
              break;
            case MINUS_EXPR:
              rts_id = set_diff;
              break;
            case MULT_EXPR:
              rts_id = set_intersection;
              break;
            case BIT_XOR_EXPR:
              rts_id = set_symdiff;
              break;
            case EQ_EXPR:
              rts_id = set_equal;
              result_type = boolean_type_node;
              break;
            case NE_EXPR:
              rts_id = set_equal;
              result_type = boolean_type_node;
              negate = 1;
              break;
            case LT_EXPR:
              rts_id = set_less;
              result_type = boolean_type_node;
              break;
            case GT_EXPR:
              rts_id = set_less;
              result_type = boolean_type_node;
              temp = exp1;
              exp1 = exp2;
              exp2 = temp;
              break;
            case LE_EXPR:
              rts_id = set_le;
              result_type = boolean_type_node;
              break;
            case GE_EXPR:
              rts_id = set_le;
              result_type = boolean_type_node;
              temp = exp1;
              exp1 = exp2;
              exp2 = temp;
              break;
            default:
              /* No other set operations are defined. */
              binary_op_error (code);
              return error_mark_node;
          }
      params = actual_set_parameters (exp1);
      if (rts_id == set_isempty)
        result = rts_call (rts_id, boolean_type_node, ptype_set (1, NULL_TREE), params);
      else
        {
          tree temp;
          params = chainon (params, actual_set_parameters (exp2));
          if (!result_type)
            {
              /* For the result type, use the union of ranges of the operands */
              tree domain1 = TYPE_DOMAIN (TREE_TYPE (exp1)),
                   domain2 = TYPE_DOMAIN (TREE_TYPE (exp2)),
                   main1 = TYPE_MAIN_VARIANT (TREE_TYPE (TREE_TYPE (exp1))),
                   main2 = TYPE_MAIN_VARIANT (TREE_TYPE (TREE_TYPE (exp2))),
                   low1  = convert (main1, TYPE_MIN_VALUE (domain1)),
                   low2  = convert (main2, TYPE_MIN_VALUE (domain2)),
                   high1 = convert (main1, TYPE_MAX_VALUE (domain1)),
                   high2 = convert (main2, TYPE_MAX_VALUE (domain2)),
                   low   = build_pascal_binary_op (MIN_EXPR, low1, low2),
                   high  = build_pascal_binary_op (MAX_EXPR, high1, high2),
                   range = build_pascal_range_type (low, high);
              result_type = build_set_type (build_tree_list (range, convert_type_to_range (range)), 0);
              result = make_new_variable ("set_result", result_type);
              params = chainon (params, actual_set_parameters (result));
              /* The ADDR_EXPR and build_indirect_ref are there so it remains an lvalue */
              temp = build_unary_op (ADDR_EXPR, result, 0);
              result = build_indirect_ref (
                         build (COMPOUND_EXPR, TREE_TYPE (temp),
                           rts_call (rts_id, void_type_node,
                                     ptype_set (0, ptype_set (1, ptype_set (1, NULL_TREE))), params),
                           temp),
                         "set result");
              return result;
            }
          else
            result = rts_call (rts_id, result_type, ptype_set (1, ptype_set (1, NULL_TREE)), params);
        }
      if (negate)
        result = build_pascal_unary_op (TRUTH_NOT_EXPR, result, 0);
      return result;
    }

  if (! flag_extended_syntax
      && (t1 == POINTER_TYPE || t2 == POINTER_TYPE)
      && code != EQ_EXPR
      && code != NE_EXPR)
    {
      error ("only `=' and `<>' operators are allowed for pointers");
      ptrarith_inform ();
    }

  /* @@@@ Do some type-checking (fjf547*.pas). This is by far not enough.
     What's the real mechanism for type-checking? -- Frank */
  switch (code)
    {
      case PLUS_EXPR:
      case MINUS_EXPR:
      case MULT_EXPR:
      case RDIV_EXPR:
      case TRUNC_DIV_EXPR:
      case TRUNC_MOD_EXPR:
      case FLOOR_MOD_EXPR:
      case EXPON_EXPR:
      case LSHIFT_EXPR:
      case RSHIFT_EXPR:
        if (!((IS_NUMERIC (t1) && IS_NUMERIC (t2))
             || (flag_extended_syntax && t1 == POINTER_TYPE && (t2 == POINTER_TYPE || t2 == INTEGER_TYPE))))
          binary_op_error (code);
      default: /* NOTHING */;
    }

  /* Check for division by zero. */
  switch (code)
    {
      case RDIV_EXPR:
      case TRUNC_DIV_EXPR:
      case FLOOR_DIV_EXPR:
        if (integer_zerop (exp2))
          error ("division by zero");
        break;
      case FLOOR_MOD_EXPR:
        {
          tree temp;
          if (integer_zerop (exp2))
            error ("zero modulus");
          exp1 = save_expr (exp1);
          exp2 = save_expr (exp2);
          temp = build_binary_op (code, exp1, exp2, 0);
          /* @@@@ This is kind of a kludge for the case when signed
             and unsigned types are used together to avoid wrap-around.
             (fjf434c.pas) It might be better to select a longer result
             type (unless both are the longest types already)-:. */
          if (!TREE_UNSIGNED (TREE_TYPE (exp1)))
            {
              /* (-exp1) mod exp2 */
              tree temp2 = convert (TREE_TYPE (exp1),
                             build_binary_op (code, build_unary_op (NEGATE_EXPR, exp1, 0), exp2, 0));
              /* exp1 >= 0 ? exp1 mod exp2 : temp2 == 0 ? 0 : exp2 - temp2 */
              temp = build (COND_EXPR, TREE_TYPE (exp1),
                       build_pascal_binary_op (GE_EXPR, exp1, integer_zero_node),
                       convert (TREE_TYPE (exp1), temp),
                       build (COND_EXPR, TREE_TYPE (exp1),
                         build_pascal_binary_op (EQ_EXPR, temp2, integer_zero_node),
                           convert (TREE_TYPE (exp1), integer_zero_node),
                           convert (TREE_TYPE (exp1), build_binary_op (MINUS_EXPR, exp2, temp2, 0))));
            }
          return build (COND_EXPR, TREE_TYPE (exp1), build_pascal_binary_op (LE_EXPR, exp2, integer_zero_node),
                        build (COMPOUND_EXPR, TREE_TYPE (exp1),
                               rts_call (gpc_RUNTIME_ERROR, void_type_node,
                                         ptype_int,
                                         build_tree_list (NULL_TREE,
                                                          build_int_2 (714, 0))),  /* second operand of `mod'' is <= 0 */
                               integer_zero_node),
                        temp);
        }
      case TRUNC_MOD_EXPR:
        if (integer_zerop (exp2))
          error ("zero modulus");
        /* @@@@ This is kind of a kludge for the case when signed
           and unsigned types are used together to avoid wrap-around.
           (fjf434a.pas) It might be better to select a longer result
           type (unless both are the longest types already)-:. */
        if (! TREE_UNSIGNED (TREE_TYPE (exp2)))
          exp2 = build_unary_op (ABS_EXPR, exp2, 0);
        if (! TREE_UNSIGNED (TREE_TYPE (exp1)))
          {
            exp1 = save_expr (exp1);
            exp2 = save_expr (exp2);
            result = build (COND_EXPR, TREE_TYPE (exp1), build_pascal_binary_op (LT_EXPR, exp1, integer_zero_node),
                       build_unary_op (NEGATE_EXPR,
                         convert (TREE_TYPE (exp1),
                           build_binary_op (code,
                             build_unary_op (NEGATE_EXPR, exp1, 0), exp2, 0)), 0),
                       convert (TREE_TYPE (exp1),
                         build_binary_op (code, exp1, exp2, 0)));
            result = build (COMPOUND_EXPR, TREE_TYPE (result), exp2, result);
            return result;
          }
        break;
      default: /* NOTHING */;
    }
  result = build_binary_op (code, exp1, exp2, 0);
  switch (TREE_CODE (result))
    {
      case INTEGER_CST:
      case REAL_CST:
      case COMPLEX_CST:
        if (TREE_OVERFLOW (result))
          error ("arithmetical overflow");
        break;
      default: /* NOTHING */;
    }
  return result;
}

tree
build_pascal_unary_op (code, xarg, noconvert)
     enum tree_code code;
     tree xarg;
     int noconvert;
{
  if (PASCAL_TYPE_RESTRICTED (TREE_TYPE (xarg)))
    error ("invalid unary operation with restricted object");

  if (TREE_CODE (xarg) == FUNCTION_DECL)
    xarg = probably_call_function (xarg);

  if (code == NEGATE_EXPR
      && TREE_CODE (TREE_TYPE (xarg)) == INTEGER_TYPE
      && TREE_UNSIGNED (TREE_TYPE (xarg)))
    {
      if (TREE_CODE (xarg) == INTEGER_CST)
        {
          /* Convert unsigned constants to signed ones here
           * and do some trivial range checking.
           */
          if (INT_CST_LT_UNSIGNED (
                  build_int_2 (0, (HOST_WIDE_INT) -1 << (HOST_BITS_PER_WIDE_INT - 1)),
                  xarg))
            {
              error ("value does not fit in longest integer type");
              return integer_zero_node;
            }
          else
            {
              int fresh = PASCAL_TREE_FRESH_INT_CST (xarg);

              xarg = copy_node (xarg);
              TREE_TYPE (xarg) = long_long_integer_type_node;
              TREE_UNSIGNED (xarg) = 0;
              xarg = build_unary_op (code, xarg, 0);

              /* -0x8000000000000000 is no overflow, although
               * 0x8000000000000000 does not fit in a signed integer type.
               */
              TREE_OVERFLOW (xarg) = 0;

              /* Note whether the type of this integer constant was guessed
               * by the lexical analzyer.
               */
              PASCAL_TREE_FRESH_INT_CST (xarg) = fresh;

              return xarg;
            }
        }
      else if (! noconvert)
        xarg = convert (select_signed_integer_type (TREE_TYPE (xarg)), xarg);
    }
  return build_unary_op (code, xarg, noconvert);
}

/* Convert a PXSC operator expression to a function call.
 * The functions are not (yet) built-in but must be supported
 * (overloaded) by the user program.
 */
tree
build_pxsc_operator (op_name, arg1, arg2)
     char *op_name;
     tree arg1, arg2;
{
  /* pxsc operator expression */
  /* is converted to a function call */
  tree op_func, arg;
  if (PASCAL_TYPE_RESTRICTED (TREE_TYPE (arg1))
      || PASCAL_TYPE_RESTRICTED (TREE_TYPE (arg2)))
    error ("invalid binary operation with restricted object");
  if (PEDANTIC (PASCAL_SC))
    error ("overloaded and rounding operators are Pascal-SC extensions");
  op_func = lookup_name (get_operator_identifier (op_name, arg1, arg2, 0));
  if (op_func)
    {
      arg = build_tree_list (NULL_TREE, arg1);
      arg = chainon (arg, build_tree_list (NULL_TREE, arg2));
      return build_function_call (op_func, arg);
    }
  else
    {
      error ("PXSC operators are not built-in but must be overloaded");
      return error_mark_node;
    }
}

/* Dereference a pointer. This may result in a function call. */
tree
build_pascal_pointer_reference (pointer)
     tree pointer;
{
  tree result, fun_type;

  if (TREE_CODE (pointer) == TYPE_DECL)
    {
      error ("trying to dereference a type rather than an expression");
      return error_mark_node;
    }

  if (PASCAL_TYPE_RESTRICTED (TREE_TYPE (pointer)))
    error ("dereferencing a restricted object is not allowed");

  if (TREE_CODE (TREE_TYPE (pointer)) == FILE_TYPE)
    result = build_buffer_ref (pointer, r_LAZYTRYGET);
  else if (MAYBE_CALL_FUNCTION (pointer)
           && (fun_type = (TREE_CODE (pointer) == FUNCTION_DECL
                           ? TREE_TYPE (pointer)
                           : TREE_TYPE (TREE_TYPE (pointer))),
               TYPE_ARG_TYPES (fun_type)
               && TREE_CODE (TREE_VALUE (TYPE_ARG_TYPES (fun_type))) == VOID_TYPE))
    {
      int function_calls = allow_function_calls ();
      result = maybe_call_function (pointer, empty_arglist);
      resume_function_calls (function_calls);
      if (TREE_CODE (pointer) == FUNCTION_DECL)  /* direct function call */
        result = build_pascal_pointer_reference (result);
    }
  else
    {
      warn_void_dereference++;
      result = build_indirect_ref (pointer, "`^'");
      warn_void_dereference--;
    }

  return result;
}

/* Return an expression for the address of FACTOR.
 * In most cases, this is an ADDR_EXPR, but it may also be a cast of a
 * reference to a pointer.
 */
tree
build_pascal_address_expression (factor)
     tree factor;
{
  tree result;

  /* Undo implicit schema dereferences. */
  if (TREE_CODE (factor) == COMPONENT_REF
      && TREE_CODE (TREE_OPERAND (factor, 1)) == FIELD_DECL
      && (DECL_NAME (TREE_OPERAND (factor, 1)) == schema_id
                        || DECL_NAME (TREE_OPERAND (factor, 1)) == string_id))
    factor = TREE_OPERAND (factor, 0);

  /* If `foo' is a procedure reference, `@foo' is a
   * type cast to a procedure pointer.
   */
  if (PASCAL_PROCEDURAL_VARIABLE (factor))
    {
      tree ftype = TREE_TYPE (TREE_TYPE (factor));
      result = convert (build_pointer_type (ftype), factor);
    }
  else
    {
      /* Don't call build_pascal_unary_op() which would
       * call the function.
       */
      result = build_unary_op (ADDR_EXPR, factor, 0);

      /* Mark constant addresses as such (for initialization). */
      if (TREE_CODE (factor) == VAR_DECL
          || TREE_CODE (factor) == FUNCTION_DECL)
        TREE_CONSTANT (result) = 1;
    }

  if (!flag_typed_address)
    result = build_c_cast (ptr_type_node, result);

  return result;
}

static int
is_pascal_loop_check (op)
     tree op;
{
  STRIP_NOPS (op);
  return PASCAL_LOOP_CHECK (op);
}

/* Subroutine of build_binary_op, used for comparison operations.
   See if the operands have both been converted from subword integer types
   and, if so, perhaps change them both back to their original type.
   This function is also responsible for converting the two operands
   to the proper common type for comparison.

   The arguments of this function are all pointers to local variables
   of build_binary_op: OP0_PTR is &OP0, OP1_PTR is &OP1,
   RESTYPE_PTR is &RESULT_TYPE and RESCODE_PTR is &RESULTCODE.

   If this function returns nonzero, it means that the comparison has
   a constant value.  What this function returns is an expression for
   that value.  */
tree
shorten_compare (op0_ptr, op1_ptr, restype_ptr, rescode_ptr)
     tree *op0_ptr, *op1_ptr;
     tree *restype_ptr;
     enum tree_code *rescode_ptr;
{
  register tree type;
  tree op0 = *op0_ptr;
  tree op1 = *op1_ptr;
  int unsignedp0, unsignedp1;
  int real1, real2;
  tree primop0, primop1;
  enum tree_code code = *rescode_ptr;

  /* Throw away any conversions to wider types
     already present in the operands.  */

  primop0 = get_narrower (op0, &unsignedp0);
  primop1 = get_narrower (op1, &unsignedp1);

  /* @@@@ This is probably wrong (or not optimal) in general, but it fixes
          (couper1.pas); the whole stuff is just too confusing (seems to
          be still more C than Pascal) to make any sense of. :-( -- Frank */
  if (TYPE_PRECISION (TREE_TYPE (primop0)) % BITS_PER_UNIT != 0)
    {
      primop0 = op0;
      unsignedp0 = TREE_UNSIGNED (op0);
    }
  if (TYPE_PRECISION (TREE_TYPE (primop1)) % BITS_PER_UNIT != 0)
    {
      primop1 = op1;
      unsignedp1 = TREE_UNSIGNED (op1);
    }

  /* Handle the case that OP0 does not *contain* a conversion
     but it *requires* conversion to FINAL_TYPE.  */

  if (op0 == primop0 && TREE_TYPE (op0) != *restype_ptr)
    unsignedp0 = TREE_UNSIGNED (TREE_TYPE (op0));
  if (op1 == primop1 && TREE_TYPE (op1) != *restype_ptr)
    unsignedp1 = TREE_UNSIGNED (TREE_TYPE (op1));

  /* If one of the operands must be floated, we cannot optimize.  */
  real1 = TREE_CODE (TREE_TYPE (primop0)) == REAL_TYPE;
  real2 = TREE_CODE (TREE_TYPE (primop1)) == REAL_TYPE;

  /* If first arg is constant, swap the args (changing operation
     so value is preserved), for canonicalization.  Don't do this if
     the second arg is 0.  */

  if (TREE_CONSTANT (primop0)
      && ! integer_zerop (primop1) && ! real_zerop (primop1))
    {
      register tree tem = primop0;
      register int temi = unsignedp0;
      primop0 = primop1;
      primop1 = tem;
      tem = op0;
      op0 = op1;
      op1 = tem;
      *op0_ptr = op0;
      *op1_ptr = op1;
      unsignedp0 = unsignedp1;
      unsignedp1 = temi;
      temi = real1;
      real1 = real2;
      real2 = temi;

      switch (code)
        {
        case LT_EXPR:
          code = GT_EXPR;
          break;
        case GT_EXPR:
          code = LT_EXPR;
          break;
        case LE_EXPR:
          code = GE_EXPR;
          break;
        case GE_EXPR:
          code = LE_EXPR;
          break;
        default:
          break;
        }
      *rescode_ptr = code;
    }

  /* If comparing an integer against a constant more bits wide,
     maybe we can deduce a value of 1 or 0 independent of the data.
     Or else truncate the constant now
     rather than extend the variable at run time.

     This is only interesting if the constant is the wider arg.
     Also, it is not safe if the constant is unsigned and the
     variable arg is signed, since in this case the variable
     would be sign-extended and then regarded as unsigned.
     Our technique fails in this case because the lowest/highest
     possible unsigned results don't follow naturally from the
     lowest/highest possible values of the variable operand.
     For just EQ_EXPR and NE_EXPR there is another technique that
     could be used: see if the constant can be faithfully represented
     in the other operand's type, by truncating it and reextending it
     and see if that preserves the constant's value.  */

  if (!real1 && !real2
      && TREE_CODE (primop1) == INTEGER_CST
      && TYPE_PRECISION (TREE_TYPE (primop0)) < TYPE_PRECISION (*restype_ptr))
    {
      int min_gt, max_gt, min_lt, max_lt;
      tree maxval, minval;
      /* 1 if comparison is nominally unsigned.  */
      int unsignedp = TREE_UNSIGNED (*restype_ptr);
      tree val;

      type = signed_or_unsigned_type (unsignedp0, TREE_TYPE (primop0));

      maxval = TYPE_MAX_VALUE (type);
      minval = TYPE_MIN_VALUE (type);

      if (unsignedp && !unsignedp0)
        *restype_ptr = signed_type (*restype_ptr);

      if (TREE_TYPE (primop1) != *restype_ptr)
        primop1 = convert (*restype_ptr, primop1);
      if (type != *restype_ptr)
        {
          minval = convert (*restype_ptr, minval);
          maxval = convert (*restype_ptr, maxval);
        }

      if (unsignedp && unsignedp0)
        {
          min_gt = INT_CST_LT_UNSIGNED (primop1, minval);
          max_gt = INT_CST_LT_UNSIGNED (primop1, maxval);
          min_lt = INT_CST_LT_UNSIGNED (minval, primop1);
          max_lt = INT_CST_LT_UNSIGNED (maxval, primop1);
        }
      else
        {
          min_gt = INT_CST_LT (primop1, minval);
          max_gt = INT_CST_LT (primop1, maxval);
          min_lt = INT_CST_LT (minval, primop1);
          max_lt = INT_CST_LT (maxval, primop1);
        }

      val = 0;
      /* This used to be a switch, but Genix compiler can't handle that.  */
      if (code == NE_EXPR)
        {
          if (max_lt || min_gt)
            val = boolean_true_node;
        }
      else if (code == EQ_EXPR)
        {
          if (max_lt || min_gt)
            val = boolean_false_node;
        }
      else if (code == LT_EXPR)
        {
          if (max_lt)
            val = boolean_true_node;
          if (!min_lt)
            val = boolean_false_node;
        }
      else if (code == GT_EXPR)
        {
          if (min_gt)
            val = boolean_true_node;
          if (!max_gt)
            val = boolean_false_node;
        }
      else if (code == LE_EXPR)
        {
          if (!max_gt)
            val = boolean_true_node;
          if (min_gt)
            val = boolean_false_node;
        }
      else if (code == GE_EXPR)
        {
          if (!min_lt)
            val = boolean_true_node;
          if (max_lt)
            val = boolean_false_node;
        }

      /* If primop0 was sign-extended and unsigned comparison specd,
         we did a signed comparison above using the signed type bounds.
         But the comparison we output must be unsigned.

         Also, for inequalities, VAL is no good; but if the signed
         comparison had *any* fixed result, it follows that the
         unsigned comparison just tests the sign in reverse
         (positive values are LE, negative ones GE).
         So we can generate an unsigned comparison
         against an extreme value of the signed type.  */

      if (unsignedp && !unsignedp0)
        {
          if (val != 0)
            switch (code)
              {
              case LT_EXPR:
              case GE_EXPR:
                primop1 = TYPE_MIN_VALUE (type);
                val = 0;
                break;

              case LE_EXPR:
              case GT_EXPR:
                primop1 = TYPE_MAX_VALUE (type);
                val = 0;
                break;

              default:
                break;
              }
          type = unsigned_type (type);
        }

      /* Don't complain of the limited range comparisons when checking `for' loop bounds. */
      if (! is_pascal_loop_check (op0)
          && !max_gt && !unsignedp0 && TREE_CODE (primop0) != INTEGER_CST)
        {
          if (val == boolean_false_node)
            warning ("comparison always yields `False' due to limited range of data type");
          if (val == boolean_true_node)
            warning ("comparison always yields `True' due to limited range of data type");
        }

      if (! is_pascal_loop_check (op0)
          && !min_lt && unsignedp0 && TREE_CODE (primop0) != INTEGER_CST)
        {
          if (val == boolean_false_node)
            warning ("comparison is always `False' due to limited range of data type");
          if (val == boolean_true_node)
            warning ("comparison is always `True' due to limited range of data type");
        }

      if (val != 0)
        {
          /* Don't forget to evaluate PRIMOP0 if it has side effects.  */
          if (TREE_SIDE_EFFECTS (primop0))
            return build (COMPOUND_EXPR, TREE_TYPE (val), primop0, val);
          return val;
        }

      /* Value is not predetermined, but do the comparison
         in the type of the operand that is not constant.
         TYPE is already properly set. */
    }
  else if (real1 && real2
           && (TYPE_PRECISION (TREE_TYPE (primop0))
               == TYPE_PRECISION (TREE_TYPE (primop1))))
    type = TREE_TYPE (primop0);

  /* If args' natural types are both narrower than nominal type
     and both extend in the same manner, compare them
     in the type of the wider arg.
     Otherwise must actually extend both to the nominal
     common type lest different ways of extending
     alter the result.
     (eg, (short)-1 == (unsigned short)-1  should be 0.)  */

  else if (unsignedp0 == unsignedp1 && real1 == real2
           && TYPE_PRECISION (TREE_TYPE (primop0)) < TYPE_PRECISION (*restype_ptr)
           && TYPE_PRECISION (TREE_TYPE (primop1)) < TYPE_PRECISION (*restype_ptr))
    {
      type = common_type (TREE_TYPE (primop0), TREE_TYPE (primop1));
      type = signed_or_unsigned_type (unsignedp0
                                      || TREE_UNSIGNED (*restype_ptr),
                                      type);
      /* Make sure shorter operand is extended the right way
         to match the longer operand.  */
      primop0 = convert (signed_or_unsigned_type (unsignedp0, TREE_TYPE (primop0)),
                         primop0);
      primop1 = convert (signed_or_unsigned_type (unsignedp1, TREE_TYPE (primop1)),
                         primop1);
    }
  else
    {
      /* Here we must do the comparison on the nominal type
         using the args exactly as we received them. */
      type = *restype_ptr;
      primop0 = op0;
      primop1 = op1;

      if (!real1 && !real2 && integer_zerop (primop1)
          && TREE_UNSIGNED (*restype_ptr))
        {
          tree value = 0;
          switch (code)
            {
            case GE_EXPR:
              /* All unsigned values are >= 0, so we warn if extra warnings
                 are requested. However, if OP0 is a constant that is >= 0,
                 the signedness of the comparison isn't an issue, so suppress
                 the warning. */
              if (! is_pascal_loop_check (op0)
                  && TREE_CODE (primop0) != INTEGER_CST)
                warning ("comparison `unsigned value >= 0' always yields `True'");
              value = boolean_true_node;
              break;

            case LT_EXPR:
              if (! is_pascal_loop_check (op0)
                  && TREE_CODE (primop0) != INTEGER_CST)
                warning ("comparison `unsigned value < 0' always yields `False'");
              value = boolean_false_node;
              break;

            default:
              break;
            }

          if (value != 0)
            {
              /* Don't forget to evaluate PRIMOP0 if it has side effects.  */
              if (TREE_SIDE_EFFECTS (primop0))
                return build (COMPOUND_EXPR, TREE_TYPE (value),
                              primop0, value);
              return value;
            }
        }
    }
  *op0_ptr = convert (type, primop0);
  *op1_ptr = convert (type, primop1);
  *restype_ptr = boolean_type_node;
  return 0;
}

/* Prepare expr to be an argument of a TRUTH_NOT_EXPR,
   or validate its data type for an `if' or `while' statement or ?..: exp.
   This preparation consists of taking the ordinary
   representation of an expression expr and producing a valid tree
   boolean expression describing whether expr is nonzero.  We could
   simply always do build_binary_op (NE_EXPR, expr, boolean_false_node, 1),
   but we optimize comparisons, &&, ||, and !.
   The resulting type should always be `boolean_type_node'.  */
tree
truthvalue_conversion (expr)
     tree expr;
{
  switch (TREE_CODE (expr))
    {
    case ERROR_MARK:
      return expr;

    case EQ_EXPR:
    case NE_EXPR: case LE_EXPR: case GE_EXPR: case LT_EXPR: case GT_EXPR:
    case TRUTH_ANDIF_EXPR:
    case TRUTH_ORIF_EXPR:
    case TRUTH_AND_EXPR:
    case TRUTH_OR_EXPR:
    case TRUTH_XOR_EXPR:
    case TRUTH_NOT_EXPR:
      assert (TREE_TYPE (expr) == boolean_type_node);
      return expr;

    case INTEGER_CST:
      return integer_zerop (expr) ? boolean_false_node : boolean_true_node;

    case REAL_CST:
      return real_zerop (expr) ? boolean_false_node : boolean_true_node;

    case ADDR_EXPR:
      /* If we are taking the address of a external decl, it might be zero
         if it is weak, so we cannot optimize.  */
      if (TREE_CODE_CLASS (TREE_CODE (TREE_OPERAND (expr, 0))) == 'd'
          && DECL_EXTERNAL (TREE_OPERAND (expr, 0)))
        break;

      if (TREE_SIDE_EFFECTS (TREE_OPERAND (expr, 0)))
        return build (COMPOUND_EXPR, boolean_type_node,
                      TREE_OPERAND (expr, 0), boolean_true_node);
      else
        return boolean_true_node;

    case COMPLEX_EXPR:
      return build_binary_op ((TREE_SIDE_EFFECTS (TREE_OPERAND (expr, 1))
                               ? TRUTH_OR_EXPR : TRUTH_ORIF_EXPR),
                              truthvalue_conversion (TREE_OPERAND (expr, 0)),
                              truthvalue_conversion (TREE_OPERAND (expr, 1)),
                              0);

    case NEGATE_EXPR:
    case ABS_EXPR:
    case FLOAT_EXPR:
    case FFS_EXPR:
      /* These don't change whether an object is non-zero or zero.  */
      return truthvalue_conversion (TREE_OPERAND (expr, 0));

    case LROTATE_EXPR:
    case RROTATE_EXPR:
      /* These don't change whether an object is zero or non-zero, but
         we can't ignore them if their second arg has side-effects.  */
      if (TREE_SIDE_EFFECTS (TREE_OPERAND (expr, 1)))
        return build (COMPOUND_EXPR, boolean_type_node, TREE_OPERAND (expr, 1),
                      truthvalue_conversion (TREE_OPERAND (expr, 0)));
      else
        return truthvalue_conversion (TREE_OPERAND (expr, 0));

    case COND_EXPR:
      /* Distribute the conversion into the arms of a COND_EXPR.  */
      return fold (build (COND_EXPR, boolean_type_node, TREE_OPERAND (expr, 0),
                          truthvalue_conversion (TREE_OPERAND (expr, 1)),
                          truthvalue_conversion (TREE_OPERAND (expr, 2))));

    case CONVERT_EXPR:
      /* Don't cancel the effect of a CONVERT_EXPR from a REFERENCE_TYPE,
         since that affects how `default_conversion' will behave.  */
      if (TREE_CODE (TREE_TYPE (expr)) == REFERENCE_TYPE
          || TREE_CODE (TREE_TYPE (TREE_OPERAND (expr, 0))) == REFERENCE_TYPE)
        break;
      /* fall through...  */
    case NOP_EXPR:
      /* If this is widening the argument, we can ignore it.  */
      if (TYPE_PRECISION (TREE_TYPE (expr))
          >= TYPE_PRECISION (TREE_TYPE (TREE_OPERAND (expr, 0))))
        return truthvalue_conversion (TREE_OPERAND (expr, 0));
      break;

    case MINUS_EXPR:
      /* With IEEE arithmetic, x - x may not equal 0, so we can't optimize
         this case.  */
      if (TARGET_FLOAT_FORMAT == IEEE_FLOAT_FORMAT
          && TREE_CODE (TREE_TYPE (expr)) == REAL_TYPE)
        break;
      /* fall through...  */
    case BIT_XOR_EXPR:
      /* This and MINUS_EXPR can be changed into a comparison of the
         two objects.  */
      if (TREE_TYPE (TREE_OPERAND (expr, 0))
          == TREE_TYPE (TREE_OPERAND (expr, 1)))
        return build_binary_op (NE_EXPR, TREE_OPERAND (expr, 0),
                                TREE_OPERAND (expr, 1), 1);
      return build_binary_op (NE_EXPR, TREE_OPERAND (expr, 0),
                              fold (build1 (NOP_EXPR,
                                            TREE_TYPE (TREE_OPERAND (expr, 0)),
                                            TREE_OPERAND (expr, 1))), 1);

    case BIT_AND_EXPR:
      if (integer_onep (TREE_OPERAND (expr, 1))
          && TREE_TYPE (expr) != boolean_type_node)
        /* Using convert here would cause infinite recursion.  */
        return build1 (NOP_EXPR, boolean_type_node, expr);
      break;

    case MODIFY_EXPR:
      if (warn_parentheses && C_EXP_ORIGINAL_CODE (expr) == MODIFY_EXPR)
        warning ("suggest parentheses around assignment used as truth value");
      break;

    default:
      break;
    }

  if (TREE_CODE (TREE_TYPE (expr)) == COMPLEX_TYPE)
    return (build_binary_op
            ((TREE_SIDE_EFFECTS (expr)
              ? TRUTH_OR_EXPR : TRUTH_ORIF_EXPR),
             truthvalue_conversion (build_unary_op (REALPART_EXPR, expr, 0)),
             truthvalue_conversion (build_unary_op (IMAGPART_EXPR, expr, 0)),
             0));

  return build_binary_op (NE_EXPR, expr, integer_zero_node, 1);
}

#ifdef EGCS
/* Make a variant type in the proper way for C/C++, propagating qualifiers
   down to the element type of an array.  */
tree
c_build_qualified_type (type, type_quals)
     tree type;
     int type_quals;
{
  /* A restrict-qualified pointer type must be a pointer to object or
     incomplete type.  Note that the use of POINTER_TYPE_P also allows
     REFERENCE_TYPEs, which is appropriate for C++.  Unfortunately,
     the C++ front-end also use POINTER_TYPE for pointer-to-member
     values, so even though it should be invalid to use `restrict'
     with such an entity we don't flag that here.  Thus, special case
     code for that case is required in the C++ front-end.  */
  if ((type_quals & TYPE_QUAL_RESTRICT)
      && (!POINTER_TYPE_P (type)
          || !C_TYPE_OBJECT_OR_INCOMPLETE_P (TREE_TYPE (type))))
    {
      error ("invalid use of `restrict'");
      type_quals &= ~TYPE_QUAL_RESTRICT;
    }

  if (TREE_CODE (type) == ARRAY_TYPE)
    return build_array_type (c_build_qualified_type (TREE_TYPE (type),
                                                     type_quals),
                             TYPE_DOMAIN (type));
  return build_qualified_type (type, type_quals);
}
#else
tree
c_build_type_variant (type, constp, volatilep)
     tree type;
     int constp, volatilep;
{
  if (TREE_CODE (type) == ARRAY_TYPE)
    return build_array_type (c_build_type_variant (TREE_TYPE (type),
                                                   constp, volatilep),
                             TYPE_DOMAIN (type));
  return p_build_type_variant (type, constp, volatilep);
}
#endif

/* Give TYPE a new main variant. This is the right thing to do only when
   something else about TYPE is modified in place. */
void
new_main_variant (type)
     tree type;
{
  tree t;
  for (t = TYPE_MAIN_VARIANT (type); TYPE_NEXT_VARIANT (t); t = TYPE_NEXT_VARIANT (t))
    if (TYPE_NEXT_VARIANT (t) == type)
      {
        TYPE_NEXT_VARIANT (t) = TYPE_NEXT_VARIANT (type);
        break;
      }
  TYPE_MAIN_VARIANT (type) = type;
  TYPE_NEXT_VARIANT (type) = NULL_TREE;
}

/* Build a Pascal variant of the TYPE.
 *
 * QUALIFIER is a bit mask of the following type qualifiers:
 *   `qualified' is not implemented yet.
 *
 *   `restricted', `protected', `bindable', `packed' are ISO-10206 standard
 *   qualifiers for types. (`protected' is not handled here.)
 *
 *   CONFORMANT is an internal flag for conformant array parameters
 *   and their indices (@@@ Will be re-implemented)
 */
tree
pascal_type_variant (type, qualifier)
     tree type;
     int qualifier;
{
#if 0
  /* handled by c_build_type_variant */
  int protected   = !!(qualifier & TYPE_QUALIFIER_PROTECTED);   /* TREE_READONLY */

                                                                /* 0 & 1 reserved */
  /* handled differently; flag reserved */
  int string      = !!(qualifier & TYPE_QUALIFIER_STRING);      /* 4 */
#endif

  int restricted  = !!(qualifier & TYPE_QUALIFIER_RESTRICTED);  /* 2 */
  int bindable    = !!(qualifier & TYPE_QUALIFIER_BINDABLE);    /* 3 */
                                                                /* 4 strings */
  int conformant  = !!(qualifier & TYPE_QUALIFIER_CONFORMANT);  /* 5 */
  int packed      = !!(qualifier & TYPE_QUALIFIER_PACKED);      /* 6 */
  int qualified   = !!(qualifier & TYPE_QUALIFIER_QUALIFIED);   /* ???? OUT OF BITS */

  int need_new_variant = (bindable   && !PASCAL_TYPE_BINDABLE (type))
#if 0
                      || (qualified  && !PASCAL_TYPE_QUALIFIED (type))
#endif
                      || (packed     && !PASCAL_TYPE_PACKED (type));

  /* Restricted types should not get a new TYPE_MAIN_VARIANT so they're
     compatible to the unrestricted ones in parameter lists. Same
     for conformant -- which is *not* used for the conformant arrays
     themselves, but for their index domain which, of course, must
     be compatible to the ordinary domain so that the domain of
     actual parameters will be compatible. */
  int need_copy = need_new_variant
                  || (conformant && !PASCAL_TYPE_CONFORMANT_INDEX (type))
                  || (restricted && !PASCAL_TYPE_RESTRICTED (type));

  /* @@ Aux info??? */

  if (TREE_CODE (type) == ERROR_MARK)
    return type;

  if (need_copy)
    {
      tree new_type = build_type_copy (type);
      /* This is a new type, so remove it from the variants of the old type */
      if (need_new_variant)
        new_main_variant (new_type);
      type = new_type;
    }

  if (bindable)
    PASCAL_TYPE_BINDABLE (type) = 1;

  if (conformant)
    PASCAL_TYPE_CONFORMANT_INDEX (type) = 1;

  if (restricted)
    PASCAL_TYPE_RESTRICTED (type) = 1;

  if (qualified)
    warning ("`qualified' is not yet implemented - ignored");

  if (packed)
    {
      tree tscan = type;
      /* Packed packs the arrays given in the same index list
       * @@@@@@ This packs all subsequent arrays: this does not
       * @@@@@@ know which of the arrays are defined in the same index list.
       * @@@@@@ Parser should flag them somehow...
       */
      do
        {
          PASCAL_TYPE_PACKED (tscan) = 1;
          if (TREE_CODE (tscan) == ARRAY_TYPE)
            tscan = TREE_TYPE (tscan);
        }
      while (TREE_CODE (tscan) == ARRAY_TYPE);
    }

  assert ((qualifier & ~(TYPE_QUALIFIER_RESTRICTED   | TYPE_QUALIFIER_BINDABLE
                         | TYPE_QUALIFIER_CONFORMANT | TYPE_QUALIFIER_PACKED
                         | TYPE_QUALIFIER_QUALIFIED)) == 0);

  /* Not allowed */
  if (PASCAL_TYPE_BINDABLE (type) && PASCAL_TYPE_RESTRICTED (type))
    error ("restricted types must not be bindable");

  return type;
}

/* @@ This is just a wrapper around build_type_variant that resets
      TYPE_LANG_SPECIFIC. Shouldn't it be merged with
      pascal_type_variant() (what's the difference, anyway)? -- Frank */
tree
p_build_type_variant (type, constp, volatilep)
     tree type;
     int constp, volatilep;
{
  tree new_type;
  if (TREE_CODE (type) == ERROR_MARK)
    return type;
  new_type = build_type_variant (type, constp, volatilep);
  if (new_type != type
      && TYPE_LANG_SPECIFIC (new_type)
      && TYPE_LANG_SPECIFIC (new_type) == TYPE_LANG_SPECIFIC (type))
    {
      TYPE_LANG_SPECIFIC (new_type) = allocate_type_lang_specific ();
      memcpy (TYPE_LANG_SPECIFIC (new_type), TYPE_LANG_SPECIFIC (type),
              sizeof (struct lang_type));
    }
  return new_type;
}
