{ Program to dump the contents of GPI files.
  Useful for GPC developers.

  @@ Some additional (probably irrelevant) output is contained in
     `$if False'. One may want to active it (possibly with command
     line options, using GetOpt) ...

  To build it, make sure that tree.inc exists and matches your GCC
  version (otherwise (re-)build it using mk-t-inc), and that GPC can
  find this tree.inc as well as gpi.h in the GPC source directory.
  (Or simply let the Makefile take care of all that. ;-)

  Copyright (C) 2002 Free Software Foundation, Inc.

  Author: Frank Heckenbach <frank@pascal.gnu.de>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License as
  published by the Free Software Foundation, version 2.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING. If not, write to
  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA 02111-1307, USA. }

{$gnu-pascal,I+}
{$if __GPC_RELEASE__ < 20020225}
{$error This prorgam requires GPC release 20020225 or newer.}
{$endif}

program GPIDump;

uses GPC, StringUtils;

var
  Hex: Boolean;

procedure ProcessFile (const GPIFileName: String);
type
  TTreeCode = ({$define DEFTREECODE(ID, NAME, CLASS, ARGS) ID, }
               {$include "tree.inc"}
               {$undef DEFTREECODE}
               LAST_AND_UNUSED_TREE_CODE,
               UnknownTreeCode);

const
  TreeCodeNames: array [TTreeCode] of record
    User, Internal: ^String
  end =
    ({$define DEFTREECODE(ID, NAME, CLASS, ARGS) (@NAME, @#ID), }
     {$include "tree.inc"}
     {$undef DEFTREECODE}
     (@'interface_name_node', @'INTERFACE_NAME_NODE'),
     (@'unknown tree code', @'UNKNOWN'));

{ tree.inc sets some defines used by gpi.h (`EGCS' etc.), so it must
  be included above. }
{$ifndef GCC_VERSION_SET}
{$error tree.inc not included or not correctly generated}
{$endif}
{$include "gpi.h"}

type
  TChunkCode = ({$define GPI_CHUNK(ID, REQUIRED, UNIQUE, NAME) ID }
                GPI_CHUNKS
                {$undef GPI_CHUNK});

const
  Chunks: array [TChunkCode] of record
    Name: PString;
    Required, Unique: Integer
  end = ({$define GPI_CHUNK(ID, REQUIRED, UNIQUE, NAME) (@NAME, REQUIRED, UNIQUE) }
         GPI_CHUNKS
         {$undef GPI_CHUNK});

type
  GPIInt = MedCard;
  TOffsets (Size: Integer) = array [0 .. Size] of GPIInt;

  TBytesChars (Size: Integer) = record
  case Boolean of
    False: (Bytes: array [0 .. Size - 1] of Byte);
    True:  (Chars: array [0 .. Size - 1] of Char);
  end;

var
  f: File;
  Header: array [1 .. Length (GPI_HEADER)] of Char;
  ChunkCodeNum: Byte;
  ChunkCode: TChunkCode;
  ChunksSeen: set of TChunkCode;
  ChunkLength, CheckSum, NodesPos1, NodesPos2, Pos1, Pos2, i: GPIInt;
  Buf, Nodes: ^TBytesChars;
  Offsets: ^TOffsets;
  DebugKey: Boolean;
  SpecialNodes: PPStrings;

  procedure Error (const Msg: String);
  begin
    WriteLn (StdErr, ParamStr (0), ': ', Msg);
    Halt (1)
  end;

  function IsSpecialNode (i: GPIInt; const Name: String): Boolean;
  begin
    IsSpecialNode := (i < SpecialNodes^.Count) and (SpecialNodes^[i + 1]^ = Name)
  end;

  function Number (i: LongestCard; MinDigits: Integer) = s: TString;
  const HexDigits: array [0 .. $f] of Char = '0123456789abcdef';
  var j, k: GPIInt;
  begin
    if not Hex then
      WriteStr (s, i)
    else
      begin
        s := '';
        j := 1;
        for k := 2 to MinDigits do
          j := $10 * j;
        while j <= i div $10 do
          j := $10 * j;
        while j > 0 do
          begin
            s := s + HexDigits[i div j];
            i := i mod j;
            j := j div $10
          end
      end
  end;

  procedure DumpBytes (const Prefix: String; Lo, Hi: Integer);
  var i: Integer;
  begin
    Write (Prefix);
    for i := Lo to Hi do
      begin
        if ((i - Lo) mod 16 = 0) and (i > Lo) then
          begin
            WriteLn;
            Write ('' : Length (Prefix))
          end;
        Write (Number (Buf^.Bytes[i], 2) : 4 - Ord (Hex))
      end
  end;

  procedure DumpChars (const Prefix: String; Lo, Hi: Integer);
  var
    i: Integer;
    SpecialChar, Quote: Boolean;
  begin
    SpecialChar := False;
    for i := Lo to Hi do
      if not IsPrintable (Buf^.Chars[i]) then
        SpecialChar := True;
    if not SpecialChar then
      Write (Prefix, '`', Buf^.Chars[Lo .. Hi], '''')
    else
      begin
        Write (Prefix);
        Quote := False;
        for i := Lo to Hi do
          if IsPrintable (Buf^.Chars[i]) then
            begin
              if not Quote then Write ('`');
              Quote := True;
              Write (Buf^.Chars[i])
            end
          else
            begin
              if Quote then Write ('''');
              Quote := False;
              Write ('#', Number (Ord (Buf^.Chars[i]), 0))
            end;
        if Quote then Write ('''')
      end
  end;

  procedure GetSize (var Pos: GPIInt; var Dest; Size: Integer);
  begin
    Move (Nodes^.Bytes[Pos], Dest, Size);
    Inc (Pos, Size)
  end;

  function GetNumber (var Pos: GPIInt) = Res: GPIInt;
  begin
    GetSize (Pos, Res, SizeOf (Res))
  end;

  { Must match compute_checksum() in module.c }
  function ComputeChecksum (const Buf: array of Byte) = Sum: GPIInt;
  var n: GPIInt;
  begin
    Sum := 0;
    for n := 0 to High (Buf) do
      { @@ $local R- } Inc (Sum, n * Buf[n]) { @@ $endlocal }
  end;

  { @@ This must correspond precisely to what module.c does. }
  procedure DumpNode (NodeNumber, Pos1, Pos2: GPIInt);
  const
    LangCodeNormal         = 0;
    LangCodeVariantRecord  = 1;
    LangCodeObject         = 2;
    LangCodeAbstractObject = 3;
    LangCodeString         = 4;
    LangCodeSchema         = 5;
  var
    First: Boolean;
    a, b, i: GPIInt;
    InterfaceNode, InterfaceUID: GPIInt;
    TreeCodeNum, LangCode: Byte;
    TreeCode: TTreeCode;

    procedure Comma;
    begin
      if First then
        Write (' ')
      else
        Write (', ');
      First := False
    end;

    procedure Comma0;
    begin
      if not First then Write (', ');
      First := False
    end;

    procedure Str (const Prefix: String);
    var l: GPIInt;
    begin
      Comma;
      l := GetNumber (Pos1);
      DumpChars (Prefix, Pos1, Pos1 + l - 1);
      Inc (Pos1, l)
    end;

    procedure Str0 (const Prefix: String);
    var l: GPIInt;
    begin
      Comma;
      l := GetNumber (Pos1) - 1;
      DumpChars (Prefix + ' ', Pos1, Pos1 + l - 1);
      Inc (Pos1, l);
      if Nodes^.Bytes[Pos1] <> 0 then Error ('no #0 terminator');
      Inc (Pos1)
    end;

    procedure OutputRef (i: GPIInt);
    begin
      { Don't write `null_tree_node', but simply 0. }
      if (i > 0 ) and (i < SpecialNodes^.Count) then
        Write (SpecialNodes^[i + 1]^)
      else
        Write (Number (i, 0))
    end;

    procedure Ref (const Prefix: String);
    begin
      Comma;
      Write (Prefix, ' <');
      OutputRef (GetNumber (Pos1));
      Write ('>')
    end;

    procedure OptRef (const Prefix: String);
    var n: GPIInt;
    begin
      n := GetNumber (Pos1);
      if n <> 0 then
        begin
          Comma;
          Write (Prefix, ' <');
          OutputRef (n);
          Write ('>')
        end
    end;

    procedure OutputFlag (const Flag: String);
    begin
      if Flag[1] = '!' then Error ('unexpected flag `' + Copy (Flag, 2) + '''');
      Comma;
      Write (Flag)
    end;

    procedure Flags;
    const
      FlagNames: array [0 .. 23] of PString =
        (@'side_effects',
         @'constant',
         @'permanent',
         @'addressable',
         @'volatile',
         @'readonly',
         @'unsigned',
         @'asm_written',
         @'used',
         @'raises',
         @'static',
         @'public',
         @'private',
         @'protected',
         @'!unused lang flag 0',
         @'#l1',
         @'c_any_field_volatile',
         @'#l3',
         @'#l4',
         @'#l5',
         @'#l6',
         @'!unknown flag #1',
         @'!unknown flag #2',
         @'!unknown flag #3');
      FlagConstant = 1;
      FlagPermanent = 2;
      FlagAsmWritten = 7;
    var
      f: packed record
        Code: Byte;
        Bits: Cardinal (24)
      end;
      i: Integer;
      Constant: Boolean;
      Flag: PString;
    const
      Assert = 1 / Ord (SizeOf (f) = 4);
    begin
      GetSize (Pos1, f, SizeOf (f));
      Constant := TreeCode in [INTEGER_CST, REAL_CST, COMPLEX_CST, STRING_CST];
      if f.Code <> Ord (TreeCode) then
        Error ('tree code in flags does not match');
      if Constant and (f.Bits and (1 shl FlagConstant) = 0) then
        Error ('constant has `constant'' flag not set');
      if f.Bits and (1 shl FlagPermanent) = 0 then
        Error ('`permanent'' flag not set');
      if TreeCode in [RECORD_TYPE, UNION_TYPE, QUAL_UNION_TYPE, ENUMERAL_TYPE] then
        if f.Bits and (1 shl FlagAsmWritten) <> 0 then
          f.Bits := f.Bits and not (1 shl FlagAsmWritten)
        else
          Error ('`asm_written'' flag not set');
      for i := 0 to 23 do
        if not (i in [FlagPermanent, FlagAsmWritten]) and not (Constant and (i = FlagConstant))
           and (f.Bits and (1 shl i) <> 0) then
          begin
            Flag := FlagNames[i];
            if Flag^ = '#l1' then
              case TreeCode of
                RECORD_TYPE,
                UNION_TYPE:    OutputFlag ('c_any_field_readonly');
                LABEL_DECL:    OutputFlag ('c_label_declared');
                else           OutputFlag ('!lang flag 1')
              end
            else if Flag^ = '#l3' then
              if IsSuffix ('_TYPE', TreeCodeNames [TreeCode].Internal^) then
                OutputFlag ('value_parameter_by_reference')
              else
                case TreeCode of
                  VAR_DECL,
                  FIELD_DECL,
                  CONVERT_EXPR:  OutputFlag ('discriminant');
                  FUNCTION_DECL,
                  CALL_EXPR:     OutputFlag ('operator');
                  else           OutputFlag ('!lang flag 3')
                end
            else if Flag^ = '#l4' then
              case TreeCode of
                ARRAY_TYPE,
                INTEGER_TYPE:  OutputFlag ('open_array');
                FIELD_DECL,
                FUNCTION_DECL,
                CALL_EXPR:     OutputFlag ('structor');
                else           OutputFlag ('!lang flag 4')
              end
            else if Flag^ = '#l5' then
              if (TreeCode = VAR_DECL) or IsSuffix ('_EXPR', TreeCodeNames [TreeCode].Internal^) then
                OutputFlag ('packed_access')
              else
                case TreeCode of
                  INTEGER_CST:   OutputFlag ('fresh_int');
                  FUNCTION_DECL: OutputFlag ('virtual');
                  else           OutputFlag ('!lang flag 5')
                end
            else if Flag^ = '#l6' then
              if IsSuffix ('_EXPR', TreeCodeNames [TreeCode].Internal^) then
                OutputFlag ('absolute')
              else
                case TreeCode of
                  FUNCTION_DECL: OutputFlag ('abstract');
                  VAR_DECL:      OutputFlag ('prog_heading');
                  else           OutputFlag ('!lang flag 6')
                end
            else
              OutputFlag (Flag^)
          end
    end;

    procedure TypeFlags;
    const
      FlagNames: array [0 .. 15] of PString =
        (@'string',
         @'no_force_blk',
         @'needs_constructing',
         @'transparent_union',
         @'packed',
         {$ifdef EGCS}
         @'restrict',
         {$endif}
         @'c_type_being_defined',
         @'c_variable_size',
         @'restricted',
         @'bindable',
         @'text_file',
         @'conformant_index',
         @'bp_typed_const',
         @'!unknown type flag #1',
         @'!unknown type flag #2',
         @'!unknown type flag #3'
         {$ifndef EGCS},
         @'!unknown type flag #4'
         {$endif});
    var
      f: packed record
        UID: Cardinal;
        Precision, MachineMode: Byte;
        Bits: packed array [0 .. 15] of Boolean;
        Align: Cardinal
      end;
      i: Integer;
    const
      Assert = 1 / Ord (SizeOf (f) = 4 + 2 * SizeOf (Cardinal));
    begin
      GetSize (Pos1, f, SizeOf (f));
      Comma;
      Write ('type_uid ', Number (f.UID, 0),
             ', precision ', Number (f.Precision, 0),
             ', machine_mode ', Number (f.MachineMode, 0));
      for i := 0 to 15 do
        if f.Bits[i] then OutputFlag (FlagNames[i]^);
      Write (', align ', Number (f.Align, 0))
    end;

    procedure DeclFlags;
    const
      FlagNames: array [0 .. 23] of PString =
        (@'c_external',
         @'nonlocal',
         @'regdecl',
         @'inline',
         @'bit_field',
         @'c_virtual',
         @'ignored',
         @'c_abstract',
         @'in_system_header',
         @'common',
         @'defer_output',
         @'transparent_union',
         @'static_ctor',
         @'static_dtor',
         @'artificial',
         @'weak',
         @'c_variable_size',
         @'!c_typedef_explicitly_signed',
         @'!c_missing_prototype',
         @'!c_decl_anticipated',
         @'c_bitfield',
         @'procedure_parameter',
         @'value_assigned',
         @'redefinable_decl');
    var
      f: packed record
        UID: Cardinal;
        MachineMode: Byte;
        Bits: Cardinal (24);
        FrameSize: Integer
      end;
      i: Integer;
    const
      Assert = 1 / Ord (SizeOf (f) = 4 + SizeOf (Integer) + SizeOf (Cardinal));
    begin
      GetSize (Pos1, f, SizeOf (f));
      Comma;
      Write ('decl_uid ', Number (f.UID, 0),
             ', machine_mode ', Number (f.MachineMode, 0));
      for i := 0 to 23 do
        if f.Bits and (1 shl i) <> 0 then OutputFlag (FlagNames[i]^);
      {$ifdef EGCS97}
      {$error DeclFlags not yet implemented for EGCS97}
      {$else}
      Write (', frame_size ', Number (f.FrameSize, 0))
      {$endif}
    end;

    procedure DeclMode;
    var Mode: Integer;
    begin
      Comma;
      GetSize (Pos1, Mode, SizeOf (Mode));
      Write ('decl_mode ', Number (Mode, 0))
    end;

    procedure IntConst (const Prefix: String);
    var
      Lo: GPIInt;
      Num: LongestInt;
    begin
      Comma;
      Write (Prefix, ' ');
      Lo := GetNumber (Pos1);
      Num := GetNumber (Pos1);
      if Num >= $80000000 then Dec (Num, $100000000);
      Num := $100000000 * Num + Lo;
      if Num = High (Integer) then
        Write ('MaxInt')
      else if Num = Low (Integer) then
        Write ('MinInt')
      else if Num = High (Cardinal) then
        Write ('MaxCard')
      else if Num = High (LongInt) then
        Write ('MaxLongInt')
      else if Num = Low (LongInt) then
        Write ('MinLongInt')
      else if Num = High (LongCard) then
        Write ('MaxLongCard')
      else
        begin
          if Num < 0 then
            begin
              Num := -Num;
              Write ('-')
            end;
          Write (Number (Num, 0))
        end
    end;

    procedure RealConst (const Prefix: String);
    const RealSize = 12;
    begin
      Comma;
      DumpBytes (Prefix + ' (', Pos1, Pos1 + RealSize - 1);
      Write (')');
      Inc (Pos1, RealSize)
    end;

  begin
    if DebugKey and (GetNumber (Pos1) <> GPI_DEBUG_KEY) then
      Error ('invalid debug key');
    InterfaceNode := GetNumber (Pos1);
    InterfaceUID := 0;
    if InterfaceNode <> 0 then InterfaceUID := GetNumber (Pos1);
    GetSize (Pos1, TreeCodeNum, SizeOf (TreeCodeNum));
    if TreeCodeNum >= Ord (High (TreeCode)) then
      TreeCode := High (TreeCode)
    else
      TreeCode := TTreeCode (TreeCodeNum);
    Write ('<', Number (NodeNumber, 0), '>',
           {$if False}
           ' (offset: ', Number (Pos1 + NodesPos1, 0), ' .. ', Number (Pos2 + NodesPos1, 0), ')',
           ', tree code ', Number (TreeCodeNum, 2),
           {$endif}
           ': ', TreeCodeNames [TreeCode].User^);
    if InterfaceNode <> 0 then
      begin
        Write (', interface <');
        OutputRef (InterfaceNode);
        Write ('>: ', Number (InterfaceUID, 0))
      end;
    WriteLn;
    First := True;
    if not (TreeCode in [INTERFACE_NAME_NODE, TREE_LIST]) then Flags;
    if TreeCode in [VOID_TYPE, REAL_TYPE, COMPLEX_TYPE, BOOLEAN_TYPE, CHAR_TYPE,
                    LANG_TYPE, INTEGER_TYPE, ENUMERAL_TYPE, SET_TYPE, POINTER_TYPE,
                    REFERENCE_TYPE, FILE_TYPE, RECORD_TYPE, UNION_TYPE] then
      begin
        TypeFlags;
        Ref ('size');
        {$ifdef EGCS}
        Ref ('size_unit');
        {$endif}
      end;
    case TreeCode of
      INTERFACE_NAME_NODE:
                        begin
                          Str ('interface ');
                          Str ('module ');
                          Write (', checksum ', Number (GetNumber (Pos1), 0))
                        end;
      IDENTIFIER_NODE:  begin
                          Str ('');
                          OptRef ('value')
                        end;
      TREE_LIST:        for i := 1 to GetNumber (Pos1) do
                          begin
                            if (i mod 10 = 1) and (i > 1) then
                              begin
                                WriteLn (',');
                                First := True
                              end;
                            Flags;
                            Comma;
                            a := GetNumber (Pos1);
                            b := GetNumber (Pos1);
                            Write ('<');
                            OutputRef (b);
                            if a <> 0 then
                              begin
                                Write (', ');
                                OutputRef (a)
                              end;
                            Write ('>')
                          end;
      VOID_TYPE,
      REAL_TYPE,
      COMPLEX_TYPE,
      BOOLEAN_TYPE,
      CHAR_TYPE,
      LANG_TYPE,
      INTEGER_TYPE,
      ENUMERAL_TYPE:    begin
                          OptRef ('subrange_of');
                          Ref ('min');
                          Ref ('max');
                          OptRef ('main_variant')
                        end;
      SET_TYPE:         begin
                          Ref ('element_type');
                          Ref ('domain');
                          OptRef ('main_variant')
                        end;
      POINTER_TYPE,
      REFERENCE_TYPE:   begin
                          Ref ('target_type');
                          OptRef ('main_variant')
                        end;
      FILE_TYPE:        begin
                          Ref ('element_type');
                          OptRef ('domain');
                          OptRef ('main_variant')
                        end;
      ARRAY_TYPE:       begin
                          Ref ('element_type');
                          Ref ('domain');
                          TypeFlags;
                          OptRef ('main_variant')
                        end;
      RECORD_TYPE,
      UNION_TYPE:       begin
                          GetSize (Pos1, LangCode, SizeOf (LangCode));
                          if LangCode <> LangCodeNormal then Comma;
                          case LangCode of
                            LangCodeNormal:         ;
                            LangCodeVariantRecord:  Write ('variant_record');
                            LangCodeObject:         Write ('object');
                            LangCodeAbstractObject: Write ('abstract_object');
                            LangCodeString:         Write ('string');
                            LangCodeSchema:         Write ('schema');
                            else                    Error ('unknown lang code')
                          end;
                          case LangCode of
                            LangCodeVariantRecord:  Ref ('variant_tag');
                            LangCodeObject,
                            LangCodeAbstractObject: Ref ('object_name');
                            LangCodeString:         Ref ('declared_capacity');
                            else
                              if GetNumber (Pos1) <> 0 then Error ('unexpected lang_info')
                          end;
                          OptRef ('lang_base');
                          Write (', fields (');
                          First := True;
                          i := 0;
                          repeat
                            a := GetNumber (Pos1);
                            if a = 0 then Break;
                            Inc (i);
                            if (i mod 3 = 1) and (i > 1) then
                              begin
                                WriteLn (',');
                                Write ('  ');
                                First := True;
                              end;
                            Comma0;
                            Write ('<');
                            OutputRef (a);
                            {.$ifndef EGCS97}
                            Write (', bitpos ');
                            OutputRef (GetNumber (Pos1));
                            {.$endif}
                            Write ('>')
                          until False;
                          Write (')');
                          First := False;
                          OptRef ('main_variant')
                        end;
      FUNCTION_TYPE:    begin
                          Ref ('result_type');
                          Ref ('arg_types');
                          TypeFlags;
                          First := True;
                          repeat
                            a := GetNumber (Pos1);
                            if a = 0 then Break;
                            b := GetNumber (Pos1);
                            if First then Write (', attributes (');
                            Comma0;
                            Write ('<');
                            OutputRef (a);
                            Write (', ');
                            OutputRef (b);
                            Write ('>')
                          until False;
                          if not First then Write (')');
                          First := False;
                          OptRef ('main_variant')
                        end;
      INTEGER_CST:      begin
                          IntConst ('value');
                          Ref ('type')
                        end;
      REAL_CST:         begin
                          RealConst ('value');
                          Ref ('type')
                        end;
      COMPLEX_CST:      begin
                          Ref ('re');
                          Ref ('im');
                          Ref ('type')
                        end;
      STRING_CST:       begin
                          Str0 ('value');
                          Ref ('type')
                        end;
      FUNCTION_DECL:    begin
                          Ref ('name');
                          Str ('asmname ');
                          Ref ('result_type');
                          Write (', arguments <');
                          First := True;
                          i := 0;
                          repeat
                            a := GetNumber (Pos1);
                            if (a = 0) or IsSpecialNode (a, 'void_type_node') then Break;
                            Inc (i);
                            if (i mod 10 = 1) and (i > 1) then
                              begin
                                WriteLn (',');
                                Write ('  ');
                                First := True;
                              end;
                            Comma0;
                            OutputRef (a)
                          until False;
                          if a = 0 then Write (', ...');
                          Write ('>');
                          First := True;
                          repeat
                            a := GetNumber (Pos1);
                            if IsSpecialNode (a, 'error_mark_node') then Break;
                            b := GetNumber (Pos1);
                            if First then Write (', attributes (');
                            Comma0;
                            Write ('<');
                            OutputRef (a);
                            Write (', ');
                            OutputRef (b);
                            Write ('>')
                          until False;
                          if not First then Write (')');
                          First := False
                        end;
      LABEL_DECL,
      PARM_DECL,
      RESULT_DECL:      begin
                          Ref ('name');
                          Ref ('type')
                        end;
      FIELD_DECL:       begin
                          DeclFlags;
                          DeclMode;
                          Ref ('name');
                          Ref ('type');
                          Ref ('decl_size');
                          {.$ifndef EGCS97}
                          Ref ('bitpos');
                          {.$endif}
                          OptRef ('fixuplist')
                        end;
      CONST_DECL,
      TYPE_DECL:        begin
                          Ref ('name');
                          Ref ('type');
                          OptRef ('initializer')
                        end;
      VAR_DECL:         begin
                          Ref ('name');
                          Str ('asmname ');
                          Ref ('type');
                          OptRef ('absolute')
                        end;
      CONSTRUCTOR:      begin
                          Ref ('op');
                          Ref ('type')
                        end;
      BIT_FIELD_REF,
      COND_EXPR,
      METHOD_CALL_EXPR: begin
                          Ref ('op0');
                          Ref ('op1');
                          Ref ('op2');
                          Ref ('type')
                        end;
      COMPONENT_REF,
      ARRAY_REF,
      COMPOUND_EXPR,
      MODIFY_EXPR,
      INIT_EXPR,
      CALL_EXPR,
      PLUS_EXPR,
      MINUS_EXPR,
      MULT_EXPR,
      TRUNC_DIV_EXPR,
      CEIL_DIV_EXPR,
      FLOOR_DIV_EXPR,
      ROUND_DIV_EXPR,
      TRUNC_MOD_EXPR,
      FLOOR_MOD_EXPR,
      CEIL_MOD_EXPR,
      ROUND_MOD_EXPR,
      RDIV_EXPR,
      EXACT_DIV_EXPR,
      MIN_EXPR,
      MAX_EXPR,
      LSHIFT_EXPR,
      RSHIFT_EXPR,
      LROTATE_EXPR,
      RROTATE_EXPR,
      BIT_IOR_EXPR,
      BIT_XOR_EXPR,
      BIT_AND_EXPR,
      BIT_ANDTC_EXPR,
      TRUTH_ANDIF_EXPR,
      TRUTH_ORIF_EXPR,
      TRUTH_AND_EXPR,
      TRUTH_OR_EXPR,
      TRUTH_XOR_EXPR,
      LT_EXPR,
      LE_EXPR,
      GT_EXPR,
      GE_EXPR,
      EQ_EXPR,
      NE_EXPR,
      IN_EXPR,
      SET_LE_EXPR,
      RANGE_EXPR,
      COMPLEX_EXPR:     begin
                          Ref ('lhs');
                          Ref ('rhs');
                          Ref ('type')
                        end;
      INDIRECT_REF,
      BUFFER_REF,
      FIX_TRUNC_EXPR,
      FIX_CEIL_EXPR,
      FIX_FLOOR_EXPR,
      FIX_ROUND_EXPR,
      FLOAT_EXPR,
      NEGATE_EXPR,
      ABS_EXPR,
      FFS_EXPR,
      BIT_NOT_EXPR,
      TRUTH_NOT_EXPR,
      CARD_EXPR,
      CONVERT_EXPR,
      NOP_EXPR,
      PLACEHOLDER_EXPR,
      NON_LVALUE_EXPR,
      SAVE_EXPR,
      UNSAVE_EXPR,
      ADDR_EXPR,
      REFERENCE_EXPR,
      ENTRY_VALUE_EXPR,
      CONJ_EXPR,
      REALPART_EXPR,
      IMAGPART_EXPR:    begin
                          Ref ('op');
                          Ref ('type')
                        end;
      else              Error ('unknown tree code')
    end;
    if not First then WriteLn;
    if Pos1 <= Pos2 then
      Error ('extra bytes in node');
    if Pos1 > Pos2 + 1 then
      Error ('too few bytes in node')
  end;

begin
  Reset (f, GPIFileName, 1);
  BlockRead (f, Header, SizeOf (Header));
  if Header <> GPI_HEADER then
    Error ('invalid header `' + Header + '''');
  BlockRead (f, i, SizeOf (i));
  if i = GPI_INVERSE_ENDIANNESS_MARKER then
    if BytesBigEndian then
      Error ('GPI file was created for a little endian host, but this system is big endian')
    else
      Error ('GPI file was created for a big endian host, but this system is little endian')
  else if i <> GPI_ENDIANNESS_MARKER then
    Error ('invalid endianness marker');
  WriteLn (GPIFileName, ': valid GPI file header');
  Buf := nil;
  ChunksSeen := [];
  Nodes := nil;
  Offsets := nil;
  DebugKey := False;
  NodesPos1 := 0;
  NodesPos2 := 0;
  while not EOF (f) do
    begin
      BlockRead (f, ChunkCodeNum, SizeOf (ChunkCodeNum));
      BlockRead (f, ChunkLength, SizeOf (ChunkLength));
      if ChunkCodeNum > Ord (High (ChunkCode)) then
        ChunkCode := GPI_CHUNK_INVALID
      else
        ChunkCode := TChunkCode (ChunkCodeNum);
      Write ('Chunk `', Chunks[ChunkCode].Name^, ''''
             {$if False}
             , ' (chunk code: ', Number (ChunkCodeNum, 0),
             ', offset: ', Number (FilePos (f), 0),
             ', length: ', Number (ChunkLength, 0), ')'
             {$endif});
      New (Buf, ChunkLength);
      BlockRead (f, Buf^.Bytes, ChunkLength);
      if (ChunkCode in ChunksSeen) and (Chunks[ChunkCode].Unique <> 0) then
        Error ('duplicate chunk');
      ChunksSeen := ChunksSeen + [ChunkCode];
      case ChunkCode of
        GPI_CHUNK_INVALID:        begin
                                    WriteLn;
                                    DumpBytes (' content:', 0, ChunkLength - 1);
                                    WriteLn
                                  end;
        GPI_CHUNK_VERSION:        begin
                                    WriteLn;
                                    DumpChars (' ', 0, ChunkLength - 1);
                                    if not IsSuffix (GCC_VERSION, Buf^.Chars[0 .. ChunkLength - 1]) then
                                      begin
                                        WriteLn;
                                        Error ('backend version in GPI file does not match (rebuild GPIDump for your compiler version)')
                                      end;
                                    if Pos (' D ', Buf^.Chars[0 .. ChunkLength - 1]) <> 0 then
                                      begin
                                        Write (' (file uses debug keys)');
                                        DebugKey := True
                                      end;
                                    WriteLn
                                  end;
        GPI_CHUNK_MODULE_NAME,
        GPI_CHUNK_SRCFILE,
        GPI_CHUNK_LINK,
        GPI_CHUNK_LIB,
        GPI_CHUNK_INITIALIZER,
        GPI_CHUNK_GPC_MAIN_NAME:  begin
                                    WriteLn;
                                    DumpChars (' ', 0, ChunkLength - 1);
                                    WriteLn
                                  end;
        GPI_CHUNK_IMPORT:         begin
                                    WriteLn;
                                    DumpChars (' ', 0, ChunkLength - SizeOf (CheckSum) - 1);
                                    Move (Buf^.Bytes[ChunkLength - SizeOf (CheckSum)], CheckSum, SizeOf (CheckSum));
                                    WriteLn (' (checksum ', Number (CheckSum, 0), ')')
                                  end;
        GPI_CHUNK_NODES:          begin
                                    Move (Buf^.Bytes[ChunkLength - SizeOf (CheckSum)], CheckSum, SizeOf (CheckSum));
                                    WriteLn (' (contents: see below, checksum ', Number (CheckSum, 0), ')');
                                    if ComputeChecksum (Buf^.Bytes[0 .. ChunkLength - SizeOf (CheckSum) - 1]) <> CheckSum then
                                      begin
                                        WriteLn (StdErr, GPIFileName, ': checksum mismatch (GPI file corrupt)');
                                        WriteLn ('The following information may be bogus due to GPI file corruption')
                                      end;
                                    NodesPos2 := FilePos (f);
                                    NodesPos1 := NodesPos2 - ChunkLength;
                                    Nodes := Buf;
                                    Buf := nil
                                  end;
        GPI_CHUNK_OFFSETS:        begin
                                    WriteLn (' (contents: implicit in the node list below)');
                                    if ChunkLength mod SizeOf (GPIInt) <> 0 then
                                      Error ('size of offset table chunk is not a multiple of the word size');
                                    New (Offsets, ChunkLength div SizeOf (GPIInt) - 1);
                                    Move (Buf^.Bytes[0], Offsets^[0], ChunkLength)
                                  end;
        GPI_CHUNK_IMPLEMENTATION: begin
                                    WriteLn;
                                    if ChunkLength <> 0 then
                                      Error ('implementation flag chunk contains unexpected data')
                                  end;
      end;
      Dispose (Buf)
    end;
  Close (f);
  for ChunkCode := Low (ChunkCode) to High (ChunkCode) do
    if Chunks[ChunkCode].Required <> 0 then
      if not (ChunkCode in ChunksSeen) then
        Error ('no ' + Chunks[ChunkCode].Name^ + ' chunk found');
  Buf := Nodes;
  SpecialNodes := TokenizeString (SpecialNodesString, [' ']);
  i := Offsets^[Offsets^.Size];
  WriteLn ('Main node: <', Number (i, 0), '>');
  if i <> SpecialNodes^.Count + Offsets^.Size - 1 then
    Error ('unexpected main node (rebuild GPIDump for your compiler version)');
  {$if False}
  for i := 1 to SpecialNodes^.Count do
    WriteLn ('<', Number (i - 1, 0), '>: ', SpecialNodes^[i]^, ' (implicit)');
  {$endif}
  for i := 0 to Offsets^.Size - 1 do
    begin
      Pos1 := Offsets^[i];
      if i = Offsets^.Size - 1 then
        Pos2 := NodesPos2 - NodesPos1 - SizeOf (CheckSum) - 1
      else
        Pos2 := Offsets^[i + 1] - 1;
      DumpNode (SpecialNodes^.Count + i, Pos1, Pos2)
    end;
  WriteLn
end;

var
  i, j: Integer;

begin
  Hex := False;
  i := 1;
  if ParamStr (i) = '-x' then
    begin
      Hex := True;
      Inc (i)
    end;
  if (ParamCount < i)
     or (ParamStr (i) = '-h') or (ParamStr (i) = '--help')
     or (ParamStr (i) = '-v') or (ParamStr (i) = '--version') then
    begin
      WriteLn (StdErr, 'GPI file dump');
      WriteLn (StdErr, 'Usage: ', ParamStr (0), ' [-x] filename...');
      WriteLn (StdErr, '  -x  Output numbers in hexadecimal');
      Halt (1)
    end;
  for j := i to ParamCount do ProcessFile (ParamStr (j))
end.
