/*
 * VIDEO.CPP - Contains video functions.
 * Copyright (C) 1998, 1999 Prashant TR
 *
 * Special thanks to Shawn Hargreaves. Borrowed lots of code from Allegro.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING.TR for more details.
*/

// ID for this file.
#define _VIDEO_CC_

#include "video.h"

int errflag = 0;
FILE *fp;
char cmdline[20];

#define ENTRIES 13

char display[ENTRIES][80] =
{
 "None",
 "Monochrome",
 "CGA, Color",
 "(Unknown) - Reserved",
 "EGA, Color",
 "EGA, Mono",
 "PGA, Color",
 "VGA, Mono",
 "VGA, Color",
 "(Unknown) - Reserved",
 "MCGA, Digital Color",
 "MCGA, Monochrome",
 "MCGA, Analog Color",
};

typedef struct VESA_INFO 
{ 
   unsigned char  VESASignature[4];
   unsigned short VESAVersion;
   unsigned short OEMStringPtrOff;
   unsigned short OEMStringPtrSeg;
   unsigned char  Capabilities[4];
   unsigned short VideoModePtrOff;
   unsigned short VideoModePtrSeg;
   unsigned short TotalMemory;
   unsigned short OemSoftwareRev;
   unsigned short OemVendorNamePtrOff;
   unsigned short OemVendorNamePtrSeg;
   unsigned short OemProductNamePtrOff;
   unsigned short OemProductNamePtrSeg;
   unsigned short OemProductRevPtrOff;
   unsigned short OemProductRevPtrSeg;
   unsigned char  Reserved[222];
   unsigned char  OemData[256];
} VESA_INFO;

typedef struct MODE_INFO
{
   unsigned short ModeAttributes;
   unsigned char  WinAAttributes;
   unsigned char  WinBAttributes;
   unsigned short WinGranularity;
   unsigned short WinSize;
   unsigned short WinASegment;
   unsigned short WinBSegment;
   unsigned long  WinFuncPtr;
   unsigned short BytesPerScanLine;
   unsigned short XResolution;
   unsigned short YResolution;
   unsigned char  XCharSize;
   unsigned char  YCharSize;
   unsigned char  NumberOfPlanes;
   unsigned char  BitsPerPixel;
   unsigned char  NumberOfBanks;
   unsigned char  MemoryModel;
   unsigned char  BankSize;
   unsigned char  NumberOfImagePages;
   unsigned char  Reserved_page;
   unsigned char  RedMaskSize;
   unsigned char  RedMaskPos;
   unsigned char  GreenMaskSize;
   unsigned char  GreenMaskPos;
   unsigned char  BlueMaskSize;
   unsigned char  BlueMaskPos;
   unsigned char  ReservedMaskSize;
   unsigned char  ReservedMaskPos;
   unsigned char  DirectColorModeInfo;
   unsigned long  PhysBasePtr;
   unsigned long  OffScreenMemOffset;
   unsigned short OffScreenMemSize;
   unsigned char  Reserved[206];
} MODE_INFO;

typedef struct PM_INFO           /* VESA 2.0 protected mode interface */
{
   unsigned short setWindow;
   unsigned short setDisplayStart;
   unsigned short setPalette;
   unsigned short IOPrivInfo;
   /* pmode code is located in this space */
} PM_INFO;

#define MAXMODES 1024

// Write any string to the file and check for successfulness.
void writestring(const char *string)
{
 if (fprintf(fp, "%s", string) == EOF) {
    errflag = 2;
    checkerrors();
 }
}

short unsigned mode[MAXMODES];
VESA_INFO vesa_info;
MODE_INFO mode_info;
PM_INFO pm_info;

int sysinfo()
{
 char output[256], buffer[20], *ptr;
 int i, j, k, nmodes, seg, off;
 union REGS regs;
 struct SREGS segs;

 FILE *stream = fopen("NUL", "w");
 _streams[1] = *stream;

 // Create output file.
 if ((fp = fopen("video.txt", "w")) == NULL) {
    errflag = 1;
    checkerrors();
 }

 writestring("\nVIDEO :\n");

 // Get Display type.
 regs.x.ax = 0x1a00;
 int86(0x10, &regs, &regs);

 if (regs.h.al != 0x1a)
    writestring("\tDisplay type                          : (Unknown)\n");
 else {
      // We have a display type.
      // Write display.
      sprintf(output, "\tDisplay type                          : %s\n",
		      display[regs.h.bl]);
      writestring(output);
      // Alternate display type.
      sprintf(output, "\tAlternate display type                : %s\n",
                      display[regs.h.bh]);
      writestring(output);
 }

 // Scan PCI devices.
 regs.x.ax = 0xb101;
 int86(0x1a, &regs, &regs);
 asm {
	MOV	AX, 0B101H
	INT	1AH
	OR	AH, AH
	JNZ	label1
	DB	66H, 81H, 0FAH	// CMP EDX, 20494350
	DD	20494350H
	JNZ	label1
 }
 if ((!(regs.x.flags & 1)) && (!regs.h.ah)) {
    int nbuses = regs.h.cl, major = regs.h.bh, minor = regs.h.bl;
    int quit = 0;
    for(i = 0; ((i <= 0x1f) && (!quit)); i++)
    for(j = 0; ((j <= nbuses) && (!quit)); j++)
    for(k = 0; ((k <= 7) &&(!quit)); k++)
    {
     regs.x.ax = 0xb108;
     regs.x.di = 0xb;
     if ((major < 2) || (minor < 1)) {
	regs.h.bl = j;
	regs.h.bh = (i << 3) | k;
     }
     else {
	  regs.h.bh = j;
	  regs.h.bl = (i << 3);
	  regs.x.bx |= k;
     }
     int86(0x1a, &regs, &regs);
     if (regs.h.cl == 3) {
	writestring("\tDisplay Device type                   : PCI/AGP\n");
	quit = 1;
     }
    }
    if (!quit)
       writestring("\tDisplay Device type                   : ISA\n");
 }
 else {
label1:
	writestring("\tDisplay Device type                   : ISA\n");
 }

 strcpy(output, "\tDisplay type                          : ");
 // Check for VESA.
 regs.x.ax = 0x4f00;
 segs.es = FP_SEG(&vesa_info);
 regs.x.di = FP_OFF(&vesa_info);
 int86x(0x10, &regs, &regs, &segs);
 // Check if present.
 if (regs.x.ax != 0x4f) {
    // Check for VGA.
    regs.x.ax = 0x101a;
    regs.x.bx = 0xffff;
    int86(0x10, &regs, &regs);
    if (regs.x.bx != 0xffff)
       strcat(output, "VGA\n");
    else {
	// Check for EGA.
	regs.h.ah = 0x12;
	regs.x.bx = 0xff10;
	int86(0x10, &regs, &regs);
	if (regs.h.bh != 0xff) strcat(output, "EGA\n");
	else {
		regs.h.ah= 0xf;
		int86(0x10, &regs, &regs);
		if (regs.h.al != 7) strcat(output, "CGA\n");
		else {
			// Check for HGA Vertical Retrace.
			j = inportb(0x3ba) & 0x80;
			for(i = 0; i < 30000; i++)
			if (j != (inportb(0x3ba) & 0x80))
				strcat(output, "Hercules Graphics Adapter\n");
			else strcat(output, "Monochrome Display Adapter\n");
		}
	}
    }
    writestring(output);
 }
 else {
    // Check for VESA modes.
    if (strncmp((char *)vesa_info.VESASignature, "VESA", 4)) {
       strcat(output, "(Unknown)\n");
       writestring(output);
       fclose(fp);
       return 0;
    }
    strcat(output, "VESA\n");
    writestring(output);

    // Get BIOS data if Trident.
    /* Not supported at present due to problems with bp. */
    asm {
	MOV	AH, 12H
	MOV	BL, 11H
	PUSH	BP
	INT	10H
	MOV	BX, BP
	POP	BP
	CMP	AL, 12H
	JNZ	no_bios_date
    }
    seg = _ES;
    off = _BX;
    movedata(seg, off + 4, FP_SEG(buffer), FP_OFF(buffer), 10);
    buffer[9] = 0;
    sprintf(output, "\tVideo BIOS date                       : %s\n",
		buffer);
    writestring(output);

no_bios_date:
    // VESA version.
    sprintf(output, "\tVESA Version                          : %d.%02d\n",
		    vesa_info.VESAVersion >> 8,
		    vesa_info.VESAVersion & 0xff);
    writestring(output);

    // Video memory size.
    sprintf(output, "\tVideo Memory Installed                : %dK\n",
		    vesa_info.TotalMemory * 64);
    writestring(output);

    // Get VESA capabilities.
    writestring("\nVIDEO CHIPSET INFORMATION :\n\n");
    sprintf(output, "\tVideo DAC type                        : %s DAC\n",
		    (vesa_info.Capabilities[0] & 1) ?
		    "Switchable" :
                    "Fixed");
    writestring(output);

    sprintf(output, "\tVGA Compatible Display                : %s\n",
                    (vesa_info.Capabilities[0] & 2) ?
		    "No" :
                    "Yes");
    writestring(output);

    sprintf(output, "\tRAMDAC type                           : %s\n",
                    (vesa_info.Capabilities[0] & 4) ?
                    "VBLANK" :
                    "Normal");
    writestring(output);

    // Get other information.
    ptr = (char *)MK_FP(vesa_info.OEMStringPtrSeg, vesa_info.OEMStringPtrOff);
    sprintf(output, "\tAdapter Description                   : %s\n",
		    ptr);
    writestring(output);

    sprintf(output, "\tSoftware Revision                     : %d.%d\n",
                    vesa_info.OemSoftwareRev >> 8,
                    vesa_info.OemSoftwareRev & 0xff);
    writestring(output);
    ptr = (char *)MK_FP(vesa_info.OemVendorNamePtrSeg,
		vesa_info.OemVendorNamePtrOff);
    sprintf(output, "\tVendor Name                           : %s\n",
		    ptr);
    writestring(output);
    ptr = (char *)MK_FP(vesa_info.OemProductNamePtrSeg,
		vesa_info.OemProductNamePtrOff);
    sprintf(output, "\tProduct Name                          : %s\n",
		    ptr);
    writestring(output);
    ptr = (char *)MK_FP(vesa_info.OemProductRevPtrSeg,
		vesa_info.OemProductRevPtrOff);
    sprintf(output, "\tProduct Revision                      : %s\n",
		    ptr);
    writestring(output);

    // Get Video Mode list.
    movedata(vesa_info.VideoModePtrSeg, vesa_info.VideoModePtrOff,
		FP_SEG(&mode),
		FP_OFF(&mode),
		MAXMODES * 2);
    nmodes = 0;
    while (mode[nmodes] != 0xffff) nmodes++;
    sprintf(output, "\tNumber of Modes supported             : %d\n",
							     nmodes);
    writestring(output);

    if (vesa_info.VESAVersion >= 0x200) {       /* have we got VESA 2.0? */
       // Check VBE/AF support.
       regs.x.ax = 0x4f0a;
       regs.x.bx = 0;
       int86x(0x10, &regs, &regs, &segs);
       if (!regs.h.ah) {
	  writestring("\tVBE/AF Protected Mode Interface       : "
				"Supported\n");
	  sprintf(output ,"\tSize of VBE/AF header                 : %d bytes\n",
			 regs.x.cx);
	  writestring(output);
	  movedata(segs.es, regs.x.di,
			FP_SEG(&pm_info),
			FP_OFF(&pm_info),
			sizeof(PM_INFO));
	  sprintf(output, "\tRequires Memory mapped I/O            : %s\n",
			  (pm_info.IOPrivInfo) ? "Yes" : "No");
          writestring(output);
       }
       else writestring("\tVBE/AF Protected Mode Interface       : "
                                  "Not Supported\n");
    }

    if (nmodes) {
       // Write individual modes information.
       i = 0;
       while (i < nmodes)
       {
        int c;
        writestring("\nVIDEO MODE(S) INFORMATION :\n\n");
	sprintf(output, "\tVideo mode number                     : %Xh\n",
                                      mode[i]);
        writestring(output);

	// Clear buffer just in case ...
	memset(&mode_info, 0, sizeof(MODE_INFO));

	// Get the mode information.
	regs.x.ax = 0x4f01;
	segs.es = FP_SEG(&mode_info);
	regs.x.di = FP_OFF(&mode_info);
	regs.x.cx = mode[i];
	int86x(0x10, &regs, &regs, &segs);
	// Check for errors - who knows - this is a diagnostic anyway.
	if (regs.h.ah) {
	   fprintf(stderr,
	   "\nError: Could not get video information for mode %04xh\n",
			    mode[i]);
	   continue;
	}

	// Write mode attributes.
	writestring("\n\tMODE ATTRIBUTES :\n\n");
	sprintf(output, "\tMode Supported                        : %s\n",
				(mode_info.ModeAttributes & 1) ? "Yes"
							       : "No");
	writestring(output);
	sprintf(output, "\tBIOS Output Supported                 : %s\n",
				(mode_info.ModeAttributes & 4) ? "Yes"
							       : "No");
	writestring(output);
	sprintf(output, "\tVideo Mode type (Color/Mono)          : %s\n",
				 (mode_info.ModeAttributes & 8) ? "Color"
                                                                : "Mono");
        writestring(output);
	sprintf(output, "\tVideo Mode type (GFX/Text)            : %s\n",
                                 (mode_info.ModeAttributes & 16) ? "Graphics"
                                                                 : "Text");
        writestring(output);
	sprintf(output, "\tVideo Mode type (VGA/Non-VGA)         : %s\n",
                                 (mode_info.ModeAttributes & 32) ? "Non_VGA"
                                                                 : "VGA");
        writestring(output);
	sprintf(output, "\tVideo Memory is Banked                : %s\n",
                                 (mode_info.ModeAttributes & 64) ? "Yes"
                                                                 : "No");
        writestring(output);
	sprintf(output, "\tLinear Frame Buffer type              : %s\n",
                                  (mode_info.ModeAttributes & 128) ? "Yes"
                                                                   : "No");
        writestring(output);

        // Write window attributes.
        writestring("\n\tWINDOW INFORMATION :\n\n");
	sprintf(output, "\tWindow A is Movable                   : %s\n",
                                  (mode_info.WinAAttributes & 1) ? "Yes"
								 : "No");
        writestring(output);
	sprintf(output, "\tWindow A is Readable                  : %s\n",
                                  (mode_info.WinAAttributes & 2) ? "Yes"
                                                                 : "No");
        writestring(output);
	sprintf(output, "\tWindow A is Writeable                 : %s\n",
                                  (mode_info.WinAAttributes & 4) ? "Yes"
                                                                 : "No");
        writestring(output);
	sprintf(output, "\tWindow B is Movable                   : %s\n",
                                  (mode_info.WinBAttributes & 1) ? "Yes"
                                                                 : "No");
	writestring(output);
	sprintf(output, "\tWindow B is Readable                  : %s\n",
                                  (mode_info.WinBAttributes & 2) ? "Yes"
                                                                 : "No");
        writestring(output);
	sprintf(output, "\tWindow B is Writeable                 : %s\n",
                                  (mode_info.WinBAttributes & 4) ? "Yes"
                                                                 : "No");
        writestring(output);

        // Other Window information.
	sprintf(output, "\tWindow Granularity                    : %dK\n",
                                  mode_info.WinGranularity);
	writestring(output);
	sprintf(output, "\tWindow Size                           : %dK\n",
                                  mode_info.WinSize);
        writestring(output);
	sprintf(output, "\tSegment address of Window A           : %Xh\n",
                                  mode_info.WinASegment);
        writestring(output);
	sprintf(output, "\tSegment address of Window B           : %Xh\n",
                                  mode_info.WinBSegment);
        writestring(output);

        writestring("\n");
	sprintf(output, "\tBytes Per Scan Line                   : %d\n",
				  mode_info.BytesPerScanLine);
        writestring(output);
	sprintf(output, "\tX Resolution                          : %d pixels\n",
                                  mode_info.XResolution);
        writestring(output);
	sprintf(output, "\tY Resolution                          : %d pixels\n",
                                  mode_info.YResolution);
        writestring(output);
	sprintf(output, "\tX Character Size                      : %d pixels\n",
                                  mode_info.XCharSize);
        writestring(output);
	sprintf(output, "\tY Character Size                      : %d pixels\n",
                                  mode_info.YCharSize);
	writestring(output);
	sprintf(output, "\tNumber of Planes                      : %d\n",
                                  mode_info.NumberOfPlanes);
        writestring(output);
	sprintf(output, "\tBits Per Pixel                        : %d\n",
                                  mode_info.BitsPerPixel);
        writestring(output);
	sprintf(output, "\tNumber of Banks                       : %d\n",
                                  mode_info.NumberOfBanks);
        writestring(output);

        strcpy(output, "\tMemory Model                          : ");
        switch (mode_info.MemoryModel) {
	       case 0:  strcat(output, "Text"); break;
	       case 1:  strcat(output, "CGA"); break;
	       case 2:  strcat(output, "Hercules"); break;
	       case 3:  strcat(output, "Planar"); break;
	       case 4:  strcat(output, "Packed pixel"); break;
	       case 5:  strcat(output, "Non-chain 4, 256 color"); break;
	       case 6:  strcat(output, "Direct color"); break;
	       case 7:  strcat(output, "YUV"); break;
	       default: strcat(output, "(Unknown)"); break;
        }
        strcat(output, "\n");
        writestring(output);
	sprintf(output, "\tBank Size                             : %d\n",
				  mode_info.BankSize);
        writestring(output);
	sprintf(output, "\tNumber of Image Pages                 : %d\n",
                                  mode_info.NumberOfImagePages);
        writestring(output);
	sprintf(output, "\tReserved Page                         : %d\n",
                                  mode_info.Reserved_page);
        writestring(output);
	sprintf(output, "\tRED Mask Size                         : %d\n",
                                  mode_info.RedMaskSize);
        writestring(output);
	sprintf(output, "\tRED Mask Position                     : %d\n",
                                  mode_info.RedMaskPos);
	writestring(output);
	sprintf(output, "\tGREEN Mask Size                       : %d\n",
                                  mode_info.GreenMaskSize);
        writestring(output);
	sprintf(output, "\tGREEN Mask Position                   : %d\n",
                                  mode_info.GreenMaskPos);
        writestring(output);
	sprintf(output, "\tBLUE Mask Size                        : %d\n",
                                  mode_info.BlueMaskSize);
        writestring(output);
	sprintf(output, "\tBLUE Mask Position                    : %d\n",
                                  mode_info.BlueMaskPos);
        writestring(output);
	sprintf(output, "\tReserved Mask Size                    : %d\n",
                                  mode_info.ReservedMaskSize);
        writestring(output);
	sprintf(output, "\tReserved Mask Position                : %d\n",
                                  mode_info.ReservedMaskPos);
        writestring(output);

	sprintf(output, "\tDirect Color Mode Information type    : %s\n",
                                  (mode_info.DirectColorModeInfo & 1) ?
                                  "Programmable" :
                                  "Fixed");
        writestring(output);
	sprintf(output, "\tDirect Color Mode Usable              : %s\n",
				  (mode_info.DirectColorModeInfo & 2) ?
                                  "Yes" :
                                  "No");
        writestring(output);
	sprintf(output, "\tPhysical Base Pointer                 : %lXh\n",
                                  mode_info.PhysBasePtr);
        writestring(output);
	sprintf(output, "\tOff Screen Memory Offset              : %lXh\n",
                                  mode_info.OffScreenMemOffset);
        writestring(output);
	sprintf(output, "\tOff Screen Memory Size                : %uK\n",
                                  mode_info.OffScreenMemSize);
        writestring(output);
	writestring("\n\n");
	i++;
       }
    }
 }

 fclose(fp);
 return 0;
}

void open_stderr()
{
	fclose(stdout);
	fclose(&_streams[2]);
	if (fopen("nul", "wb") == NULL) exit(0x7f);
	if (fopen("nul", "wb") == NULL) exit(0x7f);
	if ((stderr = fopen("errors.$$$", "ab")) == NULL) exit(0x7f);
}

void get_cmdline()
{
 if ((fp = fopen("cmdline.$$$", "rb")) == NULL) exit (0x7f);

 if (fscanf(fp, "%s", cmdline) != 1) {
		fclose(fp);
		exit (0x7f);
 }

 fclose(fp);
 unlink("cmdline.$$$");

}

#pragma argsused

// The main function.
int main(int argc, char **argv)
{
 open_stderr();
 get_cmdline();

 if (!strcmp(cmdline, "sysinfo")) return(sysinfo());

 return 0;
}
