#include <ctype.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "stone.h"

char *stone_compiler_default_step [] =
{
  "&prefix;gcc -c &src-dir; -I. -W -Wall -Werror -Wno-unused -O3 "
  "-ffast-math -fomit-frame-pointer -m486 -o &obj-dir;&option;&suffix;"
};

stone_compiler stone_compiler_default =
{
  NULL, "gcc", "gcc", 1, stone_compiler_default_step, ".c", ".o"
};

int stone_nobj = 0;
stone_object *stone_obj = NULL;
int stone_nerror = 0; /* Number of errors reported */
stone_compiler *stone_compiler_list = &stone_compiler_default;
stone_search *stone_search_list = NULL; /* Linked list of search paths */
int stone_is_setup = 0;

extern void stone_badcall_default (void);

static void stone_error_default (char *str)
{
  fputs ("Error: ", stderr);
  fputs (str, stderr);
  fputc ('\n', stderr);
}

void (*stone_error_cur) (char *str) = stone_error_default;
int (*stone_sym_proc) (char *id, int *sym) = stone_sym_default;
void (*stone_badcall_proc) (void) = stone_badcall_default;

void stone_badcall (void)
{
  stone_badcall_proc ();
}

void stone_report (char *str, ...)
{
  char buf [1024];
  va_list arg;

  stone_setup ();
  stone_nerror ++;
  if (stone_error_cur == NULL)
    return;

  va_start (arg, str);
  vsprintf (buf, str, arg);
  stone_error_cur (buf);
  va_end (arg);
}

void stone_setup (void)
{
  if (stone_is_setup)
    return;
  stone_is_setup = 1;
  stone_nobj = 1;
  stone_resize (stone_obj, stone_nobj);
  stone_obj->code = NULL;
  stone_obj->len = 0;
  stone_obj->link = 0;
  stone_obj->nsym = 0;
  stone_obj->sym = NULL;
  stone_obj->nrel = 0;
  stone_obj->rel = NULL;
  stone_obj->user = 1;
  stone_export ("%die", stone_badcall);
}

/* Add compiler to the list. */
void stone_add_compiler (char *id, char *name, char *srcext, char *objext, /* char *step [] */.../* NULL */)
{
  stone_compiler *com;
  va_list arg;
  int c;

  stone_setup ();
  for (com = stone_compiler_list; com->next != NULL; com = com->next);
  com->next = malloc (sizeof (*com->next));
  com = com->next;
  com->next = NULL;
  com->id = strdup (id);
  com->name = strdup (name);
  com->srcext = srcext;
  com->objext = objext;

  va_start (arg, objext);
  for (com->nstep = 0; va_arg (arg, char *) != NULL; com->nstep ++);
  com->step = malloc (sizeof (*com->step) * com->nstep);
  va_end (arg);

  va_start (arg, objext);
  for (c = 0; c < com->nstep; c ++)
    com->step [com->nstep] = strdup (va_arg (arg, char *));
  va_end (arg);
}

/* Add search path to front of list */
void stone_add_search (char *path)
{
  stone_search *search;

  stone_allot (search, 1);
  search->next = stone_search_list;
  stone_search_list = search;
  search->path = strdup (path);
}

int stone_export (char *name, void *ptr)
{
  stone_object_sym sym;

  stone_setup ();
  if (name == NULL)
    return 0;

  sym.name = strdup (name);
  sym.off = (int) ptr;
  sym.type = csst_export;
  if (sym.name == NULL)
    return 0;

  stone_resize (stone_obj->sym, stone_obj->nsym + 1);
  stone_obj->sym [stone_obj->nsym ++] = sym;
  return 1;
}

int stone_load (char *filename)
{
  FILE *file;
  int idx;

  stone_setup ();
  if ((file = fopen (filename, "rb")) == NULL)
    return -1;
   
#if stone_coff || stone_pe
  if ((idx = stone_load_coff (file, filename)) >= 0)
    return idx;
#endif /* stone_coff || stone_pe */

#if stone_elf
  if ((idx = stone_load_elf (file, filename)) >= 0)
    return idx;
#endif /* stone_elf */

#if stone_omf
  if ((idx = stone_load_omf (file, filename)) >= 0)
    return idx;
#endif /* stone_omf */

  fclose (file);
  return -1;
}

int stone_link (int objidx)
{
  stone_object *obj;
  int c, i;
  int reply = 1;

  int link_symbol (stone_object_sym *sym)
  {
    int rel_sym;
    int rel_obj;

    if (sym->type == csst_export || sym->type == csst_unknown)
    {
      sym->obj = ((int) obj - (int) stone_obj) / sizeof (*obj);
      sym->sym = ((int) sym - (int) obj->sym) / sizeof (*sym);
      return 1;
    }

    if (sym->obj > -1)
      return 1;

    rel_obj = stone_sym_find (sym->name, &rel_sym);
    if (rel_obj == -1)
    {
      rel_obj = 0;
      rel_sym = 0;
    }

    sym->obj = rel_obj;
    sym->sym = rel_sym;
    return 1;
  }

  stone_setup ();
  if (objidx < 0 || objidx >= stone_nobj)
    return 0;
  obj = &stone_obj [objidx];
  if (obj->link)
    return 1;

  /* link relocations */
  for (i = 0; i < obj->nrel; i ++)
  {
    int rel_off = obj->rel [i].off;
    int rel_sym = obj->rel [i].sym;
    int rel_type = obj->rel [i].type;
    int rel_base = obj->rel [i].base;

    stone_object_sym *isym = &obj->sym [rel_sym];
    stone_object_sym *esym = NULL;
    stone_object *eobj = NULL;

    int setup ()
    {
      if (link_symbol (isym))
      {
        eobj = &stone_obj [isym->obj];
        esym = &eobj->sym [isym->sym];
        return 1;
      }
      reply = 0;
      return 0;
    }

    switch (rel_type)
    {
      case csr_relative:
        if (setup ())
          *(int *) &obj->code [rel_off] = rel_base + (int) &eobj->code [esym->off] - (int) obj->code;
        break;

      case csr_absolute:
        if (isym->type == csst_import)
        {
          if (setup ())
            *(int *) &obj->code [rel_off] = rel_base + (int) &stone_obj [isym->obj].code [esym->off];
        }
        else
          *(int *) &obj->code [rel_off] = rel_base + (int) &obj->code [0];
        break;

      case csr_relative2:
        if (setup ())
          *(int *) &obj->code [rel_off] = rel_base + (int) &eobj->code [esym->off] - (int) &obj->code [rel_off];
        break;

      case csr_absolute2:
        if (setup ())
          *(int *) &obj->code [rel_off] = rel_base + (int) &stone_obj [isym->obj].code [esym->off];
        break;

      default:
        stone_report ("Unknown relocation type %d", rel_type);
        reply = 0;
        break;
    }
  }

  if (reply)
  {
    obj->link = 1;
    for (c = 0; c < obj->nsym; c ++)
      if (obj->sym [c].obj >= 0)
        stone_obj [obj->sym [c].obj].user ++;
  }
  return reply;
}

int stone_link_all (void)
{
   stone_setup ();
   return stone_link_range (0, stone_nobj - 1);
}

int stone_link_range (int low, int high)
{
   int reply = 1;
   int c;
   
   stone_setup ();
   if (low > high)
   {
      int mid = low;
      low = high;
      high = mid;
   }
   if (low < 1)
      low = 1;
   if (high >= stone_nobj)
      high = stone_nobj - 1;
   if (low > high)
      return 1;

   for (c = low; c <= high; c ++)
      if (!stone_link (c))
         reply = 0;
   return reply;
}

int stone_free (int objidx)
{
   stone_object *obj;
   int c;
   
   stone_setup ();
   if (objidx < 1 || objidx >= stone_nobj)
      return 0;

   obj = &stone_obj [objidx];
   if (obj->user == 0)
      return 0;

   obj->user --;
   if (obj->user > 0)
      return 0;

   /* Hit rock bottom, free the object */
   for (c = 0; c < obj->nsym; c ++)
      stone_free (obj->sym [c].obj);

   free (obj->code);
   free (obj->sym);
   free (obj->rel);
   return 1;
}

int stone_sym_match (char *a, char *b)
{
   stone_setup ();
   if (a == NULL || b == NULL)
      return 0;
   return strcmp (a + (a [0] == '_' ? 1 : 0),
                  b + (b [0] == '_' ? 1 : 0)) == 0;
}

/* Get symbol pointer throughout system */
void *stone_ptr (char *id)
{
   int obj, sym;

   stone_setup ();
   if (id == NULL)
      return NULL;
   obj = stone_sym_find (id, &sym);
   if (obj < 0 || obj >= stone_nobj || sym < 0 || sym >= stone_obj [obj].nsym)
      return NULL;
   return stone_obj [obj].code + stone_obj [obj].sym [sym].off;
}

int stone_sym_find (char *id, int *sym)
{
   stone_setup ();
   if (id == NULL || sym == NULL || stone_sym_proc == NULL)
      return -1;
   return stone_sym_proc (id, sym);
}

int stone_sym_default (char *name, int *symptr)
{
  stone_object_sym *sym;
  stone_object *obj;
  int s, i;

  stone_setup ();

  /* Exact match */
  for (i = 0, obj = stone_obj; i < stone_nobj; i ++, obj ++)
  for (s = obj->nsym - 1, sym = &obj->sym [s]; s >= 0; s --, sym --)
  {
    if (sym->type == csst_export && name != NULL && sym->name != NULL && !strcmp (sym->name, name))
    {
      (*symptr) = s;
      return i;
    }
  }

  /* Inexact match (Strip first _s) */
  for (i = 0, obj = stone_obj; i < stone_nobj; i ++, obj ++)
  for (s = obj->nsym - 1, sym = &obj->sym [s]; s >= 0; s --, sym --)
  {
    if (sym->type == csst_export && stone_sym_match (sym->name, name))
    {
      (*symptr) = s;
      return i;
    }
  }

  return -1;
}

void *stone_sym (char *id)
{
   int obj, sym;

   stone_setup ();
   obj = stone_sym_find (id, &sym);
   if (obj < 0 || obj >= stone_nobj)
      return NULL;
   return stone_obj [obj].code + stone_obj [obj].sym [sym].off;
}

void *stone_sym_ptr (int objidx, char *name)
{
  stone_object_sym *sym;
  stone_object *obj;
  int s;

  stone_setup ();
  if (objidx < 0 || objidx >= stone_nobj || name == NULL)
    return NULL;
      
  obj = &stone_obj [objidx];
  for (s = 0, sym = obj->sym; s < obj->nsym; s ++, sym ++)
  if (stone_sym_match (sym->name, name))
    return obj->code + sym->off;
   
  return NULL;
}

static int sort_sym (const void *a, const void *b)
{
  return strcmp (*(char **) a, *(char **) b);
}

/* List unlinked symbols, sorted */
int stone_list_unlink (void)
{
  char **sym = NULL;
  int nsym = 0;
  int c, d, e;

  for (c = 0; c < stone_nobj; c ++)
  for (d = 0; d < stone_obj [c].nsym; d ++)
  if (stone_obj [c].sym [d].obj == 0 && stone_obj [c].sym [d].sym == 0)
  {
    char *name = stone_obj [c].sym [d].name;

    if (name != NULL)
    {
      for (e = 0; e < nsym; e ++)
        if (!strcmp (name, sym [e]))
          break;
      if (e >= nsym)
      {
        sym = realloc (sym, sizeof (*sym) * (nsym + 1));
        sym [nsym ++] = name;
      }
    }
  }

  if (nsym == 0)
    return 1;
  qsort (sym, nsym, sizeof (*sym), sort_sym);
  stone_report ("%d unlinked symbols:", nsym);
  for (c = 0; c < nsym; c ++)
    stone_report ("%s", sym [c]);
  free (sym);
  return 0;
}

/* Search the objects/symbols for the closest symbol to ptr */
int stone_inside (void *ptr, int *objidx, int *symidx)
{
  stone_object_sym *sym;
  stone_object *obj;
  int c, d;
  int close;
  int cdist = 0;
  
  if (ptr == NULL || objidx == NULL || symidx == NULL)
    return 0;
    
  for (c = 0, obj = stone_obj; c < stone_nobj; c ++, obj ++)
    if (ptr >= (void *) obj->code && ptr < (void *) (obj->code + obj->len))
    {
      close = -1;
      for (d = 0, sym = obj->sym; d < obj->nsym; d ++, sym ++)
        if (ptr >= (void *) (obj->code + sym->off)
         && (close == -1 || (int) (ptr - (void *) (obj->code + sym->off)) < cdist))
        {
          cdist = ptr - (void *) (obj->code + sym->off);
          close = d;
        }

      if (close != -1)
      {
        *objidx = c;
        *symidx = close;
      }
    }
  return 0;
}

/* Clear the object data */
void stone_clear_obj (stone_object *obj)
{
  memset (obj, 0, sizeof (*obj));
  if ((unsigned long) NULL != 0UL)
  {
    obj->src = NULL;
    obj->code = NULL;
    obj->sym = NULL;
    obj->rel = NULL;
  }
}

/* Free the object data */
void stone_free_obj (stone_object *obj)
{
  int c;
  
  if (obj->src != NULL)
    free (obj->src);
  if (obj->code != NULL)
    free (obj->code);
    
  if (obj->nsym != 0 && obj->sym != NULL)
    for (c = 0; c < obj->nsym; c ++)
      if (obj->sym [c].name != NULL)
        free (obj->sym [c].name);

  if (obj->sym != NULL)
    free (obj->sym);
  if (obj->rel != NULL)
    free (obj->rel);
  stone_clear_obj (obj);
}
