/****************************************************************************/
/* THELPTZ                                                                  */
/*--------------------------------------------------------------------------*/
/* Class THelpTextZone (Help display text zone)				    */
/*--------------------------------------------------------------------------*/
/* Author      : DELPRAT Jean-Pierre                                        */
/* Created on  : 29-Oct-96                                                  */
/****************************************************************************/

#include <fcntl.h>
#include <share.h>
#include <string.h>
#include <sys\stat.h>

#include "Debug.h"

#include "Const.h"
#include "Compat.h"
#include "Files.h"
#include "Mouse.h"
#include "Vocab.h"

#include "THelpWnd.h"
#include "TPushBut.h"
#include "Cursor.h" // HideTextCursor()

#include "THelpTZ.h"


/*ͻ*/
/*                   PROPRIETES DES LIGNES DE LA ZONE                     */
/*ͼ*/

class THelpLineProperties:public TLineProperties
{
  public:
       TLinkList f_links,
		 f_last_link;

	 void m_add_link(int col1,int col2,short int help);

	     THelpLineProperties();
    virtual ~THelpLineProperties();
};

THelpLineProperties::THelpLineProperties()
{
  f_links=NULL;
  f_last_link=NULL;
}

THelpLineProperties::~THelpLineProperties()
{
  PLinkNode node,next_node;

  node=f_links;
  while (node!=NULL)
    {
      next_node=node->next;
      delete node;
      node=next_node;
    }
}

void THelpLineProperties::m_add_link(int col1,int col2,short int help)
{
  PLinkNode new_node;

  new_node=new TLinkNode;

  new_node->col1=col1;
  new_node->col2=col2;
  new_node->help=help;
  new_node->last=f_last_link;
  new_node->next=NULL;

  if (f_links==NULL)
    f_links=new_node;
  else
    f_last_link->next=new_node;

  f_last_link=new_node;
}


/*ͻ*/
/*                               CONSTANTES                               */
/*ͼ*/


#define MAX_LINE_LENGTH        133    // Sans les tags  (avec \n)
#define MAX_TAGGED_LINE_LENGTH 301    // Avec les tags  (avec \n)

#define DEFAULT_WIDTH          70     // (sans \n)
#define MAX_WIDTH             ((MAX_LINE_LENGTH)-1)  // (sans \n)

#define MAX_TOPIC_LENGTH       80


// Help file signature

#define HELP_FILE_SIGNATURE    "JPHLP39"
#define SIGNATURE_LENGTH       (sizeof(HELP_FILE_SIGNATURE)-1)

// Special topics

#define INDEX_TOPIC            "Index"
#define CONTENTS_TOPIC         "Contents"


/*ͻ*/
/*                 INITIALISATION DES VARIABLES STATIQUES                 */
/*ͼ*/

/*ͻ*/
/*                           METHODES PUBLIQUES                           */
/*ͼ*/

/****************************************************************************/
/* Constructeur                                                             */
/*--------------------------------------------------------------------------*/
/* parent           : Objet auquel appartient l'objet                       */
/* rel_x,rel_y      : Coordonnes de la zone texte p/r au groupe            */
/* width,height     : Dimensions de la zone                                 */
/****************************************************************************/

THelpTextZone::THelpTextZone(PObject parent,
			     int rel_x,int rel_y,
			     int width,int height)
	      :TTextZone(parent,
			 rel_x,rel_y,
			 width,height,
			 " ",
			 MAX_WIDTH,
			 SBC_BOTH,
			 ENABLED)

{
  // Help file

  f_help_file=NULL;
  f_help_file_name=NULL;

  // History

  f_help_history=NULL;

  // Decompressing

  f_help_tree=NULL;
  f_root=0;
  f_in8=0;
  f_ct8 = 8;

  // Help

  f_first_help=NULL;
  f_this_help=NULL;
  f_help_count=0;

//  f_key_word_count=0;

  // Selected link

  f_selected_link=NULL;
  f_selected_link_line=0;

  // Disable text modification

  m_disable_modification();

  saved_cursor_style = f_cursor_style;
  f_cursor_style = _NOCURSOR;
}

/****************************************************************************/
/* Destructeur                                                              */
/*--------------------------------------------------------------------------*/
/****************************************************************************/

THelpTextZone::~THelpTextZone()
{
  m_free_help_file();
  f_cursor_style = saved_cursor_style;
}


/****************************************************************************/
/* m_load_help_file                                                         */
/*--------------------------------------------------------------------------*/
/****************************************************************************/

boolean THelpTextZone::m_load_help_file(const char *file_name)
{
  long int   where;
  int            i;
  short int    len;

  boolean index;
  boolean contents;

  char *index_topic=INDEX_TOPIC;
  char *contents_topic=CONTENTS_TOPIC;


  m_unload_help_file();

  if (!m_open_help_file(file_name))
    return(FALSE);


  f_help_file_name=new char[strlen(file_name)+1];
  strcpy(f_help_file_name,file_name);


  fseek(f_help_file,-(long)sizeof(where),SEEK_END);
  fread(&where,sizeof(where),1,f_help_file);
  fseek(f_help_file,where,SEEK_SET);

  fread(&f_help_count,sizeof(f_help_count),1,f_help_file);

  f_first_help = new THelpTopic [f_help_count];

  for (i = 0; i < f_help_count; i++)
    {
      fread(&len,sizeof(len),1,f_help_file);
      (f_first_help + i)->name = new char [len + 1];
      if (len)
	fread((f_first_help + i)->name,len + 1,1,f_help_file);
      else
	((f_first_help+i)->name)[0]=0;

      fread(&(f_first_help + i)->byte_pos   ,sizeof(f_first_help->byte_pos),1,f_help_file);
      fread(&(f_first_help + i)->bit_pos    ,sizeof(f_first_help->bit_pos),1,f_help_file    );
      fread(&(f_first_help + i)->next_help,sizeof(f_first_help->next_help),1,f_help_file);
      fread(&(f_first_help + i)->previous_help,sizeof(f_first_help->previous_help),1,f_help_file);
    }

  fclose(f_help_file);
  f_help_file=NULL;

  index=m_topic_exists(index_topic);
  contents=m_topic_exists(contents_topic);

  if (index)
    ((PHelpWindow)f_window)->f_index_button->m_enable();
  if (contents)
    ((PHelpWindow)f_window)->f_contents_button->m_enable();

  if (contents)
    m_display_topic(contents_topic,TRUE);
  else
    {
      if (index)
	m_display_topic(index_topic,TRUE);
    }

  return(TRUE);
}

/****************************************************************************/
/* m_unload_help_file                                                       */
/*--------------------------------------------------------------------------*/
/****************************************************************************/

void THelpTextZone::m_unload_help_file()
{
  // Voir si un fichier d'aide est toujours ouvert

  if (f_help_file_name==NULL)
    return;

  m_enable_modification();
  m_clear_text();
  m_disable_modification();
  m_set_caption(" ");

  ((PHelpWindow)f_window)->f_contents_button->m_disable();
  ((PHelpWindow)f_window)->f_index_button   ->m_disable();
  ((PHelpWindow)f_window)->f_back_button    ->m_disable();
  ((PHelpWindow)f_window)->f_previous_button->m_disable();
  ((PHelpWindow)f_window)->f_next_button    ->m_disable();

  m_free_help_file();
}

boolean THelpTextZone::m_topic_exists(const char *topic)
{
  return(((m_find_topic(topic)) != NULL)?TRUE:FALSE);
}

boolean THelpTextZone::m_display_topic(const char *topic, boolean push_current_topic)
{
  THelpTopic  *new_topic;

  if (topic==NULL)
    new_topic=NULL;
  else
    new_topic= m_find_topic(topic);

  if (new_topic != NULL)
    {
      if (f_this_help!=new_topic)
	{
	  if (push_current_topic)
	    m_push_current_topic();

	  f_this_help = new_topic;

	  if (m_open_help_file(f_help_file_name))
	    {
	      m_read_topic();
	      fclose(f_help_file);
	      f_help_file=NULL;
	      if (!stricmp(topic,INDEX_TOPIC))
		m_set_caption(GetString(VOC_INDEX));
	      if (f_selected_link == NULL) { // Fix: jump to first item
                TKey k; k.character = RIGHT;
  	        boolean ff = f_focused; f_focused = TRUE;
  	        m_key_pressed_event(k);
  	        f_focused = ff;
	      }
	      return(TRUE);
	    }
	}
    }
  else
    m_display_string("",VOC_HELP_TOPIC_NOT_FOUND);

  return (FALSE);
}

void THelpTextZone::m_select_back_topic()
{
  PHistoryNode node;

  if (f_help_history!=NULL)
    {
      node=f_help_history;
      f_help_history=f_help_history->next;
      m_select_topic(f_first_help + node->topic_index,FALSE);
      m_set_first_visible_line(node->first_line);
      m_set_first_visible_col(node->first_col);
      m_set_cursor_at(node->col,node->line,TRUE);

      delete node;

      if (f_help_history==NULL)
	((PHelpWindow)f_window)->f_back_button->m_disable();
    }
}

void THelpTextZone::m_select_contents_topic()
{
  m_display_topic(CONTENTS_TOPIC,TRUE);
}

void THelpTextZone::m_select_index_topic()
{
  m_display_topic(INDEX_TOPIC,TRUE);
}

void THelpTextZone::m_select_previous_topic()
{
  if (f_this_help!=NULL)
    if (f_this_help->previous_help!=-1)
      m_select_topic(f_first_help + (f_this_help->previous_help),TRUE);
}

void THelpTextZone::m_select_next_topic()
{
  if (f_this_help!=NULL)
    if (f_this_help->next_help!=-1)
      m_select_topic(f_first_help + (f_this_help->next_help),TRUE);
}


/****************************************************************************/
/* m_display_string_node                                                    */
/*--------------------------------------------------------------------------*/
/* Affiche une ligne du texte.                                              */
/* La position d'affichage doit dj tre tablie                           */
/****************************************************************************/

void THelpTextZone::m_display_string_node(PStringNode node,int line)
{
  char *string;
  int  string_length;
  THelpLineProperties *properties;
  PLinkNode link_node;
  int col;


  unsigned normal_attr,selected_attr;
  unsigned link_normal_attr,link_selected_attr;
  unsigned selected_link_normal_attr,selected_link_selected_attr;

  int  min_selected_col,max_selected_col;


  string=node->string;
  string_length=strlen(string);
  properties=(THelpLineProperties *)node->properties;

  // Couleurs d'affichage

  if ((f_focused)&&(f_window->m_is_active()))
    {
      selected_attr=f_window->m_get_inverse_attr(f_background);
      link_selected_attr=(BLACK<<4)+(unsigned)GREEN;
      selected_link_selected_attr=(BLACK<<4)+(unsigned)YELLOW;
    }
  else
    {
      selected_attr=((DARKGRAY<<4)+(unsigned)WHITE);
      link_selected_attr=(DARKGRAY<<4)+(unsigned)GREEN;
      selected_link_selected_attr=(DARKGRAY<<4)+(unsigned)YELLOW;
    }
  normal_attr  =f_window->m_get_normal_attr(f_background);
  link_normal_attr=(f_background << 4) + (unsigned)GREEN;
  selected_link_normal_attr=(GREEN << 4) + (unsigned)YELLOW;

  // Partie selectionnee de la ligne

  m_get_line_selected_part(line,min_selected_col,max_selected_col);

  // Affichage de la ligne

  col=1;
  if (properties==NULL)
    link_node=NULL;
  else
    link_node=properties->f_links;

  while (link_node!=NULL)
    {
      m_display_help_string_part(string,string_length,
				 col,(link_node->col1)-1,
				 min_selected_col,max_selected_col,
				 normal_attr,selected_attr);
      m_display_help_string_part(string,string_length,
				 link_node->col1,link_node->col2,
				 min_selected_col,max_selected_col,
				 (f_selected_link==link_node)?selected_link_normal_attr
							    :link_normal_attr,
				 (f_selected_link==link_node)?selected_link_selected_attr
							    :link_selected_attr);
      col=(link_node->col2)+1;
      link_node=link_node->next;
    }

  m_display_help_string_part(string,string_length,
			     col,f_first_visible_col+f_text_width-1,
			     min_selected_col,max_selected_col,
			     normal_attr,selected_attr);
}

void THelpTextZone::m_display_help_string_part(const char *string, int string_length,
  int col1, int col2, int min_selected_col, int max_selected_col, 
  unsigned normal_attr, unsigned selected_attr)
{
  if (col2<col1)
    return;

  if ((min_selected_col>col2) || (max_selected_col<col1))
    max_selected_col=0;
  else
    {
      if (min_selected_col<col1)
	min_selected_col=col1;
      if (max_selected_col>col2)
	max_selected_col=col2;
    }

  if (max_selected_col!=0)
    {
      m_display_string_part(string,string_length,col1,min_selected_col-1,normal_attr);
      m_display_string_part(string,string_length,min_selected_col,max_selected_col,selected_attr);
      m_display_string_part(string,string_length,max_selected_col+1,col2,normal_attr);
    }
  else
    m_display_string_part(string,string_length,col1,col2,normal_attr);

}

/*ͻ*/
/*                          METHODES PROTEGEES                            */
/*ͼ*/

/****************************************************************************/
/* m_cursor_pos_changed_callback                                            */
/*--------------------------------------------------------------------------*/
/* Appel en cas de changement de position du curseur. 			    */
/****************************************************************************/

void THelpTextZone::m_cursor_pos_changed_callback()
{
  PLinkNode           selected_link;
  int 		      old_selected_link_line;

  // Recherche d'un lien sous le curseur

  selected_link=m_pos_to_link(f_cursor_col,f_cursor_line);

  // Le lien a change

  if (selected_link!=f_selected_link)
    {
      old_selected_link_line=f_selected_link_line;
      f_selected_link=selected_link;
      if (f_selected_link!=NULL)
	f_selected_link_line=f_cursor_line;

      if (old_selected_link_line!=0)
	m_display_lines(old_selected_link_line,old_selected_link_line);

      if (f_selected_link!=NULL)
	 {
	   if (f_selected_link_line!=old_selected_link_line)
	     m_display_lines(f_selected_link_line,f_selected_link_line);
	 }
    }

  // Appelle de la methode du parent

  TTextZone::m_cursor_pos_changed_callback();
}

/****************************************************************************/
/* m_left_button_double_click_event                                         */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a double-cliqu dans l'objet avec le bouton gauche         */
/* (l'objet tant activable et ayant dj subi l'vnement                  */
/* m_left_button_pressed_event).                                            */
/* Retourne TRUE si l'objet est intress par cet vnement.                */
/****************************************************************************/

boolean THelpTextZone::m_left_button_double_click_event(int x,int y)
{
  int x1,y1,x2,y2;
  PLinkNode link;

  if (TObject::m_left_button_double_click_event(x,y))
    return(TRUE);

  x1=m_get_x()+f_text_rel_x;
  y1=m_get_y()+f_text_rel_y;
  x2=x1+f_text_width-1;
  y2=y1+f_text_height-1;

  if ((x<x1) || (x>x2) || (y<y1) || (y>y2))
    return(FALSE);

  link=m_pos_to_link(x-x1+f_first_visible_col,
		     y-y1+f_first_visible_line);

 if (link!=NULL)
   {
     m_select_topic((link->help==-1)?NULL:f_first_help+link->help,TRUE);
   }

  return(TRUE);
}

/****************************************************************************/
/* m_key_pressed_event                                                      */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur une touche qui est propose  l'objet         */
/* (qui est activable).                                                     */
/* Retourne TRUE si l'objet est intress par cette touche.                 */
/****************************************************************************/

boolean THelpTextZone::m_key_pressed_event(TKey key)
{
  PLinkNode selected_link;
  int line;
  PStringNode string_node,initial_string_node;
  PLinkNode link_node;

  if (TObject::m_key_pressed_event(key))
    return(TRUE);

  if (!f_focused)
    return(FALSE);

  switch (key.character)
  {
    case RETURN : if (f_selected_link!=NULL)
		    m_select_topic((f_selected_link->help==-1)?NULL:f_first_help+f_selected_link->help,TRUE);
		  break;

    case ESC    : f_window->m_close();
		  break;

    case RIGHT: case DOWN: // Fix: to jump to next help topic
    case TAB    : if (f_nb_lines == 0)
		    break;

		  string_node=m_line_to_string(f_cursor_line);
		  line=f_cursor_line;
		  selected_link=NULL;

		  if (string_node->properties!=NULL)
		    {
		      link_node=((THelpLineProperties *)(string_node->properties))->f_links;
		      while (link_node!=NULL)
			{
			  if (f_cursor_col<link_node->col1)
			    {
			      selected_link=link_node;
			      break;
			    }
			  link_node=link_node->next;
			}
		    }

		  if (selected_link==NULL)
		    {
		      initial_string_node=string_node;
		      string_node=string_node->next;
		      line++;

		      if (string_node==NULL)
			{
			  string_node=f_text;
			  line=1;
			}

		      while ((string_node!=initial_string_node) && (selected_link==NULL))
			{
			  if (string_node->properties!=NULL)
			    selected_link=((THelpLineProperties *)(string_node->properties))->f_links;
			  else
			    {

			      string_node=string_node->next;
			      if (string_node!=NULL)
				line++;
			      else
				{
				  string_node=f_text;
				  line=1;
				}
			    }
			}

		      if (string_node==initial_string_node)
			if (string_node->properties!=NULL)
			  selected_link=((THelpLineProperties *)(string_node->properties))->f_links;
		    }


		  if (selected_link!=NULL) 
		    m_set_cursor_at(selected_link->col1,line,TRUE);

		  break;

    case LEFT: case UP: // Fix: to jump to previous help topic
    case SHIFT_TAB :
		  if (f_nb_lines == 0)
		    break;

		  string_node=m_line_to_string(f_cursor_line);
		  line=f_cursor_line;
		  selected_link=NULL;

		  if (string_node->properties!=NULL)
		    {
		      link_node=((THelpLineProperties *)(string_node->properties))->f_last_link;
		      while (link_node!=NULL)
			{
			  if (f_cursor_col>link_node->col2)
			    {
			      selected_link=link_node;
			      break;
			    }
			  link_node=link_node->last;
			}
		    }

		  if (selected_link==NULL)
		    {
		      initial_string_node=string_node;
		      string_node=string_node->last;
		      line--;

		      if (string_node==NULL)
			{
			  string_node=f_last_string;
			  line=f_nb_lines;
			}

		      while ((string_node!=initial_string_node) && (selected_link==NULL))
			{
			  if (string_node->properties!=NULL)
			    selected_link=((THelpLineProperties *)(string_node->properties))->f_last_link;
			  else
			    {

			      string_node=string_node->last;
			      if (string_node!=NULL)
				line--;
			      else
				{
				  string_node=f_last_string;
				  line=f_nb_lines;
				}
			    }
			}

		      if (string_node==initial_string_node)
			if (string_node->properties!=NULL)
			  selected_link=((THelpLineProperties *)(string_node->properties))->f_last_link;
		    }


		  if (selected_link!=NULL)
		    m_set_cursor_at(selected_link->col1,line,TRUE);

		  break;
    case INSERT: break; // Fix: not allow cursor redefinition



    default     : return (TTextZone::m_key_pressed_event(key));
  }

  return(TRUE);
}

/*ͻ*/
/*                           METHODES PRIVEES                             */
/*ͼ*/

void THelpTextZone::m_free_help_file()
{
  PHistoryNode node,next_node;
  int i;

  // Voir si un fichier d'aide est toujours ouvert

  if (f_help_file_name==NULL)
    return;

  for (i = 0; i < f_help_count; i++)
    {
      delete []((f_first_help + i)->name);
    }

  delete (f_first_help);  f_first_help = NULL;
  delete []f_help_tree;   f_help_tree = NULL;


  node=f_help_history;
  while (node!=NULL)
    {
      next_node=node->next;
      delete node;
      node=next_node;
    }
  f_help_history=NULL;

  delete []f_help_file_name;
  f_help_file_name=NULL;
}

boolean THelpTextZone::m_open_help_file(const char *file_name)
{
  int     help_file_handle;
  word    tree_count;
  word    bytectr;

  int     i;
  char    help_file_name[MAX_PATH];
  boolean success=TRUE;

  char    signature[SIGNATURE_LENGTH];

  // On recherche le fichier d'aide dans le rpertoire de l'excutable

  FullPathOfFileInExeDir(help_file_name,file_name);

  // Ouverture du fichier

  if ((help_file_handle = sopen(help_file_name,
				  O_RDONLY | O_BINARY,
				  SH_DENYWR, S_IREAD)) <0)
    success=FALSE;

  else
    {
      f_help_file=fdopen(help_file_handle,"rb");
      if (f_help_file==NULL)
	{
	  close(help_file_handle);
	  success=FALSE;
	}
    }

  if (!success)
    {
      m_display_string(file_name,VOC_HELP_FILE_NOT_FOUND);
      return(FALSE);
    }

  // Vrification de la signature

  // Si l'arbre de huffman n'a pas encore t charg...

  if (f_help_tree == NULL)
    {
      if ((fread(signature,SIGNATURE_LENGTH,1,f_help_file))!=1)
	success=FALSE;
      else
	{
	  if (memcmp(signature,HELP_FILE_SIGNATURE,SIGNATURE_LENGTH))
	    success=FALSE;
	}

      if (!success)
	{
	  m_display_string(file_name,VOC_INVALID_HELP_FILE);
	  return(FALSE);
	}


      fread(&bytectr, sizeof(bytectr),1,f_help_file);
      fread(&tree_count , sizeof(tree_count),1,f_help_file );
      fread(&f_root , sizeof(f_root),1,f_help_file );

      f_help_tree = new THuffmanTree [tree_count - 256];

      if (f_help_tree != NULL)
	{
	  for (i = 0; i < tree_count - 256; i++)
	    {
	      fread(&f_help_tree[i].left, sizeof(short int),1,f_help_file);
	      fread(&f_help_tree[i].right,sizeof(short int),1,f_help_file);
	    }
	}
    }
  return(TRUE);
}

THelpTopic *THelpTextZone::m_find_topic(const char *topic)
{
  int             i;
  THelpTopic   *thishelp = NULL;

  for (i = 0; i < f_help_count; i++)
    {
      if (stricmp(topic,(f_first_help + i)->name) == 0)
	{
	  thishelp = f_first_help + i;
	  break;
	}
    }
  return (thishelp);
}

void THelpTextZone::m_read_topic()
{
  THelpLineProperties *line_properties;
  TMousePointer pointer;

  int line_nb;
     char  *cp;
      char  *cp1;
  char *ptr;
  int off1,off2;
  char *pt,*pt1;
  unsigned int help;
  char   *ncp;




  char hline[MAX_TAGGED_LINE_LENGTH+1];

  pointer=GetMousePointer();
  SetMousePointer(MP_HOURGLASS);

  m_enable_modification();
  m_clear_text();

  f_selected_link=NULL;
  f_selected_link_line=0;

  /* -- seek to the first line of the help text -- */
  m_seek_help_line(f_this_help->byte_pos,f_this_help->bit_pos);

  /* ----- read the title ----- */
  if (m_get_help_line(hline)==NULL)
    {
      m_set_caption(" ");
      SetMousePointer(pointer);
      return;
    }
  hline[strlen(hline) - 1] = '\0';
  m_set_caption(hline);

 /* ---- disable ineffective buttons ---- */

 if (f_this_help->previous_help == -1)
   ((PHelpWindow)f_window)->f_previous_button->m_disable();
 else
   ((PHelpWindow)f_window)->f_previous_button->m_enable();

 if (f_this_help->next_help == -1)
   ((PHelpWindow)f_window)->f_next_button->m_disable();
 else
   ((PHelpWindow)f_window)->f_next_button->m_enable();


  /* ----- read the help text ------- */
  line_nb=2;
  while (TRUE)
    {
      cp = hline;

      if (m_get_help_line(hline) == NULL) break;

      if ( (*hline == '<') && (strchr(hline,'>')!=NULL))
	break;

      hline[strlen(hline) - 1] = '\0';
      /* --- add help text to the help window --- */

      line_properties=NULL;

      while (cp != NULL)
	{
	  if ((cp = strchr(cp,'[')) != NULL)
	    {
	      /* ----- hit a new key word ----- */

	      cp++;
	      if ((*cp) != '.')
		continue;

	      cp++;
	      if ((*cp) != '.')
		continue;

	      cp++;
	      ptr=cp-3;
	      off1 = (int)(cp - hline)-2;

	      pt1 = cp;
	      if ((ncp = strchr(cp,']')) != NULL)
		{
		  pt=ncp;

		  if ((ncp = strchr(ncp,'<')) != NULL)
		    {
		      cp1 = strchr(ncp,'>');

		      if (cp1 == ncp+5)
			{
			  off2 = off1 + (int)(pt - pt1) -1;

			  memmove(ptr,ptr+3,(int)(pt-pt1));
			  ptr+=(int)(pt-pt1);

			  sscanf(ncp+1,"%04X",&help);

			  memmove(ptr,cp1 + 1,strlen(cp1+1)+1);
			  cp=ptr;

			  if (off2>=off1)
			    {
			      if (line_properties==NULL)
				line_properties=new THelpLineProperties;
			      line_properties->m_add_link(off1,off2,(short int)help);
			    }
			}
		    }

		}
	    }
	}
      m_insert_line(line_nb++,hline,line_properties);
    }

  m_delete_line(1);
  m_disable_modification();
  SetMousePointer(pointer);
}

void THelpTextZone::m_seek_help_line(long offset,int bit)
{
  DEBUG(long fs);

  /* On se positionne sur le bon octet */

  DEBUG_TEST(f_help_file!=NULL);

  DEBUG(fs =)  fseek(f_help_file,offset,0);
  DEBUG_TEST(fs!=-1);

  f_ct8 = bit;
  if (f_ct8 < 8)
    {
      f_in8 = fgetc(f_help_file);
      f_in8 <<= bit;
    }
}

char *THelpTextZone::m_get_help_line(char *line)
{
  int h;              /* Index in HelpTree */
  int i;

  *line = '\0';
  i=0;

  while (TRUE)
    {
      h = f_root;
      while (h > 255)
	{
	  if (f_ct8 == 8)
	    {
	      if ((f_in8 = fgetc(f_help_file)) == EOF)
		{
		  *line = '\0';
		  return (NULL);
		}
	      f_ct8 = 0;
	    }

	  if (f_in8 & 0x80)
	    h = f_help_tree[h - 256].left;
	  else
	    h = f_help_tree[h - 256].right;

	  f_in8 <<= 1;
	  f_ct8++;
	}

      if (i<MAX_TAGGED_LINE_LENGTH)
        {
          *line++ = h;
          i++;
        }

      if (h == '\n') break;
    }

  *line = '\0';
  return (line);
}

void THelpTextZone::m_select_topic(THelpTopic *new_topic,boolean push_current_topic)
{
  m_display_topic( ((new_topic==NULL)?NULL:new_topic->name),
		   push_current_topic);
}


PLinkNode THelpTextZone::m_pos_to_link(int col,int line)
{
  THelpLineProperties *properties;
  PLinkNode           selected_link,link_node;

  // Recherche d'un lien sous le curseur

  PStringNode node;

  if ((line<1) || (line>f_nb_lines))
    return(NULL);

  node=m_line_to_string(line);

  properties=(THelpLineProperties *)node->properties;

  selected_link=NULL;
  if (properties!=NULL)
    {
      link_node=properties->f_links;
      while ((link_node!=NULL) && (selected_link==NULL))
	{
	  if (   (col>=link_node->col1)
	      && (col<=link_node->col2))
	    selected_link=link_node;
	  link_node=link_node->next;
	}
    }

  return(selected_link);
}

void THelpTextZone::m_push_current_topic()
{
  PHistoryNode new_node;
  if (f_this_help!=NULL)
    {
      new_node=new THistoryNode;
      new_node->topic_index=(int)(f_this_help-f_first_help);
      new_node->first_col=f_first_visible_col;
      new_node->first_line=f_first_visible_line;
      new_node->col=f_cursor_col;
      new_node->line=f_cursor_line;

      new_node->next=f_help_history;
      f_help_history=new_node;

      ((PHelpWindow)f_window)->f_back_button->m_enable();
    }
}

void THelpTextZone::m_display_string(const char *caption, int string_number)
{
  ((PHelpWindow)f_window)->f_previous_button->m_disable();
  ((PHelpWindow)f_window)->f_next_button->m_disable();
  m_set_caption(caption);

  m_push_current_topic();
  m_enable_modification();
  m_clear_text();
  m_set_line(1,GetString(string_number));
  m_disable_modification();

  f_this_help=NULL;
}