/*
 * @(#)dir.c 1.4 87/11/06 Public Domain.
 *
 *  A public domain implementation of BSD directory routines for
 *  MS-DOS.  Written by Michael Rendell ({uunet,utai}michael@garfield),
 *  August 1897
 *
 *  Ported to OS/2 by Kai Uwe Rommel
 *  December 1989, February 1990
 *  Change for HPFS support, October 1990
 *  Updated for better error checking and real rewinddir, February 1992
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>

#ifdef OS2

#define far     _far
#define near    _near
#define pascal  _pascal
#define cdecl   _cdecl

#define INCL_NOPM
#include <os2.h>

static HDIR hdir;
static USHORT count;
static FILEFINDBUF find;

#else

#include <dos.h>

#define TRUE    1
#define FALSE   0

static struct find_t find;

#define achName name
#define DosFindFirst(p, pd, a, pf, sf, pc, o)    _dos_findfirst(p, a, pf)
#define DosFindNext(d, pf, sf, pc)               _dos_findnext(pf)
#define DosFindClose(d)

#endif


int attributes = A_DIR | A_HIDDEN;

static char *getdirent(char *);
static void free_dircontents(struct _dircontents *);


DIR *opendir(char *name)
{
  struct stat statb;
  DIR *dirp;
  char c;
  char *s;
  struct _dircontents *dp;
  char nbuf[MAXPATHLEN + 1];
  int len;

  strcpy(nbuf, name);
  len = strlen (nbuf);
  s = nbuf + len;

  if ( ((c = nbuf[strlen(nbuf) - 1]) == '\\' || c == '/') &&
       (strlen(nbuf) > 1) )
  {
    nbuf[strlen(nbuf) - 1] = 0;

    if ( nbuf[strlen(nbuf) - 1] == ':' )
      strcat(nbuf, "\\.");
  }
  else
    if ( nbuf[strlen(nbuf) - 1] == ':' )
      strcat(nbuf, ".");

  if ( stat(nbuf, &statb) < 0 || (statb.st_mode & S_IFMT) != S_IFDIR )
    return NULL;

  if ( nbuf[strlen(nbuf) - 1] == '.'
       && (strlen(nbuf) < 2 || nbuf[strlen(nbuf) - 2] != '.') )
    strcpy(nbuf + strlen(nbuf) - 1, "*.*");
  else
    if ( ((c = nbuf[strlen(nbuf) - 1]) == '\\' || c == '/') &&
         (strlen(nbuf) == 1) )
      strcat(nbuf, "*.*");
    else
      strcat(nbuf, "\\*.*");

  if ( (dirp = malloc(sizeof(DIR))) == NULL ||
       (dirp -> dd_name = malloc(strlen(nbuf) + 1)) == NULL )
  {
    if ( dirp )
      free(dirp);

    return NULL;
  }

  dirp -> dd_loc = 0;
  dirp -> dd_contents = dirp -> dd_cp = NULL;
  strcpy(dirp -> dd_name, nbuf);

  if ( (s = getdirent(nbuf)) == NULL )
    return dirp;

  do
  {
    if ( (dp = malloc(sizeof(struct _dircontents))) == NULL ||
         (dp -> _d_entry = malloc(strlen(s) + 1)) == NULL      )
    {
      if ( dp )
        free(dp);
      free_dircontents(dirp -> dd_contents);
      free(dirp -> dd_name);
      free(dirp);

      return NULL;
    }

    if ( dirp -> dd_contents )
    {
      dirp -> dd_cp -> _d_next = dp;
      dirp -> dd_cp = dirp -> dd_cp -> _d_next;
    }
    else
      dirp -> dd_contents = dirp -> dd_cp = dp;

    strcpy(dp -> _d_entry, s);
    dp -> _d_next = NULL;

#ifdef OS2
    dp -> _d_size = find.cbFile;
    dp -> _d_mode = find.attrFile;
    dp -> _d_time = *(unsigned *) &(find.ftimeLastWrite);
    dp -> _d_date = *(unsigned *) &(find.fdateLastWrite);
#else
    dp -> _d_size = find.size;
    dp -> _d_mode = find.attrib;
    dp -> _d_time = find.wr_time;
    dp -> _d_date = find.wr_date;
#endif
  }
  while ( (s = getdirent(NULL)) != NULL );

  dirp -> dd_cp = dirp -> dd_contents;

  return dirp;
}


int closedir(DIR *dirp)
{
  if ( dirp == NULL )
    return -1;

  free_dircontents(dirp -> dd_contents);
  free(dirp -> dd_name);
  free(dirp);

  return 0;
}


struct dirent *readdir(DIR *dirp)
{
  static struct dirent dp;

  if ( dirp == NULL || dirp -> dd_cp == NULL )
    return NULL;

  dp.d_namlen = dp.d_reclen =
    strlen(strcpy(dp.d_name, dirp -> dd_cp -> _d_entry));

  dp.d_ino = 0;

  dp.d_size = dirp -> dd_cp -> _d_size;
  dp.d_mode = dirp -> dd_cp -> _d_mode;
  dp.d_time = dirp -> dd_cp -> _d_time;
  dp.d_date = dirp -> dd_cp -> _d_date;

  dirp -> dd_cp = dirp -> dd_cp -> _d_next;
  dirp -> dd_loc++;

  return &dp;
}


void seekdir(DIR *dirp, long off)
{
  long i = off;
  struct _dircontents *dp;

  if ( dirp == NULL )
    return;

  if ( off >= 0 )
  {
    for ( dp = dirp -> dd_contents; (--i >= 0) && dp; dp = dp -> _d_next );

    dirp -> dd_loc = off - (i + 1);
    dirp -> dd_cp = dp;
  }
}


long telldir(DIR *dirp)
{
  if ( dirp == NULL )
    return -1;

  return dirp -> dd_loc;
}


void rewinddir(DIR *dirp)
{
  char *s;
  struct _dircontents *dp;

  if ( dirp == NULL )
    return;

  free_dircontents(dirp -> dd_contents);
  dirp -> dd_loc = 0;
  dirp -> dd_contents = dirp -> dd_cp = NULL;

  if ( (s = getdirent(dirp -> dd_name)) == NULL )
    return;

  do
  {
    if ( (dp = malloc(sizeof(struct _dircontents))) == NULL ||
         (dp -> _d_entry = malloc(strlen(s) + 1)) == NULL      )
    {
      if ( dp )
        free(dp);
      free_dircontents(dirp -> dd_contents);
      dirp -> dd_contents = dirp -> dd_cp = NULL;

      return;
    }

    if ( dirp -> dd_contents )
    {
      dirp -> dd_cp -> _d_next = dp;
      dirp -> dd_cp = dirp -> dd_cp -> _d_next;
    }
    else
      dirp -> dd_contents = dirp -> dd_cp = dp;

    strcpy(dp -> _d_entry, s);
    dp -> _d_next = NULL;

#ifdef OS2
    dp -> _d_size = find.cbFile;
    dp -> _d_mode = find.attrFile;
    dp -> _d_time = *(unsigned *) &(find.ftimeLastWrite);
    dp -> _d_date = *(unsigned *) &(find.fdateLastWrite);
#else
    dp -> _d_size = find.size;
    dp -> _d_mode = find.attrib;
    dp -> _d_time = find.wr_time;
    dp -> _d_date = find.wr_date;
#endif
  }
  while ( (s = getdirent(NULL)) != NULL );

  dirp -> dd_cp = dirp -> dd_contents;
}


static void free_dircontents(struct _dircontents *dp)
{
  struct _dircontents *odp;

  while (dp)
  {
    if (dp -> _d_entry)
      free(dp -> _d_entry);

    odp = dp;
    dp = odp -> _d_next;
    free(odp);
  }
}


static char *getdirent(char *dir)
{
  int done;
  static int lower = TRUE;

  if (dir != NULL)
  {				       /* get first entry */
    lower = IsFileSystemFAT(dir);

#ifdef OS2
    hdir = HDIR_CREATE;
    count = 1;
#endif
    done = DosFindFirst(dir, &hdir, attributes,
			&find, sizeof(find), &count, 0L);
  }
  else				       /* get next entry */
    done = DosFindNext(hdir, &find, sizeof(find), &count);

  if (done == 0)
  {
    if ( lower )
      strlwr(find.achName);

    return find.achName;
  }
  else
  {
    DosFindClose(hdir);
    return NULL;
  }
}


int IsFileSystemFAT(char *dir)
{
#ifdef OS2
  USHORT nDrive;
  ULONG lMap;
  BYTE bData[64], bName[3];
  USHORT cbData;
  static USHORT nLastDrive = -1, nResult;

  if ( _osmode == DOS_MODE )
    return TRUE;

  /* We separate FAT and HPFS file systems here. */

  if ( isalpha(dir[0]) && (dir[1] == ':') )
    nDrive = toupper(dir[0]) - '@';
  else
    DosQCurDisk(&nDrive, &lMap);

  if ( nDrive == nLastDrive )
    return nResult;

  bName[0] = (char) (nDrive + '@');
  bName[1] = ':';
  bName[2] = 0;

  nLastDrive = nDrive;
  cbData = sizeof(bData);

  nResult = (DosQFSAttach(bName, 0U, 1U, bData, &cbData, 0L) == 0) 
         && (strcmp(bData + (*(USHORT *) (bData + 2) + 7), "FAT") == 0);

  return nResult;  
#else
  return TRUE;
#endif
}
