-----------------------------------------------------------------------
--
--  File:        shows.adb
--  Description: Shows how to use the VGA_Grapics package
--  Rev:         0.7
--  Date:        01-feb-98
--  Author:      Jerry van Dijk
--  Mail:        jdijk@acm.org
--
--  Copyright (c) Jerry van Dijk, 1996, 1997, 1998
--  Billie Holidaystraat 28
--  2324 LK Leiden
--  THE NETHERLANDS
--  tel int + 31 71 531 4365
--
--  Permission granted to use for any purpose, provided this copyright
--  remains attached and unmodified.
--
--  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
--  IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
--
-----------------------------------------------------------------------

with Ada.Characters.Latin_1;
with Ada.Text_Io; use Ada.Text_Io;

with VGA_Graphics; use VGA_Graphics;
with PC_Keyboard;  use PC_Keyboard;

procedure Shows is

   Continue : Boolean := False;  -- For protection test

   -- Display a message and wait for <Enter> pressed
   procedure Message(S : in String) is
      X : constant Horizontal_Location := (80 - S'Length) * 4;
      Y : constant Vertical_Location   := Vertical_Maximum - 28;
   begin
      Write_String(X, Y, White, S);
      while Get_Char /= Ada.Characters.Latin_1.CR loop
         null;
      end loop;
      Fill_Box(X, Y, X + S'Length * 8, Y + Font_Height - 1, Get_Pixel(0,0));
   end Message;

   --  Fill the whole screen Cyan using the Put_Pixel procedure
   procedure Pixel_Test is
   begin
      for Y in 0..Vertical_Maximum loop
         For X in 0..Horizontal_Maximum loop
            Put_Pixel(X, Y, Cyan);
         end loop;
      end loop;
   end Pixel_Test;

   --  Fill the whole screen red using the Vertical_Line procedure
   procedure Vertical_Test is
   begin
      for X in 0..Horizontal_Maximum loop
         Vertical_Line(X, 0, Vertical_Maximum, Red);
      end loop;
   end Vertical_Test;

   --  Fill the whole screen green using the Horizontal_Line procedure
   procedure Horizontal_Test is
   begin
      for Y in 0..Vertical_Maximum loop
         Horizontal_Line(0, Horizontal_Maximum, Y, Green);
      end loop;
   end Horizontal_Test;

   --  Check if screen is completely green using Get_Pixel
   procedure Test_Color is
      Wrong_Color : exception;
   begin
      for Y in 0..Vertical_Maximum loop
         for X in 0.. Horizontal_Maximum loop
            if Get_Pixel(X, Y) /= Green then
               raise Wrong_Color;
            end if;
         end loop;
      end loop;
   exception
      when Wrong_Color =>
         Write_String(0, 0, Red, "Screen not completely filled !");
   end Test_Color;

   procedure Box_Test is
   begin
      -- Outer border
      Vertical_Line( 96, 47, 204, Black);
      Vertical_Line(304, 48, 204, White);
      Horizontal_Line(96, 304,  46, Black);
      Horizontal_Line(97, 303, 204, White);
      -- Border filling
      Vertical_Line(97, 49, 201, Light_Gray);
      Vertical_Line(98, 49, 201, Light_Gray);
      Vertical_Line(302, 49, 201, Light_Gray);
      Vertical_Line(303, 49, 201, Light_Gray);
      Horizontal_Line(97, 303,  47, Dark_Gray);
      Horizontal_Line(97, 303,  48, Light_Gray);
      Horizontal_Line(97, 303, 202, Light_Gray);
      Horizontal_Line(97, 303, 203, Light_Gray);
      -- Inner border
      Vertical_Line( 99, 50, 201, White);
      Vertical_Line(301, 50, 200, Black);
      Horizontal_Line( 99, 301,  49, White);
      Horizontal_Line(100, 301, 201, Black);
      -- Fill window
      Fill_Box(100, 50, 300, 200, Dark_Gray);
   end Box_Test;

   --  Draw a number of brown circles
   procedure Circle_Test is
   begin
      Fill_Box(349, 250, 451, 350, Blue);
      for Radius in 1..50 loop
         Draw_Circle(400, 300, Radius, Brown);
      end loop;
   end Circle_Test;

   -- Draw a number of yellow diagonal lines in a box
   procedure Line_Test is
   begin
      -- Draw Box
      Draw_Line(349, 250, 451, 250, Yellow);
      Draw_Line(349, 350, 451, 350, Yellow);
      Draw_Line(349, 250, 349, 350, Yellow);
      Draw_Line(451, 250, 451, 350, Yellow);
      -- Draw diagonals
      Draw_Line(349, 250, 451, 350, Yellow);
      Draw_Line(451, 250, 349, 350, Yellow);
   end Line_Test;

   --  Test the screen buffer
   procedure Copy_Test is
      Buffer : Screen_Buffer(103, 101);
   begin
      Get_Buffer(Buffer, 349, 250);
      Put_Buffer(Buffer, 120, 280);
   end Copy_Test;

   --  Cycle palette colors
   procedure Palette_Test is
      Pal  : Color_Palette;
   begin
      Pal := Read_All_Palette;
      for I in 0..Screen_Color'Pos(Screen_Color'Last)-1 loop
         delay 0.3;
         Set_Palette(Pal, Border, Get_Palette(Pal, Black));
         for J in Black..White loop
            Set_Palette(Pal, J, Get_Palette(Pal, Screen_Color'Succ(J)));
         end loop;
         Set_Palette(Pal, Border, Get_Palette(Pal, Black));
         Write_All_Palette(Pal);
      end loop;
   end Palette_Test;

   procedure Flood_Fill_Test is
   begin
      Clear_Screen(Brown);
      Horizontal_Line(60, 160, 60, Yellow);
      Horizontal_Line(60, 160, 160, Yellow);
      Vertical_Line(60, 60, 160, Yellow);
      Vertical_Line(160, 60, 160, Yellow);
      Draw_Line(60, 60, 110, 40, Yellow);
      Draw_Line(160, 60, 210, 40, Yellow);
      Horizontal_Line(110, 210, 40, Yellow);
      Draw_Line(160, 160, 210, 140, Yellow);
      Vertical_Line(210, 40, 140, Yellow);
      Draw_Line(300, 40, 400, 100, Blue);
      Draw_Line(400, 100, 500, 40, Blue);
      Draw_Line(300, 40, 400, 250, Blue);
      Draw_Line(400, 250, 500, 40, Blue);
      Message("Press <Enter> to floodfill the figures");
      Flood_Fill(61, 61, Light_Blue);
      Flood_Fill(100, 50, Light_Cyan);
      Flood_Fill(161, 61, Blue);
      Flood_Fill(305, 46, Light_Green);
   end Flood_Fill_Test;

begin

   begin                --  Check protection
      Clear_Screen;
   exception
      when Not_In_VGA_Error => Continue := True;
   end;

   if Continue = False then
      Put_Line(Standard_Error, "Protection is failing!");
   else

      VGA_Mode;            --  Switch to VGA mode

      Message("Press <Enter> to clear the screen");
      Clear_Screen;        --  Clear screen to default background

      Message("Press <Enter> to fill the screen pixel by pixel");
      Pixel_Test;          --  Put_Pixel test

      Message("Press <Enter> to fill the screen with vertical lines");
      Vertical_Test;       --  Vertical_Line test

      Message("Press <Enter> to fill the screen with horizontal lines");
      Horizontal_Test;     --  Horizontal_Line test

      Message("Press <Enter> to read the screen pixel by pixel");
      Test_Color;          --  Get_Pixel test

      Message("Press <Enter> to draw a window");
      Box_Test;            --  Fill_Box test

      Message("Press <Enter> to draw circles");
      Circle_Test;         --  Draw_Circle test

      Message("Press <Enter> to draw lines");
      Line_Test;           --  Draw_Line test

      Message("Press <Enter> to copy the boxed circle");
      Copy_Test;           --  Test the screen buffer stuff

      Message("Press <Enter> to test the color palette");
      Palette_Test;        --  Test palette functions;

      Message("Press <Enter> to test the floodfill function");
      Flood_Fill_Test;     --  Test floodfill function;

      Message("Press <Enter> to return to DOS");
      TXT_Mode;            --  Return to the original video mode

   end if;  -- Continue

exception

   -----------------------------------------------
   --  In case we could not switch to VGA mode  --
   -----------------------------------------------
   when No_VGA_Error =>
      Put_Line(Standard_Error, "Couldn't switch to VGA mode");

   -------------------------------------------------------
   -- In case we could not return to the original mode  --
   -------------------------------------------------------
   when No_TXT_Mode_Error =>
      Put_Line(Standard_Error, "Couldn't switch back to text mode");

   -------------------------------------------------------------
   --  In case we get/put a Screen_Buffer outside the screen  --
   -------------------------------------------------------------
   when Screen_Buffer_Error =>
      TXT_Mode;
      Put_Line(Standard_Error, "Screen_Buffer outside screen or empty");

   ----------------------------------------------------
   --  In case we execute VGA commands in Text mode  --
   ----------------------------------------------------
   when Not_In_VGA_Error =>
      Put_Line(Standard_Error, "Executing graphics command in text mode");

   ---------------------------------------
   -- In case we move beyond the screen --
   ---------------------------------------
   when Outside_Screen_Error =>
      TXT_Mode;
      Put_Line(Standard_Error, "Trying to access beyond the screen limits");

   ---------------------------------------------
   -- In case the Flood_Fill procedure failed --
   ---------------------------------------------
   when Flood_Fill_Failed =>
      TXT_Mode;
      Put_Line(Standard_Error, "The floodfill algoritm failed");

end Shows;
