WITH Ada.Text_IO; USE Ada.Text_IO;
WITH Currency; USE Currency;
WITH Dates; USE Dates;
WITH Persons; USE Persons;
WITH Personnel; USE Personnel;
WITH Payroll; USE Payroll;
WITH Lists_Generic;
PROCEDURE Payroll_List IS
------------------------------------------------------------------------
--| Demonstrates the use of a heterogeneous list.
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: September 1995                                     
------------------------------------------------------------------------
    
  TYPE PayrollPointer IS ACCESS ALL Person'Class;
  -- as before, this can designate a Person or anything
  -- derived from Person
  
  PACKAGE PayrollLists IS NEW Lists_Generic
    (ElementType => PayrollPointer);
  USE PayrollLists;
  -- The list element type is now a classwide pointer
  
  Company: List;
  Which  : Position;
  Temp   : PayrollPointer;
  
BEGIN -- Payroll_List

  -- Construct all the people dynamically, and add each one
  -- to the end of the list as it is constructed. We no longer
  -- need an explicit variable for each person.
  
  Temp := NEW Person'(Persons.Constructors.MakePerson(
              Name      => "George",
              Gender    => Male,
              BirthDate => MakeDate(1971,11,2)));
  AddToRear(Company, Temp);
  
  Temp := NEW Employee'(Personnel.Constructors.MakeEmployee(
                Name      => "Mary",  
                Gender    => Female,
                BirthDate => MakeDate(1950,10,21),
                ID        => 1234,
                StartDate => MakeDate(1989,7,1)));
  AddToRear(Company, Temp);
  
  Temp := NEW Professional'(Payroll.Constructors.MakeProfessional(
             Name        => "Martha",
             Gender      => Female,
             BirthDate   => MakeDate(1947,7,8),
             ID          => 2222,
             StartDate   => MakeDate(1985,6,6),
             MonthSalary => MakeCurrency(50000.00)));
  AddToRear(Company, Temp);
  
  Temp := NEW Sales'(Payroll.Constructors.MakeSales(
             Name       => "Virginia",
             Gender     => Female,
             BirthDate  => MakeDate(1955,2,1),
             ID         => 3456,
             StartDate  => MakeDate(1990,1,1),
             WeekSalary => MakeCurrency(2500.00),
             CommRate   => 0.25)); 
  AddToRear(Company, Temp);
  
  Temp := NEW Clerical'(Payroll.Constructors.MakeClerical(
             Name       => "Herman",
             Gender     => Male,
             BirthDate  => MakeDate(1975,5,13),
             ID         => 1557,
             StartDate  => MakeDate(1991,7,1),
             HourlyWage => MakeCurrency(7.50)));
  AddToRear(Company, Temp);
  
  -- Now we can traverse the list. Note again that Put is a 
  -- dispatching operation; the correct Put is dispatched at
  -- execution time.

  Which := First(Company);
  WHILE NOT IsPastEnd(Company, Which) LOOP
    Put(Item => Retrieve(Company,Which).ALL); -- dispatching
    Ada.Text_IO.Put_Line(Item => "------------------------");
    GoAhead(Company, Which);
  END LOOP;  
 
END Payroll_List; 
