 title Mc_Core.Asm is the resident control module for Mystic

; Copyright (c) ASN 1998-99 (See Read.Me & Mc_Init.Asm for details)

; This module contains the resident interrupt handlers required for keeping
; track of time elapsed, keyboard strikes and activating the screen saver.

; Mystic v1.0 Application Program Interface (API) is INT 69h. Service code to
; be passed in AX. Calls to the API normally destroy register values.

; Tested with TASM v3.1 normal settings (no additional options). Should
; assemble with MASM without any modifications. NASM/A86 will require some
; touching up.

.8086                                  ; pure 86/88 assembly code here!
.radix 10                              ; default number base is 10

;----------------------------------------------------------------------------;
 VIDEO_INT        equ       10h        ; video interrupt
 GET_VIDEO_MODE   equ       0fh        ; get video mode service
 STACK_SIZE       equ       256d       ; 128 words for the stack
 KEY_IN_PORT      equ       60h        ; keyboard read port
 KEY_ACK_PORT     equ       61h        ; keyboard acknowledgement port
 NONSPECIFIC_EOI  equ       20h        ; non specific end-of-interrupt
 PIC              equ       20h        ; port address of interrupt controller

 MC_API_ERROR     equ      0fefeh      ; API error

 NO               equ        0         ; define constants for
 YES              equ     not (NO)     ;   intuitive programming

;----------------------------------------------------------------------------;
 _TEXT segment byte public use16 'CODE'
     assume cs:_TEXT, ds:nothing, es:nothing, ss:nothing

 org 100h                              ; this is a .COM program

     extrn MC_Load_Stub:near

 MC_Begin:
     call MC_Load_Stub                 ; call stub load code (Mc_Init.Asm)

;----------------------------------------------------------------------------;
     public Old_Timer_Handler
     public Old_Key_Handler
     public Old_DOS_Idle_Handler
     public Old_INT69h_Handler
     public In_DOS_Flag

 Old_Timer_Handler      dd     0       ; old INT 08h (timer) handler
 Old_Key_Handler        dd     0       ; old INT 09h (keyboard) handler
 Old_DOS_Idle_Handler   dd     0       ; old INT 28h (DOS idle) handler
 Old_INT69h_Handler     dd     0       ; old INT 69h handler
 In_DOS_Flag            dd     0       ; address of inDOS flag

 Tick_Count             dw     0       ; keeps track of time (volatile)
 Old_Stack_Segment      dw     0       ; we save SS & SP in these buffers
 Old_Stack_Pointer      dw     0       ;  to preserve current program's stack

 MC_API_List label word                ; jump table based on fn-code
     dw MC_SS_API_Call                 ; fn 0 - screen saver specific options
     dw MC_Get_Info                    ; fn 1 - get screen saver info
     dw MC_Set_Delay                   ; fn 2 - set screen saver delay
     dw MC_Set_Modes                   ; fn 3 - set valid video modes
     dw MC_Unlink                      ; fn 4 - unlink screen saver
     dw MC_Suspend                     ; fn 5 - suspend screen saver
     dw MC_Enable                      ; fn 6 - enable screen saver

 MC_API_MAX equ (($-MC_API_List)/2)-1  ; highest API service supported

 Is_Triggered?          db    NO       ; 1=trigger_scheduled, 0=not_scheduled
 Is_Active?             db    NO       ; 1=screen_saver_active, 0=inactive
 Is_Key_Hit?            db    NO       ; 1=key_hit, 0=no_key_hit
 Is_Enabled?            db   YES       ; 1=enabled, 0=disabled

 Stack_Buffer db STACK_SIZE dup (0)    ; our program's stack

 Stack_Top equ this byte               ; stacktop

;----------------------------------------------------------------------------;
     public MC_DOS_Idle_Handler

     extrn SS_Main_Body_Code:near
     extrn SS_Start_Up_Code:near
     extrn SS_Clean_Up_Code:near
     extrn SS_Time_Delay:word          ; timer ticks before activation
     extrn SS_Valid_Mode_List:dword    ; bit-pattern : 1=valid, 0=invalid

 MC_DOS_Idle_Handler proc far          ; INT 28h (DOS idle) handler
     cmp [Is_Enabled?], NO             ; is screen saver disabled ?
     je Normal_DOS_Idle_Flow           ; yes, so quit

     cmp [Is_Triggered?], NO           ; activate screen saver ?
     jne Trigger_Now                   ; yes, activate it

 Normal_DOS_Idle_Flow:
     jmp [Old_DOS_Idle_Handler]        ; branch to old INT 28h routine

 Trigger_Now:                          ; come here if triggered
     mov [Is_Active?], YES             ; currently active
     sti                               ; enable interrupts

     mov [Old_Stack_Segment], ss       ; save stack of currently
     mov [Old_Stack_Pointer], sp       ;   executing program

     push cs                           ; push CS onto stack
     cli                               ; stack switch - disable interrupts
     pop ss                            ; get segment address
     lea sp, [Stack_Top]               ; load stack top in SP
     sti                               ; re-enable interrupts

     push ax                           ; save regs
     push bx
     push cx
     push dx
     push si
     push di
     push bp
     push ds
     push es

     mov ah, GET_VIDEO_MODE            ; service 0fh - get video mode
     int VIDEO_INT                     ; call video BIOS
     mov cl, al                        ; use mode id as rotate count
     mov ax, 1                         ; 1-bit in AX
     rol ax, cl                        ; swing 1-bit to position
     test ax, word ptr [SS_Valid_Mode_List] ; is mode supported?
     jnz Mode_Is_Valid                 ; yes, it is
     test ax, word ptr [SS_Valid_Mode_List+2] ; if bit not set,
     jz Mode_Is_Invalid                ;   mode is unsupported

 Mode_Is_Valid:
     push cs                           ; set DS to point
     pop ds                            ;   to our data segment

     call SS_Start_Up_Code             ; screen saver init code

 Do_It_Again:
     call SS_Main_Body_Code            ; call screen saver proc
     cmp [Is_Key_Hit?], NO             ; has a key been pressed ?
     je Do_It_Again                    ; if not, repeat

     call SS_Clean_Up_Code             ; screen saver cleanup code

 Mode_Is_Invalid:
     sub ax, ax                        ; zero AX
     mov [Tick_Count], ax              ; re-initialize time elapsed counter
     mov [Is_Triggered?], al           ; not triggered
     mov [Is_Key_Hit?], al             ; no keypress
     mov [Is_Active?], al              ; inactive

     pop es                            ; restore the regs we saved
     pop ds
     pop bp
     pop di
     pop si
     pop dx
     pop cx
     pop bx
     pop ax

     cli                               ; disable interrupts
     mov ss, [Old_Stack_Segment]       ; restore stack of
     mov sp, [Old_Stack_Pointer]       ;   interrupted program
     sti                               ; enable interrupts

     iret                              ; interrupt return
 MC_DOS_Idle_Handler endp

;----------------------------------------------------------------------------;
     public MC_Timer_INT_Handler

 MC_Timer_INT_Handler proc far         ; hooks on to INT 08h
     pushf                             ; push flags
     cli                               ; disable interrupts
     call [Old_Timer_Handler]          ; call the old INT 08h handler routine

     cmp [Is_Active?], NO              ; is screen saver active ?
     je Is_Not_Active                  ; if not, proceed
     iret                              ; return to calling procedure

 Is_Not_Active:
     sti                               ; re-enable interrupts
     push ax
     push bx                           ; save regs
     push ds

     cmp [Is_Triggered?], NO           ; if currently triggered
     jne See_If_DOS_Busy               ;   don't change anything

     mov bx, [Tick_Count]              ; load count in BX
     inc bx                            ; increment count
     mov [Tick_Count], bx              ; store count
     cmp [SS_Time_Delay], bx           ; check if count > delay
     ja Cannot_Activate                ; if not, we guessed right
     mov [Is_Triggered?], YES          ; else we activate

 See_If_DOS_Busy:
     lds bx, [In_DOS_Flag]             ; inDOS flag address
     cmp byte ptr [bx], 0              ; check inDOS flag
     jne Cannot_Activate               ; if DOS is busy, exit

     pushf                             ; save flags
     cli                               ; disable interrupts
     call MC_DOS_Idle_Handler          ; vector to our routine

 Cannot_Activate:
     pop ds                            ; restore regs
     pop bx
     pop ax

     iret
 MC_Timer_INT_Handler endp

;----------------------------------------------------------------------------;
     public MC_Key_INT_Handler

 MC_Key_INT_Handler proc far           ; hooks on to INT 09h
     cmp [Is_Active?], NO              ; is the screen saver active ?
     je Normal_Key_Flow                ; if not, don't do anything

     push ax                           ; else save AX
     cli                               ; disable interrupts
     in al, KEY_IN_PORT                ; read keystroke from port

     in al, KEY_ACK_PORT               ; read byte from clear port
     mov ah, al                        ; save in AH
     or al, 80h                        ; these ops are necessary to
     out KEY_ACK_PORT, al              ;   clear kbd on XTs

     mov al, NONSPECIFIC_EOI           ; non specific eoi
     out PIC, al                       ; signal interrupt controller
     sti                               ; enable interrupts
     pop ax                            ; restore reg
     mov [Is_Key_Hit?], YES            ; record keyboard activity

     iret                              ; interrupt return

 Normal_Key_Flow:
     mov [Tick_Count], 0               ; reset count to 0
     jmp [Old_Key_Handler]             ; branch to original INT 09h handler
 MC_Key_INT_Handler endp

;----------------------------------------------------------------------------;
     public MC_API_Handler             ; grabs INT 69h (trashes regs)

     extrn SS_Module_Name:byte         ; screen saver name
     extrn SS_Author_Name:byte         ; who wrote it
     extrn SS_API_Handler:near         ; screen saver specific API (.SCR)

 MC_API_Handler proc far               ; Mystic v1.0 API function dispatcher
     sti                               ; on entry, AX = API ID, other params
     mov bx, ax                        ; load API function ID in BX
     mov ax, MC_API_ERROR              ; assume API call isn't valid
     cmp bx, MC_API_MAX                ; cross-check assumption
     ja Invalid_API_Call               ; in case we guessed right
                                       ; fall thru in case API call is valid
     add bx, bx                        ; scale for word look-up
     call [MC_API_List+bx]             ; call appropriate function
                                       ; come back here after calling fn
 Invalid_API_Call:                     ; come here directly if invalid call
     iret                              ; return

 MC_API_Handler endp                   ; function dispatcher ends

;----------------------------------------------------------------------------;
;                     Code for Mystic v1.0 API services                      ;
;----------------------------------------------------------------------------;
;    API services may be added to the following list by writing the necessa- ;
; ry code under a label as illustrated below. The code should terminate with ;
; a NEAR return (RETN) instruction. There is no need to preserve registers,  ;
; except possibly SS, SP, BP, DS in case you're issuing the API call from a  ;
; high level language such as C or Pascal. You must also add a DW to the     ;
; MC_API_List array (in the data section above) referencing the label under  ;
; which your code appears. Always append this label at the END of the list   ;
; so that your API service is the last one. For example, to add the service  ;
; My_API_Service to Mystic, proceed as follows.                              ;
;                                                                            ;
; 1. Declare a label called My_API_Service below and write the code for it.  ;
;                                                                            ;
; My_API_Service:                                                            ;
;    .                                                                       ;
;    .                                 ; code for your API service           ;
;    retn                                                                    ;
;                                                                            ;
; 2. Add this label to the end of the MC_API_List array declared above.      ;
;                                                                            ;
; MC_API_List label word                                                     ;
;    dw MC_SS_API_Call                                                       ;
;    .                                 ; other API functions                 ;
;    .                                                                       ;
;    dw My_API_Service                 ; ALWAYS APPEND AT THE END!           ;
;                                                                            ;
; That's it! The next time you build a screen saver, the new API service will;
; be available for use. It's ID will be one greater than the previous API ID ;
;                                                                            ;
;----------------------------------------------------------------------------;
 MC_SS_API_Call:                       ; fn 0 - screen saver specific call
     call SS_API_Handler               ; xfer control to .SCR module
     retn

 MC_Get_Info:                          ; fn 1 - get screen saver info
     mov ah, 0abh                      ; AH = Mystic ID
     mov al, [Is_Enabled?]             ; AL = enabled/suspended status
     push cs                           ; load address of screen saver
     pop es                            ;   name in ES:DI
     lea di, [SS_Module_Name]
     lea si, [SS_Author_Name]          ; author's name in ES:SI
     mov cx, [SS_Time_Delay]           ; CX = time delay in timer ticks
     mov bx, word ptr [SS_Valid_Mode_List]   ; BX = low word
     mov dx, word ptr [SS_Valid_Mode_List+2] ; DX = high word
     retn

 MC_Set_Delay:                         ; fn 2 - set screen saver delay
     mov [SS_Time_Delay], cx           ; CX = time delay in timer ticks
     retn

 MC_Set_Modes:                         ; fn 3 - set valid modes
     mov word ptr [SS_Valid_Mode_List], bx   ; BX = low word
     mov word ptr [SS_Valid_Mode_List+2], dx ; DX = high word
     retn

 MC_Unlink:                            ; fn 4 - unlink screen saver
     retn                              ; function not yet supported

 MC_Suspend:                           ; fn 5 - suspend screen saver
     mov [Is_Enabled?], NO             ; set flag to suspend screen saver
     retn

 MC_Enable:                            ; fn 6 - enable screen saver
     mov [Is_Enabled?], YES            ; set flag to enable screen saver
     mov [Tick_Count], 0               ; start counting
     mov [Is_Triggered?], NO           ; is not triggered
     retn

;----------------------------------------------------------------------------;
 _TEXT ends

     end MC_Begin

;x------------------------------ Mc_Core.Asm -------------------------------x;
