/***************************************************************

	sfsm.c		Space Flight Simulator
			Main Menu and Orbital Modeling Module

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

***************************************************************/

#include "stdio.h"
#include "time.h"
#include "signal.h"
#include "ctype.h"
#include "math.h"

#ifdef __STDC__
#include "malloc.h"
#else
extern char * malloc();
#endif

#include "sfsm.h"

#ifndef   __STDC__
#define   time_t    long
#define   size_t   int
#endif

extern  int sfs_sigint();

#define MM_TITLES       4
/* #define SMALL_DEBUG */
#define MAXYSIZE	375

struct dir_ent *d_entries[ MAX_ENTRIES ];
char *d_titles[ MAX_ENTRIES ];

char *mm_titles[ MM_TITLES ] = {
	 MM_SETPROGFILE,
	 MM_OM,
	 MM_OS,
	 MM_EXIT };

/***    Colors */

int   cl_mback;      /* Main background color   */
int   cl_mfore;      /* Main foreground color   */
int   cl_marker;      /* Color for prompt markers   */
int   cl_grid;      /* Color for planetary grid   */
int   cl_surface;      /* Color for planetary surface   */
int   cl_orbits[ OR_COLORS ];      /* Colors for different orbits   */

char    sfs_program[ 36 ];
char    sfs_tbuf[ 512 ];
int     sfs_curorbit = 0;
long    sfs_insertion;
long    sfs_inc;         
long    sfs_tfactor;      /* time factor         */

#define LOGOPAUSE       5

struct  uiwindow *main_window;          /* structure for main window    */
struct  menu_box w1, w2, w3, w4;    /* structures quad windows      */

int   stat_y1;         /* line for status display   */
int   stat_y2;         /* top of status display   */
int   stat_x1;         /* column for status display   */
int   stat_x2;         /* max col for status display   */

char sfs_prname[ 128 ];
char bw_ebuf[ BW_EBUFSIZE ];

static    int midx, midy;
static   FILE *data;

main( argc, argv )
   int argc;
   char **argv;
   {
   int x1, y1, x2, y2, incr, sfs_resume;
   register int c, goodbye;
   static char tbuf[ 64 ];
   static struct sfs_orbit *sorbit_array[ ORBITS ];

   /***    First some precautionary measures               */
   /***    Load "chain" so that the loader (sfs) will exit */
   /***    in the case of any unexpected exit              */

   if ( ( data = fopen( SFS_CHAINFN, "w" ) ) != 0 )
      {
      fprintf( data, "%s \n", SFSCH_EXIT );
      fclose( data );
      }

   /***    Now set a signal handler so that a SIGINT       */
   /***    (control-C under DOS, DEL under Unix) will      */
   /***    exit gracefully                                 */

   signal( SIGINT, sfs_sigint );

   /* Process arguments */

   strcpy( sfs_program, SFS_DEFAULTPROG );
   if ( argc > 1 )
      {
      if ( ( data = fopen( argv[ 1 ], "r" ) ) != 0 )
         {
         fclose( data );
         strcpy( sfs_program, argv[ 1 ] );
         }
#ifdef  DEBUG
      else
	 {
	 fprintf( stderr, "Failed to open file %s\n", argv[ 1 ] );
	 exit( 0 );
	 }
#endif
      }

   sfs_resume = 0;
   if ( argc > 2 )
      {
      if ( strncmp( argv[ 2 ], SFS_CONTARG, (size_t) 4 ) == 0 )
         {
         sfs_resume = TRUE;
         }
      }

   /* Initialize the toolkit */

   ui_init();

   /* Return if initialization failed */

   if ( ui_grwind->initialized == FALSE )
      {
      fprintf( stderr, "Graphics system is not initialized.\n" );
      return -1;
      }

   /* assign colors */

   sfs_colors();         /* Set colors       */


   /* reset icon colors */

   if ( gr_colors > 2 )
      {
      ui_seticons( cl_mback, cl_marker );
      }

   /* Initialize variable-precision trig */

   vpt_init();
   sfs_font( 0 );

   /* get memory for directory/menu arrays */

   for ( c = 0; c < MAX_ENTRIES; ++c )
      {
      if ( ( d_entries[ c ] =
	 (struct dir_ent *) malloc( sizeof( struct dir_ent ))) == NULL )
	 {
	 gr_deinit();
	 kb_deinit();
	 fprintf( stderr, MEMERR_DS, c );
	 exit( 1 );
	 }
      if ( ( d_titles[ c ] = malloc( TITLE_SIZE )) == NULL )
	 {
	 gr_deinit();
	 kb_deinit();
	 fprintf( stderr, MEMERR_TA, c );
	 exit( 1 );
	 }
      }

   /* Clear the screen */

   gr_cls( GR_PRIMARY );

   sfs_font( 0 );

   /* Draw the main screen */

   sfs_ms();

#ifdef	OLD_DEBUG
   kb_rx();
#endif

   if ( sfs_resume == FALSE )
      {

      /* Show the program logo */

      sfs_logo();
      sfs_wait( LOGOPAUSE );

      }

#ifdef  OLD_DEBUG
   bw_error( "Test error message" );
#endif

   /* Set default values */

   sfs_curorbit = 0;
   strcpy( sfs_prname, UNTPROG );
   sfs_tfactor = 1;
   sfs_inc = 30;
   sfs_insertion = 0;

   /* Read the initial program file */

   sfs_pr( sfs_program, sfs_prname, sorbit_array, ORBITS );
   bw_message( " " );

   /* Cycle through the main menu */

   goodbye = 0;
   while( goodbye == 0 )
      {

      c = sfs_mm();

      switch( c )
         {
         case 0:
	    sfs_setpr( sorbit_array, ORBITS );
	    bw_message( " " );
	    sfs_curorbit = 0;
            break;
         case 1:
	    sfs_om( sorbit_array, ORBITS );
            break;
         case 2:
	    if ( ( data = fopen( SFS_CHAINFN, "w" ) ) != 0 )
               {
#ifdef OLD_DEBUG
               bw_debug( "Setting up chain file" );
#endif
	       fprintf( data, "%s %s %s\n", SFSCH_SIMUL, sfs_program,
		  SFS_CONTARG );
               fclose( data );
               }
            sfs_exit();
            exit( 1 );
            break;
#ifdef  BLOCKEDOUT
	 case 3:
	    if ( ( data = fopen( SFS_CHAINFN, "w" ) ) != 0 )
               {
	       fprintf( data, "%s %s %s %s\n", SFSCH_ATLAS, sfs_fdfile,
		  SFS_CONTARG, sfs_program );
               fclose( data );
               }
            sfs_exit();
	    exit( 1 );
            break;
#endif
	 case TK_EXIT:
	 case 3:
	    if ( ( data = fopen( SFS_CHAINFN, "w" ) ) != 0 )
               {
	       fprintf( data, "%s \n", SFS_EXITARG );
               fclose( data );
               }
            sfs_exit();
            exit( 1 );
            goodbye = 1;
            break;
         }
      }

   sfs_exit();

   }

sfs_colors()
   {
   register int c;
   int n;

   if ( gr_colors > 8 )
      {
      cl_mback = WHITE;
      cl_mfore = BLACK;
      cl_marker = DARK_CYAN;
      cl_grid = DARK_BLUE;
      cl_surface = DARK_GREEN;
      n = LIGHT_RED;
      for ( c = 0; c < OR_COLORS; ++c )
         {
         cl_orbits[ c ] = n;
         ++n;
         }
      }
   else if ( gr_colors > 2 )
      {
      cl_mback = WHITE;
      cl_mfore = BLACK;
      cl_marker = BLACK;
      cl_grid = WHITE;
      cl_surface = WHITE;
      for ( c = 0; c < OR_COLORS; ++c )
         {
         cl_orbits[ c ] = WHITE;
         }
      }
   else
      {
      cl_mback = WHITE;
      cl_mfore = BLACK;
      cl_marker = WHITE;
      cl_grid = WHITE;
      cl_surface = WHITE;
      for ( c = 0; c < OR_COLORS; ++c )
         {
         cl_orbits[ c ] = WHITE;
         }
      }

   }

/***  Main SFS Screen      */

sfs_ms()
   {

   /* Draw the window itself */

   sprintf( sfs_tbuf, "%s, %s %s", SFS_TITLE, VERABB, VERSION );
   main_window = ui_window( 0, 0, ui_grwind->xmax, ui_grwind->ymax,
      TRUE, cl_marker, cl_mfore,
      sfs_tbuf, FALSE, cl_mback,
      FALSE, BLACK, cl_mback, GRID, BUT_CLOSE );
   ui_fbox( main_window->u_x1, main_window->u_y1,
      main_window->u_x2, main_window->u_y2, cl_mback, GRID );

   /* Draw two lines demarcating Status area */

   gr_line( GR_PRIMARY, ui_grwind->xmax / 2, main_window->tbar_y1,
      ui_grwind->xmax / 2, main_window->y2, cl_mfore, SOLID );
   gr_line( GR_PRIMARY, ( ui_grwind->xmax / 2 ) + 2,
      main_window->tbar_y1,
      ( ui_grwind->xmax / 2 ) + 2,
      main_window->y2, cl_mfore, SOLID );

   /* Record the status area */

   stat_y1 = main_window->tbar_y1;
   stat_y2 = main_window->y2 + ( ( ui_grwind->fysize ) * 0.3 );
   stat_x1 = ( main_window->x2 / 2 ) + 5;
   stat_x2 = main_window->tbar_x2;

   }

/*** Display Program Logo */

sfs_logo()
   {
   struct uiwindow *logo_window;
   int x1, x2, y1, y2, height, length, midx, midy;

   midx = main_window->u_x1 + ( ( main_window->u_x2 - main_window->u_x1 ) / 2 );
   midy = main_window->u_y1 + ( ( main_window->u_y2 - main_window->u_y1 ) / 2 );

   length = ( main_window->u_x2 * 7 ) / 10;
   x1 = midx - length / 2;
   x2 = midx + length / 2;
   height = main_window->u_y2 / 2;
   y1 = midy - height / 2;
   y2 = midy + height / 2;

   logo_window = ui_window( x1, y1, x2, y2, FALSE, cl_marker, cl_mfore,
      "", TRUE, cl_mfore, TRUE, cl_mfore, cl_mback, SOLID, FALSE );

   sfs_font( 2 ); 
   y1 = midy + ui_grwind->fysize;
   gr_text( GR_PRIMARY, midx - ( gr_strlen( SFS_TITLE ) / 2 ),
      y1, SFS_TITLE, cl_mfore, cl_mback );
   sfs_font( 0 );
   y1 -= ui_grwind->fysize * 2;
   gr_text( GR_PRIMARY, midx - ( gr_strlen( SFSM_PAX ) / 2 ),
      y1, SFSM_PAX, cl_mfore, cl_mback );
   y1 -= ui_grwind->fysize * 2;
   gr_text( GR_PRIMARY, midx - ( gr_strlen( SFSM_CPRT ) / 2 ),
      y1, SFSM_CPRT, cl_mfore, cl_mback );
   y1 -= ( ui_grwind->fysize * 4 ) / 3;
   gr_text( GR_PRIMARY, midx - ( gr_strlen( SFSM_BWS ) / 2 ),
      y1, SFSM_BWS, cl_mfore, cl_mback );

   }

/***  Main Menu */

sfs_mm()
   {
   static struct uiwindow *pr_box;
   static struct menu_box m_box;

   /* Clear the entire area */

   ui_fbox( main_window->u_x1, main_window->u_y1,
      main_window->u_x2, main_window->u_y2,
      BLACK, SOLID );
   ui_fbox( main_window->u_x1, main_window->u_y1,
      main_window->u_x2, main_window->u_y2, 
      cl_mback, GRID );

   bw_message( MES_MM );

   /* Display the program box */

   if ( pr_box == NULL )
      {
      pr_box = ui_window( main_window->u_x1 + 2 +
	    (( main_window->u_x2 - main_window->u_x1 )/2),
	 main_window->u_y2 -
	    ( ui_grwind->fysize * ( MM_TITLES + 3 )),
	 main_window->u_x2, main_window->u_y2,
	 TRUE, cl_marker, cl_mfore, SFSM_CPFX,
	 TRUE, cl_mfore, FALSE, BLACK, cl_mback, SOLID, FALSE );
      }
   else
      {
      ui_rewindow( pr_box, SFSM_CPFX );
      }
   sprintf( sfs_tbuf, SFSM_CPFI,
      sfs_program, sfs_prname );
   ui_text( pr_box->u_x1, pr_box->u_y1, pr_box->u_x2, pr_box->u_y2,
      64, cl_mback, cl_mfore, sfs_tbuf );

   /* get the main menu item and return it */

   m_box.is_drawn = FALSE;
   return ui_list( MENU_SLIDERS, main_window->u_x1,
      main_window->u_y2 -
	 ( ui_grwind->fysize * ( MM_TITLES + 3 )),
      main_window->u_x2 -
         (( main_window->u_x2 - main_window->u_x1 )/2),
      main_window->u_y2,
      MM_TTL, MM_TITLES, mm_titles,
      cl_mfore, cl_mback, cl_marker, &m_box );

   }

sfs_setpr( sorbit_array, n_orbits )
   struct sfs_orbit **sorbit_array;
   int n_orbits;
   {
   static struct menu_box setp_box;
   int x1, x2, y1, y2, height, length, midx, midy;
   int item;

   midx = main_window->u_x1 + ( ( main_window->u_x2 - main_window->u_x1 ) / 2 );
   midy = main_window->u_y1 + ( ( main_window->u_y2 - main_window->u_y1 ) / 2 );

   length = ( main_window->u_x2 * 7 ) / 10;
   x1 = midx - length / 2;
   x2 = midx + length / 2;
   height = main_window->u_y2 / 2;
   y1 = midy - height / 2;
   y2 = midy + height / 2;

   setp_box.is_drawn = FALSE;
   item = ui_ftext( x1, y1, x2, y2, "*.sfs", SFSM_SPF,
      &setp_box, d_titles, d_entries, MAX_ENTRIES,
      cl_mback, cl_mfore, cl_marker );

   fntolc( d_entries[ item ] );
   strcpy( sfs_program, d_entries[ item ] );
   sfs_pr( sfs_program, sfs_prname, sorbit_array, n_orbits );

   }

sfs_font( size )
   int size;
   {
   static oldsize = 159;
   int rq_height, calcy;

   if ( size == oldsize )
      {
      return;
      }

   if ( ui_grwind->ymax > MAXYSIZE )
      {
      calcy = MAXYSIZE;
      }
   else
      {
      calcy = ui_grwind->ymax;
      }

   switch( size )
      {
      case 1:
	 rq_height = calcy / 25;
         break;
      case 2:
	 rq_height = calcy / 15;
         break;
      default:
	 rq_height = calcy / 30;
         break;
      }

   gr_font( GR_PRIMARY, F_DEFAULT, rq_height );
   }

bw_error( mes )
   char *mes;
   {
   static char buffer[ BW_EBUFSIZE ];
   static int x_mem;
   static size_t x_size;
   static int x1, x2, y1, y2, height, length, midx, midy, c;
   static struct uiwindow *err_window;

   if ( x_mem == NULL )
      {

      midx = main_window->u_x1 + ( ( main_window->u_x2 - main_window->u_x1 ) / 2 );
      midy = main_window->u_y1 + ( ( main_window->u_y2 - main_window->u_y1 ) / 2 );

      length = main_window->u_x2 / 2;
      x1 = midx - length / 2;
      x2 = midx + length / 2;
      height = main_window->u_y2 / 2;
      y1 = midy - height / 2;
      y2 = midy + height / 2;

      }

   gr_imsave( GR_PRIMARY, TRUE, x1, y1, x2, y2, &x_mem );
   buffer[ 0 ] = 0;
   sprintf( buffer, ERR_PROMPT, mes );
   c = ui_yn( x1, y1, x2, y2, cl_mback, cl_mfore, cl_marker, TRUE,
      buffer, ERRP_NO, ERRP_YES, &err_window );
   gr_imsave( GR_PRIMARY, FALSE, x1, y1, x2, y2, &x_mem );
   gr_imfree( x_mem );

   if ( c == FALSE )
      {
      sfs_exit();
      exit( 1 );
      }
   }

#ifdef  DEBUG
bw_debug( mes )
   char *mes;
   {
   static char buffer[ 256 ];
   static int x_mem;
   static size_t x_size;
   static int x1, x2, y1, y2, height, length, midx, midy, c;
   static struct uiwindow *deb_window;

#ifdef  SMALL_DEBUG
   sprintf( buffer, "DEBUG: %s     ", mes );
   gr_text( GR_PRIMARY, 5, 5, buffer, BLACK, WHITE );
   ui_wait();
#else
   if ( x_mem == NULL )
      {

      midx = main_window->u_x1 + ( ( main_window->u_x2 - main_window->u_x1 ) / 2 );
      midy = main_window->u_y1 + ( ( main_window->u_y2 - main_window->u_y1 ) / 2 );

      length = main_window->u_x2 / 2;
      x1 = midx - length / 2;
      x2 = midx + length / 2;
      height = main_window->u_y2 / 2;
      y1 = midy - height / 2;
      y2 = midy + height / 2;

      }

   gr_imsave( GR_PRIMARY, TRUE, x1, y1, x2, y2, &x_mem );
   buffer[ 0 ] = 0;
   sprintf( buffer, "DEBUG:  %s ", mes );
   c = ui_alarm( x1, y1, x2, y2, cl_mback, cl_mfore, cl_marker, TRUE,
      buffer, "Press Any Key to Continue", &deb_window );
   gr_imsave( GR_PRIMARY, FALSE, x1, y1, x2, y2, &x_mem );
   gr_imfree( x_mem );

#endif

   }
#endif

sfs_exit()
   {
   gr_cls( GR_PRIMARY );
   gr_deinit();
   }

sfs_wait( seconds )
   int seconds;
   {
   static time_t start, current;
   time( &start );
   current = start;
   while ( current < ( start + seconds ) )
      {
      time( &current );
      }
   }

tl_window( title )
   char *title;
   {

   midx = main_window->u_x1 + ( ( main_window->u_x2 - main_window->u_x1 ) / 2 );
   midy = main_window->u_y1 + ( ( main_window->u_y2 - main_window->u_y1 ) / 2 );

   if ( w1.window == NULL )
      {
      w1.window = ui_window( main_window->u_x1 + 1, midy + 1,
	 midx - 1, main_window->u_y2 - 1, TRUE, cl_marker, BLACK,
	 title, 1, 0, 0, 0, 1, 1, BUT_CLOSE );
      }
   else
      {
      ui_rewindow( w1.window, title );
      }
   }

tr_window( title )
   char *title;
   {

   midx = main_window->u_x1 + ( ( main_window->u_x2 - main_window->u_x1 ) / 2 );
   midy = main_window->u_y1 + ( ( main_window->u_y2 - main_window->u_y1 ) / 2 );

   if ( w2.window == NULL )
      {
      w2.window = ui_window( midx + 1, midy + 1,
	 main_window->u_x2 - 1, main_window->u_y2 - 1, TRUE, cl_marker, cl_mfore,
	 title, 1, 0, 0, 0, 1, 1, BUT_CLOSE );
      }
   else
      {
      ui_rewindow( w2.window, title );
      }
   }

bl_window( title )
   char *title;
   {

   midx = main_window->u_x1 + ( ( main_window->u_x2 - main_window->u_x1 ) / 2 );
   midy = main_window->u_y1 + ( ( main_window->u_y2 - main_window->u_y1 ) / 2 );

   if ( w3.window == NULL )
      {
      w3.window = ui_window( main_window->u_x1 + 1, main_window->u_y1 + 1,
	 midx - 1, midy - 1, TRUE, cl_marker, BLACK,
	 title, TRUE, BLACK, FALSE, BLACK, WHITE, GRID, BUT_CLOSE );
      }
   else
      {
      ui_rewindow( w3.window, title );
      }
   sfsm_clpe();
   }

br_window( title )
   char *title;
   {

   midx = main_window->u_x1 + ( ( main_window->u_x2 - main_window->u_x1 ) / 2 );
   midy = main_window->u_y1 + ( ( main_window->u_y2 - main_window->u_y1 ) / 2 );

   if ( w4.window == NULL )
      {
      w4.window = ui_window( midx + 1, main_window->u_y1 + 1,
	 main_window->u_x2 - 1, midy - 1, TRUE, cl_marker, cl_mfore,
	 title, 1, 0, 0, 0, 1, 1, BUT_CLOSE );
      }
   else
      {
      ui_rewindow( w4.window, title );
      }
   }

sfs_sigint()
   {
   if ( ( data = fopen( SFS_CHAINFN, "w" ) ) != 0 )
      {
      fprintf( data, "%s \n", SFS_EXITARG );
      fclose( data );
      }
   sfs_exit();
   exit( 1 );
   }

ui_poll()
   {
   ;
   }

bw_message( m )
   char *m;
   {
   ui_fbox( stat_x1, main_window->tbar_y1 + 1,
      main_window->tbar_x2, main_window->y2 - 1,
      main_window->t_bcolor, SOLID );

   ui_str( stat_x1, stat_y1, stat_x2, cl_marker, cl_mfore, m );

   }

matherr( xstruct )
   struct exception *xstruct;
   {
   sprintf( bw_ebuf, ERR_MATH, xstruct->name );
   bw_error( bw_ebuf );
   }


