/***************************************************************

	sfsx.c		Space Flight Simulator
			Orbital Simulation Module -- Main Segment

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

***************************************************************/

/***  Include Files */

#include "stdio.h"
#include "signal.h"
#include "time.h"
#include "math.h"
#include "sfsx.h"

#ifdef  __STDC__
#include "malloc.h"
#include "process.h"
#else
extern  char * malloc();
#define size_t   int
#define time_t   long
#endif

/***  Local Data */

int     fo_xcenter, fo_ycenter;

static  char *em_titles[ EM_TITLES ];
static  FILE *data;
static int x_zoom = FALSE;   /* boolean: use zoom display? */
static int d_zoom;           /* zoom window display */
static int d_upper;          /* upper window display */
static int d_lower;          /* lower window display */
struct uiwindow *e_wind;     /* window for elements */
static x_init = FALSE;       /* is video system initialized? */
static int *x_elements;      /* pointer to elements to toggle */
static struct sfs_display **x_displays;
static int n_displays = 0;   /* number of displays */
static char **t_array;       /* title array */
static char pr_file[ SFS_FILESIZE ];

extern int sfsx_sigint();
extern int *sfsx_curel();
extern int sfsx_poll();

/****************************************************************

	main()          Main Program Function

****************************************************************/

main( argc, argv )
   int argc;
   char **argv;
   {
   int x_resume;                /* resume execution? */

   /***    First some precautionary measures               */
   /***    Load "chain" so that the loader (sfs) will exit */
   /***    in the case of any unexpected exit              */

   if ( ( data = fopen( SFS_CHAINFN, "w" ) ) != NULL )
      {
      fprintf( data, "%s \n", SFSCH_EXIT );
      fclose( data );
      }

   /***    Now set a signal handler so that a SIGINT       */
   /***    (control-C under DOS, DEL under Unix) will      */
   /***    exit gracefully                                 */

   signal( SIGINT, sfsx_sigint );

   /***    Set up initial program file */

   if ( argc > 1 )
      {
      strcpy( pr_file, argv[ 1 ] );
      }
   else
      {
      strcpy( pr_file, SFS_DEFAULTPROG );
      }

   /***  Initialize Bywater user interface */

   ui_init();
 
   /* Return if initialization failed */

   if ( ui_grwind->initialized == FALSE )
      {
      fprintf( stderr, "Graphics system is not initialized.\n" );
      return -1;
      }

#ifdef  OLD_DEBUG
   fprintf( stderr, "ui is now initialized\n" );
#endif

   /***  Initialize colors */

   sfsx_colors();

   /* reset icon colors */

   if ( gr_colors > 2 )
      {
      ui_seticons( cl_mback, cl_marker );
      }

   /***  Initialize parameters */

   vpt_init();
   vpt_level = 1;
   sfsx_font( 0 );

   /***  Set up the screen */

   sfsx_screen();

   fo_xcenter = fo_window->x1 + (( fo_window->x2 - fo_window->x1 ) / 2 );
   fo_ycenter = fo_window->y1 + (( fo_window->y2 - fo_window->y1 ) / 2 );

#ifdef  OLD_DEBUG
   fprintf( stderr, "screen is now drawn\n" );
#endif

   /***  Initialize error handler */

   err_init();

#ifdef  OLD_DEBUG
   fprintf( stderr, "error handler is now initialized\n" );
#endif

   sfsx( pr_file );

   }

sfsx( program )
   char *program;
   {
   register int orbit;          /* cycle through orbits */
   register int c;              /* a multipurpose counter */
   int x_continue;              /* continue main loop */
   long x_target;               /* target time (met) for next display update */
   long x_next;                 /* next target time (to calculate rotation) */
   static struct sfs_orbit *sorbit_array[ ORBITS ];
				/* array of orbit structures */
   static int x_int;		/* a dummy for initial setting of element */
   
   /***  Set up dummy int pointer for elements */

   x_elements = &x_int;

   /***  Allocate memory for Escape menu */

   for ( c = 0; c < EM_TITLES; ++c )
      {
      if ( ( em_titles[ c ] = malloc( 64 ) ) == NULL)
         {
	 bw_error( MEMERR_ESC );
         }
      }

#ifdef  OLD_DEBUG
   fprintf( stderr, "escape menu is now initialized\n" );
#endif

   /***  Send initializing message on status/help line */

   sfsx_help( SFSX_INIT );

   /***  Set up default values in case they are not read 
	 by the program file. */

   for ( orbit = 0; orbit < ORBITS; ++orbit )
      {
      sorbit_array[ orbit ] = NULL;      /* set all orbits to NULL */
      }

   sfs_tfactor = 1;                     /* default time factor = 1 (real time) */
   sfs_inc = 120;                       /* default display update = 120
					   seconds (2 minutes) */
   sfs_insertion = 0;                   /* default insertion point = 0 seconds */

   /***  Interpret the program file */

   if ( sfs_pr( program, pr_name, sorbit_array, ORBITS ) == BW_ERROR )
      {
      sprintf( sfsx_tbuf, SFSX_PRERR, program );
      bw_error( sfsx_tbuf );
      sfs_exit( );
      exit( 1 );
      }   
#ifdef  OLD_DEBUG
   else
      {
      sprintf( sfsx_tbuf, "Successfully interpreted program file <%s>", program );
      bw_debug( sfsx_tbuf );
      }
#endif

   /***  Set up program screen */

   cp_prog( program );

   /***  Initialize displays */

   di_init( sorbit_array, ORBITS, &x_displays, &n_displays, &t_array );

   /***  Set initial elements to toggle */

   x_elements = sfsx_curel();

#ifdef  OLD_DEBUG
   for ( c = 0; c < n_displays; ++c )
      {
      sprintf( sfsx_tbuf, "Display title %d: %s", c, t_array[ c ] );
      bw_debug( sfsx_tbuf );
      }
#endif

   /***  find first visual display, and set zoom and upper windows to it */

   c = 0;
   x_continue = TRUE;
   while( ( c < n_displays ) && ( x_continue == TRUE ))
      {
      if ( x_displays[ c ]->type == SFSD_VISUAL )
	 {
	 d_zoom = c;
	 d_upper = c;
	 x_continue = FALSE;
	 }
      ++c;
      }

   /***  find first ground track display, and set lower window to it */

   c = 0;
   x_continue = TRUE;
   while( ( c < n_displays ) && ( x_continue == TRUE ))
      {
      if ( x_displays[ c ]->type == SFSD_GROUND )
	 {
	 d_lower = c;
	 x_continue = FALSE;
	 }
      ++c;
      }

   /***  Read spd files for all active orbits */

   for ( c = 0; c < ORBITS; ++c )
      {
      if ( sorbit_array[ c ] != NULL )
	 {

	 /* first read orb file for this orbit */

	 spj_readspd( sorbit_array[ c ]->orbfile,
	    &(sorbit_array[ c ]->o_start), &(sorbit_array[ c ]->o_end) );

	 /* read grid file for this orbit */

	 spj_readspd( sorbit_array[ c ]->gridfile,
	    &(sorbit_array[ c ]->g_start), &(sorbit_array[ c ]->g_end) );

	 /* read surface data file for this orbit */

	 spj_readspd( sorbit_array[ c ]->surfile,
	    &(sorbit_array[ c ]->s_start), &(sorbit_array[ c ]->s_end) );

	 /* read point data file for this orbit */

#ifdef  USEPOINTS
	 spj_readspd( sorbit_array[ c ]->pdfile,
	    &(sorbit_array[ c ]->p_start), &(sorbit_array[ c ]->p_end) );
#endif

	 }
      }

   /***  Send help message */

   sfsx_help( SFSX_HELP );

   /***  Initialize ground track and forward track for each orbit */

   for ( orbit = 0; orbit < ORBITS; ++orbit )
      {
      if ( sorbit_array[ orbit ] != NULL )
         {

	 /***  Initialize ground track system */

	 gt_init( sorbit_array, orbit  );

	 /***  Initialize the forward orbital track */

	 ft_init( sorbit_array, orbit, sfsx_poll );

         }
      }

   /***  Poll to allow early exit with escape key */

   sfsx_poll();

   x_target = sfsx_getmet();            /* initialize mission elapsed time */

   /***  Main Program Loop */

   x_continue = TRUE;
   while( x_continue == TRUE )
      {

      /***  1. See if a new focus and/or orbit needs to be set up */
      /***     This is a stub presently, but will eventually allow */
      /***     "true" space flight, i.e., the ability to switch to */
      /***     a transfer orbit, and eventually to another orbital */
      /***     focus. */

      sfsx_poll();

      /***  2.  Perform necessary calculations */
      /***  2a. first calculate the current orbital positions
		(subsatellite points) and update ground tracks */

#ifdef  OLD_DEBUG
      sprintf( sfsx_tbuf, "[pr:] main(): orbit %d, focus %s, mass %le",
	 x_current, sorbit_array[ x_current ]->aorbit->focus->name,
	 sorbit_array[ x_current ]->aorbit->focus->mass );
      bw_debug( sfsx_tbuf );
#endif

      x_next = x_target + ( sfs_inc * sfs_tfactor );
      for ( orbit = 0; orbit < ORBITS; ++orbit )
         {
         if ( sorbit_array[ orbit ] != NULL )
            {
	    or_ssp( sorbit_array[ orbit ]->aorbit,
	       x_target, &(sorbit_array[ orbit ]->ssp_lat),
	       &(sorbit_array[ orbit ]->ssp_lon),
	       &(sorbit_array[ orbit ]->ssp_r),
	       &(sorbit_array[ orbit ]->ssp_n) );
#ifdef  OLD_DEBUG
	    sprintf( sfsx_tbuf, "main(): ssp, time %ld lat %.2lf lon %.2lf no %ld",
               x_target, x_lat, x_lon, x_n );
	    bw_debug( sfsx_tbuf );
#endif
	    sprintf( sfsx_tbuf, "Updating ground track, orbit %d", orbit + 1 );
	    bw_message( sfsx_tbuf );
            gt_update( x_target, sorbit_array, ORBITS );       /* Update Ground Track */
#ifdef	OLD_DEBUG
            bw_debug( "return from gt_update()" );
#endif
	    sfsx_poll();

            /***  2b. calculate the next orbital position to allow
      	          the rotation to be set appropriately */

	    or_ssp( sorbit_array[ orbit ]->aorbit,
	       x_next, &(sorbit_array[ orbit ]->nssp_lat),
	       &(sorbit_array[ orbit ]->nssp_lon),
	       &(sorbit_array[ orbit ]->nssp_r),
	       &(sorbit_array[ orbit ]->nssp_n) );
#ifdef  OLD_DEBUG
	    sprintf( sfsx_tbuf, "main(): next ssp, lat %.2lf lon %.2lf",
               nssp_lat, nssp_lon );
	    bw_debug( sfsx_tbuf );
#endif

	    sfsx_poll();

            /***  2c. calculate the rotation based on the current position
      	          and the next position */

	    sorbit_array[ orbit ]->ssp_rot =
	       vi_rotate( sorbit_array[ orbit ]->ssp_lon,
	       sorbit_array[ orbit ]->ssp_lat,
	       sorbit_array[ orbit ]->nssp_lon,
	       sorbit_array[ orbit ]->nssp_lat );

#ifdef  OLD_DEBUG
	    sprintf( sfsx_tbuf, "main(): rotation = %.2lf deg",
               x_rot );
	    bw_debug( sfsx_tbuf );
#endif

	    sfsx_poll();
	    }
	 }

      /***  2d. calculate perspective points */

      pe_calc( sorbit_array, ORBITS, sfsx_poll );
      sfsx_poll();

      /***      calculations for each orbit */

      for ( orbit = 0; orbit < ORBITS; ++orbit )
	 {
	 if ( sorbit_array[ orbit ] != NULL )
	    {

	    /***  2d+. calculate forward track */

	    ft_calc( x_target, sorbit_array, orbit, sfsx_poll );

	    /***  2e. calculate positions for the surface data */

	    sprintf( sfsx_tbuf, CALCSD, orbit + 1 );
	    bw_message( sfsx_tbuf );
	    spj_calc( &( sorbit_array[ orbit ]->s_start ), 
               &( sorbit_array[ orbit ]->s_end ), 
	       sorbit_array[ orbit ]->ssp_lat,
	       sorbit_array[ orbit ]->ssp_lon,
	       (double) sorbit_array[ orbit ]->ssp_r,
	       sorbit_array[ orbit ]->aorbit->focus->radius,
	       sorbit_array[ orbit ]->ssp_rot, SPJ_ALLSIDES, sfsx_poll );
	    sfsx_poll();

	    /***  2f. calculate positions for the point data */

#ifdef  USEPOINTS
	    sprintf( sfsx_tbuf, "Calculating point data points for orbit %d",
	       orbit + 1 );
	    bw_message( sfsx_tbuf );
	    spj_calc( &( sorbit_array[ orbit ]->p_start ), 
               &( sorbit_array[ orbit ]->p_end ), 
	       sorbit_array[ orbit ]->ssp_lat,
	       sorbit_array[ orbit ]->ssp_lon,
	       (double) sorbit_array[ orbit ]->ssp_r,
	       sorbit_array[ orbit ]->aorbit->focus->radius,
	       sorbit_array[ orbit ]->ssp_rot, SPJ_ALLSIDES, sfsx_poll );
	    sfsx_poll();
#endif

	    /***  2g. calculate positions for the grid data */

	    sprintf( sfsx_tbuf, CALCGR, orbit + 1 );
	    bw_message( sfsx_tbuf );
	    spj_calc( &( sorbit_array[ orbit ]->g_start ),
               &( sorbit_array[ orbit ]->g_end ),
               sorbit_array[ orbit ]->ssp_lat, 
	       sorbit_array[ orbit ]->ssp_lon,
	       (double) sorbit_array[ orbit ]->ssp_r,
	       sorbit_array[ orbit ]->aorbit->focus->radius,
	       sorbit_array[ orbit ]->ssp_rot, SPJ_ALLSIDES, sfsx_poll );
	    sfsx_poll();

	    /***  2h. calculate positions for the orb. Note that the
		      orb has to be shown from viewer latitude 0.0,
		      longitude 0.0 */

	    sprintf( sfsx_tbuf, CALCORB, orbit + 1 );
	    bw_message( sfsx_tbuf );
	    spj_calc( &( sorbit_array[ orbit ]->o_start ),
               &( sorbit_array[ orbit ]->o_end ), 
               0.0, 0.0, (double) sorbit_array[ orbit ]->ssp_r,
	       sorbit_array[ orbit ]->aorbit->focus->radius,
	       sorbit_array[ orbit ]->ssp_rot, SPJ_ALLSIDES, sfsx_poll );
	    sfsx_poll();

	    }
	 }

      /***  3. Wait until time for next display */

      bw_message( SFSX_WAIT );
      sfsx_poll();
      while( sfsx_getmet() < x_target )
	 {
#ifdef  OLD_DEBUG
	 sprintf( sfsx_tbuf, "wait: %ld / %ld", sfsx_getmet(), x_target );
	 bw_message( sfsx_tbuf );
#endif
	 sfsx_poll();
	 }

      /***  4. Update the simulation display */

#ifdef  OLD_DEBUG
      sprintf( sfsx_tbuf, "call update() display %d, n %d, orbit_n %ld",
	 x_display, ORBITS, x_n );
      bw_debug( sfsx_tbuf );
      sprintf( sfsx_tbuf, "call update() lat %.1lf lon %.1lf, rad %.1lf dis %.1lf",
	 x_lat, x_lon,
	 sorbit_array[ x_current ]->aorbit->focus->radius,
	 (double) x_r );
      bw_debug( sfsx_tbuf );
      sprintf( sfsx_tbuf, "call update() focal radius %.1lf",
	 sorbit_array[ 0 ]->aorbit->focus->radius );
      bw_debug( sfsx_tbuf );
      sfsx_test( x_display, ORBITS, x_lat, x_lon,
	 sorbit_array[ x_current ]->aorbit->focus->radius,
	 (double) x_r, x_n, sorbit_array );
#endif

      sfsx_update( ORBITS,
         sorbit_array, x_displays, n_displays, x_target,
         x_zoom, d_zoom, d_upper, d_lower );

      /***  display is initialized */

      x_init = TRUE;

      /***  5. Set the next target time */

      x_target += ( sfs_inc * sfs_tfactor );

#ifdef  BLOCKEDOUT
      bw_message( "end of main loop cycle: ESCAPE to exit" );
      if ( kb_rx() == 0x1b )
	 {
	 sfs_exit();
	 }
#endif

      }

   }

/****************************************************************

	sfs_exit()             Exit from Simulation Module

****************************************************************/

sfs_exit()
   {
/*   gr_cls( GR_PRIMARY ); */
   if ( ( data = fopen( SFS_CHAINFN, "w" ) ) != NULL )
      {
      fprintf( data, "%s %s %s\n", SFSCH_MODEL, pr_file, SFS_CONTARG );
      fclose( data );
      }
   else
      {
      fprintf( stderr, ERR_CHAIN );
      }
   gr_deinit();
   exit( 0 );
   }

/****************************************************************

	sfsx_colors()           Initialize Colors for Simulation

****************************************************************/

sfsx_colors()
   {
   register int c;
   int n;

   if ( gr_colors > 8 )
      {
      cl_mback = WHITE;
      cl_mfore = BLACK;
      cl_marker = DARK_CYAN;
      cl_grid = DARK_BLUE;
      cl_surface = DARK_GREEN;
      n = LIGHT_RED;
      for ( c = 0; c < OR_COLORS; ++c )
         {
         cl_orbits[ c ] = n;
         ++n;
         }
      }
   else if ( gr_colors > 2 )
      {
      cl_mback = WHITE;
      cl_mfore = BLACK;
      cl_marker = BLACK;
      cl_grid = WHITE;
      cl_surface = WHITE;
      for ( c = 0; c < OR_COLORS; ++c )
         {
         cl_orbits[ c ] = WHITE;
         }
      }
   else
      {
      cl_mback = WHITE;
      cl_mfore = BLACK;
      cl_marker = WHITE;
      cl_grid = WHITE;
      cl_surface = WHITE;
      for ( c = 0; c < OR_COLORS; ++c )
         {
         cl_orbits[ c ] = WHITE;
         }
      }

   }

/****************************************************************

	sfsx_poll()

****************************************************************/

sfsx_poll()
   {
   register int c;
   static int x, y, b;

   /* Check for keyboard input */

   while ( kb_rxstat() == TRUE )
      {
      c = kb_rx();
      if ( c == 0x1b )
         {
	 sfsx_esc();
         x_elements = sfsx_curel();
         }
      else
         {
	 el_toggle( c, x_elements );
	 if ( x_init == TRUE )
	    {
	    el_show( GR_PRIMARY, e_wind, *x_elements );
	    }
         }
      }

   /* Check for mouse input */

   if ( gr_ismouse == TRUE )
      {
      if ( gr_mouse( SAMPLE, &x, &y, &b ) == TRUE )
	 {

	 gr_mouse( WAIT, &x, &y, &b );           /* click down */
	 gr_mouse( WAIT, &x, &y, &b );           /* click up */

#ifdef  OLD_DEBUG
	 sprintf( sfsx_tbuf, "Mouse hit: %d %d", x, y );
	 bw_debug( sfsx_tbuf );
#endif

	 /* Is the release within bounds ? */

	 if ( uil_bounds( x, y, main_window->tbar_x1, main_window->tbar_y1,
	    main_window->tbar_x2, main_window->tbar_y2) == TRUE )
	    {
	    sfsx_esc();
	    }

	 }
      }
   }

/****************************************************************

	sfsx_esc()

****************************************************************/

sfsx_esc()
   {
   int c;
   static struct menu_box esc_box;
   int y, test;
   static int x_x1 = 0, x_y1 = 0, x_x2 = 0, x_y2 = 0, item;
   static int x_ebuf;
   
   ui_push();                   /* save current ui screen */
   ui_setscreen( GR_PRIMARY );  /* set primary screen */
   if ( x_x1 == 0 )
      {
      x_x1 = (int) fo_xcenter - ( ui_grwind->xmax / 4 );
      x_y1 = (int) fo_ycenter - ( ui_grwind->fysize * 7 );
      x_x2 = (int) fo_xcenter + ( ui_grwind->xmax / 4 );
      x_y2 = (int) fo_ycenter + ( ui_grwind->fysize * 5 );
      }

   strcpy( em_titles[ 0 ], ZOOMTITLE );
   strcpy( em_titles[ 1 ], UPPERTITLE );
   strcpy( em_titles[ 2 ], LOWERTITLE );
   strcpy( em_titles[ 3 ], SYSTITLE );
   strcpy( em_titles[ 4 ], ABORTTITLE );
   strcpy( em_titles[ 5 ], EXITTITLE );

   gr_imsave( GR_PRIMARY, TRUE, x_x1, x_y1, x_x2, x_y2, &x_ebuf );

   sfsx_help( SFSX_MM );

   esc_box.is_drawn = FALSE;
   item = ui_list( MENU_SLIDERS, x_x1, x_y1, x_x2, x_y2,
      ESCTITLE, EM_TITLES,
      em_titles, cl_mfore, cl_mback, cl_marker,
      &esc_box );

   gr_imsave( GR_PRIMARY, FALSE, x_x1, x_y1, x_x2, x_y2, &x_ebuf );
   gr_imfree( x_ebuf );
   sfsx_help( SFSX_HELP );

   switch( item )
      {
      case 0:
	 d_zoom = sdi( t_array, n_displays, ZOOMTITLE );
	 x_zoom = TRUE;
	 break;
      case 1:
	 d_upper = sdi( t_array, n_displays, UPPERTITLE );
	 x_zoom = FALSE;
         break;
      case 2:
	 d_lower = sdi( t_array, n_displays, LOWERTITLE );
	 x_zoom = FALSE;
         break;
      case 3:
         ssp();
	 break;
      case 4:
	 if ( ( data = fopen( SFS_CHAINFN, "w" ) ) != NULL )
	    {
	    fprintf( data, "%s\n", SFS_EXITARG );
	    fclose( data );
	    }
	 else
	    {
	    fprintf( stderr, ERR_CHAIN );
	    }
	 gr_deinit();
	 exit( 0 );
	 break;
#ifdef  ABORT_ON_ESCAPE
      case TK_EXIT:
#endif
      case 5:
	 sfs_exit();
	 exit(0);
      default:
         break;
      }

   ui_pop();                            /* restore ui screen */

   }

/****************************************************************

	sfsx_getmet()           Get Mission Elapsed Time

****************************************************************/

long
sfsx_getmet()
   {
   static int x_initialized = FALSE;
   static time_t x_now;

   if ( x_initialized == FALSE )
      {
      x_initialized = TRUE;
      x_now = time( NULL );
      sfsx_epoch = x_now;
#ifdef  OLD_DEBUG
      sprintf( sfsx_tbuf, "sfs_getmet(): epoch = %ld", sfsx_epoch );
      bw_debug( sfsx_tbuf );
#endif
      return (long) sfs_insertion;
      }

   x_now = time( NULL );
   return (long) ( ( ( x_now + sfs_insertion ) - sfsx_epoch ) * sfs_tfactor );
   }

#ifdef  OLD_DEBUG
sfsx_test( display, n, lat, lon, radius, distance, orbit_n, sorbit_array )
   int display;
   int n;
   double lat, lon;
   double radius, distance;
   long int orbit_n;
   struct sfs_orbit **sorbit_array;
   {

   sprintf( sfsx_tbuf, "sfsx_test() display %d, n %d, orbit_n %ld",
      display, n, orbit_n );
   bw_debug( sfsx_tbuf );
   sprintf( sfsx_tbuf, "sfsx_test() lat %.1lf lon %.1lf, rad %.1lf dis %.1lf",
      lat, lon,
      radius, distance );
   bw_debug( sfsx_tbuf );
   sprintf( sfsx_tbuf, "sfsx_test() focal radius %.1lf",
      sorbit_array[ 0 ]->aorbit->focus->radius );
   bw_debug( sfsx_tbuf );
   }
#endif

/****************************************************************

	sfsx_update()          Update Simulation Display

****************************************************************/

sfsx_update( n_orbits,
   sorbit_array, x_displays, n_displays, newtime, 
   zoom, d_zoom, d_upper, d_lower )
   int n_orbits;
   struct sfs_orbit **sorbit_array;
   struct sfs_display **x_displays;
   int n_displays;
   long newtime;
   int zoom, d_zoom, d_upper, d_lower;
   {
   static int redraw = TRUE;            /* redraw Ground Track? */
   static int old_zoom = 0xff;
   static int old_zd = 0xff;
   static int old_ud = 0xff;
   static int old_ld = 0xff;

#ifdef  OLD_DEBUG
   sprintf( sfsx_tbuf, "sfsx_update() display %d, n %d, orbit_n %ld",
      display, n, orbit_n );
   bw_debug( sfsx_tbuf );
   sprintf( sfsx_tbuf, "sfsx_update() lat %.1lf lon %.1lf, rad %.1lf dis %.1lf",
      lat, lon,
      radius, distance );
   bw_debug( sfsx_tbuf );
   sprintf( sfsx_tbuf, "sfsx_update() focal radius %.1lf",
      sorbit_array[ 0 ]->aorbit->focus->radius );
   bw_debug( sfsx_tbuf );
   return TRUE;
#endif

#ifdef  OLD_DEBUG
   if ( distance == 0.0 )
      {
      sprintf( sfsx_tbuf, "[pr: sfsx_update() received distance = %.2lf",
	 distance );
      bw_error( sfsx_tbuf );
      return;
      }
   if ( radius == 0.0 )
      {
      sprintf( sfsx_tbuf, "[pr: sfsx_update() received radius = %.2lf",
	 radius );
      bw_error( sfsx_tbuf );
      return;
      }
#endif

#ifdef  OLD_DEBUG
   bw_message( "DEBUG: update control panel..." );
#endif

   /***  Update timing panel */

   cp_sti();

   /***  Call the appropriate display routine */

   if ( zoom == TRUE )
      {
      if ( ( zoom != old_zoom ) || ( d_zoom != old_zd )) /* Redraw the screen(s)? */
         {
	 cp_lwin( &z_window, "  " );
         old_zoom = zoom;
	 old_zd = d_zoom;
	 e_wind = z_window;
	 redraw = TRUE;
         }
      sfsx_draw( z_window, sorbit_array, x_displays, d_zoom, n_orbits,
	 newtime, redraw );
      redraw = FALSE;
      }
   else
      {
      if ( zoom != old_zoom ) /* Redraw the screen(s)? */
	 {
	 ui_fbox( fo_window->u_x1, fo_window->u_y1,
	    fo_window->u_x2, fo_window->u_y2, BLACK, SOLID );
	 cp_twin( &t_window, " " );
	 old_ud = d_upper;
	 cp_bwin( &b_window, " " );
	 old_ld = d_lower;
	 old_zoom = zoom;
	 e_wind = t_window;
	 redraw = TRUE;
	 }
      else if ( d_upper != old_ud )
	 {
	 cp_twin( &t_window, " " );
	 old_ud = d_upper;
	 redraw = TRUE;
	 }
      else if ( d_lower != old_ld )
	 {
	 cp_bwin( &b_window, " " );
	 old_ld = d_lower;
	 redraw = TRUE;
	 }
      sfsx_draw( t_window, sorbit_array, x_displays, d_upper, n_orbits,
	 newtime, redraw );
      sfsx_draw( b_window, sorbit_array, x_displays, d_lower, n_orbits,
	 newtime, redraw );
      redraw = FALSE;
      }
   }

/****************************************************************

	sfsx_draw()           draw display in a single window

****************************************************************/

sfsx_draw( uiwind, sorbit_array, x_displays, display, n_orbits, newtime, redraw )
   struct uiwindow *uiwind;
   struct sfs_orbit **sorbit_array;
   struct sfs_display **x_displays;
   int display, n_orbits;
   long newtime;
   int redraw;
   {
   int orbit;

   orbit = x_displays[ display ]->orbit;

   switch( x_displays[ display ]->type )
      {
      case SFSD_VISUAL:
	 vi_draw( uiwind, sorbit_array[ orbit ]->aorbit->focus,
	    x_displays[ display ]->elements,
	    &(sorbit_array[ orbit ]->g_start),
	    &(sorbit_array[ orbit ]->g_end),
	    &(sorbit_array[ orbit ]->s_start),
	    &(sorbit_array[ orbit ]->s_end),
	    &(sorbit_array[ orbit ]->o_start),
	    &(sorbit_array[ orbit ]->o_end),
	    sorbit_array[ orbit ]->aorbit->focus->radius,
	    (double) sorbit_array[ orbit ]->ssp_r,
	    sorbit_array, orbit );
	 break;
      case SFSD_GROUND:
#ifdef  OLD_DEBUG
	 sprintf( sfsx_tbuf, "sfsx_draw() received redraw %d", redraw );
	 bw_debug( sfsx_tbuf );
#endif
	 gt_draw( x_displays[ display ], uiwind,
	    sorbit_array[ orbit ]->aorbit->focus,
	    x_displays[ display ]->elements, redraw,
	    &(sorbit_array[ orbit ]->s_start),
	    &(sorbit_array[ orbit ]->s_end),
	    sorbit_array, n_orbits );
	 break;
      case SFSD_PERSP:
	 pe_draw( uiwind, sorbit_array[ orbit ]->aorbit->focus,
	    sorbit_array, n_orbits,
	    &(sorbit_array[ orbit ]->o_start),
	    &(sorbit_array[ orbit ]->o_end), newtime, 
            x_displays[ display ]->elements );
	 break;
      default:
#ifdef  DEBUG
	 sprintf( sfsx_tbuf, "sfsx_draw() received display %d", display );
	 bw_error( sfsx_tbuf );
#endif
	 return BW_ERROR;
	 break;
      }
   }

/****************************************************************

	sfsx_sigint()           Interrupt signal handler

****************************************************************/

sfsx_sigint()
   {
   sfs_exit();
   exit( 0 );
   }

/****************************************************************

	sfsx_curel()           Return current elements to poll

****************************************************************/

int *
sfsx_curel()
   {
   if ( x_zoom == TRUE )
      {
      return &( x_displays[ d_zoom ]->elements );
      }
   else
      {
      return &( x_displays[ d_upper ]->elements );
      }
   }

/****************************************************************

	ui_poll()               Null poll for ui compatibility

****************************************************************/

ui_poll()
   {
   ;
   }

