/*
   Module:  main.c
   Date:    3/9/92
   Version: 1.0b
   Author:  Dave Lutz
   Email:   lutz@psych.rochester.edu
   Copyright: 1992 University of Rochester, Psychology Dept.

   Disclaimer:  This software is distributed free of charge.  As such, it
                comes with ABSOLUTELY NO WARRANTY.  The user of the software
                assumes ALL RISKS associated with its use.

                Your rights to modify and/or distribute this software are
                outlined in the file ADI7221.DOC.

   Purpose: This module provides the main processing routine for the generic
            ADI driver.  This handles program flow, and determines which of
            the device specific procedures to call based on the ADI code
            found in the source file.

            To effectively use this module, it should be compiled and linked
            with a set of obj files that provide the following functions:

                int dev_beginplot(PLTFILE *outfile)
                int dev_endplot(PLTFILE *outfile)
                int dev_move(unsigned xcoord,ycoord; PLTFILE *outfile)
                int dev_draw(unsigned xcoord,ycoord; PLTFILE *outfile)
                int dev_newpen(unsigned pennum; PLTFILE *outfile)
                   Note: dev_newpen should initiate a hardware autoselect
                int dev_setspeed(unsigned speed; PLTFILE *outfile)
                int dev_linetype(unsigned linetype; PLTFILE *outfile)
                int dev_penchange(PLTFILE *outfile)
                   Note: The main program calls dev_penchange and then pauses
                         for a manual pen change.  Since this version is not
                         a direct feed to the device, dev_penchange will
                         never be called.  However, future versions may use
                         this function.
                int dev_abortplot(PLTFILE *outfile)

            Each of these functions must produce a char array of codes
            and output those codes via the putarr function.  The putarr 
            function and the PLTFILE data type are part of the inout module.

            Each function should return an int value indicating it's success
            or failure.  Values returned should be as follows:

                TRUE   - args (if any) converted successfully, no error was
                         returned by putarr()
                BADIO  - putarr returned with an error
                BADFMT - could not convert arguments to device format

            The #defines for the codes listed above can be found in
            retcodes.h.

            The device dependant code should also include a #define for
            DEV_WELCOME.  This should be a set of arguments for printf
            that will be printed at program startup.

   Functions Provided:

        main

   Functions Required:

        (the following functions are part of the generic package)
        openin
        openout
        closein
        closeout
        getstr
        parse
        terminate

        (the following functions are device dependant)
        dev_beginplot
        dev_endplot
        dev_move
        dev_draw
        dev_newpen      (device autoselect)
        dev_newspeed
        dev_linetype
        (dev_penchange is not used by this version)
        dev_abortplot
*/

#include <stdio.h>
#include <stdlib.h>
#include "adicodes.h"
#include "retcodes.h"
#include "inout.h"
#include "parse.h"
#include "terminat.h"
#include "hp7221.h"

#define MAXLINE 80  /* maximum line length in input file */

main(argc,argv)
   int argc;
   char *argv[];
{
   char line[MAXLINE+1];
   int done, cnt, cmd;
   unsigned arg1, arg2;
   PLTFILE *infile = (PLTFILE *)NULL, *outfile = (PLTFILE *)NULL;


   /* print welcome message */
   printf (DEV_WELCOME);

   if (argc != 3) {
       fprintf (stderr, "Usage: %s infile outfile\n",argv[0]);
       exit (BADARGS);
   }

   /* From this point on, program termination should be accomplished via
      terminate()
   */

   switch (openin(argv[1], &infile)) {
      case TRUE:
         break;
      case NOBUFF:
         terminate (BADIO, &infile, &outfile, "can't allocate input buffer");
         break;
      case BADOPEN:
         terminate (BADIO, &infile, &outfile,
                    "can't open input file for reading");
         break;
   }
   switch (openout(argv[2], &outfile)) {
      case TRUE:
         break;
      case NOBUFF:
         terminate (BADIO, &infile, &outfile, "can't allocate output buffer");
         break;
      case BADOPEN:
         terminate (BADIO, &infile, &outfile,
                    "can't open output file for writing");
         break;
   }

   if (getstr(infile,MAXLINE,line) != 4) {
      terminate (NOTADI, &infile, &outfile, "input file not a valid ADI file");
   } else if (line[0]-'0' != BEGIN_PLOT) {
      terminate (NOTADI, &infile, &outfile, "input file not a valid ADI file");
   } else if (line[2]-'0' != ADIVERSION) {
      fprintf (stderr, "Warning: input file not a version 1 ADI file.\n");
   }
   if (!dev_beginplot(outfile))
      terminate (BADIO, &infile, &outfile, "I/O error on output");

   done = FALSE;
   while  (!done && ((cnt = getstr(infile, MAXLINE, line)) > 0)) {
      parse(line,&cmd,&arg1,&arg2);
      switch(cmd) {
         case BEGIN_PLOT:
            terminate (BADCMD, &infile, &outfile,
                       "Multiple BEGIN_PLOT commands found");
            break;
         case END_PLOT:
            if (!dev_endplot(outfile))
               terminate (BADIO, &infile, &outfile, "I/O error on output");
            else
               done=TRUE;
            break;
         case MOVE:
            switch (dev_move(arg1,arg2,outfile)) {
               case TRUE:
                  break;
               case BADIO:
                  terminate (BADIO, &infile, &outfile, "I/O error on output");
                  break;
               case BADFMT:
                  terminate (BADCMD, &infile, &outfile, "invalid MOVE command");
                  break;
            }
            break;
         case DRAW:
            switch (dev_draw(arg1,arg2,outfile)) {
               case TRUE:
                  break;
               case BADIO:
                  terminate (BADIO, &infile, &outfile, "I/O error on output");
                  break;
               case BADFMT:
                  terminate (BADCMD, &infile, &outfile, "invalid DRAW command");
                  break;
            }
            break;
         case NEW_PEN:
            switch (dev_newpen(arg1,outfile)) {
               case TRUE:
                  break;
               case BADIO:
                  terminate (BADIO, &infile, &outfile, "I/O error on output");
                  break;
               case BADFMT:
                  terminate (BADCMD, &infile, &outfile,
                             "invalid NEW_PEN command");
                  break;
            }
            break;
         case SET_SPEED:
            switch (dev_setspeed(arg1,outfile)) {
               case TRUE:
                  break;
               case BADIO:
                  terminate (BADIO, &infile, &outfile, "I/O error on output");
                  break;
               case BADFMT:
                  terminate (BADCMD, &infile, &outfile,
                             "invalid SET_SPEED command");
                  break;
            }
            break;
         case LINE_TYPE:
            switch (dev_linetype(arg1,outfile)) {
               case TRUE:
                  break;
               case BADIO:
                  terminate (BADIO, &infile, &outfile, "I/O error on output");
                  break;
               case BADFMT:
                  terminate (BADCMD, &infile, &outfile,
                             "invalid LINE_TYPE command");
                  break;
            }
            break;
         case PEN_CHANGE:
#ifdef WARN_PENCHANGE
            fprintf (stderr, "Warning: PEN_CHANGE encountered but not supported.\n");
#endif
            break;
         case ABORT_PLOT:
            if (dev_abortplot(outfile))
               terminate (ABORTED, &infile, &outfile,
                          "input file contains ABORT_PLOT command");
            else
               terminate (BADIO, &infile, &outfile, "I/O error on output");
            break;
         default:
            terminate (BADCMD, &infile, &outfile,
                       "unknown command encountered");
            break;
      }
   }
   if (cnt < 0)
      terminate (BADIO, &infile, &outfile, "I/O error on input");

   if (!done)
      terminate (NOEND, &infile, &outfile, "no END_PLOT encountered.\n");

   if (!closein(&infile))
      terminate (BADIO, &infile, &outfile, "can't close input file");

   if (!closeout(&outfile))
      terminate (BADIO, &infile, &outfile, "can't close output file");

   exit (ALLOK);
}
