/*
   Module:  comio.c
   Date:    3/9/92
   Version: 1.0b
   Author:  Dave Lutz
   Email:   lutz@psych.rochester.edu
   Copyright: 1992 University of Rochester, Psychology Dept.

   Disclaimer:  This software is distributed free of charge.  As such, it
                comes with ABSOLUTELY NO WARRANTY.  The user of the software
                assumes ALL RISKS associated with its use.

                Your rights to modify and/or distribute this software are
                outlined in the file SEND7221.DOC.

   Purpose: This module provides the functions needed to do serial io
            operations on an IBM compatible PC.  Functions provided are
            cominit, comin, comout, comstat, comlstat.

*/

#include "comio.h"

#ifndef NULL
#  define NULL 0
#endif

/* dos interrupt and function codes */
#define SERIAL_IO 0x14
#define INIT      0x00
#define SEND      0x01
#define RECV      0x02
#define STAT      0x03


/*
   Function: cominit
   Purpose:  set up a COMPORT for subsequent io operations

   Pre: cpp is a pointer to a COMPORT data structure.
        portdes is one of the port designators defined in comio.h
        config is the desired configuration for the com port, and was
        obtained by combining the configuration constants defined in
        comio.h with a bitwise or.
        timeout is the number of seconds to wait for the first character
        during an input operation.

   Post: The portdes, config, and timeout paramaters are transferred to the 
         cpp structure.
         A dos interrupt is called to open and configure the desired
         serial port.
         The bitmapped status of the port is returned.

         The status of the port can be determined by comparing it to the
         LCS_* and MCS_* values defined in comio.h.  For example:

                (MCS_DSRSTAT & retval) 
         
         will be nonzero if Modem Control Status indicates Data Set Ready.
*/

unsigned cominit (cpp, portdes, config, timeout)
   COMPORT *cpp;
   int portdes;
   unsigned config;
   time_t timeout;
{
   /* transfer the configuration options to the appropriate registers */
   cpp->inregs.x.dx = portdes;
   cpp->inregs.h.al = config;
   cpp->timeout     = timeout;

   /* set up for an init call */
   cpp->inregs.h.ah = INIT;

   /* perform the call, returning the contents of the ax register */
   return (int86(SERIAL_IO,&(cpp->inregs),&(cpp->outregs)));
}



/*
   Function: comin
   Purpose:  read a block of characters from the serial port.

   Pre: cpp is a pointer to a COMPORT structure.
        cpp has been prepared via a call to cominit.
        Nobody has been mucking around with the internals of the
        cpp structure...
        buff is a pointer to storage for the bytes read from the
        port.  There is enough storage for max characters.
        max is the maximum number of chars to read.
        term is a character which, when read, signals the end of
        the input stream.
        If there is no real term character, term = -1.

   Post:  Characters are read from the serial port and stored in buff
          until: max chars have been read, the term char is read, or an
          I/O error occurs.
          The function will wait the predefined (by cominit) number of 
          seconds for the first character to appear.
          After the first character appears, the remaining chars are
          expected to follow immediately.  If they don't, the function
          will exit.
          The number of characters actually read is returned.
          The serial port should be checked with comlstat to determine
          if an error has occurred.
*/
int comin (cpp, buff, max, term)
   COMPORT *cpp;
   char *buff;
   int max,
       term;
{
   time_t starttime;
   int readcnt = 0;
   unsigned testmask;
   testmask = (LCS_RDR | LCS_OVRN | LCS_PAR | LCS_FRM );

   /* bail out if they don't really want to read anything */
   if (max == 0)
      return (readcnt);

   /* wait cpp->timeout seconds for the first char to appear (or an error) */
   starttime = time ((time_t *)NULL);
   cpp->inregs.h.ah = STAT;
   while ((int86(SERIAL_IO,&(cpp->inregs),&(cpp->outregs)) & testmask) == 0)
      if (time ((time_t *)NULL) > starttime + cpp->timeout)
         return (readcnt);

   /* make sure the while loop terminated due to "Receive Data Ready" */
   if ((cpp->outregs.x.ax & LCS_RDR) != LCS_RDR)
      return (readcnt);

   /* read from the port up to max chars, an error, or a term char */
   cpp->inregs.h.ah = RECV;
   do {
      (void) int86(SERIAL_IO,&(cpp->inregs),&(cpp->outregs));
      *buff = cpp->outregs.h.al;
   } while ((cpp->outregs.h.ah == 0) && (++readcnt < max) && (*buff++ != term));
   return (readcnt);
}



/*
   Function: comout
   Purpose:  Send a block of characters through the serial port.

   Pre: cpp is a pointer to a COMPORT structure.
        cpp has been prepared via a call to cominit.
        buff is a pointer to the block of chars to be transmitted.
        buffcnt is the number of chars to send.

   Post: An attempt is made to send the block of chars through the
         serial port.
         The number of chars actually sent is returned.
         If all bytes were not sent, it should be possible to determine
         the reason by making a call to comlstat.
*/
int comout (cpp, buff, buffcnt)
   COMPORT *cpp;
   char *buff;
   int buffcnt;
{
   int sentcnt = 0;
   cpp->inregs.h.ah = SEND;
   while (sentcnt < buffcnt) {
      cpp->inregs.h.al = *buff++;
      if ((int86(SERIAL_IO,&(cpp->inregs), &(cpp->outregs)) & LCS_TIM) != 0)
         break;
      sentcnt++;
   }
   return (sentcnt);
}



/*
   Function: comstat
   Purpose:  Report the current status of the com port.

   Pre: cpp is a pointer to a COMPORT structure.
        cpp has been set up via a call to comopen.
        Nobody has been mucking around with the internals of the
        cpp structure...

   Post:  A DOS interrupt is called to check the status of the port
          specified in *cpp.
          The status of the port is returned as a bitmapped value.
          This value can be interpreted by comparing it to the LCS_*
          and MCS_* masks defined in comio.h
*/
unsigned comstat (cpp)
   COMPORT *cpp;
{
   cpp->inregs.h.ah=STAT;
   /* perform the call, returning the value of the AX register */
   return (int86(SERIAL_IO, &(cpp->inregs), &(cpp->outregs)));
}


/*
   Function: comlstat
   Purpose:  Report the status of the com port as it was after the last
             call to one of the other comio functions.

   Pre: cpp is a pointer to a COMPORT stucture.
        cpp has been set up via a call to comopen.
        Nobody has been mucking around with the internals of the
        cpp structure...

   Post: The last status of the com port is returned as a bitmapped
         value.
         This value can be interpreted by comparing it to the LCS_*
         and MCS_* masks defined in comio.h
*/
unsigned comlstat (cpp)
   COMPORT *cpp;
{
   return (cpp->outregs.x.ax);
}
