/****************************************************************************/
/* PROGRAM:  dpv (Drive Path Volume)                                        */
/* VERSION:  1.2                                                            */
/*    DATE:  December 8, 1990                                               */
/*  AUTHOR:  Sydney M. Willett                                              */
/* DESCRIPTION:  Checks if a formatted disk is inserted in a drive, if a    */
/*               drive or path is valid, or if a volume label matches.      */
/*               Returns an errorlevel upon exit to indicate the status.    */
/* INPUTS:  DPV [drive:][path] [/Vvolume]                                   */
/*              drive:    drive                                             */
/*              path:     path for search - preceding \ searches from root  */
/*              /Vvolume  volume label for search                           */
/* OUTPUTS:  Returns an errorlevel, each value indicating:                  */
/*              0 = disk in drive okay -or- subdirectory exists             */
/*                  -or- volume label matches                               */
/*              1 = volume label doesn't match                              */
/*              2 = invalid path                                            */
/*              4 = invalid drive                                           */
/*              5 = no diskette in drive                                    */
/*              6 = unformatted diskette in drive                           */
/*              7 = invalid parameter(s)                                    */
/****************************************************************************/


/*============================================================================
INCLUDES
----------------------------------------------------------------------------*/
#include <dir.h>
#include <dos.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>


/*============================================================================
EQUATES
----------------------------------------------------------------------------*/
#define BACKSLASH 92


/*============================================================================
DATA DECLARATIONS
----------------------------------------------------------------------------*/
typedef unsigned char BYTE;


/*============================================================================
FUNCTION DECLARATIONS
----------------------------------------------------------------------------*/
void volcat(char *vol,char *str);


/*============================================================================
MAIN
----------------------------------------------------------------------------*/
void main(int argc,char **argv)
{
BYTE drive;
char *buffer;
char chk_drive = 0;
char chk_path = 0;
char chk_vol = 0;
char path[81];
char volume[16];
int arg;
int argndx;
int errorlevel = 7; /*  0 = disk in drive okay            */
                    /*      -or- subdirectory exists      */
                    /*      -or- volume label matches     */
                    /*  1 = volume label doesn't match    */
                    /*  2 = invalid path                  */
                    /*  4 = invalid drive                 */
                    /*  5 = no diskette in drive          */
                    /*  6 = unformatted diskette in drive */
                    /*  7 = invalid parameter(s)          */
int i;
struct ffblk fblk;
struct SREGS sregs;
union REGS regs;

  /* ... Get current drive and directory, and setup strings */
  getcwd(path,80);
  i = strlen(path);
  if ( path[i-1] != BACKSLASH ) /* need trailing '\' */
  {
    path[i] = BACKSLASH;
    path[i+1] = NULL;
  }
  strncpy(volume,path,3);
  volume[3] = NULL;
  drive = path[0] - 'A';

  /* ... Parse command line arguments ... */
  for ( arg=1 ; arg<argc ; arg++ )
  {
    strupr(argv[arg]);
    if ( chk_vol ) /* next part of volume label, append to volume string */
    {
      volcat(&volume[3]," "); /* add space */
      volcat(&volume[3],argv[arg]);
    }
    else
    {
      argndx = 0;
      if ( argv[arg][1] == ':' ) /* drive specified */
      {
        argndx = 2;
        if ( argv[arg][0] != path[0] ) /* not current drive */
        {
          drive = argv[arg][0] - 'A';
          path[0] = argv[arg][0];
          volume[0] = argv[arg][0];
          chk_drive++;
        }
      }
      switch ( argv[arg][argndx] )
      {
        case NULL      : errorlevel = 0; /* drive only */
                         break;
        case '-'       :
        case '/'       : switch ( argv[arg][1] ) /* volume label */
                         {
                           case 'V' : volcat(&volume[3],&argv[arg][2]);
                                      chk_vol++;
                                      errorlevel = 0;
                                      break;
                           default  : break; /* invalid argument */
                         }
                         break;
        case BACKSLASH : path[2] = NULL; /* full path, shorten path string */
        default        : strcat(path,&argv[arg][argndx]); /* append arg */
                         if ( strlen(path) > 3 ) /* if not root directory, */
                           chk_path++;                       /* check path */
                         errorlevel = 0;
                         break;
      }
    }
  }

  /* ... Perform actions, if any ... */
  if ( errorlevel == 7 ) /* invalid or no arguments */
  {
    printf("DPV 1.2   Copyright 1990 by S.M. Willett \n\n");
    printf("Usage: DPV [drive:][path] [/Vvolume]\n");
    printf("           drive:     drive\n");
    printf("           path:      path (preceding \\ searches from root)\n");
    printf("           /Vvolume   volume label\n\n");
    printf("Example:  DPV a:\\work /Vwordstar\n");
    printf("  This searches drive A: for directory named \\WORK and volume label WORDSTAR\n\n");
  }
  else
  {
    if ( chk_drive ) /* check drive */
    {
      buffer = malloc(8192); /* set up dummy buffer area */
      regs.h.al = drive; /* set up regs for int 25h, absolute sector read */
      regs.x.cx = 1;
      regs.x.dx = 1;
      sregs.ds = FP_SEG(buffer);
      regs.x.bx = FP_OFF(buffer);
      int86x(0x25,&regs,&regs,&sregs); /* absolute sector read */
      free(buffer); /* release buffer memory */
      if ( regs.x.cflag ) /* error reading disk */
      {
        switch ( regs.x.ax ) /* evalute disk read */
        {
          case 0x020C :
          case 0x040C :
          case 0x100C : errorlevel = 6; /* unformatted disk */
                        break;
          case 0x8002 : errorlevel = 5; /* no diskette in drive */
                        break;
          default     : errorlevel = 4; /* invalid drive */
                        break;
        }
      }
    }
    if ( !errorlevel )
    {
      if ( chk_path ) /* search for subdirectory */
      {
        if ( findfirst(path,&fblk,FA_DIREC) != 0 ) /* if unsuccessful, */
          errorlevel |= 2;               /* subdirectory doesn't exist */
      }
      if ( chk_vol ) /* search for volume label */
      {
        for ( i=strlen(volume) ; i>10 ; i--) /* insert '.' if needed */
        {             
          volume[i+1] = volume[i];
          if ( i == 11 )
            volume[11] = '.';
        }
        if ( findfirst(volume,&fblk,FA_LABEL) != 0 ) /* if unsuccessful, */
          errorlevel |= 1;                      /* not same volume label */
      }
    }
  }

  /* ... Return errorlevel ... */
  exit(errorlevel);
}


/*============================================================================
VOLCAT
----------------------------------------------------------------------------*/
void volcat(char *vol,char *str)
{
int remaining;

  remaining = 11 - strlen(vol);
  if ( remaining > 0 )
    strncat(vol,str,remaining);
}



