/*
   CVBD.C - Convert to Base and Dump (version 2.00)

            A program that converts the contents of a file or series of files
            into a formatted sequence of values in hex, dec, octal.

   This program is classified as "FREE SOFTWARE".
   This means you can use it (modified or not) in any way you wish.
   However I would appreciate any reference to my name.

   Please e-mail any comments, suggestions or bugs you may have found.

   Created by Christos Evaggelou <christos.e@cytanet.com.cy>
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "cvbd.h"
#include "misc.h"

/* ======================================================================= */
/* MAIN                                                                    */
/* ======================================================================= */
int main(int argc,char **argv)
{
     OPTIONS options;
     FILE    *fp;
     int     i,j;

     ProcessCommandLine(argc,argv,&options);

     if (options.bEchoOptions)
     {
        printf("COMMAND-LINE OPTIONS:\n");
        printf("    cvBase:       %-15s iWidth:         %d\n"
               "    iColumns:     %-3d             bLeading0:      %-3s\n"
               "    bLowerHex:    %-3s             bCHex:          %-3s\n"
               "    bLeftJust:    %-3s             bUseAutoWidth:  %-3s\n"
               "    bShowFNames:  %-3s             bEchoOptions:   %-3s\n"
               "    bPause:       %-3s             bySeparator:    \"%c\" (ASCII %d)\n"
               ,ConvBaseStr(options.cvBase),options.iWidth,options.iColumns
               ,BOOL_STR(options.bLeading0),BOOL_STR(options.bLowerHex)
               ,BOOL_STR(options.bCHex),BOOL_STR(options.bLeftJust)
               ,BOOL_STR(options.bUseAutoWidth),BOOL_STR(options.bShowFNames)
               ,BOOL_STR(options.bEchoOptions),BOOL_STR(options.bPause)
               ,options.bySeparator,options.bySeparator);
     }
     i = 0;
     while (++i < argc)
         if (argv[i][0] != '/' && argv[i][0] != '-')
            if (!(fp = fopen(argv[i],"rb")))
                 printf("\nERROR: cannot open %s (skipping)\n",argv[i]);
            else
            {
                 if (options.bShowFNames)
                 {
                    printf("\nListing %s\n========",argv[i]);
                    for (j = 0; j < strlen(argv[i]); j++)
                        putchar('=');
                    putchar('\n');
                 }
                 if (options.bPause)
                 {
                    printf("Paused, press any key to continue, 'Q' to abort... ");
                    fflush(NULL);
                    if (toupper(getchar()) == 'Q')
                    {
                         printf("*** Aborted!\n");
                         exit(ABORTED);
                    }
                 }
                 PerformConversion(fp,options);
                 fclose(fp);
            }
     return(0);
}
/* ======================================================================= */
/* Handles command-line arguments, stores options in *options */
/* ======================================================================= */
void ProcessCommandLine(int argc,char **argv,OPTIONS *options)
{
     CmdLnEval clEval;             /* Command-line processing result */
     Boolean   bAnyFilesSpecified; /* YES if the user specified any files */
     int       i;

     /* Set default options */
     options->cvBase = defBase;
     options->iColumns = defColumns;
     options->iWidth = defWidth(defBase);
     options->bLeftJust = defLeftJust;
     options->bLeading0 = defLeading0;
     options->bCHex = defCHex;
     options->bLowerHex = defLowerHex;
     options->bySeparator = defSeparator;
     options->bUseAutoWidth = defUseAutoWidth;
     options->bEchoOptions = defEchoOptions;
     options->bShowFNames = defShowFNames;
     options->bPause = defPause;

     /* Initialise variables */
     clEval = ceOK;
     bAnyFilesSpecified = NO;
     /* Handles all command-line arguments */
     while (--argc && clEval == ceOK)
     {
           /* If this parameter is a flag (switch) */
           if (argv[argc][0] == '-' || argv[argc][0] == '/')
           {
              /* Handle each flag individually */
              switch(toupper(argv[argc][1]))
              {
                 case 'E': switch(argv[argc][2])
                           {
                              case '+': case '\0': options->bEchoOptions = YES; break;
                              case '-': options->bEchoOptions = NO; break;
                              default : clEval = ceECHO_OPTIONS_ERROR; break;
                           }
                           break;
                 case 'P': switch(argv[argc][2])
                           {
                              case '+': case '\0': options->bPause = YES; break;
                              case '-': options->bPause = NO; break;
                              default : clEval = cePAUSE_ERROR; break;
                           }
                           break;
                 case 'F': switch(argv[argc][2])
                           {
                              case '+': case '\0': options->bShowFNames = YES; break;
                              case '-': options->bShowFNames = NO; break;
                              default : clEval = ceSHOW_FNAMES_ERROR; break;
                           }
                           break;
                 case 'D': options->cvBase = cbDEC; break;
                 case 'O': options->cvBase = cbOCT; break;
                 case 'B': options->cvBase = cbBIN;
                           /* this should be replaced by the actual code  */
                           /* for binary conversion when implemented */
                           printf("I'm sorry, but binary conversion has not been implemented yet!\n");
                           exit(ABORTED);
                           break;
                 /* The [/H[[L|U],C]] flag requires complex handling */
                 case 'H': options->cvBase = cbHEX;
                           options->bLowerHex = NO;
                           options->bCHex = NO;
                           /* Check subflags */
                           for (i = 2; argv[argc][i] && clEval == ceOK; i++)
                              /* Handle each subflag individually */
                              switch (toupper(argv[argc][i]))
                              {
                                   case 'L':
                                       options->bLowerHex = YES; break;
                                   case 'C':
                                       options->bCHex = YES; break;
                                   case 'U':
                                       options->bLowerHex = NO; break;
                                   default: /* Unknown sub-flag */
                                       clEval = ceHEX_ERROR; break;
                              }
                           break;
                 case 'L': switch(argv[argc][2])
                           {
                              case '+': case '\0': options->bLeftJust = YES; break;
                              case '-': options->bLeftJust = NO; break;
                              default : clEval = ceLEFTJUST_ERROR; break;
                           }
                           break;
                 case '0': switch(argv[argc][2])
                           {
                              case '+': case '\0': options->bLeading0 = YES; break;
                              case '-': options->bLeading0 = NO; break;
                              default : clEval = ceLEADING0_ERROR; break;
                           }
                           break;
                           /* Handle width specification */
                 case 'W': if (toupper(argv[argc][2]) == 'A')
                              options->bUseAutoWidth = YES;
                           else
                           if (isdigit(argv[argc][2]))
                           {
                               options->bUseAutoWidth = NO;
                               if ((options->iWidth = atoi(&argv[argc][2])) <= 0)
                                   options->iWidth = 0;
                           }
                           else
                           if (argv[argc][2] == '\0')
                           {
                              options->bUseAutoWidth = NO;
                              options->iWidth = 0;
                           }
                           else
                               clEval = ceWIDTH_SPEC_ERROR;
                           break;
                           /* Number of values in every line */
                 case 'C': if ((options->iColumns = atoi(&argv[argc][2])) <= 0)
                              options->iColumns = 0;
                           break;
                           /* Handle separator specification */
                 case 'S': if (isdigit(argv[argc][2]))
                              options->bySeparator = atoi(&argv[argc][2]);
                           else
                           if (argv[argc][2] == ':')
                              options->bySeparator = argv[argc][3];
                           else
                           if (argv[argc][2] == '\0')
                              options->bySeparator = 0;
                           else
                               clEval = ceSEPAR_SPEC_ERROR;
                           break;
                           /* Help request */
                 case '?': case '!': clEval = ceSHOW_HELP; break;
                           /* All other are unknown flags */
                 default : clEval = ceUNKNOWN_FLAG; break;
              }  /* Switch closes here */
           }
           else  /* if not a flag then it's (probably) a file specification */
               bAnyFilesSpecified = YES;
     }

     /* If no files specified then we've got an error! */
     if (!bAnyFilesSpecified && clEval == ceOK)
        clEval = ceNO_FILES_SPECIFIED;

     /* Handle command-line processing result value */
     switch(clEval) {
        case ceNO_FILES_SPECIFIED:
             printf("Command-line error: must specify at least one file to convert.\n");
             exit(ABORTED);
        case ceUNKNOWN_FLAG:
             printf("Command-line error: unknown flag specified.\n"
                    "Use %s /! or /? for more help on command-line syntax.\n",argv[0]);
             exit(ABORTED);
        case ceLEFTJUST_ERROR:
        case ceLEADING0_ERROR:
        case ceECHO_OPTIONS_ERROR:
        case ceSHOW_FNAMES_ERROR:
        case cePAUSE_ERROR:
        case ceHEX_ERROR:
        case ceWIDTH_SPEC_ERROR:
        case ceSEPAR_SPEC_ERROR:
             printf("Command-line error: unexpected text after flag.\n"
                    "Use %s /! or /? for more help on command-line syntax.\n",argv[0]);
             exit(ABORTED);
        case ceSHOW_HELP:
             printf("%s - Convert to Base and Dump version 2.00\n\n",argv[0]);
             printf("Shows contents of files as sequence of values (HEX/DEC/OCT)\n"
                    "by Christos Evaggelou, compiled on "__DATE__"\n\n"
                    "Command-line argument syntax:\n"
                    "   <file1>..[fileN]  [/O] [/D] [/H[[L|U],C]] [/L[+|-]] [/E[+|-]]\n"
                    "                     [/F[+|-]] [/0[+|-]] [/P[+|-]] [/!] [/?]\n"
                    "                     [/S[:c|nnn]] [/W[nnn|A]] [/C[nnn]]\n"
                    "Flags:\n"
                    "   /O /D and /H      convert to OCT, DEC or HEX(normal style)\n"
                    "   /H[L|U]           use lowercase or uppercase letters in hex\n"
                    "   /HC               use C/C++ hex style (0xhhhh) - usable with above\n"
                    "   /L[+|-]           left-justify values\n"
                    "   /E[+|-]           show (echo) user options at startup\n"
                    "   /F[+|-]           show file names before every conversion\n"
                    "   /P[+|-]           pause before every conversion\n"
                    "   /0[+|-]           display leading 0s\n"
                    "   /! and /?         show this help screen\n"
                    "   /Snnn             separation character 'nnn' (ASCII value)\n"
                    "   /S:c              separation character 'c' (literal value)\n"
                    "   /S or /S0         don't use separation character\n");
             printf("Press any key to view next page...");
             fflush(NULL);
             getchar();
             printf("\n   /Wnnn             set formatting width to nnn (decimal)\n"
                    "   /WA               auto-set widths for each base (O:3,D:3,H:2)\n"
                    "   /W or W0          use as much width required for each value\n"
                    "   /Cnnn             show nnn values per line\n"
                    "   /C or /C0         no limit to number of values per line\n\n"
                    "Notes:\n"
                    "   Both '-' and '/' can be used for flags\n\n"
                    "Defaults:\n"
                    "   /HCU /WA /C10 /S:, /0+ /L- /P- /E- /F-\n");
             exit(0);
       case ceOK: break;
       default:
             printf("Internal error in ProccessCommandLine: switch(clEval) { ...");
             exit(ABORTED);
     }
}
/* ======================================================================= */
/* Does the actual conversion according to the defined options */
/* ======================================================================= */
void PerformConversion(FILE *fp,OPTIONS o)
{
    Byte byByte[IOBUFSIZE];
    long i;
    long lBytesDisplayed,           /* Number of values displayed in line */
         lBytesRead,                /* Bytes read at each buffering */
         lFileSize,                 /* File size  */
         lTotalBytesDisplayed;      /* Total bytes displayed */
    char fmt[50];                   /* Format string */
    char tmp[18];                   /* Used by ce_itoa */

    switch(o.cvBase)
    {
       case cbHEX: sprintf(fmt,"%s%%%s%s%s%s"
                               ,o.bCHex ? "0x" : ""
                               ,o.bLeftJust ? "-" : ""
                               ,o.bLeading0 ? "0" : ""
                               ,o.bUseAutoWidth ? ce_itoa(defWidth(cbHEX),tmp) :
                                  (o.iWidth ? ce_itoa(o.iWidth,tmp) : "")
                               ,o.bLowerHex ? "x" : "X");
                   break;
       case cbDEC: sprintf(fmt,"%%%s%s%sd"
                              ,o.bLeftJust ? "-" : ""
                              ,o.bLeading0 ? "0" : ""
                              ,o.bUseAutoWidth ? ce_itoa(defWidth(cbDEC),tmp) :
                                 (o.iWidth ? ce_itoa(o.iWidth,tmp) : ""));
                   break;
       case cbOCT: sprintf(fmt,"%%%s%s%so"
                              ,o.bLeftJust ? "-" : ""
                              ,o.bLeading0 ? "0" : ""
                              ,o.bUseAutoWidth ? ce_itoa(defWidth(cbOCT),tmp) :
                                 (o.iWidth ? ce_itoa(o.iWidth,tmp) : ""));
                   break;
       case cbBIN: /* not yet implemented for BINARY mode */ ;
                   /* if implemented in the future, help should be included */
                   break;
       default : printf("PerformConversion: internal error in switch(o.cvBase) { ...\n");
                 exit(ABORTED);
                 break;
    }

    lBytesDisplayed = lBytesRead = lTotalBytesDisplayed = 0;
    lFileSize = filesize(fp);
    while (!feof(fp))
    {
        lBytesRead = fread(byByte,sizeof(Byte),IOBUFSIZE,fp);

        for (i = 0; i < lBytesRead; i++)
        {
            printf(fmt,byByte[i]);
            lBytesDisplayed++;
            lTotalBytesDisplayed++;
            if (o.bySeparator != '\0' && lTotalBytesDisplayed != lFileSize)
               printf("%c",o.bySeparator);
            if (o.iColumns != 0 && (lBytesDisplayed % o.iColumns == 0))
            {
               printf("\n");
               lBytesDisplayed = 0;
            }
        }
    }
    printf("\n");
}
/* ======================================================================= */
