
/* convert.c   EOUE Project, 1997 by Michael Mangelsdorf */
/* THIS FILE IS PUBLIC DOMAIN.
   LEAVE THIS MESSAGE IN PLACE AS COURTESY TO THE AUTHOR */


#include <stdio.h>

typedef unsigned char byte;

unsigned *jtext = NULL;  /* Must be defined. Should be re-assigned to
			    point to array  j_resource  in main(), or
			    resource() will not work. */

			  /*JIS codes for the kana bitmaps */
unsigned kana2jis [] =    /*sequentially encoded in file KANA.16*/
  {
	0x2422, 0x2424, 0x2426, 0x2428, 0x242A, 0x2522, 0x2524, 0x2526,
	0x2528, 0x252A, 0x2463, 0x2465, 0x2467, 0x242B, 0x242D, 0x242F,
	0x2431, 0x2433, 0x252B, 0x252D, 0x252F, 0x2531, 0x2533, 0x2521,
	0x2523, 0x2525, 0x2527, 0x2529, 0x242C, 0x242E, 0x2430, 0x2432,
	0x2434, 0x252C, 0x252E, 0x2530, 0x2532, 0x2534, 0x2421, 0x2423,
	0x2425, 0x2427, 0x2429, 0x2435, 0x2437, 0x2439, 0x243B, 0x243D,
	0x2535, 0x2537, 0x2539, 0x253B, 0x253D, 0x2436, 0x2438, 0x243A,
	0x243C, 0x243E, 0x2536, 0x2538, 0x253A, 0x253C, 0x253E, 0x243F,
	0x2441, 0x2444, 0x2446, 0x2448, 0x253F, 0x2541, 0x2544, 0x2546,
	0x2548, 0x2469, 0x246A, 0x246B, 0x246C, 0x246D, 0x2569, 0x256A,
	0x256B, 0x256C, 0x256D, 0x2440, 0x2442, 0x2445, 0x2447, 0x2449,
	0x2540, 0x2547, 0x2549, 0x244A, 0x244B, 0x244C, 0x244D, 0x244E,
	0x254A, 0x254B, 0x254C, 0x254D, 0x254E, 0x245E, 0x245F, 0x2460,
	0x2461, 0x2462, 0x255E, 0x255F, 0x2560, 0x2561, 0x2562, 0x244F,
	0x2452, 0x2455, 0x2458, 0x245B, 0x254F, 0x2552, 0x2555, 0x2558,
	0x255B, 0x2450, 0x2453, 0x2456, 0x2459, 0x245C, 0x2550, 0x2553,
	0x2556, 0x2559, 0x255C, 0x2451, 0x2454, 0x2457, 0x245A, 0x245D,
	0x2551, 0x2554, 0x2557, 0x255A, 0x255D, 0x2464, 0x2466, 0x2468,
	0x2564, 0x2566, 0x2568, 0x246F, 0x2470, 0x2471, 0x2472, 0x256F,
	0x2570, 0x2571, 0x2572, 0x2473, 0x2573, 0x2443, 0x2543, 0x246E,
	0x2575, 0x2576, 0x2139, 0x2563, 0x2565, 0x2567, 0x2442, 0x2445,
	0xFFFF
  };

			  /*JIS codes for the joyo kanji bitmaps*/
			  /*sequentially encoded in file JOYO.16*/
unsigned joyo2jis[] =

     {   
	0x3021, 0x3025, 0x3026, 0x302D, 0x302E, 0x3035, 0x3037, 0x3042,
	0x3045, 0x3046, 0x304A, 0x304C, 0x304D, 0x304E, 0x304F, 0x3051,
	0x3052, 0x3053, 0x3055, 0x3056, 0x3057, 0x3059, 0x305B, 0x305C,
	0x305D, 0x305E, 0x305F, 0x3061, 0x3063, 0x3064, 0x3065, 0x3066,
	0x3068, 0x3069, 0x306C, 0x306D, 0x306F, 0x3070, 0x3072, 0x3075,
	0x3077, 0x3078, 0x3079, 0x307A, 0x307B, 0x3121, 0x3122, 0x3123,
	0x3124, 0x3126, 0x3127, 0x3129, 0x312B, 0x3132, 0x313A, 0x313F,
	0x3140, 0x3144, 0x3146, 0x3147, 0x3149, 0x314A, 0x314B, 0x3151,
	0x3152, 0x3153, 0x3154, 0x3155, 0x3156, 0x3157, 0x3158, 0x3159,
	0x315A, 0x315B, 0x315C, 0x315F, 0x3160, 0x3163, 0x3164, 0x3167,
	0x3168, 0x3169, 0x316A, 0x316C, 0x316E, 0x316F, 0x3173, 0x3174,
	0x3176, 0x3178, 0x317A, 0x317B, 0x317C, 0x317D, 0x317E, 0x3221,
	0x3223, 0x3224, 0x3225, 0x3226, 0x3227, 0x322B, 0x322D, 0x322F,
	0x3230, 0x3231, 0x3235, 0x3237, 0x3238, 0x3239, 0x323A, 0x323B,
	0x323C, 0x323D, 0x323E, 0x323F, 0x3241, 0x3242, 0x3243, 0x3244,
	0x3246, 0x3247, 0x3248, 0x3249, 0x324A, 0x324B, 0x324C, 0x324D,
	0x324E, 0x324F, 0x3250, 0x3252, 0x3254, 0x3255, 0x3256, 0x3259,
	0x325A, 0x325B, 0x325D, 0x325F, 0x3261, 0x3263, 0x3266, 0x3268,
	0x326A, 0x326C, 0x326D, 0x326E, 0x3270, 0x3271, 0x3272, 0x3273,
	0x3274, 0x3275, 0x3277, 0x3278, 0x3279, 0x327B, 0x327C, 0x327D,
	0x327E, 0x3323, 0x3324, 0x3325, 0x3326, 0x3327, 0x3328, 0x332B,
	0x332C, 0x332D, 0x332F, 0x3330, 0x3332, 0x3334, 0x3335, 0x3336,
	0x3339, 0x333A, 0x3340, 0x3345, 0x3346, 0x3348, 0x334A, 0x334B,
	0x334C, 0x334D, 0x334E, 0x334F, 0x3350, 0x3351, 0x3353, 0x3354,
	0x3355, 0x3356, 0x3357, 0x3358, 0x3359, 0x335A, 0x335B, 0x335D,
	0x3363, 0x3364, 0x3365, 0x3367, 0x3368, 0x3369, 0x336A, 0x336C,
	0x336D, 0x336E, 0x3374, 0x3422, 0x3425, 0x3427, 0x3428, 0x3429,
	0x342A, 0x342B, 0x342C, 0x342D, 0x342E, 0x3430, 0x3431, 0x3432,
	0x3433, 0x3434, 0x3435, 0x3436, 0x3437, 0x3438, 0x3439, 0x343A,
	0x343D, 0x343E, 0x343F, 0x3440, 0x3441, 0x3444, 0x3445, 0x3446,
	0x3447, 0x3449, 0x344A, 0x344B, 0x344C, 0x344E, 0x344F, 0x3451,
	0x3453, 0x3454, 0x3455, 0x3456, 0x3457, 0x3458, 0x3459, 0x345B,
	0x345D, 0x345E, 0x345F, 0x3463, 0x3464, 0x3468, 0x3469, 0x346A,
	0x346B, 0x346D, 0x346E, 0x346F, 0x3470, 0x3471, 0x3473, 0x3474,
	0x3475, 0x3476, 0x3477, 0x3478, 0x3479, 0x347A, 0x347B, 0x347C,
	0x347D, 0x347E, 0x3521, 0x3522, 0x3524, 0x3525, 0x3527, 0x3528,
	0x352A, 0x352C, 0x352D, 0x352E, 0x352F, 0x3530, 0x3531, 0x3532,
	0x3533, 0x3534, 0x3536, 0x3537, 0x3539, 0x353A, 0x353B, 0x353C,
	0x353D, 0x353E, 0x353F, 0x3541, 0x3544, 0x3546, 0x3548, 0x354A,
	0x354D, 0x3551, 0x3552, 0x3553, 0x3554, 0x3555, 0x3556, 0x3557,
	0x3559, 0x355A, 0x355B, 0x355C, 0x355D, 0x355E, 0x355F, 0x3560,
	0x3561, 0x3563, 0x3565, 0x3566, 0x3567, 0x3569, 0x356A, 0x356B,
	0x356C, 0x356D, 0x356E, 0x356F, 0x3570, 0x3571, 0x3572, 0x3573,
	0x3575, 0x3576, 0x3577, 0x3579, 0x357B, 0x357D, 0x357E, 0x3621,
	0x3625, 0x3626, 0x3627, 0x3628, 0x362B, 0x362D, 0x362E, 0x362F,
	0x3632, 0x3633, 0x3634, 0x3635, 0x3636, 0x3637, 0x3638, 0x3639,
	0x363A, 0x363B, 0x363C, 0x363D, 0x363F, 0x3640, 0x3641, 0x3643,
	0x3644, 0x3645, 0x3647, 0x3648, 0x3649, 0x364A, 0x364B, 0x364C,
	0x3650, 0x3651, 0x3654, 0x3657, 0x3658, 0x365A, 0x365B, 0x365D,
	0x365F, 0x3660, 0x3661, 0x3662, 0x3663, 0x3664, 0x3665, 0x3667,
	0x3668, 0x366C, 0x366E, 0x3671, 0x3672, 0x3673, 0x3675, 0x3676,
	0x3678, 0x3679, 0x367E, 0x3721, 0x3724, 0x372B, 0x372C, 0x372E,
	0x372F, 0x3730, 0x3731, 0x3732, 0x3733, 0x3734, 0x3738, 0x3739,
	0x373A, 0x373B, 0x373C, 0x373F, 0x3740, 0x3741, 0x3742, 0x3743,
	0x3744, 0x3746, 0x3747, 0x3748, 0x3749, 0x374A, 0x374C, 0x374F,
	0x3750, 0x3751, 0x3754, 0x3756, 0x3757, 0x3759, 0x375A, 0x375C,
	0x375D, 0x375E, 0x375F, 0x3760, 0x3762, 0x3763, 0x3766, 0x3767,
	0x3768, 0x3769, 0x376A, 0x376B, 0x376C, 0x376E, 0x376F, 0x3770,
	0x3772, 0x3773, 0x3774, 0x3775, 0x3777, 0x3778, 0x3779, 0x377A,
	0x377B, 0x377C, 0x3821, 0x3822, 0x3824, 0x3825, 0x3826, 0x3828,
	0x3829, 0x382A, 0x382B, 0x382C, 0x382D, 0x382E, 0x382F, 0x3831,
	0x3832, 0x3833, 0x3835, 0x3836, 0x3837, 0x3838, 0x3839, 0x383A,
	0x383B, 0x383C, 0x383D, 0x3840, 0x3842, 0x3844, 0x3845, 0x3846,
	0x3847, 0x3849, 0x384A, 0x384B, 0x384C, 0x384D, 0x384E, 0x384F,
	0x3850, 0x3858, 0x385B, 0x385C, 0x385D, 0x385E, 0x385F, 0x3861,
	0x3862, 0x3864, 0x3865, 0x3866, 0x3867, 0x386B, 0x386C, 0x386D,
	0x386E, 0x3872, 0x3874, 0x3875, 0x3877, 0x3878, 0x3879, 0x387A,
	0x387C, 0x387D, 0x387E, 0x3921, 0x3923, 0x3925, 0x3926, 0x3927,
	0x3929, 0x392A, 0x392C, 0x392D, 0x392F, 0x3931, 0x3932, 0x3933,
	0x3934, 0x3935, 0x3936, 0x3939, 0x393B, 0x393D, 0x393E, 0x393F,
	0x3941, 0x3942, 0x3943, 0x3944, 0x3945, 0x3946, 0x3948, 0x394A,
	0x394B, 0x394C, 0x394D, 0x394E, 0x3952, 0x3953, 0x3954, 0x3955,
	0x3956, 0x3957, 0x3958, 0x3959, 0x395A, 0x395B, 0x395D, 0x395F,
	0x3960, 0x3961, 0x3962, 0x3964, 0x3966, 0x3967, 0x3969, 0x396B,
	0x396E, 0x396F, 0x3970, 0x3971, 0x3972, 0x3973, 0x3975, 0x3976,
	0x3978, 0x397C, 0x397E, 0x3A23, 0x3A24, 0x3A26, 0x3A27, 0x3A28,
	0x3A29, 0x3A2B, 0x3A2C, 0x3A2E, 0x3A30, 0x3A32, 0x3A34, 0x3A36,
	0x3A38, 0x3A39, 0x3A3A, 0x3A3D, 0x3A3E, 0x3A3F, 0x3A42, 0x3A44,
	0x3A45, 0x3A46, 0x3A47, 0x3A4A, 0x3A4B, 0x3A4C, 0x3A4D, 0x3A4E,
	0x3A4F, 0x3A50, 0x3A51, 0x3A52, 0x3A55, 0x3A57, 0x3A58, 0x3A59,
	0x3A5A, 0x3A5B, 0x3A5C, 0x3A5D, 0x3A5E, 0x3A5F, 0x3A60, 0x3A61,
	0x3A62, 0x3A64, 0x3A69, 0x3A6A, 0x3A6E, 0x3A6F, 0x3A71, 0x3A72,
	0x3A76, 0x3A77, 0x3A78, 0x3A79, 0x3A7D, 0x3A7E, 0x3B21, 0x3B23,
	0x3B24, 0x3B25, 0x3B26, 0x3B28, 0x3B2E, 0x3B30, 0x3B31, 0x3B32,
	0x3B33, 0x3B34, 0x3B36, 0x3B37, 0x3B3A, 0x3B3B, 0x3B3D, 0x3B3F,
	0x3B40, 0x3B43, 0x3B44, 0x3B45, 0x3B47, 0x3B48, 0x3B49, 0x3B4A,
	0x3B4B, 0x3B4C, 0x3B4D, 0x3B4E, 0x3B4F, 0x3B50, 0x3B51, 0x3B52,
	0x3B54, 0x3B55, 0x3B56, 0x3B57, 0x3B58, 0x3B59, 0x3B5C, 0x3B5D,
	0x3B5E, 0x3B5F, 0x3B60, 0x3B61, 0x3B63, 0x3B64, 0x3B65, 0x3B66,
	0x3B67, 0x3B68, 0x3B69, 0x3B6A, 0x3B6B, 0x3B6C, 0x3B6D, 0x3B6E,
	0x3B6F, 0x3B70, 0x3B71, 0x3B72, 0x3B73, 0x3B74, 0x3B75, 0x3B76,
	0x3B77, 0x3B78, 0x3B79, 0x3B7A, 0x3B7B, 0x3B7C, 0x3B7D, 0x3B7E,
	0x3C21, 0x3C22, 0x3C23, 0x3C25, 0x3C27, 0x3C28, 0x3C2A, 0x3C2B,
	0x3C2D, 0x3C30, 0x3C31, 0x3C34, 0x3C37, 0x3C39, 0x3C3A, 0x3C3C,
	0x3C3E, 0x3C3F, 0x3C40, 0x3C41, 0x3C42, 0x3C47, 0x3C4B, 0x3C4C,
	0x3C4D, 0x3C4E, 0x3C4F, 0x3C50, 0x3C51, 0x3C52, 0x3C54, 0x3C55,
	0x3C56, 0x3C57, 0x3C58, 0x3C59, 0x3C5A, 0x3C5B, 0x3C5C, 0x3C5F,
	0x3C60, 0x3C61, 0x3C63, 0x3C64, 0x3C65, 0x3C67, 0x3C68, 0x3C69,
	0x3C6A, 0x3C6B, 0x3C6C, 0x3C6D, 0x3C6E, 0x3C6F, 0x3C71, 0x3C72,
	0x3C73, 0x3C74, 0x3C75, 0x3C77, 0x3C78, 0x3C79, 0x3C7B, 0x3C7C,
	0x3C7D, 0x3C7E, 0x3D21, 0x3D22, 0x3D23, 0x3D24, 0x3D25, 0x3D26,
	0x3D28, 0x3D29, 0x3D2A, 0x3D2C, 0x3D2D, 0x3D2E, 0x3D30, 0x3D31,
	0x3D35, 0x3D37, 0x3D38, 0x3D39, 0x3D3B, 0x3D3C, 0x3D3D, 0x3D3E,
	0x3D40, 0x3D41, 0x3D42, 0x3D43, 0x3D44, 0x3D45, 0x3D46, 0x3D47,
	0x3D49, 0x3D4A, 0x3D4B, 0x3D4C, 0x3D4D, 0x3D4E, 0x3D4F, 0x3D50,
	0x3D51, 0x3D52, 0x3D53, 0x3D55, 0x3D56, 0x3D5A, 0x3D5B, 0x3D5C,
	0x3D5E, 0x3D60, 0x3D61, 0x3D62, 0x3D63, 0x3D64, 0x3D65, 0x3D67,
	0x3D68, 0x3D69, 0x3D6A, 0x3D6B, 0x3D6E, 0x3D6F, 0x3D70, 0x3D71,
	0x3D74, 0x3D75, 0x3D76, 0x3D77, 0x3D78, 0x3D79, 0x3D7C, 0x3D7D,
	0x3D7E, 0x3E21, 0x3E22, 0x3E23, 0x3E24, 0x3E26, 0x3E27, 0x3E29,
	0x3E2C, 0x3E2D, 0x3E2E, 0x3E2F, 0x3E30, 0x3E32, 0x3E34, 0x3E35,
	0x3E36, 0x3E37, 0x3E38, 0x3E3A, 0x3E3C, 0x3E3D, 0x3E3E, 0x3E42,
	0x3E43, 0x3E44, 0x3E46, 0x3E47, 0x3E48, 0x3E49, 0x3E4A, 0x3E4B,
	0x3E4C, 0x3E4D, 0x3E4E, 0x3E4F, 0x3E50, 0x3E51, 0x3E52, 0x3E53,
	0x3E57, 0x3E59, 0x3E5A, 0x3E5B, 0x3E5C, 0x3E5D, 0x3E5E, 0x3E62,
	0x3E63, 0x3E65, 0x3E66, 0x3E68, 0x3E69, 0x3E6A, 0x3E6B, 0x3E6C,
	0x3E6D, 0x3E6E, 0x3E6F, 0x3E70, 0x3E72, 0x3E74, 0x3E75, 0x3E76,
	0x3E78, 0x3E79, 0x3E7A, 0x3E7B, 0x3E7C, 0x3E7E, 0x3F22, 0x3F23,
	0x3F25, 0x3F26, 0x3F27, 0x3F28, 0x3F29, 0x3F2B, 0x3F2D, 0x3F2E,
	0x3F2F, 0x3F30, 0x3F31, 0x3F32, 0x3F33, 0x3F34, 0x3F35, 0x3F36,
	0x3F37, 0x3F39, 0x3F3B, 0x3F3C, 0x3F3D, 0x3F3F, 0x3F40, 0x3F42,
	0x3F43, 0x3F45, 0x3F46, 0x3F47, 0x3F48, 0x3F49, 0x3F4A, 0x3F4B,
	0x3F4C, 0x3F4D, 0x3F4E, 0x3F4F, 0x3F52, 0x3F53, 0x3F54, 0x3F57,
	0x3F58, 0x3F5D, 0x3F5E, 0x3F61, 0x3F62, 0x3F63, 0x3F64, 0x3F65,
	0x3F66, 0x3F67, 0x3F68, 0x3F6A, 0x3F6B, 0x3F6C, 0x3F6E, 0x3F6F,
	0x3F71, 0x3F72, 0x3F74, 0x3F75, 0x3F78, 0x3F79, 0x4021, 0x4023,
	0x4024, 0x4025, 0x4026, 0x4027, 0x4029, 0x402A, 0x402B, 0x402C,
	0x402D, 0x402E, 0x402F, 0x4030, 0x4031, 0x4032, 0x4035, 0x4036,
	0x4037, 0x4038, 0x4039, 0x403A, 0x403B, 0x403C, 0x403D, 0x403E,
	0x403F, 0x4040, 0x4041, 0x4042, 0x4044, 0x4045, 0x4046, 0x4047,
	0x4049, 0x404A, 0x404B, 0x404D, 0x404E, 0x404F, 0x4050, 0x4051,
	0x4052, 0x4053, 0x4055, 0x4056, 0x4057, 0x405A, 0x405B, 0x405C,
	0x405D, 0x405E, 0x405F, 0x4060, 0x4061, 0x4062, 0x4063, 0x4064,
	0x4065, 0x4067, 0x4068, 0x4069, 0x406A, 0x406B, 0x406C, 0x406E,
	0x406F, 0x4070, 0x4072, 0x4074, 0x4075, 0x4076, 0x4077, 0x4078,
	0x407B, 0x407E, 0x4121, 0x4125, 0x4126, 0x4129, 0x412A, 0x412B,
	0x412C, 0x412D, 0x412F, 0x4130, 0x4131, 0x4132, 0x4133, 0x4134,
	0x4135, 0x4136, 0x413A, 0x413C, 0x4142, 0x4143, 0x4144, 0x4145,
	0x4146, 0x4147, 0x4148, 0x414A, 0x414B, 0x414E, 0x414F, 0x4150,
	0x4152, 0x4153, 0x4154, 0x4155, 0x4158, 0x415B, 0x415C, 0x415D,
	0x415E, 0x4160, 0x4161, 0x4162, 0x4163, 0x4165, 0x4167, 0x4168,
	0x416A, 0x416B, 0x416D, 0x4170, 0x4171, 0x4172, 0x4174, 0x4175,
	0x4176, 0x4177, 0x4178, 0x417A, 0x417B, 0x417C, 0x417D, 0x417E,
	0x4221, 0x4222, 0x4223, 0x4224, 0x4225, 0x4226, 0x4227, 0x4228,
	0x4229, 0x422B, 0x422C, 0x422D, 0x422E, 0x422F, 0x4230, 0x4231,
	0x4232, 0x4233, 0x4234, 0x4238, 0x4239, 0x423A, 0x423B, 0x423C,
	0x423E, 0x423F, 0x4240, 0x4244, 0x4245, 0x4246, 0x4247, 0x424C,
	0x424E, 0x4250, 0x4251, 0x4253, 0x4254, 0x4255, 0x4256, 0x4258,
	0x4259, 0x425A, 0x425B, 0x425E, 0x425F, 0x4260, 0x4261, 0x4262,
	0x4265, 0x4266, 0x4267, 0x4268, 0x426A, 0x426C, 0x426E, 0x4270,
	0x4272, 0x4273, 0x4274, 0x4275, 0x4277, 0x4279, 0x427A, 0x4322,
	0x4323, 0x4325, 0x4326, 0x432A, 0x432B, 0x4330, 0x4331, 0x4332,
	0x4334, 0x4335, 0x4338, 0x433A, 0x433B, 0x433C, 0x4340, 0x4342,
	0x4343, 0x4344, 0x4345, 0x4346, 0x4347, 0x4348, 0x434A, 0x434B,
	0x434C, 0x434D, 0x434E, 0x434F, 0x4351, 0x4353, 0x4354, 0x4355,
	0x4356, 0x4357, 0x4359, 0x435B, 0x435C, 0x435D, 0x435F, 0x4360,
	0x4361, 0x4362, 0x4363, 0x4364, 0x4365, 0x4366, 0x4367, 0x4368,
	0x4369, 0x436A, 0x436B, 0x436C, 0x436D, 0x436E, 0x436F, 0x4372,
	0x4373, 0x4378, 0x4379, 0x437A, 0x437B, 0x4422, 0x4423, 0x4424,
	0x4425, 0x4426, 0x4427, 0x4428, 0x4429, 0x442B, 0x442C, 0x442E,
	0x442F, 0x4430, 0x4431, 0x4432, 0x4434, 0x4436, 0x4437, 0x4439,
	0x443A, 0x443B, 0x443C, 0x443E, 0x443F, 0x4440, 0x4441, 0x4442,
	0x4443, 0x4444, 0x4445, 0x4446, 0x4449, 0x444B, 0x444C, 0x444D,
	0x4452, 0x445A, 0x4460, 0x4462, 0x4463, 0x4464, 0x4465, 0x4467,
	0x4468, 0x4469, 0x446A, 0x446B, 0x446C, 0x446D, 0x446E, 0x446F,
	0x4471, 0x4473, 0x4478, 0x4479, 0x447A, 0x447B, 0x447E, 0x4521,
	0x4525, 0x4526, 0x4528, 0x4529, 0x452A, 0x452B, 0x452C, 0x452F,
	0x4530, 0x4531, 0x4533, 0x4534, 0x4535, 0x4537, 0x4538, 0x4539,
	0x453A, 0x453E, 0x4540, 0x4541, 0x4542, 0x4544, 0x4545, 0x4547,
	0x4549, 0x454C, 0x454D, 0x454F, 0x4550, 0x4553, 0x4554, 0x4558,
	0x4559, 0x455A, 0x455B, 0x455C, 0x455D, 0x455E, 0x455F, 0x4560,
	0x4561, 0x4562, 0x4563, 0x4567, 0x4569, 0x456A, 0x456B, 0x456C,
	0x456D, 0x456F, 0x4570, 0x4572, 0x4574, 0x4576, 0x4577, 0x4579,
	0x457A, 0x457B, 0x457C, 0x457D, 0x457E, 0x4624, 0x4625, 0x4626,
	0x4627, 0x4628, 0x4629, 0x462B, 0x462C, 0x462D, 0x462E, 0x462F,
	0x4630, 0x4631, 0x4632, 0x4633, 0x4636, 0x4638, 0x4639, 0x463B,
	0x463C, 0x463D, 0x463F, 0x4640, 0x4641, 0x4643, 0x4644, 0x4646,
	0x4647, 0x4648, 0x4649, 0x464C, 0x464D, 0x464F, 0x4656, 0x465A,
	0x465E, 0x465F, 0x4662, 0x466C, 0x466E, 0x4670, 0x4671, 0x4673,
	0x4674, 0x4675, 0x4679, 0x467C, 0x467D, 0x467E, 0x4721, 0x4722,
	0x4724, 0x4725, 0x4726, 0x4727, 0x472B, 0x472D, 0x472E, 0x472F,
	0x4730, 0x4733, 0x4734, 0x473A, 0x473B, 0x473C, 0x473D, 0x473E,
	0x4740, 0x4744, 0x4746, 0x4748, 0x4749, 0x474B, 0x474C, 0x474F,
	0x4750, 0x4751, 0x4752, 0x4753, 0x4754, 0x4755, 0x4758, 0x4759,
	0x475A, 0x475B, 0x475C, 0x475D, 0x475E, 0x475F, 0x4763, 0x4764,
	0x4765, 0x4766, 0x476C, 0x476E, 0x476F, 0x4771, 0x4772, 0x4775,
	0x4776, 0x4777, 0x4779, 0x477A, 0x477B, 0x477E, 0x4822, 0x4829,
	0x482A, 0x482C, 0x482D, 0x482F, 0x4831, 0x4832, 0x4833, 0x4834,
	0x4836, 0x483C, 0x483D, 0x483E, 0x483F, 0x4841, 0x4842, 0x4844,
	0x4847, 0x4848, 0x4849, 0x484A, 0x484B, 0x484C, 0x484D, 0x484E,
	0x484F, 0x4851, 0x4852, 0x4853, 0x4855, 0x4856, 0x4857, 0x485A,
	0x485C, 0x485D, 0x485E, 0x4860, 0x4861, 0x4862, 0x4863, 0x4864,
	0x4866, 0x4867, 0x4868, 0x4869, 0x486A, 0x486B, 0x486D, 0x486E,
	0x486F, 0x4871, 0x4872, 0x4873, 0x4874, 0x4877, 0x4878, 0x4879,
	0x487E, 0x4921, 0x4924, 0x492C, 0x492E, 0x4931, 0x4934, 0x4936,
	0x4938, 0x4939, 0x493A, 0x493C, 0x493D, 0x493E, 0x4941, 0x4942,
	0x4943, 0x4944, 0x494A, 0x494D, 0x494F, 0x4950, 0x4951, 0x4952,
	0x4953, 0x4954, 0x4955, 0x4957, 0x4958, 0x4959, 0x495B, 0x495C,
	0x495D, 0x495E, 0x495F, 0x4961, 0x4962, 0x4963, 0x4964, 0x4965,
	0x4966, 0x4968, 0x4969, 0x496A, 0x496B, 0x496D, 0x496E, 0x4970,
	0x4971, 0x4974, 0x4975, 0x4977, 0x497A, 0x497B, 0x497C, 0x497D,
	0x497E, 0x4A21, 0x4A22, 0x4A23, 0x4A24, 0x4A27, 0x4A28, 0x4A29,
	0x4A2A, 0x4A2C, 0x4A2E, 0x4A2F, 0x4A30, 0x4A33, 0x4A34, 0x4A36,
	0x4A37, 0x4A38, 0x4A39, 0x4A3A, 0x4A3B, 0x4A3C, 0x4A3D, 0x4A3E,
	0x4A3F, 0x4A40, 0x4A41, 0x4A42, 0x4A44, 0x4A45, 0x4A46, 0x4A49,
	0x4A4A, 0x4A4C, 0x4A50, 0x4A51, 0x4A52, 0x4A54, 0x4A55, 0x4A56,
	0x4A57, 0x4A58, 0x4A59, 0x4A5B, 0x4A5D, 0x4A5E, 0x4A61, 0x4A62,
	0x4A64, 0x4A66, 0x4A67, 0x4A68, 0x4A69, 0x4A6B, 0x4A6C, 0x4A6D,
	0x4A6F, 0x4A70, 0x4A71, 0x4A73, 0x4A74, 0x4A75, 0x4A76, 0x4A78,
	0x4A7A, 0x4A7C, 0x4A7D, 0x4B21, 0x4B22, 0x4B24, 0x4B25, 0x4B26,
	0x4B27, 0x4B2B, 0x4B2C, 0x4B2D, 0x4B2E, 0x4B30, 0x4B33, 0x4B34,
	0x4B35, 0x4B36, 0x4B37, 0x4B38, 0x4B39, 0x4B3A, 0x4B3B, 0x4B3C,
	0x4B3D, 0x4B3E, 0x4B3F, 0x4B40, 0x4B41, 0x4B42, 0x4B43, 0x4B44,
	0x4B45, 0x4B47, 0x4B49, 0x4B4C, 0x4B4D, 0x4B4F, 0x4B50, 0x4B51,
	0x4B52, 0x4B57, 0x4B59, 0x4B5B, 0x4B5C, 0x4B5D, 0x4B5E, 0x4B5F,
	0x4B60, 0x4B61, 0x4B62, 0x4B63, 0x4B64, 0x4B65, 0x4B67, 0x4B68,
	0x4B6B, 0x4B6C, 0x4B74, 0x4B75, 0x4B76, 0x4B7A, 0x4B7C, 0x4B7D,
	0x4B7E, 0x4C21, 0x4C23, 0x4C24, 0x4C25, 0x4C28, 0x4C29, 0x4C2E,
	0x4C2F, 0x4C31, 0x4C32, 0x4C33, 0x4C34, 0x4C35, 0x4C37, 0x4C38,
	0x4C3B, 0x4C3C, 0x4C3E, 0x4C3F, 0x4C40, 0x4C41, 0x4C42, 0x4C43,
	0x4C44, 0x4C47, 0x4C48, 0x4C4A, 0x4C4C, 0x4C4F, 0x4C50, 0x4C51,
	0x4C53, 0x4C54, 0x4C55, 0x4C56, 0x4C57, 0x4C5A, 0x4C5B, 0x4C5C,
	0x4C61, 0x4C64, 0x4C66, 0x4C67, 0x4C68, 0x4C6B, 0x4C6E, 0x4C70,
	0x4C71, 0x4C72, 0x4C73, 0x4C74, 0x4C75, 0x4C76, 0x4C78, 0x4C7B,
	0x4C7D, 0x4C7E, 0x4D21, 0x4D22, 0x4D23, 0x4D25, 0x4D26, 0x4D27,
	0x4D29, 0x4D2A, 0x4D2B, 0x4D2D, 0x4D31, 0x4D33, 0x4D35, 0x4D36,
	0x4D37, 0x4D39, 0x4D3A, 0x4D3B, 0x4D3C, 0x4D3D, 0x4D3E, 0x4D3F,
	0x4D40, 0x4D42, 0x4D44, 0x4D46, 0x4D47, 0x4D48, 0x4D49, 0x4D4A,
	0x4D4B, 0x4D4D, 0x4D4E, 0x4D4F, 0x4D51, 0x4D52, 0x4D53, 0x4D55,
	0x4D57, 0x4D58, 0x4D59, 0x4D5B, 0x4D5C, 0x4D5E, 0x4D5F, 0x4D61,
	0x4D62, 0x4D63, 0x4D65, 0x4D67, 0x4D68, 0x4D6A, 0x4D6B, 0x4D6D,
	0x4D6E, 0x4D6F, 0x4D70, 0x4D71, 0x4D73, 0x4D74, 0x4D77, 0x4D78,
	0x4D79, 0x4D7A, 0x4D7D, 0x4E21, 0x4E22, 0x4E24, 0x4E25, 0x4E26,
	0x4E27, 0x4E28, 0x4E29, 0x4E2C, 0x4E2E, 0x4E31, 0x4E32, 0x4E33,
	0x4E34, 0x4E35, 0x4E38, 0x4E39, 0x4E3A, 0x4E3B, 0x4E3D, 0x4E3E,
	0x4E40, 0x4E41, 0x4E43, 0x4E44, 0x4E45, 0x4E48, 0x4E49, 0x4E4C,
	0x4E4D, 0x4E4E, 0x4E4F, 0x4E50, 0x4E51, 0x4E52, 0x4E53, 0x4E57,
	0x4E58, 0x4E59, 0x4E5D, 0x4E5E, 0x4E5F, 0x4E60, 0x4E61, 0x4E63,
	0x4E64, 0x4E65, 0x4E69, 0x4E6B, 0x4E6C, 0x4E6D, 0x4E6E, 0x4E6F,
	0x4E70, 0x4E71, 0x4E72, 0x4E73, 0x4E74, 0x4E75, 0x4E76, 0x4E77,
	0x4E78, 0x4E7D, 0x4F22, 0x4F23, 0x4F27, 0x4F29, 0x4F2A, 0x4F2B,
	0x4F2D, 0x4F2F, 0x4F30, 0x4F32, 0x4F33, 0x4F37, 0x4F3A, 0x4F3B,
	0x4F3F, 0x4F40, 0x4F42, 0x4F43, 0x4F45, 0x4F47, 0x4F48, 0x4F51,
	0x4F53, 0xFFFF
     };


/* input:  Array of unsigned with first element containing length.
	   Elements correspond to indexes into the kanji/kana font.
   output: Each element is replaced by its JIS equivalent.
*/
unsigned* jis (unsigned *codestr) {
 unsigned i;
 for (i=1;i<codestr[0];i++)
    if (codestr[i]>=2000) codestr[i]=kana2jis[codestr[i]-2000];
    else codestr[i]=joyo2jis[codestr[i]];
 return codestr;
}


/* Reverse jis(), see above.
*/
unsigned* internal (unsigned *codestr) {
 unsigned i,j=0;
 for (i=1;i<codestr[0];i++)
   if (codestr[i]<0x3000) {
    j=0;
    while (kana2jis[j]!=0xFFFF) if (kana2jis[j++]==codestr[i]) break;
    codestr[i]=2000+j-1;
   }
   else {
    j=0;
    while (joyo2jis[j]!=0xFFFF) if (joyo2jis[j++]==codestr[i]) break;
    codestr[i]=j-1;
   }
 return codestr;
}


/* Retrieves a japanese string (of unsigneds) from a 'resource'
   array created by JIS2C.EXE. Refer to documentation.
   The array always terminates with 0xFFFF. The value 0x0000 corresponds
   to a line break or JIS/ASCII transition in the original JIS input file.
*/
unsigned* resource (unsigned index, unsigned *codestr) {
  unsigned i=0,j=0, ptr=0;
  if (!index) return NULL;
  while (1) {
     if (jtext[ptr]==0xFFFF) return NULL;
     if (jtext[ptr++]) {j++; continue;}
     if (++i==index) break;
     else j=0;
  }
  codestr[0]=j+1; ptr--;
  for (i=0;i<=j;i++) codestr[i+1]=jtext[ptr-j+i];
  return internal (codestr);
}

