/* Copyright (C) 1999 Tobias Romann, see flat.asm for details */
/* 32-bit dynamic memory management system for the flat real mode for C/C++
   library.
 */

#include <stdlib.h>

#ifndef __FLAT_H
#include "../../flat.h"
#endif

/* available heap memory */
long heap32_avail;

/* base adress of heap */
flatptr heap32_base;

/* maximum number of heap nodes, items allocated, heap error code */
int heap32_max = 64, heap32_items, heap32_errno;

/* pointer to the heap nodes */
heap32_node *heap32;

/* optimize the heap, by joining sequences of free blocks into a big one */
void heap32_sort (void);

/* misc. local stuff */
static void heap_init (void);
static int find_block (long);
static long get_len (int idx);
static flatptr get_adress (int idx);
static int is_free (int idx);
static int resize_heap (int new_size);
static int insert_heap (int idx);
static void delete_heap (int idx);

/* macros:
   get_len -- return the length of a heap node
   is_free -- return whether the block is free
 */
#define get_len(idx) ((long)(heap32[(idx)].size & ~HEAP32_USED_BIT))
#define is_free(idx) ((int)!(heap32[(idx)].size & HEAP32_USED_BIT))

/* heap_init -- allocate memory for the heap */
void
heap_init (void)
{
  heap32 = (heap32_node *) malloc (sizeof (heap32_node) * heap32_max);

  /* init some variables */
  heap32_avail = em_avail;
  heap32_base = em_base;

  if (!heap32)
    heap32_errno = HEAP32_NOCONV;
  else
    {
      /* create a dummy item; the entire extended memory in a free block */
      heap32_errno = HEAP32_ZERO;
      heap32[0].size = heap32_avail;
      heap32_items = 1;
    }
}

/* malloc32 -- try to allocate extended memory
   Returns 0L on error, else the allocated flat pointer.
 */
flatptr
malloc32 (long size)
{
  register int idx;
  long remaining_length;
  flatptr p;

  /* if no heap */
  if (!heap32)
    {
      heap_init ();

      /* exit on error */
      if (HEAP32_NOCONV == heap32_errno)
        return (0L);
    }

  /* try to allocate the block */
  idx = find_block (size);

  /* if it couldn't be allocated, sort the heap */
  if (-1 == idx)
    {
      heap32_sort ();

      /* try again */
      idx = find_block (size);
      if (-1 == idx)
        return (0);
    }

  heap32_avail -= size;

  p = get_adress (idx);
  remaining_length = get_len (idx) - size;

  heap32[idx].size = size | HEAP32_USED_BIT;

  insert_heap (1 + idx);
  heap32[1 + idx].size = remaining_length;
  return (p);
}

/* free32 -- free a memory block */
void
free32 (flatptr p)
{
  flatptr i = em_base;
  long s;
  register int foo;

  /* search for a matching block */
  for (foo = 0; foo < heap32_items; ++foo)
    {
      /* found it, then set the "free" bit */
      if (i == p)
        {
          heap32_avail += get_len (foo);
          heap32[foo].size &= ~HEAP32_USED_BIT;
          return;
        }

      i += get_len (foo);
    }
  heap32_errno = HEAP32_NOTFOUND;
}

/* find_block -- search for a free block that can be split into a used and
   a free one
 */
int
find_block (long size)
{
  register int foo;

  /* enough memory available */
  if (heap32_avail - size < 0)
    {
      heap32_errno = HEAP32_NOEXT;
      return (-1);
    }

  /* scan all heap items */
  for (foo = 0; foo < heap32_items; ++foo)
    {
      /* a free block */
      if (is_free (foo))
        {
          /* and with the right size */
          if (size <= get_len (foo))
            {
              return (foo);
            }
        }
    }

  /* search failed */
  return (-1);
}

/* heap32_sort -- join following free heap nodes to one big */
void
heap32_sort (void)
{
  register int foo;

  /* from the first to the last-1 item */
  for (foo = 0; foo < heap32_items - 1; ++foo)
    {
      if (is_free (foo) && is_free (1 + foo))
        {
          /* make one item with the size of two */
          heap32[foo].size += heap32[1 + foo].size;
          delete_heap (1 + foo);

          /* redo this item */
          --foo;
        }
    }
}

/* resize_heap -- resize the heap node table */
int
resize_heap (int new_size)
{
  heap32_node *p = heap32;

  if (NULL == (p = (heap32_node *) realloc (p,
                                          new_size * sizeof (heap32_node))))
    {
      heap32_errno = HEAP32_NOCONV;
      return (-1);
    }

  heap32 = p;
  heap32_max = new_size;
  return (0);
}

/* insert_heap -- insert a new node into the heap table, resizing it if
   needed
 */
int
insert_heap (int idx)
{
  register int foo;

  /* is the current table too small */
  if (heap32_items + 1 > heap32_max)
    {
      /* try to double the size */
      if (resize_heap (heap32_max << 1))
        return (-1);
      heap32_max <<= 1;
    }

  /* copy all following items to the next position */
  for (foo = idx; foo <= heap32_items; ++foo)
    heap32[foo + 1].size = heap32[foo].size;

  /* create a zero size item */
  heap32[idx].size = 0;

  /* one more now */
  ++heap32_items;
  return (0);
}

/* delete_heap -- remove an item from the table
   Note: This does *not* downsize the table, even if it's possible!
   Use heap32_downsize for that.
 */
void
delete_heap (int idx)
{
  register int foo;

  for (foo = idx + 1; foo <= heap32_items; ++foo)
    heap32[foo - 1].size = heap32[foo].size;

  --heap32_items;
}

/* get_adress -- return the adress of a heap node */
flatptr
get_adress (int idx)
{
  flatptr p = heap32_base;

  --idx;
  while (idx >= 0)
    p += get_len (idx--);

  return (p);
}

/* heap32_downsize -- downsize the heap node table */
int
heap32_downsize (int new_size)
{
  heap32_node *n;

  /* if the new size would kill any block, exit */
  if (new_size < heap32_items)
    return (-1);

  /* don't enlarge the table */
  if (new_size >= heap32_max)
    return (-1);

  /* try to downsize it */
  n = (heap32_node *) realloc (heap32, new_size * sizeof (heap32_node));
  if (!n)
    {
      heap32_errno = HEAP32_NOCONV;
      return (-1);
    }

  heap32_max = new_size;
  return (0);
}
