/*************************************************************************
** interpcom-1.1 (command interpreter - tutorial)                        **
** functcmd.c : Commands of manipulation of functions                    **
**                                                                       **
** Copyright (C) 1998  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**									 **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details. 			 **
**									 **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free		 **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      Aile 45-55                                                       **
**      2, place Jussieu                                                 **
**      75251 Paris Cedex 05                                             **
**      France								 **
**                                                                       **
 *************************************************************************/


#include "interp.h"
#include "funct.h"

funct_t F_COPY = {
    f_funct_copy,
    d_funct_copy,
    C_funct_copy,
    dC_funct_copy
};

funct_t F_INTEG = {
    integ_funct_f,
    integ_funct_d,
    integ_funct_C,
    integ_funct_dC
};

funct_t F_DIFF = {
    diff_funct_f,
    diff_funct_d,
    diff_funct_C,
    diff_funct_dC
};

funct2_t F_ADD = {
    f_funct_add,
    d_funct_add,
    C_funct_add,
    dC_funct_add
};

funct2_t F_SUB = {
    f_funct_sub,
    d_funct_sub,
    C_funct_sub,
    dC_funct_sub
};

funct2_t F_MUL = {
    f_funct_mul,
    d_funct_mul,
    C_funct_mul,
    dC_funct_mul
};

funct2_t F_DIV = {
    f_funct_div,
    d_funct_div,
    C_funct_div,
    dC_funct_div
};

extern	int  _NBFONC;

#ifdef _SECURE_FUNC
#define TX_f(c) test_correct_x_f((funct_f *)((c)))
#define TX_d(c) test_correct_x_d((funct_d *)((c)))
#define TX_C(c) test_correct_x_C((funct_C *)((c)))
#define TX_dC(c) test_correct_x_dC((funct_dC *)((c)))
#else
#define TX_f(c) 1
#define TX_d(c) 1
#define TX_C(c) 1
#define TX_dC(c) 1
#endif




/*---------------------------------------------------------------------------
    Returns a string containing the integer argument.
---------------------------------------------------------------------------*/
char *
ch_copy_int(int i0)
{
    char 	    h[20];
    
    memset(h, 0, 20);
    sprintf(h, "%d", i0);
    return ch_copy(h);
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
	Command of definition of a real function in simple precision. 
	Syntax :
	    defunc_f f xr
        where 'f' is the name of the function and 'xr' the name of a 
        "xrange" in simple precision.
---------------------------------------------------------------------------*/
int
def_funct_f(int argc, char *argv[])
{
    int		    i0,
		    iw,
                    i0_b,
                    iw_b,
		   *d;
    funct_f	   *a;
    float	   *x_r;
    char 	  **e,
		   *k[3];

    iw = sketch_obj(argv[1], &i0);
    if (iw != 0) {
	error_mess(_FUNC_MESS);
        return 1;
    }
    k[0] = ch_copy("objdef");
    k[1] = ch_copy_int(_RFUNC_F - 1);
    k[2] = argv[1];
    i0 = obj_create(3, k);
    if (i0 == -1) {
	error_mess(_FUNC_MESS);
        return 1;
    }
    free(k[0]);
    free(k[1]);
    iw_b = sketch_obj_restr(argv[2], &i0_b, _XRANGE_F);
    if (iw_b != _XRANGE_F) {
	error_mess(1 + _FUNC_MESS);
    	k[0] = ch_copy("destroy");
    	k[1] = argv[1];
        detruit_obj(2, k);
        free(k[0]);
        return 1;
    }
    a = (funct_f *) malloc((size_t) sizeof(funct_f));
    x_r = (float *) Obj[_XRANGE_F - 1][i0_b].adresse;
    d = Obj[_XRANGE_F - 1][i0_b].dim;
    a->type = x_r[0];
    if (a->type == 0)
        a->nb = d[0];
    else
	a->nb = x_r[7];
    a->x = x_r;
    a->f = float_alloc1(a->nb);
    a->nom = ch_copy(argv[2]);
    e = (char **) Obj[_RFUNC_F - 1][i0].adresse;
    e[0] = (char *) a;
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
	Command of definition of a real function in double precision. 
	Syntax :
	    defunc_d f xr
        where 'f' is the name of the function and 'xr' the name of a 
        "xrange" in double precision.
---------------------------------------------------------------------------*/
int
def_funct_d(int argc, char *argv[])
{
    int		    i0,
		    iw,
                    i0_b,
                    iw_b,
		   *d;
    funct_d	   *a;
    double	   *x_r;
    char 	  **e,
		   *k[3];

    iw = sketch_obj(argv[1], &i0);
    if (iw != 0) {
	error_mess(_FUNC_MESS);
        return 1;
    }
    k[0] = ch_copy("objdef");
    k[1] = ch_copy_int(_RFUNC_D - 1);
    k[2] = argv[1];
    i0 = obj_create(3, k);
    if (i0 == -1) {
	error_mess(_FUNC_MESS);
        return 1;
    }
    free(k[0]);
    free(k[1]);
    iw_b = sketch_obj_restr(argv[2], &i0_b, _XRANGE_D);
    if (iw_b != _XRANGE_D) {
	error_mess(1 + _FUNC_MESS);
    	k[0] = ch_copy("destroy");
    	k[1] = argv[1];
        detruit_obj(2, k);
        free(k[0]);
        return 1;
    }
    a = (funct_d *) malloc((size_t) sizeof(funct_f));
    x_r = (double *) Obj[_XRANGE_D - 1][i0_b].adresse;
    d = Obj[_XRANGE_D - 1][i0_b].dim;
    a->type = x_r[0];
    if (a->type == 0)
        a->nb = d[0];
    else
	a->nb = x_r[7];
    a->x = x_r;
    a->f = double_alloc1(a->nb);
    a->nom = ch_copy(argv[2]);
    e = (char **) Obj[_RFUNC_D - 1][i0].adresse;
    e[0] = (char *) a;
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
	Command of definition of a complex function in simple precision. 
	Syntax :
	    defunc_C f xr
        where 'f' is the name of the function and 'xr' the name of a 
        "xrange" in simple precision.
---------------------------------------------------------------------------*/
int
def_funct_C(int argc, char *argv[])
{
    int		    i0,
		    iw,
                    i0_b,
                    iw_b,
		   *d;
    funct_C	   *a;
    float	   *x_r;
    char 	  **e,
		   *k[3];

    iw = sketch_obj(argv[1], &i0);
    if (iw != 0) {
	error_mess(_FUNC_MESS);
        return 1;
    }
    k[0] = ch_copy("objdef");
    k[1] = ch_copy_int(_CFUNC_F - 1);
    k[2] = argv[1];
    i0 = obj_create(3, k);
    if (i0 == -1) {
	error_mess(_FUNC_MESS);
        return 1;
    }
    free(k[0]);
    free(k[1]);
    iw_b = sketch_obj_restr(argv[2], &i0_b, _XRANGE_F);
    if (iw_b != _XRANGE_F) {
	error_mess(_FUNC_MESS + 1);
    	k[0] = ch_copy("destroy");
    	k[1] = argv[1];
        detruit_obj(2, k);
        free(k[0]);
        return 1;
    }
    a = (funct_C *) malloc((size_t) sizeof(funct_f));
    x_r = (float *) Obj[_XRANGE_F - 1][i0_b].adresse;
    d = Obj[_XRANGE_F - 1][i0_b].dim;
    a->type = x_r[0];
    if (a->type == 0)
        a->nb = d[0];
    else
	a->nb = x_r[7];
    a->x = x_r;
    a->f = fcomplex_alloc1(a->nb);
    a->nom = ch_copy(argv[2]);
    e = (char **) Obj[_CFUNC_F - 1][i0].adresse;
    e[0] = (char *) a;
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
	Command of definition of a complex function in double precision. 
	Syntax :
	    defunc_dC f xr
        where 'f' is the name of the function and 'xr' the name of a 
        "xrange" in double precision.
---------------------------------------------------------------------------*/
int
def_funct_dC(int argc, char *argv[])
{
    int		    i0,
		    iw,
                    i0_b,
                    iw_b,
		   *d;
    funct_dC	   *a;
    double	   *x_r;
    char 	  **e,
		   *k[3];

    iw = sketch_obj(argv[1], &i0);
    if (iw != 0) {
	error_mess(_FUNC_MESS);
        return 1;
    }
    k[0] = ch_copy("objdef");
    k[1] = ch_copy_int(_CFUNC_D - 1);
    k[2] = argv[1];
    i0 = obj_create(3, k);
    if (i0 == -1) {
	error_mess(_FUNC_MESS);
        return 1;
    }
    free(k[0]);
    free(k[1]);
    iw_b = sketch_obj_restr(argv[2], &i0_b, _XRANGE_D);
    if (iw_b != _XRANGE_D) {
	error_mess(_FUNC_MESS + 1);
    	k[0] = ch_copy("destroy");
    	k[1] = argv[1];
        detruit_obj(2, k);
        free(k[0]);
        return 1;
    }
    a = (funct_dC *) malloc((size_t) sizeof(funct_f));
    x_r = (double *) Obj[_XRANGE_D - 1][i0_b].adresse;
    d = Obj[_XRANGE_D - 1][i0_b].dim;
    a->type = x_r[0];
    if (a->type == 0)
        a->nb = d[0];
    else
	a->nb = x_r[7];
    a->x = x_r;
    a->f = dcomplex_alloc1(a->nb);
    a->nom = ch_copy(argv[2]);
    e = (char **) Obj[_CFUNC_D - 1][i0].adresse;
    e[0] = (char *) a;
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Function that is called when a command of destruction of a function 
    is given.
---------------------------------------------------------------------------*/
int
detruit_funct(int iw, int i0)
{
    funct_f	   *a_f;
    funct_d	   *a_d;
    funct_C	   *a_C;
    funct_dC	   *a_dC;
    char	  **c;

    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == 3) {
	a_f = (funct_f *) c[0];
        if (a_f != NULL) {
            XFREE(a_f->f);
            free(a_f->nom);
	    free(a_f);
	}
    }
    else {
	if (iw == 4) {
	    a_d = (funct_d *) c[0];
	    if (a_d != NULL) {
                XFREE(a_d->f);
                free(a_d->nom);
	        free(a_d);
	    }
	}
	else {
	    if (iw == 5) {
	        a_C = (funct_C *) c[0];
		if (a_C != NULL) {
                    XFREE(a_C->f);
                    free(a_C->nom);
	            free(a_C);
	         }
	    }
	    else {
	        a_dC = (funct_dC *) c[0];
		if (a_dC != NULL) {
                    XFREE(a_dC->f);
                    free(a_dC->nom);
	            free(a_dC);
		}
	    }
	}
    }
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
	Commande of definition of a xrange in simple precision (defxr_f).
---------------------------------------------------------------------------*/
int
def_xrange_f(int argc, char *argv[])
{
    if (def_xrange_gen(argc, argv, 0) == 1)
	return 1;
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
	Commande of definition of a xrange in double precision (defxr_d).
---------------------------------------------------------------------------*/
int
def_xrange_d(int argc, char *argv[])
{
    if (def_xrange_gen(argc, argv, 1) == 1)
	return 1;
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
---------------------------------------------------------------------------*/
int
def_xrange_gen(int argc, char *argv[], int cas)
{
    int		    i0,
		    iw,
		    nb,
		    type;
    char	    h[100],
		   *k[3];
    float	   *x_r;
    double	   *x_rd;

    iw = sketch_obj(argv[1], &i0);
    if (iw != 0) {
	error_mess(_FUNC_MESS);
        return 1;
    }
    if (argc < 3) {
	print("%s", mess[_FUNC_MESS + 3]);
        read_int(&type);
        print("%s", mess[_FUNC_MESS + 4]);
        read_int(&nb);
    }
    else {
        if (argc < 4) {
	    error_mess(_FUNC_MESS + 6);
	    return 1;
        }
	type = convert_int(argv[2]);
	nb = convert_int(argv[3]);
    }
    if (nb < 2 || type < 0 || type > 1) {
	error_mess(_FUNC_MESS + 2);
	return 1;
    }
    iw = 10;
    if (type == 0) 
	iw = nb;
    memset(h, 0, 100);
    sprintf(h, "nbpoints_x=%d", iw);
    S_convert_int(h);
    k[0] = ch_copy("objdef");
    k[2] = argv[1];
    if (cas == 0) {
        k[1] = ch_copy_int(_XRANGE_F - 1);
        i0 = obj_create(3, k);
	x_r = (float *) Obj[_XRANGE_F - 1][i0].adresse;
        x_r[0] = type;
        if (type > 0)
	    x_r[7] = nb;
    }
    else {
        k[1] = ch_copy_int(_XRANGE_D - 1);
        i0 = obj_create(3, k);
	x_rd = (double *) Obj[_XRANGE_D - 1][i0].adresse;
        x_rd[0] = type;
        if (type > 0)
	    x_rd[7] = nb;
    }
    free(k[0]);
    free(k[1]);
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command of fixation of the values of a x-range in simple
    precision (fix_xr_f).
---------------------------------------------------------------------------*/
int
fix_xrange_f(int argc, char *argv[])
{
    int		    i,
		    i0,
		    iw,
		    type,
		   *d;
    float	   *x_r;

    iw = sketch_obj_restr(argv[1], &i0, _XRANGE_F);
    if (iw != _XRANGE_F) {
	error_mess(_FUNC_MESS + 1);
        return 1;
    }
    x_r = (float *) Obj[_XRANGE_F - 1][i0].adresse;
    type = x_r[0];
    if (type == 0) {
        d = (int *) Obj[_XRANGE_F - 1][i0].dim;
        if (argc == 4) {
	    iw = convert_int(argv[2]);
            if (iw < 1 || iw > d[0]) {
	        error_mess(_FUNC_MESS + 2);
		return 1;
	    }
	    x_r[iw] = convert_float(argv[3]);
	    return 0;
	}
	else {
	    for (i = 1; i <= d[0]; i++)
		read_float(&x_r[i]);
	}
    }
    else {
        if (argc < 4) {
	    error_mess(_FUNC_MESS + 20);
	    return 1;
	}
    	x_r[1] = convert_float(argv[2]);
	x_r[2] = convert_float(argv[3]);
	x_r[3] = x_r[1] + (x_r[7] - 1.)* x_r[2];
    }
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command of fixation of the values of a x-range in double
    precision (fix_xr_f).
---------------------------------------------------------------------------*/
int
fix_xrange_d(int argc, char *argv[])
{
    int		    i,
		    i0,
		    iw,
		    type,
		   *d;
    double	   *x_r;
    float	    x;

    iw = sketch_obj_restr(argv[1], &i0, _XRANGE_D);
    if (iw != _XRANGE_D) {
	error_mess(_FUNC_MESS + 1);
        return 1;
    }
    x_r = (double *) Obj[_XRANGE_D - 1][i0].adresse;
    type = x_r[0];
    if (type == 0) {
        d = (int *) Obj[_XRANGE_D - 1][i0].dim;
        if (argc == 4) {
	    iw = convert_int(argv[2]);
            if (iw < 1 || iw > d[0]) {
	        error_mess(_FUNC_MESS + 2);
		return 1;
	    }
	    x_r[iw] = convert_float(argv[3]);
	    return 0;
	}
	else {
	    for (i = 1; i <= d[0]; i++) {
		read_float(&x);
		x_r[i] = x;
	    }
	}
    }
    else {
        if (argc < 4) {
	    error_mess(_FUNC_MESS + 20);
	    return 1;
	}
	x_r[1] = convert_float(argv[2]);
	x_r[2] = convert_float(argv[3]);
	x_r[3] = x_r[1] + (x_r[7] - 1.)* x_r[2];
    }
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to set the value of a real function at a point
    (fix_func_R).
---------------------------------------------------------------------------*/
int 
fix_func_R(int argc, char *argv[])
{
    int		    i0,
		    iw,
		    n,
		    nb;
    funct_f	   *a_f;
    funct_d 	   *a_d;
    char	  **c;

    a_f = NULL;
    a_d = NULL;
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F);
    if (iw == 0)
        iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D);
    if (iw != _RFUNC_F && iw != _RFUNC_D) {
	error_mess(_FUNC_MESS + 7);
	return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _RFUNC_F) {
	a_f = (funct_f *) c[0];
	nb = a_f->nb;
    }
    else {   
	a_d = (funct_d *) c[0];
	nb = a_d->nb;
    }
    n = convert_int(argv[2]);
    if (n < 1 || n > nb) {
	error_mess(_FUNC_MESS + 8);
	return 1;
    }
    if (iw == _RFUNC_F)
	a_f->f[n] = convert_float(argv[3]);
    else
	a_d->f[n] = convert_float(argv[3]);

    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to set the value of a complex function at a point
    (fix_func_C).
---------------------------------------------------------------------------*/
int 
fix_func_C(int argc, char *argv[])
{
    int		    i0,
		    iw,
		    n,
		    nb;
    funct_C	   *a_f;
    funct_dC 	   *a_d;
    char	  **c;

    a_f = NULL;
    a_d = NULL;
    iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F);
    if (iw != _CFUNC_F) {
 	iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D);
	if (iw != _CFUNC_D) {
	    error_mess(_FUNC_MESS + 7);
	    return 1;
	}
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _CFUNC_F) {
	a_f = (funct_C *) c[0];
	nb = a_f->nb;
    }
    else {   
	a_d = (funct_dC *) c[0];
	nb = a_d->nb;
    }
    n = convert_int(argv[2]);
    if (n < 1 || n > nb) {
	error_mess(_FUNC_MESS + 8);
	return 1;
    }
    if (iw == 3) {
	a_f->f[n].r = convert_float(argv[3]);
	a_f->f[n].i = convert_float(argv[4]);
    }
    else {
	a_d->f[n].r = convert_float(argv[3]);
	a_d->f[n].i = convert_float(argv[4]);
    }

    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to save a function (save_func).
---------------------------------------------------------------------------*/
int
sauve_func(int argc, char *argv[])
{
    int		    i0,
		    iw,
		    i,
		    nb,
		    type;
    funct_f	   *a_f;
    funct_d 	   *a_d;
    funct_C	   *a_C;
    funct_dC 	   *a_dC;
    float	   *x_r;
    double	   *x_rd;
    FILE	   *s;
    char	  **c;

    a_f = NULL;
    a_d = NULL;
    a_C = NULL;
    a_dC = NULL;
    x_r = NULL;
    x_rd = NULL;
    iw = sketch_obj(argv[1], &i0);
    if (iw != _RFUNC_F && iw != _RFUNC_D
       && iw != _CFUNC_F && iw != _CFUNC_D) {
	error_mess(_FUNC_MESS + 7);
	return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _CFUNC_F) {
        a_C = (funct_C *) c[0];
	nb = a_C->nb;
        type = a_C->type;
    }
    else {
        if (iw == _CFUNC_D) {   
	    a_dC = (funct_dC *) c[0];
	    nb = a_dC->nb;
            type = a_dC->type;
	}
	else {
	    if (iw == _RFUNC_F) {
		a_f = (funct_f *) c[0];
		nb = a_f->nb;
                type = a_f->type;
	    }
	    else {
		a_d = (funct_d *) c[0];
	    	nb = a_d->nb;
                type = a_d->type;
	    }
	}
    }
    if (iw == _RFUNC_F || iw == _CFUNC_F) {
        x_r = float_alloc1(nb);
        if (type == 0) {
	    if (iw == _RFUNC_F)
	        for (i = 1; i <= nb; i++)
		    x_r[i] = a_f->x[i];
	    else
	        for (i = 1; i <= nb; i++)
		    x_r[i] = a_C->x[i];
        }
	else {
	    if (iw == _RFUNC_F)
		for (i = 1; i <= nb; i++)
		    x_r[i] = a_f->x[1] + (i - 1) * a_f->x[2];
            else
		for (i = 1; i <= nb; i++)
		    x_r[i] = a_C->x[1] + (i - 1) * a_C->x[2];
	}
    }
    else {
	x_rd = double_alloc1(nb);
        if (type == 0) {
	    if (iw == _RFUNC_D)
	        for (i = 1; i <= nb; i++)
		    x_rd[i] = a_d->x[i];
	    else
	        for (i = 1; i <= nb; i++)
		    x_rd[i] = a_dC->x[i];
        }
	else {
	    if (iw == _RFUNC_D)
	        for (i = 1; i <= nb; i++)
		    x_rd[i] = a_d->x[1] + (i - 1) * a_d->x[2];
            else
		for (i = 1; i <= nb; i++)
		    x_rd[i] = a_dC->x[1] + (i - 1) * a_dC->x[2];
	}
    }
    s = Copen(result_rep, argv[2], "w");
    if (s == NULL) {
	error_mess(_FUNC_MESS + 9);
	return 1;
    }
    if (iw == _RFUNC_F)
	for (i = 1; i <= nb; i++)
	    fprintf(s, "%f    %f\n", x_r[i], a_f->f[i]);
    else {
	if (iw == _RFUNC_D) 
	    for (i = 1; i <= nb; i++)
	        fprintf(s, "%f    %f\n", x_rd[i], a_d->f[i]);
	else {
	    if (iw == _CFUNC_F)  
	        for (i = 1; i <= nb; i++)
	            fprintf(s, "%f    %f    %f\n", x_r[i], a_C->f[i].r, 
			a_C->f[i].i);
	    else
		for (i = 1; i <= nb; i++)
	            fprintf(s, "%f    %f    %f\n", x_rd[i], a_dC->f[i].r, 
			a_dC->f[i].i);

	}
    }

    fclose(s);
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to read a function from a file (not yet done...).
---------------------------------------------------------------------------*/
int
read_func(int argc, char *argv[])
{
    return 0;
}
/*-------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------
---------------------------------------------------------------------------*/
int
exec_func2(int argc, char *argv[], funct2_t e)
{
    int		    iw;
    char	   **c1,
		   **c2,
		   **c3;

    c1 = NULL;
    c2 = NULL;
    c3 = NULL;
    if (fix_func(&c1, &c2, &c3, &iw, argv) == 1) {
	error_mess(_FUNC_MESS + 7);
	return 1;
    }
    if (iw == _RFUNC_F) {
        if (TX_f(c1[0]) * TX_f(c2[0]) * TX_f(c3[0]) == 1)
		    e.F_f((funct_f *) c1[0], (funct_f *) c2[0], 
			(funct_f *) c3[0]);
    }
    else {
	if (iw == _RFUNC_D) {
            if (TX_d(c1[0]) * TX_d(c2[0]) * TX_d(c3[0]) == 1)
	        e.F_d((funct_d *) c1[0], (funct_d *) c2[0], (funct_d *) c3[0]);
	}
	else {
	    if (iw == _CFUNC_F) {
                if (TX_C(c1[0]) * TX_C(c2[0]) * TX_C(c3[0]) == 1)
	            e.F_C((funct_C *) c1[0], (funct_C *) c2[0], 
			(funct_C *) c3[0]);
	    }
	    else {
                if (TX_dC(c1[0]) * TX_dC(c2[0]) * TX_dC(c3[0]) == 1)
	            e.F_dC((funct_dC *) c1[0], (funct_dC *) c2[0], 
		        (funct_dC *) c3[0]);
	    }
	}
    }
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to compute the sum of two functions (add_func).
---------------------------------------------------------------------------*/
int
add_func(int argc, char *argv[])
{
    return exec_func2(argc, argv, F_ADD);
}
/*-------------------------------------------------------------------------*/





/*--------------------------------------------------------------------------
--------------------------------------------------------------------------*/
int
fix_func(char ***c1, char ***c2, char ***c3, int *iw, char *argv[])
{
    int 	    iw0,
		    iw1,
		    iw2,
		    i00,
		    i01,
		    i02;

    iw0 = sketch_obj_restr(argv[1], &i00, _RFUNC_F);
    if (iw0 != _RFUNC_F) {
	iw0 = sketch_obj_restr(argv[1], &i00, _RFUNC_D);
        if (iw0 != _RFUNC_D) {
	    iw0 = sketch_obj_restr(argv[1], &i00, _CFUNC_F);
	    if (iw0 != _CFUNC_F) {
		iw0 = sketch_obj_restr(argv[1], &i00, _CFUNC_D);
                if (iw0 != _CFUNC_D) {
		    error_mess(_FUNC_MESS + 7);
		    return 1;
		}
	    }
	}
    }
    iw1 = sketch_obj_restr(argv[2], &i01, iw0);
    if (iw0 != iw1)
	return 1;
    iw2 = sketch_obj_restr(argv[3], &i02, iw0);
    if (iw0 != iw2)
	return 1;
    c1[0] = (char **) Obj[iw0 - 1][i00].adresse;
    c2[0] = (char **) Obj[iw0 - 1][i01].adresse;
    c3[0] = (char **) Obj[iw0 - 1][i02].adresse;
    iw[0] = iw0;
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to compute the difference of two functions (sub_func).
---------------------------------------------------------------------------*/
int
sub_func(int argc, char *argv[])
{
    return exec_func2(argc, argv, F_SUB);
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to compute the product of two functions (mul_func).
---------------------------------------------------------------------------*/
int
mul_func(int argc, char *argv[])
{
    return exec_func2(argc, argv, F_MUL);
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to compute the quotient of two functions (div_func).
---------------------------------------------------------------------------*/
int
div_func(int argc, char *argv[])
{
    return exec_func2(argc, argv, F_DIV);
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to describe a function (desc_func).
---------------------------------------------------------------------------*/
int
desc_func(int argc, char *argv[])
{
    int		    iw,
		    i0,
		    iw0,
		    i00,
		    nb,
                    type;
    char	  **c,
		   *nom_x,
		    h[100];
    funct_f 	   *a_f;
    funct_d 	   *a_d;
    funct_C 	   *a_C;
    funct_dC 	   *a_dC;
    double	    xmin,
		    xmax;

    iw = sketch_obj(argv[1], &i0);
    if (iw < _RFUNC_F || iw > _CFUNC_D) {
	error_mess(_FUNC_MESS + 7);
	return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;

    if (iw == _RFUNC_F) {
	print("%s", mess[_FUNC_MESS + 10]);
	a_f = (funct_f *) c[0];
        nb = a_f->nb;
        type = a_f->type;
        nom_x = a_f->nom;
        xmin = (double) a_f->x[1];
	if (type == 1) 
            xmax = (double) a_f->x[3];
	else 
	    xmax = (double) a_f->x[nb];
    }
    else {
	if (iw == _RFUNC_D) {
	    print("%s", mess[_FUNC_MESS + 11]);
	    a_d = (funct_d *) c[0];
            nb = a_d->nb;
            type = a_d->type;
            nom_x = a_d->nom;
            xmin = a_d->x[1];
	    if (type == 1) 
                xmax = a_d->x[3];
	    else 
	        xmax = a_d->x[nb];
        }
	else {
	    if (iw == _CFUNC_F) {
	        print("%s", mess[_FUNC_MESS + 12]);
	        a_C = (funct_C *) c[0];
                nb = a_C->nb;
                type = a_C->type;
                nom_x = a_C->nom;
                xmin = (double) a_C->x[1];
 	        if (type == 1) 
                    xmax = (double) a_C->x[3];
	        else 
                    xmax = (double) a_C->x[nb];
            }
	    else {
	        print("%s", mess[_FUNC_MESS + 13]);
	        a_dC = (funct_dC *) c[0];
                nb = a_dC->nb;
                type = a_dC->type;
                nom_x = a_dC->nom;
                xmin = a_dC->x[1];
	        if (type == 1) 
                    xmax = a_dC->x[3];
	        else 
                    xmax = a_dC->x[nb];
	    }
	}
    }
    iw0 = sketch_obj(nom_x, &i00);
    if (((iw == _RFUNC_F || iw == _CFUNC_F) && iw0 != _XRANGE_F) || 
	((iw == _RFUNC_D || iw == _CFUNC_D) && iw0 != _XRANGE_D)) {
	error_mess(_FUNC_MESS + 1);
	return 1;
    }
    print("%s  %d\n", mess[_FUNC_MESS + 3], type);
    print("%s  %d\n", mess[_FUNC_MESS + 4], nb);
    print("%s  %s\n", mess[_FUNC_MESS + 14], nom_x);
    print("%s  %f\n", mess[_FUNC_MESS + 21], xmin);
    print("%s  %f\n", mess[_FUNC_MESS + 22], xmax);
    memset(h, 0, 100);
    sprintf(h, "xmin=%f", xmin);
    convert_float(h);
    memset(h, 0, 100);
    sprintf(h, "xmax=%f", xmax);
    convert_float(h);
    return 0;
}
/*-------------------------------------------------------------------------*/





/*--------------------------------------------------------------------------
--------------------------------------------------------------------------*/
int
exec_func(int argc, char *argv[], funct_t e)
{
    int		    iw,
		    i0,
		    iw0,
		    i00;
    char	  **c,
		  **c0;

    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F);
    if (iw != _RFUNC_F) {
	iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D);
        if (iw != _RFUNC_D) {
	    iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F);
	    if (iw != _CFUNC_F) {
		iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D);
                if (iw != _CFUNC_D) {
		    error_mess(_FUNC_MESS + 7);
		    return 1;
		}
	    }
	}
    }
    iw0 = sketch_obj_restr(argv[2], &i00, iw);
    if (iw0 != iw) {
	error_mess(_FUNC_MESS + 15);
	return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    c0  = (char **) Obj[iw0 - 1][i00].adresse;
    if (iw == _RFUNC_F) {
	if (TX_f(c[0]) * TX_f(c0[0]) == 1)
	    e.F_f((funct_f *) c[0], (funct_f *) c0[0]);
    }
    else {
	if (iw == _RFUNC_D) {
	    if (TX_d(c[0]) * TX_d(c0[0]) == 1)
	        e.F_d((funct_d *) c[0], (funct_d *) c0[0]);
        }
	else {
	    if (iw == _CFUNC_F) { 
	        if (TX_C(c[0]) * TX_C(c0[0]) == 1)
	            e.F_C((funct_C *) c[0], (funct_C *) c0[0]);
	    }
	    else { 
	        if (TX_dC(c[0]) * TX_dC(c0[0]) == 1)
	            e.F_dC((funct_dC *) c[0], (funct_dC *) c0[0]);
	    }
	}
    }

    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to copy a function to another (copy_func).
---------------------------------------------------------------------------*/
int
copy_func(int argc, char *argv[])
{
    return exec_func(argc, argv, F_COPY);
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to compute the product of a function with a number (rmul).
---------------------------------------------------------------------------*/
int
Rmul_func(int argc, char *argv[])
{
    int		    iw,
		    i0,
		    iw0,
		    i00,
		    ik;
    char	  **c,
		  **c0;
    double	    t,
		    ti,
		    tr;

    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F);
    if (iw != _RFUNC_F) {
	iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D);
        if (iw != _RFUNC_D) {
	    iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F);
	    if (iw != _CFUNC_F) {
		iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D);
                if (iw != _CFUNC_D) {
		    error_mess(_FUNC_MESS + 7);
		    return 1;
		}
	    }
	}
    }
    if (iw > _RFUNC_D && argc < 5) {
	error_mess(_FUNC_MESS + 6);
	return 1;
    }
    t = 0.;
    ti = 0.;
    tr = 0.;
    if (iw < _CFUNC_F) {
	t = convert_float(argv[2]);
        ik = 0;
    }
    else {
	tr = convert_float(argv[2]);
	ti = convert_float(argv[3]);
	ik = 1;
    }
    iw0 = sketch_obj_restr(argv[3 + ik], &i00, iw);
    if (iw0 != iw) {
	error_mess(15);
	return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    c0  = (char **) Obj[iw0 - 1][i00].adresse;
    if (iw == _RFUNC_F) {
	if (TX_f(c[0]) * TX_f(c0[0]) == 1)
	    f_funct_Rmul((float) t, (funct_f *) c[0], (funct_f *) c0[0]);
    }
    else {
	if (iw == _RFUNC_D) { 
	    if (TX_d(c[0]) * TX_d(c0[0]) == 1)
	        d_funct_Rmul(t, (funct_d *) c[0], (funct_d *) c0[0]);
   	}
	else {
	    if (iw == _CFUNC_F) {
	        if (TX_C(c[0]) * TX_C(c0[0]) == 1)
	            C_funct_Rmul(Complex((float) tr, (float) ti), 
		        (funct_C *) c[0], (funct_C *) c0[0]);
	    }
	    else { 
	        if (TX_dC(c[0]) * TX_dC(c0[0]) == 1)
	            dC_funct_Rmul(dComplex(tr, ti), (funct_dC *) c[0], 
		        (funct_dC *) c0[0]);
	    }
	}
    }
    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to put the real or imaginary part of a complex function
    in a real one (real or imag).
---------------------------------------------------------------------------*/
int
reel_imag_Cfunc(int argc, char *argv[])
{
    int		    iw,
		    i0,
		    iw0,
		    i00;
    char	  **c,
		  **c0;

    iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F);
    if (iw != _CFUNC_F) {
	iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D);
    	if (iw != _CFUNC_D) {
	    error_mess(_FUNC_MESS + 7);
	    return 1;
	}
    }
    iw0 = sketch_obj_restr(argv[2], &i00, iw - 2);
    if (iw0 != iw - 2) {
	error_mess(_FUNC_MESS + 7);
	return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    c0  = (char **) Obj[iw0 - 1][i00].adresse;
    if (iw0 == _RFUNC_F) {
	if (TX_C(c[0]) * TX_f(c0[0]) == 1) {
            if (comp(argv[0], "real") == 1)
	        reel_f((funct_C *) c[0], (funct_f *) c0[0]);
	    else
	        imag_f((funct_C *) c[0], (funct_f *) c0[0]);
	}
    }
    else {
	if (TX_dC(c[0]) * TX_d(c0[0]) == 1) {
            if (comp(argv[0], "real") == 1)
	        reel_d((funct_dC *) c[0], (funct_d *) c0[0]);
	    else
	        imag_d((funct_dC *) c[0], (funct_d *) c0[0]);
	}
    }

    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to put a real function in the real part or the imaginary
    part of a complex one (imag_fix or real_fix).
---------------------------------------------------------------------------*/
int
reel_imag_fix_func(int argc, char *argv[])
{
    int		    iw,
		    i0,
		    iw0,
		    i00;
    char	  **c,
		  **c0;

    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F);
    if (iw != _RFUNC_F) {
	iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D);
        if (iw != _RFUNC_D) {
	    error_mess(_FUNC_MESS + 7);
	    return 1;
	}
    }
    iw0 = sketch_obj_restr(argv[2], &i00, iw + 2);
    if (iw0 != iw + 2) {
	error_mess(_FUNC_MESS + 7);
	return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    c0  = (char **) Obj[iw0 - 1][i00].adresse;
    if (iw0 == _RFUNC_F) {
	if (TX_f(c[0]) * TX_C(c0[0]) == 1) {
            if (comp(argv[0], "real_fix") == 1)
	        conv_f_Cr((funct_f *) c[0], (funct_C *) c0[0]);
	    else
	        conv_f_Ci((funct_f *) c[0], (funct_C *) c0[0]);
	}
    }
    else {
	if (TX_d(c[0]) * TX_dC(c0[0]) == 1) {
            if (comp(argv[0], "real_fix") == 1)
	        conv_d_dCr((funct_d *) c[0], (funct_dC *) c0[0]);
	    else
	        conv_d_dCi((funct_d *) c[0], (funct_dC *) c0[0]);
	}
    }

    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command of integration of a function (integ)
---------------------------------------------------------------------------*/
int
integ_func(int argc, char *argv[])
{
    return exec_func(argc, argv, F_INTEG);
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command of derivation of a function (diff)
---------------------------------------------------------------------------*/
int
diff_func(int argc, char *argv[])
{
    return exec_func(argc, argv, F_DIFF);
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command of composition of 2 functions (comp_func)
---------------------------------------------------------------------------*/
int
comp_func(int argc, char *argv[])
{
    int 	    iw,
		    i0,
		    iw0,
		    i00,
                    iw1,
		    i01,
		    iw2;
    char	  **c,
		  **c0,
		  **c1;
    
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F);
    if (iw != _RFUNC_F) 
	iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D);
    if (iw > 0)
        iw0 = sketch_obj_restr(argv[2], &i00, iw);
    else {
	iw0 = sketch_obj_restr(argv[2], &i00, _RFUNC_F);
        if (iw0 != _RFUNC_F)
	    iw0 = sketch_obj_restr(argv[2], &i00, _RFUNC_D);
    }
    if (iw > 0)
	iw2 = iw;
    else {
	if (iw0 > 0)
	    iw2 = iw0;
	else
	    iw2 = 0;
    }
    if (iw2 > 0) {
        iw1 = sketch_obj_restr(argv[3], &i01, iw2);
	if (iw1 != iw2) {
	    error_mess(_FUNC_MESS + 7);
	    return 1;
	}
    }
    else {
	iw1 = sketch_obj_restr(argv[3], &i01, _RFUNC_F);
	if (iw1 == _RFUNC_F) {
	    iw1 = sketch_obj_restr(argv[3], &i01, _RFUNC_D);
	    if (iw1 != _RFUNC_D) {
		error_mess(_FUNC_MESS + 7);
		return 1;
	    }
	}
    }
    c = NULL;
    c0 = NULL;
    if (iw > 0)
        c = (char **) Obj[iw - 1][i0].adresse;
    if (iw0 > 0)
        c0 = (char **) Obj[iw0 - 1][i00].adresse;
    c1 = (char **) Obj[iw1 - 1][i01].adresse;
    if (iw == _RFUNC_F && iw0 == iw) {
	if (TX_f(c[0]) * TX_f(c0[0]) * TX_f(c1[0]) == 1)
	    comp_funct_f((funct_f *) c[0], (funct_f *) c0[0], 
		(funct_f *) c1[0]);
    }
    else {
        if (iw == _RFUNC_D && iw0 == iw) {
	    if (TX_d(c[0]) * TX_d(c0[0]) * TX_d(c1[0]) == 1)
	        comp_funct_d((funct_d *) c[0], (funct_d *) c0[0], 
		    (funct_d *) c1[0]);
        }
  	else {
            if (iw == 0 && iw0 == _RFUNC_F) {
	        if (TX_f(c0[0]) * TX_f(c1[0]) == 1)
	            comp_functb_f(argv[1], (funct_f *) c0[0], 
			(funct_f *) c1[0]);
	    }
	    else {
    		if (iw == 0 && iw0 == _RFUNC_D) {
	            if (TX_d(c0[0]) * TX_d(c1[0]) == 1)
		    comp_functb_d(argv[1], (funct_d *) c0[0], 
			(funct_d *) c1[0]);
                }
		else {
    		    if (iw == _RFUNC_F && iw0 == 0) {
	                if (TX_f(c[0]) * TX_f(c1[0]) == 1)
			    comp_functc_f((funct_f *) c[0], argv[2], 
			        (funct_f *) c1[0]);
		    }
		    else {
    			if (iw == 0 && iw0 == _RFUNC_D) {
	                    if (TX_d(c[0]) * TX_d(c1[0]) == 1)
			        comp_functc_d((funct_d *) c[0], argv[2], 
				    (funct_d *) c1[0]);
			}
			else {
    			    if (iw == 0 && iw0 == 0 && iw1 == _RFUNC_F) {
	                        if (TX_f(c1[0]) == 1)
				    comp_functd_f(argv[1], argv[2], 
				        (funct_f *) c1[0]);
			    }
     			    else { 
	                        if (TX_d(c1[0]) == 1)
				    comp_functd_d(argv[1], argv[2], 
				        (funct_d *) c1[0]);
			    }
			}
		    }
		}
	    }
	}
    }

    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to fill a real function with a mathematical formula 
    (fill_func).
---------------------------------------------------------------------------*/
int
fill_func(int argc, char *argv[])
{
    int	 	    iw,
		    i0,
		    i,
		    j,
                    j0;
    float	   *x_r;
    double	   *x_rd;
    char	  **c,
		    h[100];
    funct_f	   *a_f;
    funct_d	   *a_d;

    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F);
    if (iw != _RFUNC_F) {
	iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D);
        if (iw != _RFUNC_D) {
	    error_mess(_FUNC_MESS + 7);
	    return 1;
	}
    }
    j0 = -1;

    for (j = 0; j < _NBFONC; j++) {
	if (comp(argv[2], Funcs[j].name) == 1) {
	    j0 = j;
            break;
        }
    }

    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _RFUNC_F) {
        if (TX_f(c[0]) == 1) {
            a_f = (funct_f *) c[0];
	    f_fill(a_f, &x_r);
            if (j0 >= 0) {
                for (i = 1; i <= a_f->nb; i++)
	            a_f->f[i] = Funcs[j0].func(x_r[i]);
            }
	    else {
                for (i = 1; i <= a_f->nb; i++) {
		    memset(h, 0, 100);
		    sprintf(h, "x=%f", x_r[i]);
	            convert_float(h);
		    a_f->f[i] = convert_float(argv[2]);
	        }
 	    }
	    if (a_f->type > 0)
	        XFREE(x_r);
	}
    }
    else {
        if (TX_d(c[0]) == 1) {
            a_d = (funct_d *) c[0];
	    d_fill(a_d, &x_rd);
            if (j0 >= 0) {
                for (i = 1; i <= a_d->nb; i++)
	            a_d->f[i] = Funcs[j0].func(x_rd[i]);
            }
	    else {
                for (i = 1; i <= a_d->nb; i++) {
		    memset(h, 0, 100);
		    sprintf(h, "x=%f", x_rd[i]);
	            convert_float(h);
		    a_d->f[i] = convert_float(argv[2]);
	        }
 	    }
	    if (a_d->type > 0)
	        XFREE(x_rd);
	}
    }

    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to give a constant value to a function (const_func).
---------------------------------------------------------------------------*/
int
const_func(int argc, char *argv[])
{
    int 	    iw,
		    i0;
    char 	  **c;
    fcomplex	    z;
    dcomplex	    zd;

    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F);
    if (iw != _RFUNC_F) {
	iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D);
	if (iw != _RFUNC_D) {
	    iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F);
	if (iw != _CFUNC_F)
	    iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D);
	}
    }
    if (iw == 0) {
	error_mess(_FUNC_MESS + 7);
	return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _RFUNC_F) {
	if (TX_f(c[0]) == 1)
	    f_funct_const(convert_float(argv[2]), (funct_f *) c[0]);
    }
    else {
	if (iw == _RFUNC_D) {
	    if (TX_d(c[0]) == 1)
	        d_funct_const((double) convert_float(argv[2]), 
		    (funct_d *) c[0]);
	}
	else {
	    if (iw == _CFUNC_F) {
	        if (TX_C(c[0]) == 1) {
		    z = Complex(convert_float(argv[2]), convert_float(argv[3]));
	            C_funct_const(z, (funct_C *) c[0]);
		}
	    }
	    else {
	        if (TX_dC(c[0]) == 1) {
		    zd = dComplex((double) convert_float(argv[2]),
 		        (double) convert_float(argv[3]));
	            dC_funct_const(zd, (funct_dC *) c[0]);
		}
	    }
	}
    }

    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to print the value of a function at a point (val_func).
---------------------------------------------------------------------------*/
int
val_func(int argc, char *argv[])
{
    int 	    iw,
		    i0;
    char	  **c;
    float	    x;
    double	    xx,
		    y;
    fcomplex	    z;
    dcomplex	    zz;

    y = convert_float(argv[2]);
    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F);
    if (iw != _RFUNC_F) {
	iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D);
        if (iw != _RFUNC_D) {
	    iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F);
            if (iw != _CFUNC_F)
	        iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D);
	}
    }
    if (iw == 0) {
	error_mess(_FUNC_MESS + 7);
	return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _CFUNC_F) {
        if (TX_C(c[0]) == 1) {
	    z = C_funct_eval((funct_C *) c[0], (float) y);
            xx = z.r;
	    SetValue("func_r", &xx);
	    xx = z.i;
	    SetValue("func_i", &xx);
            print("%f + %f.I\n", z.r, z.i);
	}
    }
    else {
	if (iw == _CFUNC_D) {
            if (TX_dC(c[0]) == 1) {
	        zz = dC_funct_eval((funct_dC *) c[0], y);
	        SetValue("func_r", &zz.r);
	        SetValue("func_i", &zz.i);
                print("%f + %f.I\n", zz.r, zz.i);
	    }
	}
	else {
	    if (iw == _RFUNC_F) {
                if (TX_f(c[0]) == 1) {
		    x = f_funct_eval((funct_f *) c[0], (float) y);
		    xx = x;
	            SetValue("func", &xx);
                    print("%f\n", x);
		}
	    }
	    else {
                if (TX_d(c[0]) == 1) {
		    xx = d_funct_eval((funct_d *) c[0], y);
	            SetValue("func", &xx);
                    print("%f\n", xx);
		}
	    }
	}
    }

    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to compute the 'generalized inverse' of a real function
---------------------------------------------------------------------------*/
int
inv_func_cmd(int argc, char *argv[])
{
    int		    i0,
		    i1,
		    iw0,
		    iw1;
    funct_f	   *a_f,
		   *b_f;
    funct_d	   *a_d,
		   *b_d;
    char 	  **c;

    iw0 = sketch_obj_restr(argv[1], &i0, _RFUNC_F);
    if (iw0 != _RFUNC_F) 
	iw0 = sketch_obj_restr(argv[1], &i0, _RFUNC_D);
    if (iw0 == 0) {
	error_mess(_FUNC_MESS + 7);
	return 1;
    }
    iw1 = sketch_obj_restr(argv[2], &i1, iw0);
    if (iw0 != iw1) {
	error_mess(_FUNC_MESS + 15);
	return 1;
    }
    if (iw0 == _RFUNC_F) {
        c = (char **) Obj[iw0 - 1][i0].adresse;
	if (TX_f(c[0]) == 1) {
	    a_f = (funct_f *) c[0];
            c = (char **) Obj[iw0 - 1][i1].adresse;
	    if (TX_f(c[0]) == 1) {
	        b_f = (funct_f *) c[0];
	        inv_sup_funct_f(a_f, b_f);
	    }
	}
    }
    else {
        c = (char **) Obj[iw0 - 1][i0].adresse;
	if (TX_d(c[0]) == 1) {
	    a_d = (funct_d *) c[0];
            c = (char **) Obj[iw0 - 1][i1].adresse;
	    if (TX_d(c[0]) == 1) {
	        b_d = (funct_d *) c[0];
	        inv_sup_funct_d(a_d, b_d);
	    }
	}
    }

    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to compute the maximum of a function
---------------------------------------------------------------------------*/
int
Max_funct_cmd(int argc, char *argv[])
{
    int 	    iw,
		    i0;
    char	  **c;
    float	    xmax_f,
		    fmax_f,
		    Rmax_C;
    double	    xmax_d,
		    fmax_d,
		    Rmax_dC;
    fcomplex	    fmax_C;
    dcomplex	    fmax_dC;

    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F);
    if (iw != _RFUNC_F) {
	iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D);
        if (iw != _RFUNC_D) {
	    iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F);
            if (iw != _CFUNC_F)
	        iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D);
	}
    }
    if (iw == 0) {
	error_mess(_FUNC_MESS + 7);
	return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _CFUNC_F) {
        if (TX_C(c[0]) == 1) {
	    Max_funct_C((funct_C *) c[0], &xmax_f, &fmax_C, &Rmax_C);
	    xmax_d = xmax_f;
	    SetValue("xmax_func" , &xmax_d);
	    fmax_d = fmax_C.r;
	    SetValue("fmax_func_r", &fmax_d);
	    fmax_d = fmax_C.i;
	    SetValue("fmax_func_i", &fmax_d);
	    fmax_d = Rmax_C;
	    SetValue("fmax_func_R", &fmax_d);
	    print("xmax = %f\n", xmax_f);
	    print("fmax = %f + %f.I\n", fmax_C.r, fmax_C.i);
	    print("Rmax = %f\n", Rmax_C);
	}
    }
    else {
	if (iw == _CFUNC_D) {
            if (TX_dC(c[0]) == 1) {
	        Max_funct_dC((funct_dC *) c[0], &xmax_d, &fmax_dC, &Rmax_dC);
	        SetValue("xmax_func" , &xmax_d);
	        SetValue("fmax_func_r", &fmax_dC.r);
	        SetValue("fmax_func_i", &fmax_dC.i);
	        SetValue("fmax_func_R", &Rmax_dC);
	        print("xmax = %f\n", xmax_d);
	        print("fmax = %f + %f.I\n", fmax_dC.r, fmax_dC.i);
	        print("Rmax = %f\n", Rmax_dC);
	    }
	}
	else {
	    if (iw == _RFUNC_F) {
                if (TX_f(c[0]) == 1) {
	      	    Max_funct_f((funct_f *) c[0], &xmax_f, &fmax_f);
	    	    xmax_d = xmax_f;
	    	    SetValue("xmax_func" , &xmax_d);
	    	    fmax_d = fmax_f;
	    	    SetValue("fmax_func", &fmax_d);
		    print("xmax = %f\n", xmax_f);
		    print("fmax = %f\n", fmax_f);
		}
	    }
	    else {
                if (TX_d(c[0]) == 1) {
	      	    Max_funct_d((funct_d *) c[0], &xmax_d, &fmax_d);
	    	    SetValue("xmax_func" , &xmax_d);
	    	    SetValue("fmax_func", &fmax_d);
		    print("xmax = %f\n", xmax_d);
		    print("fmax = %f\n", fmax_d);
		}
	    }
	}
    }

    return 0;
}
/*-------------------------------------------------------------------------*/





/*---------------------------------------------------------------------------
    Command used to compute the minimum of a function
---------------------------------------------------------------------------*/
int
Min_funct_cmd(int argc, char *argv[])
{
    int 	    iw,
		    i0;
    char	  **c;
    float	    xmin_f,
		    fmin_f,
		    Rmin_C;
    double	    xmin_d,
		    fmin_d,
		    Rmin_dC;
    fcomplex	    fmin_C;
    dcomplex	    fmin_dC;

    iw = sketch_obj_restr(argv[1], &i0, _RFUNC_F);
    if (iw != _RFUNC_F) {
	iw = sketch_obj_restr(argv[1], &i0, _RFUNC_D);
        if (iw != _RFUNC_D) {
	    iw = sketch_obj_restr(argv[1], &i0, _CFUNC_F);
            if (iw != _CFUNC_F)
	        iw = sketch_obj_restr(argv[1], &i0, _CFUNC_D);
	}
    }
    if (iw == 0) {
	error_mess(_FUNC_MESS + 7);
	return 1;
    }
    c = (char **) Obj[iw - 1][i0].adresse;
    if (iw == _CFUNC_F) {
        if (TX_C(c[0]) == 1) {
	    Min_funct_C((funct_C *) c[0], &xmin_f, &fmin_C, &Rmin_C);
	    xmin_d = xmin_f;
	    SetValue("xmin_func" , &xmin_d);
	    fmin_d = fmin_C.r;
	    SetValue("fmin_func_r", &fmin_d);
	    fmin_d = fmin_C.i;
	    SetValue("fmin_func_i", &fmin_d);
	    fmin_d = Rmin_C;
	    SetValue("fmin_func_R", &fmin_d);
	    print("xmin = %f\n", xmin_f);
	    print("fmin = %f + %f.I\n", fmin_C.r, fmin_C.i);
	    print("Rmin = %f\n", Rmin_C);
	}
    }
    else {
	if (iw == _CFUNC_D) {
            if (TX_dC(c[0]) == 1) {
	        Min_funct_dC((funct_dC *) c[0], &xmin_d, &fmin_dC, &Rmin_dC);
	        SetValue("xmin" , &xmin_d);
	        SetValue("fmin_func_r", &fmin_dC.r);
	        SetValue("fmin_func_i", &fmin_dC.i);
	        SetValue("fmin_func_R", &Rmin_dC);
	        print("xmin = %f\n", xmin_d);
	        print("fmin = %f + %f.I\n", fmin_dC.r, fmin_dC.i);
	        print("Rmin = %f\n", Rmin_dC);
	    }
	}
	else {
	    if (iw == _RFUNC_F) {
                if (TX_f(c[0]) == 1) {
	      	    Min_funct_f((funct_f *) c[0], &xmin_f, &fmin_f);
	    	    xmin_d = xmin_f;
	    	    SetValue("xmin_func" , &xmin_d);
	    	    fmin_d = fmin_f;
	    	    SetValue("fmin_func", &fmin_d);
		    print("xmin = %f\n", xmin_f);
		    print("fmin = %f\n", fmin_f);
		}
	    }
	    else {
                if (TX_d(c[0]) == 1) {
	      	    Min_funct_d((funct_d *) c[0], &xmin_d, &fmin_d);
	    	    SetValue("xmin_func" , &xmin_d);
	    	    SetValue("fmin_func", &fmin_d);
		    print("xmin = %f\n", xmin_d);
		    print("fmin = %f\n", fmin_d);
		}
	    }
	}
    }

    return 0;
}
/*-------------------------------------------------------------------------*/
