/* Copyright 1995-96 Jon Griffiths.  See the file "jlib.doc" for details. */

#include <stdio.h>
#include <stdlib.h>
#include <jlib.h>
#include "lz.h"
#include "rle.h"

/* magic number that identifies jlib files */
#define JLB_MAGIC_NUMBER 0xd00d


image * image_load_jlb(char *filename) 
{
 FILE * infile;
 image * timg;
 USHORT magic_number;
 long comp_length,filler,rlen;
 UBYTE *sp,*rp;
 int i;
 ULONG thirtytwobit;
 USHORT t_ushort;
 
 JLIB_ENTER("image_load_jlb");

   
 /* grab mem for header record */ 
 if((timg = (image *) malloc (sizeof (image))) == NULL){
      jlib_exit(jlib_msg(JLIB_EMALLOC));
 }
      
  /* grab mem for palette */ 
  if((timg->palette = (UBYTE *) malloc (768 * sizeof (UBYTE))) == NULL){
      jlib_exit(jlib_msg(JLIB_EMALLOC));
  }
   
      
  /* Open the file */ 
  if((infile = fopen (filename, "rb")) == NULL){
    JLIB_SPRINTF("%s File Open Error.",filename);
    free (timg->palette);
    
    JLIB_LEAVE;
    return (image *)NULL;
  }
   
      
  /* read in header information */ 
  fseek (infile, 0L, 0);

  /* read magic number */ 
  jio_read_elementary_type(infile,&magic_number, sizeof(USHORT));

  /* check magic number */ 
  if(magic_number != JLB_MAGIC_NUMBER){
     JLIB_SPRINTF("%s Not .JLB File.",filename);
     free (timg->palette);
     free (timg);
     
     JLIB_LEAVE;
     return (image *)NULL;
   }

  /* get width + height */
  jio_read_elementary_type(infile,&t_ushort, sizeof(USHORT));
  timg->width = (int)t_ushort;
  jio_read_elementary_type(infile,&t_ushort, sizeof(USHORT));
  timg->height = (int)t_ushort;

  JLIB_SPRINTF("JLB Width: %d.",timg->width);
  JLIB_SPRINTF("JLB Height: %d.",timg->height);

  /* get length of rle data */
  jio_read_elementary_type(infile,&thirtytwobit, sizeof(ULONG));

  rlen = (long)thirtytwobit;

  /* get length of compressed data */
  jio_read_elementary_type(infile,&thirtytwobit, sizeof(ULONG));

  comp_length = (long)thirtytwobit;

  /* grab memory to hold compressed data */
  if((sp = (UBYTE *) malloc ((comp_length) * sizeof (UBYTE))) == NULL){
    free (timg->palette);
    free (timg->data);
    free (timg);

      jlib_exit(jlib_msg(JLIB_EMALLOC));
   }

  /* grab memory to hold rle data */
  if((rp = (UBYTE *) malloc (rlen )) == NULL){
     free (timg->palette);
     free (timg->data);
     free (timg);

      jlib_exit(jlib_msg(JLIB_EMALLOC));
   }
   
   /* read compressed data into compressed buffer */
   for(i=0;i<comp_length;i++){
       sp[i]=(UBYTE)fgetc(infile);
   }
   
   /* uncompress from the compressed buffer to the rle buffer */  
   lz_decompress(sp,comp_length,rp,&filler);

  /* run length decode data into image buffer */
  rlen = rle_decode(rp,&timg->data);

  
   /* free the memory for the compressed buffer */
   free(sp);
   
  /* get length of compressed palette data */
  jio_read_elementary_type(infile,&thirtytwobit, sizeof(ULONG));

  comp_length = (long)thirtytwobit;
   
  /* grab memory to hold compressed palette data */
  if((sp = (UBYTE *) malloc ((comp_length) * sizeof (UBYTE))) == NULL){
     free (timg->palette);
     free (timg->data);
     free (timg);

      jlib_exit(jlib_msg(JLIB_EMALLOC));
   }


   /* read compressed pal data into compressed pal buffer */
   for(i=0;i<comp_length;i++){
       sp[i]=(UBYTE)fgetc(infile);
   }
   
   /* uncompress from the compressed pal buff to the uncompressed buffer */  
   lz_decompress(sp,comp_length,timg->palette,&filler);

   /* free the memory for the compressed buffer */
   free(sp);
      
   fclose (infile);
   
   image_setup(timg);

   JLIB_LEAVE;

   image_setup(timg);

   /* return a pointer to the image */
   return timg;
}


int image_save_jlb(char *filename,buffer_rec *buff,UBYTE *pal) 
{
 FILE * outfile;
 USHORT magic_number=JLB_MAGIC_NUMBER;
 long *comp_length,rlen;
 UBYTE *sp,*rp;
 int i; 
 ULONG thirtytwobit;
 USHORT t_ushort;
 
 JLIB_ENTER("image_save_jlb");

   
  /* Open the file */ 
  if((outfile = fopen (filename, "wb")) == NULL){
     JLIB_SPRINTF("%s File Open Error.",filename);
     JLIB_LEAVE
     return 0;
  }
   
  /* grab memory to hold compressed length */
  if((comp_length = (long *) malloc (sizeof(long))) == NULL){
      jlib_exit(jlib_msg(JLIB_EMALLOC));
  }
      
  fseek (outfile, 0L, 0);

  /* write magic number */ 
  jio_write_elementary_type(outfile,&magic_number, sizeof(USHORT));

  /* put width + height */
  t_ushort = (USHORT)buff->width;
  jio_write_elementary_type(outfile,&t_ushort, sizeof(USHORT));
  t_ushort = (USHORT)buff->height;
  jio_write_elementary_type(outfile,&t_ushort, sizeof(USHORT));

  JLIB_SPRINTF("Buff Width: %d.",buff->width);
  JLIB_SPRINTF("Buff Height: %d.",buff->height);
  
  /* run length encode data into rp */
  rlen = rle_encode(B_BUFF_PTR(buff),B_SIZE(buff),&rp);

  /* put length of rle data */
  thirtytwobit = (ULONG)rlen;
  
  jio_write_elementary_type(outfile,&rlen, sizeof(ULONG));
  
  /* grab memory to hold compressed rle data */
  if((sp = (UBYTE *) malloc (rlen+256)) == NULL){
      jlib_exit(jlib_msg(JLIB_EMALLOC));
   }
   
   /* compress from the rle buffer to the compressed buffer */  
   lz_compress(rp,rlen,sp,comp_length);

  /* free the rle buffer */
  free(rp);
  
  /* put length of compressed data */
  thirtytwobit = (ULONG)*comp_length;
  
  jio_write_elementary_type(outfile,&thirtytwobit, sizeof(ULONG));

   /* write compressed data into the file */
   for(i=0;i< *comp_length;i++){
       fputc(sp[i],outfile);
   }

  /* free the memory for the compressed buffer */
  free(sp);
  
  /* grab memory to hold compressed palette data */
  if((sp = (UBYTE *) malloc (256+768)) == NULL){
      jlib_exit(jlib_msg(JLIB_EMALLOC));
   }

  /* compress from the uncompressed pal buff to the compressed buffer */  
  lz_compress(pal,768,sp,comp_length);

  /* put length of compressed palette data */
  thirtytwobit = (ULONG)*comp_length;
  
  jio_write_elementary_type(outfile,&thirtytwobit, sizeof(ULONG));

   /* write compressed data into the file */
   for(i=0;i<*comp_length;i++){
       fputc(sp[i],outfile);
   }

  /* free the memory for the compressed buffer */
  free(sp);
  free(comp_length);
        
   fclose (outfile);
      
   JLIB_LEAVE
   
   /* return success */ 
   return 1;
}
