/* Copyright 1995-96 Jon Griffiths.  See the file "jlib.doc" for details. */

#include <stdio.h>
#include <stdlib.h>
#include <jlib.h>

/* color of all on screen text */
#define COLOR 15

/* color of bounding rectangles */
#define RECT_COL 15

/* key values of "y" and "Y" */
#define BIG_Y 'Y'
#define SMALL_Y 'y'


/* menu codes */
#define ERROR 1
#define QUIT 2
#define COPY 3
#define SAVE 4
#define RECT 5
#define CLRRECT 6
#define ADD_RECT 7
#define NOT_EVENT 8
#define IN_PLUS 9
#define IN_MINUS 10
#define OUT_PLUS 11
#define OUT_MINUS 12

/* stuff for menu buttons */
 typedef struct{
	  int x;
	  int y;
	  int width;
	  int height;
	  char   *text;
 }menu_button;

#define COL(x)   ((x)*CHAR_WIDTH('a'))
#define LINE(x)  ((x)*CHAR_HEIGHT('a'))
#define HALF_W   (CHAR_WIDTH('a')/2)
#define HALF_H   (CHAR_HEIGHT('a')/2)
#define DEF_W    ((COL(10))+4)
#define AT_W(x)  ((x*DEF_W)+ x*2)

/* x / y pos of in and out sprites on screen */
#define INX COL(20)
#define INY LINE(10)
#define OUTX COL(50)
#define OUTY LINE(10)

 /* global menu buttons: bad style,  good sense! */
 menu_button B_quit =   { AT_W(0),LINE(2)+HALF_H,DEF_W,LINE(1)+HALF_H,"Quit" };
 menu_button B_save =   { AT_W(1),LINE(2)+HALF_H,DEF_W,LINE(1)+HALF_H,"Save" };
 menu_button B_copy =   { AT_W(2),LINE(2)+HALF_H,DEF_W,LINE(1)+HALF_H,"Copy" };
 menu_button B_rect =   { AT_W(3),LINE(2)+HALF_H,DEF_W,LINE(1)+HALF_H,"ShowRect" };
 menu_button B_clrrect= { AT_W(4),LINE(2)+HALF_H,DEF_W,LINE(1)+HALF_H,"ClrRect" };
 menu_button B_addrect= { AT_W(5),LINE(2)+HALF_H,DEF_W,LINE(1)+HALF_H,"AddRect" };
 
 menu_button B_inplus = { INX,    INY+SPR_MAX_Y+5, COL(1)+2, LINE(1)+2,"+" };
 menu_button B_inminus= { INX+40, INY+SPR_MAX_Y+5, COL(1)+2, LINE(1)+2,"-" };
 menu_button B_outplus= { OUTX,   OUTY+SPR_MAX_Y+5,COL(1)+2, LINE(1)+2,"+" };
 menu_button B_outminus={ OUTX+40,OUTY+SPR_MAX_Y+5,COL(1)+2, LINE(1)+2,"-" };

 /* global offscreen buffer */
 buffer_rec *offscreen;
  
/* protos */
void draw_button(menu_button *m);
int is_in_button(int x,int y, menu_button *m);
void blank_space(int x1,int y1,int x2,int y2);
int get_event(void);
void do_quit(void);
void do_copy(sprite_system *in,sprite_system *out,int cin,int cout);
void cleanup(void);
void write_out_file(char *file,sprite_system *outsys);
void show_rects(sprite_system *in,sprite_system *out,int cin,int cout);
void do_rect_add(sprite_system *in,int cin);


/****************************************************************************/
int main(int argc, char *argv[])
{
  int code;
  char tmpstr[100];
  sprite_system *ssys, *outsys;
  int current_in = 0, current_out = 0;
  int changes =1; 


#if (MOUSE_AVAILABLE == 0)
     printf("ERROR: No mouse installed.\n");
     return -1;
  }
#else

  /* initialisation stuff... */
  if(argc!=3){
	      printf("spr_edit - edit sprite files.\n");
	      printf("Usage: spr_edit <infile> <outfile> \n");
	      printf("Where: <infile>  = file to get sprites from.\n");
	      printf("Where: <outfile> = file to put sprites into.\n");
	      exit(-1);
  }

  ssys=sprite_init(16,256);
  outsys=sprite_init(16,256);

  if(sprite_load(argv[1],ssys)!=SUCCESS){
     printf("ERROR: Could not load sprite file %s.\n",argv[1]);
     exit(-1);
  }

  offscreen=buff_init(SCREEN_WIDTH,SCREEN_HEIGHT);
   
  screen_set_video_mode();

  if(!mouse_present()){
     screen_restore_video_mode();
     printf("ERROR: No mouse installed.\n");
     exit(-1);
  }

  buff_draw_stringNC(offscreen,"Sprite File Editor v2.0",COL(30),0,COLOR);

  draw_button(&B_quit);
  draw_button(&B_save);
  draw_button(&B_copy);
  draw_button(&B_rect);
  draw_button(&B_clrrect);
  draw_button(&B_addrect);
  draw_button(&B_inplus);
  draw_button(&B_inminus);
  draw_button(&B_outplus);
  draw_button(&B_outminus);


  /* draw boxes to surround the sprites */
  buff_draw_boxNC(offscreen,INX,INY,INX+SPR_MAX_X+2,INY+SPR_MAX_Y+2,COLOR);
  buff_draw_boxNC(offscreen,OUTX,OUTY,OUTX+SPR_MAX_X+2,OUTY+SPR_MAX_Y+2,COLOR);
  
  /* put what we have so far on screen */
  screen_blit_fs_buffer(offscreen);

  mouse_show_pointer();

  do{
    if(changes==1){
       changes = 0;
       
       /* blank the status areas above each sprite box */
       blank_space(INX,INY-LINE(1),INX+COL(12),INY-1);
       blank_space(OUTX,OUTY-LINE(1),OUTX+COL(12),OUTY-1);

       /* update the values of each sprite */
       sprintf(tmpstr,"Frame:%3d\n",current_in);
       buff_draw_stringNC(offscreen,tmpstr,INX,INY-LINE(1),COLOR);
       sprintf(tmpstr,"Frame:%3d\n",current_out);
       buff_draw_stringNC(offscreen,tmpstr,OUTX,OUTY-LINE(1),COLOR);

       /* draw each sprite */
       if(ssys->sprite_data[current_in] !=NULL){
          blank_space(INX+1,INY+1,INX+SPR_MAX_X,INY+SPR_MAX_Y);
          buff_stamp_spriteNC(ssys,current_in,offscreen,INX+1,INY+1);
       }
       else{
          blank_space(INX+1,INY+1,INX+SPR_MAX_X,INY+SPR_MAX_Y);
          buff_draw_stringNC(offscreen,"EOF",INX+COL(1),INY+LINE(1),COLOR);
       }

       if(outsys->sprite_data[current_out] !=NULL){
          blank_space(OUTX+1,OUTY+1,OUTX+SPR_MAX_X,OUTY+SPR_MAX_Y);
          buff_stamp_spriteNC(outsys,current_out,offscreen,OUTX+1,OUTY+1);
       }
       else{
          blank_space(OUTX+1,OUTY+1,OUTX+SPR_MAX_X,OUTY+SPR_MAX_Y);
          buff_draw_stringNC(offscreen,"Free",OUTX+10,OUTY+10,COLOR);
       }

       mouse_hide_pointer();

       /* update the screen */
       screen_blit_fs_buffer(offscreen);

       mouse_show_pointer();
    }  
    
    code = get_event();

    switch(code){
	 case QUIT:
		   do_quit();
		   break;

	 case SAVE:
		   write_out_file(argv[2],outsys);
		   mouse_hide_pointer();
		   buff_draw_stringNC(offscreen,"File saved. Click to continue",COL(30),LINE(20),COLOR);
		   screen_blit_fs_buffer(offscreen);
		   mouse_show_pointer();
		   
		   get_event();

		   mouse_hide_pointer();		   
		   blank_space(COL(30),LINE(20),COL(30)+COL(30),LINE(21));
		   screen_blit_fs_buffer(offscreen);
		   mouse_show_pointer();
		   changes =1;
		   break;

	 case COPY:
		   do_copy(ssys,outsys,current_in,current_out);
		   changes=1;
		   break;

	 case RECT:
		   mouse_hide_pointer();
		   show_rects(ssys,outsys,current_in,current_out);
		   screen_blit_fs_buffer(offscreen);
		   mouse_show_pointer();
		   break;

	 case CLRRECT:
                   if(ssys->sprite_data[current_in] != NULL){        
                      free(ssys->sprite_data[current_in]->rect_coords);
                      ssys->sprite_data[current_in]->rect_coords=NULL;
                      ssys->sprite_data[current_in]->no_rects=0;
		      changes =1;		   
                   }
		   break;

	 case ADD_RECT:
		   if(ssys->sprite_data[current_in] != NULL){
  		      mouse_hide_pointer();
		      buff_draw_stringNC(offscreen,"Click on 2 corners",COL(30),LINE(20),COLOR);
		      screen_blit_fs_buffer(offscreen);              
		      mouse_show_pointer();
		   
                      do_rect_add(ssys,current_in);		   

                      blank_space(COL(30),LINE(20),COL(30)+COL(30),LINE(21));
                      screen_blit_fs_buffer(offscreen);
		      break;
		   }

	 case IN_PLUS:
		   if(ssys->sprite_data[current_in] != NULL){
		      ++current_in;
		   }
		   changes =1;
		   break;

	 case IN_MINUS:
		   if(current_in!=0){
		      --current_in;
		   }
		   changes =1;
		   break;

	 case OUT_PLUS:
		   if(outsys->sprite_data[current_out] != NULL){
		      ++current_out;
		   }
		   changes =1;
		   break;

	 case OUT_MINUS:
		   if(current_out!=0){
		      --current_out;
		   }
		   changes =1;
		   break;


	 case NOT_EVENT:
		   break;

	 default:
		   break;
    }

  }while(1);

  /* finish up */
  cleanup();
  return 1;
}

/****************************************************************************/
void draw_button(menu_button *m)
{
 buff_draw_boxNC(offscreen,m->x,m->y,m->x+m->width,m->y+m->height,COLOR);
 buff_draw_stringNC(offscreen,m->text,m->x+2,m->y+2,COLOR);
}

/****************************************************************************/
int is_in_button(int x,int y, menu_button *m)
{
 if((x>m->x)&&(y > m->y)&&(x<(m->x+m->width))&&(y<(m->y+m->height))){
    return 1;
 }
 else{
    return 0;
 }
}

/****************************************************************************/
void blank_space(int x1,int y1,int x2,int y2)
{
 buff_draw_rectNC(offscreen,x1,y1,x2,y2,0);
}

/****************************************************************************/
/* return an event code corresponding to an event */
int get_event(void)
{
 int xpos,ypos,buttons;

 /* this loop waits until the left button is pressed */
 do{
   mouse_get_status(&xpos,&ypos,&buttons);
 }while(!(BUTTON_DOWN(buttons)));

 /* this loop waits until the left button is released */
 do{
   mouse_get_status(&xpos,&ypos,&buttons);
 }while(BUTTON_DOWN(buttons));

 
 /* check the buttons */
 if(is_in_button(xpos,ypos,&B_quit)){
    return QUIT;
 }
 if(is_in_button(xpos,ypos,&B_save)){
    return SAVE;
 }
 if(is_in_button(xpos,ypos,&B_copy)){
    return COPY;
 }
 if(is_in_button(xpos,ypos,&B_rect)){
    return RECT;
 }
 if(is_in_button(xpos,ypos,&B_clrrect)){
    return CLRRECT;
 }
 if(is_in_button(xpos,ypos,&B_addrect)){
    return ADD_RECT;
 }
 if(is_in_button(xpos,ypos,&B_inplus)){
    return IN_PLUS;
 }
 if(is_in_button(xpos,ypos,&B_inminus)){
    return IN_MINUS;
 }
 if(is_in_button(xpos,ypos,&B_outplus)){
    return OUT_PLUS;
 }
 if(is_in_button(xpos,ypos,&B_outminus)){
    return OUT_MINUS;
 }

 return NOT_EVENT;
}

/****************************************************************************/
/* handle the quit button */
void do_quit(void)
{
 char c;

 mouse_hide_pointer();

 buff_draw_stringNC(offscreen,"Quit : Are you sure (Y/N) ?",COL(30),LINE(20),COLOR);
 screen_blit_fs_buffer(offscreen);

 fflush(stdin); 
 c=fgetc(stdin);

 if((c == BIG_Y) || (c == SMALL_Y)){
    cleanup();
 }
 else{
    /* we just clear the message and let program flow continue through the
    ** switch statement in the caller.
    */
    blank_space(COL(30),LINE(20),COL(30)+COL(30),LINE(21));
    screen_blit_fs_buffer(offscreen);
 
    mouse_show_pointer();
 }
}

/****************************************************************************/
void cleanup(void)
{
  mouse_hide_pointer();
  screen_restore_video_mode();
  exit(0);
}

/****************************************************************************/
void write_out_file(char *file,sprite_system *outsys)
{
 unsigned short i=0;
 int width,height;
 FILE *fp;
 int j;
 char *rc;

  if((fp=fopen(file,"wb"))==NULL){
     cleanup();
  }

 /* write out dummy value for num sprites in file */
 fwrite(&i,sizeof(unsigned short),1,fp);

 /* write out each sprite */
 while(outsys->sprite_data[i]!=NULL){
       width = outsys->sprite_data[i]->width;
       height = outsys->sprite_data[i]->height;

       fputc(width,fp);
       fputc(height,fp);

       fwrite(outsys->sprite_data[i]->data,width*height,1,fp);

      /* write out # of bounding rectangles */
      fputc(outsys->sprite_data[i]->no_rects,fp); 

      /* write out coords if any */
      if(outsys->sprite_data[i]->no_rects!=0){
         rc=outsys->sprite_data[i]->rect_coords;
      
         for(j=0;j<outsys->sprite_data[i]->no_rects;j++){
            fputc(*rc,fp);
            rc++;
            fputc(*rc,fp);
            rc++;      
            fputc(*rc,fp);
            rc++;
            fputc(*rc,fp);
            rc++;
         }
      }

       i++;            /* add 1 sprite to count */
 }

 /* write correct number of sprites */
 fseek(fp,0,SEEK_SET);
 fwrite(&i,sizeof(unsigned short),1,fp);

 fclose(fp);
}

/****************************************************************************/
void do_copy(sprite_system *in,sprite_system *out,int cin,int cout)
{
  out->sprite_data[cout]=in->sprite_data[cin];

}


/****************************************************************************/
void show_rects(sprite_system *in,sprite_system *out,int cin,int cout)
{
 UBYTE *rc;
 int i;
 UBYTE a,b,c,d;

 /* for each sprite,  if there is one,  draw it's rectangle(s). */


 if(in->sprite_data[cin] != NULL){
    rc = in->sprite_data[cin]->rect_coords;

    for(i=in->sprite_data[cin]->no_rects;i!=0;i--){
	a = *rc;
	rc++;
	b = *rc;
	rc++;
	c = *rc;
	rc++;
	d = *rc;
	rc++;

	buff_draw_boxNC(offscreen,INX+1+a,INY+1+b,INX+1+c,INY+1+d,RECT_COL);
    }
 }


 if(out->sprite_data[cout] != NULL){
    rc = out->sprite_data[cout]->rect_coords;

    for(i=out->sprite_data[cout]->no_rects;i!=0;i--){
	a = *rc;
	rc++;
	b = *rc;
	rc++;
	c = *rc;
	rc++;
	d = *rc;
	rc++;
	
	buff_draw_boxNC(offscreen,OUTX+1+a,OUTY+1+b,OUTX+1+c,OUTY+1+d,RECT_COL);
    }
 }


}


/****************************************************************************/
void do_rect_add(sprite_system *in,int cin)
{
 int x1pos,y1pos,buttons1;
 int x2pos,y2pos,buttons2;
 int width = SPR_X_SIZE(in->sprite_data[cin]);
 int height = SPR_Y_SIZE(in->sprite_data[cin]);
 unsigned char *rc;
 unsigned char *temp;
 int i;
 
 /* get a point from the user within the sprite */ 
 do{
   do{
     mouse_get_status(&x1pos,&y1pos,&buttons1);
   }while(!(BUTTON_DOWN(buttons1)));

   do{
     mouse_get_status(&x1pos,&y1pos,&buttons1);
   }while(BUTTON_DOWN(buttons1));

 }while((x1pos<INX)||(x1pos>INX+width)||(y1pos<INY)||(y1pos>INY+height));

 mouse_hide_pointer();
 buff_draw_point(offscreen,x1pos,y1pos,COLOR);
 screen_blit_fs_buffer(offscreen); 
 mouse_show_pointer();
 
 /* get another point from the user within the sprite */  
 do{
   do{
     mouse_get_status(&x2pos,&y2pos,&buttons2);
   }while(!(BUTTON_DOWN(buttons2)));

   do{
     mouse_get_status(&x2pos,&y2pos,&buttons2);
   }while(BUTTON_DOWN(buttons2));

 }while((x2pos<INX)||(x2pos>INX+width)||(y2pos<INY)||(y2pos>INY+height));

 mouse_hide_pointer();
 buff_draw_point(offscreen,x2pos,y2pos,COLOR);
 /* draw the box the user has selected */
 buff_draw_box(offscreen,x1pos,y1pos,x2pos,y2pos,RECT_COL);

 screen_blit_fs_buffer(offscreen); 
 mouse_show_pointer();
 
 /* add the rectangle to the sprite */
 in->sprite_data[cin]->no_rects++; /* increment count */
 
 /* increase memory to hold new coords */ 
 temp=in->sprite_data[cin]->rect_coords;
 
 in->sprite_data[cin]->rect_coords=malloc((in->sprite_data[cin]->no_rects * 4));

 for(i=0;i<((in->sprite_data[cin]->no_rects-1)*4);i++){
     in->sprite_data[cin]->rect_coords[i]= temp[i];
 }    
 
 if(temp != NULL){
    free(temp);
 }
      
 /* get rc to point to the last 4 chars of the rect coords */
 rc=in->sprite_data[cin]->rect_coords;
  
 rc+= ((in->sprite_data[cin]->no_rects-1) * 4);
  
 /* write out the rectangles */
 *rc= x1pos-INX-1; /* -1 because the sprite is drawn at INX+1 */
 rc++;
 *rc= y1pos-INY-1;
 rc++;
 *rc= x2pos-INX-1;
 rc++;
 *rc= y2pos-INY-1;
 
}

#endif /* mouse available */
